(function (exports) {
  'use strict';

  /**
   * @typedef {import("../types.mjs").ColorSource} ColorSource
   */

  /**
   * A representation of a color in hexadecimal format.
   * This class provides methods for transformations and manipulations of colors.
   */
  let Color$1 = class Color extends Number {

    /**
     * Is this a valid color?
     * @type {boolean}
     */
    get valid() {
      const v = this.valueOf();
      return Number.isInteger(v) && v >= 0 && v <= 0xFFFFFF;
    }

    /* ------------------------------------------ */

    /**
     * A CSS-compatible color string.
     * If this color is not valid, the empty string is returned.
     * An alias for Color#toString.
     * @type {string}
     */
    get css() {
      return this.toString(16);
    }

    /* ------------------------------------------ */

    /**
     * The color represented as an RGB array.
     * @type {[number, number, number]}
     */
    get rgb() {
      return [((this >> 16) & 0xFF) / 255, ((this >> 8) & 0xFF) / 255, (this & 0xFF) / 255];
    }

    /* ------------------------------------------ */

    /**
     * The numeric value of the red channel between [0, 1].
     * @type {number}
     */
    get r() {
      return ((this >> 16) & 0xFF) / 255;
    }

    /* ------------------------------------------ */

    /**
     * The numeric value of the green channel between [0, 1].
     * @type {number}
     */
    get g() {
      return ((this >> 8) & 0xFF) / 255;
    }

    /* ------------------------------------------ */

    /**
     * The numeric value of the blue channel between [0, 1].
     * @type {number}
     */
    get b() {
      return (this & 0xFF) / 255;
    }

    /* ------------------------------------------ */

    /**
     * The maximum value of all channels.
     * @type {number}
     */
    get maximum() {
      return Math.max(...this);
    }

    /* ------------------------------------------ */

    /**
     * The minimum value of all channels.
     * @type {number}
     */
    get minimum() {
      return Math.min(...this);
    }

    /* ------------------------------------------ */

    /**
     * Get the value of this color in little endian format.
     * @type {number}
     */
    get littleEndian() {
      return ((this >> 16) & 0xFF) + (this & 0x00FF00) + ((this & 0xFF) << 16);
    }

    /* ------------------------------------------ */

    /**
     * The color represented as an HSV array.
     * Conversion formula adapted from http://en.wikipedia.org/wiki/HSV_color_space.
     * Assumes r, g, and b are contained in the set [0, 1] and returns h, s, and v in the set [0, 1].
     * @type {[number, number, number]}
     */
    get hsv() {
      const [r, g, b] = this.rgb;
      const max = Math.max(r, g, b);
      const min = Math.min(r, g, b);
      const d = max - min;

      let h;
      const s = max === 0 ? 0 : d / max;
      const v = max;

      // Achromatic colors
      if (max === min) return [0, s, v];

      // Normal colors
      switch (max) {
        case r: h = (g - b) / d + (g < b ? 6 : 0); break;
        case g: h = (b - r) / d + 2; break;
        case b: h = (r - g) / d + 4; break;
      }
      h /= 6;
      return [h, s, v];
    }

    /* ------------------------------------------ */

    /**
     * The color represented as an HSL array.
     * Assumes r, g, and b are contained in the set [0, 1] and returns h, s, and l in the set [0, 1].
     * @type {[number, number, number]}
     */
    get hsl() {
      const [r, g, b] = this.rgb;

      // Compute luminosity, saturation and hue
      const l = Math.max(r, g, b);
      const s = l - Math.min(r, g, b);
      let h = 0;
      if ( s > 0 ) {
        if ( l === r ) {
          h = (g - b) / s;
        } else if ( l === g ) {
          h = 2 + (b - r) / s;
        } else {
          h = 4 + (r - g) / s;
        }
      }
      const finalHue = (60 * h < 0 ? 60 * h + 360 : 60 * h) / 360;
      const finalSaturation = s ? (l <= 0.5 ? s / (2 * l - s) : s / (2 - (2 * l - s))) : 0;
      const finalLuminance = (2 * l - s) / 2;
      return [finalHue, finalSaturation, finalLuminance];
    }

    /* ------------------------------------------ */

    /**
     * The color represented as a linear RGB array.
     * Assumes r, g, and b are contained in the set [0, 1] and returns linear r, g, and b in the set [0, 1].
     * @link https://en.wikipedia.org/wiki/SRGB#Transformation
     * @type {Color}
     */
    get linear() {
      const toLinear = c => (c > 0.04045) ? Math.pow((c + 0.055) / 1.055, 2.4) : (c / 12.92);
      return this.constructor.fromRGB([toLinear(this.r), toLinear(this.g), toLinear(this.b)]);
    }

    /* ------------------------------------------ */
    /*  Color Manipulation Methods                */
    /* ------------------------------------------ */

    /** @override */
    toString(radix) {
      if ( !this.valid ) return "";
      return `#${super.toString(16).padStart(6, "0")}`;
    }

    /* ------------------------------------------ */

    /**
     * Serialize the Color.
     * @returns {string}    The color as a CSS string
     */
    toJSON() {
      return this.css;
    }

    /* ------------------------------------------ */

    /**
     * Returns the color as a CSS string.
     * @returns {string}    The color as a CSS string
     */
    toHTML() {
      return this.css;
    }

    /* ------------------------------------------ */

    /**
     * Test whether this color equals some other color
     * @param {Color|number} other  Some other color or hex number
     * @returns {boolean}           Are the colors equal?
     */
    equals(other) {
      return this.valueOf() === other.valueOf();
    }

    /* ------------------------------------------ */

    /**
     * Get a CSS-compatible RGBA color string.
     * @param {number} alpha      The desired alpha in the range [0, 1]
     * @returns {string}          A CSS-compatible RGBA string
     */
    toRGBA(alpha) {
      const rgba = [(this >> 16) & 0xFF, (this >> 8) & 0xFF, this & 0xFF, alpha];
      return `rgba(${rgba.join(", ")})`;
    }

    /* ------------------------------------------ */

    /**
     * Mix this Color with some other Color using a provided interpolation weight.
     * @param {Color} other       Some other Color to mix with
     * @param {number} weight     The mixing weight placed on this color where weight is placed on the other color
     * @returns {Color}           The resulting mixed Color
     */
    mix(other, weight) {
      return new Color(Color.mix(this, other, weight));
    }

    /* ------------------------------------------ */

    /**
     * Multiply this Color by another Color or a static scalar.
     * @param {Color|number} other  Some other Color or a static scalar.
     * @returns {Color}             The resulting Color.
     */
    multiply(other) {
      if ( other instanceof Color ) return new Color(Color.multiply(this, other));
      return new Color(Color.multiplyScalar(this, other));
    }

    /* ------------------------------------------ */

    /**
     * Add this Color by another Color or a static scalar.
     * @param {Color|number} other  Some other Color or a static scalar.
     * @returns {Color}             The resulting Color.
     */
    add(other) {
      if ( other instanceof Color ) return new Color(Color.add(this, other));
      return new Color(Color.addScalar(this, other));
    }

    /* ------------------------------------------ */

    /**
     * Subtract this Color by another Color or a static scalar.
     * @param {Color|number} other  Some other Color or a static scalar.
     * @returns {Color}             The resulting Color.
     */
    subtract(other) {
      if ( other instanceof Color ) return new Color(Color.subtract(this, other));
      return new Color(Color.subtractScalar(this, other));
    }

    /* ------------------------------------------ */

    /**
     * Max this color by another Color or a static scalar.
     * @param {Color|number} other  Some other Color or a static scalar.
     * @returns {Color}             The resulting Color.
     */
    maximize(other) {
      if ( other instanceof Color ) return new Color(Color.maximize(this, other));
      return new Color(Color.maximizeScalar(this, other));
    }

    /* ------------------------------------------ */

    /**
     * Min this color by another Color or a static scalar.
     * @param {Color|number} other  Some other Color or a static scalar.
     * @returns {Color}             The resulting Color.
     */
    minimize(other) {
      if ( other instanceof Color ) return new Color(Color.minimize(this, other));
      return new Color(Color.minimizeScalar(this, other));
    }

    /* ------------------------------------------ */
    /*  Iterator                                  */
    /* ------------------------------------------ */

    /**
     * Iterating over a Color is equivalent to iterating over its [r,g,b] color channels.
     * @returns {Generator<number>}
     */
    *[Symbol.iterator]() {
      yield this.r;
      yield this.g;
      yield this.b;
    }

    /* ------------------------------------------------------------------------------------------- */
    /*                      Real-time performance Methods and Properties                           */
    /*  Important Note:                                                                            */
    /*  These methods are not a replacement, but a tool when real-time performance is needed.      */
    /*  They do not have the flexibility of the "classic" methods and come with some limitations.  */
    /*  Unless you have to deal with real-time performance, you should use the "classic" methods.  */
    /* ------------------------------------------------------------------------------------------- */

    /**
     * Set an rgb array with the rgb values contained in this Color class.
     * @param {number[]} vec3  Receive the result. Must be an array with at least a length of 3.
     */
    applyRGB(vec3) {
      vec3[0] = ((this >> 16) & 0xFF) / 255;
      vec3[1] = ((this >> 8) & 0xFF) / 255;
      vec3[2] = (this & 0xFF) / 255;
    }

    /* ------------------------------------------ */

    /**
     * Apply a linear interpolation between two colors, according to the weight.
     * @param {number}        color1       The first color to mix.
     * @param {number}        color2       The second color to mix.
     * @param {number}        weight       Weight of the linear interpolation.
     * @returns {number}                   The resulting mixed color
     */
    static mix(color1, color2, weight) {
      return (((((color1 >> 16) & 0xFF) * (1 - weight) + ((color2 >> 16) & 0xFF) * weight) << 16) & 0xFF0000)
        | (((((color1 >> 8) & 0xFF) * (1 - weight) + ((color2 >> 8) & 0xFF) * weight) << 8) & 0x00FF00)
        | (((color1 & 0xFF) * (1 - weight) + (color2 & 0xFF) * weight) & 0x0000FF);
    }

    /* ------------------------------------------ */

    /**
     * Multiply two colors.
     * @param {number}        color1       The first color to multiply.
     * @param {number}        color2       The second color to multiply.
     * @returns {number}                   The result.
     */
    static multiply(color1, color2) {
      return ((((color1 >> 16) & 0xFF) / 255 * ((color2 >> 16) & 0xFF) / 255) * 255 << 16)
        | ((((color1 >> 8) & 0xFF) / 255 * ((color2 >> 8) & 0xFF) / 255) * 255 << 8)
        | (((color1 & 0xFF) / 255 * ((color2 & 0xFF) / 255)) * 255);
    }

    /* ------------------------------------------ */

    /**
     * Multiply a color by a scalar
     * @param {number} color        The color to multiply.
     * @param {number} scalar       A static scalar to multiply with.
     * @returns {number}            The resulting color as a number.
     */
    static multiplyScalar(color, scalar) {
      return (Math.clamp(((color >> 16) & 0xFF) / 255 * scalar, 0, 1) * 255 << 16)
        | (Math.clamp(((color >> 8) & 0xFF) / 255 * scalar, 0, 1) * 255 << 8)
        | (Math.clamp((color & 0xFF) / 255 * scalar, 0, 1) * 255);
    }

    /* ------------------------------------------ */

    /**
     * Maximize two colors.
     * @param {number}        color1       The first color.
     * @param {number}        color2       The second color.
     * @returns {number}                   The result.
     */
    static maximize(color1, color2) {
      return (Math.clamp(Math.max((color1 >> 16) & 0xFF, (color2 >> 16) & 0xFF), 0, 0xFF) << 16)
        | (Math.clamp(Math.max((color1 >> 8) & 0xFF, (color2 >> 8) & 0xFF), 0, 0xFF) << 8)
        | Math.clamp(Math.max(color1 & 0xFF, color2 & 0xFF), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Maximize a color by a static scalar.
     * @param {number} color         The color to maximize.
     * @param {number} scalar        Scalar to maximize with (normalized).
     * @returns {number}             The resulting color as a number.
     */
    static maximizeScalar(color, scalar) {
      return (Math.clamp(Math.max((color >> 16) & 0xFF, scalar * 255), 0, 0xFF) << 16)
        | (Math.clamp(Math.max((color >> 8) & 0xFF, scalar * 255), 0, 0xFF) << 8)
        | Math.clamp(Math.max(color & 0xFF, scalar * 255), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Add two colors.
     * @param {number}        color1       The first color.
     * @param {number}        color2       The second color.
     * @returns {number}                   The resulting color as a number.
     */
    static add(color1, color2) {
      return (Math.clamp((((color1 >> 16) & 0xFF) + ((color2 >> 16) & 0xFF)), 0, 0xFF) << 16)
        | (Math.clamp((((color1 >> 8) & 0xFF) + ((color2 >> 8) & 0xFF)), 0, 0xFF) << 8)
        | Math.clamp(((color1 & 0xFF) + (color2 & 0xFF)), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Add a static scalar to a color.
     * @param {number} color         The color.
     * @param {number} scalar        Scalar to add with (normalized).
     * @returns {number}             The resulting color as a number.
     */
    static addScalar(color, scalar) {
      return (Math.clamp((((color >> 16) & 0xFF) + scalar * 255), 0, 0xFF) << 16)
        | (Math.clamp((((color >> 8) & 0xFF) + scalar * 255), 0, 0xFF) << 8)
        | Math.clamp(((color & 0xFF) + scalar * 255), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Subtract two colors.
     * @param {number}        color1       The first color.
     * @param {number}        color2       The second color.
     */
    static subtract(color1, color2) {
      return (Math.clamp((((color1 >> 16) & 0xFF) - ((color2 >> 16) & 0xFF)), 0, 0xFF) << 16)
        | (Math.clamp((((color1 >> 8) & 0xFF) - ((color2 >> 8) & 0xFF)), 0, 0xFF) << 8)
        | Math.clamp(((color1 & 0xFF) - (color2 & 0xFF)), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Subtract a color by a static scalar.
     * @param {number} color         The color.
     * @param {number} scalar        Scalar to subtract with (normalized).
     * @returns {number}             The resulting color as a number.
     */
    static subtractScalar(color, scalar) {
      return (Math.clamp((((color >> 16) & 0xFF) - scalar * 255), 0, 0xFF) << 16)
        | (Math.clamp((((color >> 8) & 0xFF) - scalar * 255), 0, 0xFF) << 8)
        | Math.clamp(((color & 0xFF) - scalar * 255), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Minimize two colors.
     * @param {number}        color1       The first color.
     * @param {number}        color2       The second color.
     */
    static minimize(color1, color2) {
      return (Math.clamp(Math.min((color1 >> 16) & 0xFF, (color2 >> 16) & 0xFF), 0, 0xFF) << 16)
        | (Math.clamp(Math.min((color1 >> 8) & 0xFF, (color2 >> 8) & 0xFF), 0, 0xFF) << 8)
        | Math.clamp(Math.min(color1 & 0xFF, color2 & 0xFF), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Minimize a color by a static scalar.
     * @param {number} color         The color.
     * @param {number} scalar        Scalar to minimize with (normalized).
     */
    static minimizeScalar(color, scalar) {
      return (Math.clamp(Math.min((color >> 16) & 0xFF, scalar * 255), 0, 0xFF) << 16)
        | (Math.clamp(Math.min((color >> 8) & 0xFF, scalar * 255), 0, 0xFF) << 8)
        | Math.clamp(Math.min(color & 0xFF, scalar * 255), 0, 0xFF);
    }

    /* ------------------------------------------ */

    /**
     * Convert a color to RGB and assign values to a passed array.
     * @param {number} color   The color to convert to RGB values.
     * @param {number[]} vec3  Receive the result. Must be an array with at least a length of 3.
     */
    static applyRGB(color, vec3) {
      vec3[0] = ((color >> 16) & 0xFF) / 255;
      vec3[1] = ((color >> 8) & 0xFF) / 255;
      vec3[2] = (color & 0xFF) / 255;
    }

    /* ------------------------------------------ */
    /*  Factory Methods                           */
    /* ------------------------------------------ */

    /**
     * Create a Color instance from an RGB array.
     * @param {ColorSource} color     A color input
     * @returns {Color}               The hex color instance or NaN
     */
    static from(color) {
      if ( (color === null) || (color === undefined) ) return new this(NaN);
      if ( typeof color === "string" ) return this.fromString(color);
      if ( typeof color === "number" ) return new this(color);
      if ( (color instanceof Array) && (color.length === 3) ) return this.fromRGB(color);
      if ( color instanceof Color ) return color;
      return new this(color);
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance from a color string which either includes or does not include a leading #.
     * @param {string} color                      A color string
     * @returns {Color}                           The hex color instance
     */
    static fromString(color) {
      return new this(parseInt(color.startsWith("#") ? color.substring(1) : color, 16));
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance from an RGB array.
     * @param {[number, number, number]} rgb      An RGB tuple
     * @returns {Color}                           The hex color instance
     */
    static fromRGB(rgb) {
      return new this(((rgb[0] * 255) << 16) + ((rgb[1] * 255) << 8) + (rgb[2] * 255 | 0));
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance from an RGB normalized values.
     * @param {number} r                          The red value
     * @param {number} g                          The green value
     * @param {number} b                          The blue value
     * @returns {Color}                           The hex color instance
     */
    static fromRGBvalues(r, g, b) {
      return new this(((r * 255) << 16) + ((g * 255) << 8) + (b * 255 | 0));
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance from an HSV array.
     * Conversion formula adapted from http://en.wikipedia.org/wiki/HSV_color_space.
     * Assumes h, s, and v are contained in the set [0, 1].
     * @param {[number, number, number]} hsv      An HSV tuple
     * @returns {Color}                           The hex color instance
     */
    static fromHSV(hsv) {
      const [h, s, v] = hsv;
      const i = Math.floor(h * 6);
      const f = (h * 6) - i;
      const p = v * (1 - s);
      const q = v * (1 - f * s);
      const t = v * (1 - (1 - f) * s);
      let rgb;
      switch (i % 6) {
        case 0: rgb = [v, t, p]; break;
        case 1: rgb = [q, v, p]; break;
        case 2: rgb = [p, v, t]; break;
        case 3: rgb = [p, q, v]; break;
        case 4: rgb = [t, p, v]; break;
        case 5: rgb = [v, p, q]; break;
      }
      return this.fromRGB(rgb);
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance from an HSL array.
     * Assumes h, s, and l are contained in the set [0, 1].
     * @param {[number, number, number]} hsl      An HSL tuple
     * @returns {Color}                           The hex color instance
     */
    static fromHSL(hsl) {
      const [h, s, l] = hsl;

      // Calculate intermediate values for the RGB components
      const chroma = (1 - Math.abs(2 * l - 1)) * s;
      const hue = h * 6;
      const x = chroma * (1 - Math.abs(hue % 2 - 1));
      const m = l - chroma / 2;

      let r, g, b;
      switch (Math.floor(hue)) {
        case 0: [r, g, b] = [chroma, x, 0]; break;
        case 1: [r, g, b] = [x, chroma, 0]; break;
        case 2: [r, g, b] = [0, chroma, x]; break;
        case 3: [r, g, b] = [0, x, chroma]; break;
        case 4: [r, g, b] = [x, 0, chroma]; break;
        case 5:
        case 6:[r, g, b] = [chroma, 0, x]; break;
        default: [r, g, b] = [0, 0, 0]; break;
      }

      // Adjust for luminance
      r += m;
      g += m;
      b += m;
      return this.fromRGB([r, g, b]);
    }

    /* ------------------------------------------ */

    /**
     * Create a Color instance (sRGB) from a linear rgb array.
     * Assumes r, g, and b are contained in the set [0, 1].
     * @link https://en.wikipedia.org/wiki/SRGB#Transformation
     * @param {[number, number, number]} linear   The linear rgb array
     * @returns {Color}                           The hex color instance
     */
    static fromLinearRGB(linear) {
      const [r, g, b] = linear;
      const tosrgb = c => (c <= 0.0031308) ? (12.92 * c) : (1.055 * Math.pow(c, 1 / 2.4) - 0.055);
      return this.fromRGB([tosrgb(r), tosrgb(g), tosrgb(b)]);
    }
  };

  /** @module constants */

  /**
   * The shortened software name
   * @type {string}
   */
  const vtt$1 = "Foundry VTT";

  /**
   * The full software name
   * @type {string}
   */
  const VTT = "Foundry Virtual Tabletop";

  /**
   * The software website URL
   * @type {string}
   */
  const WEBSITE_URL = "https://foundryvtt.com";

  /**
   * The serverless API URL
   */
  const WEBSITE_API_URL = "https://api.foundryvtt.com";

  /**
   * An ASCII greeting displayed to the client
   * @type {string}
   */
  const ASCII = `_______________________________________________________________
 _____ ___  _   _ _   _ ____  ______   __ __     _______ _____ 
|  ___/ _ \\| | | | \\ | |  _ \\|  _ \\ \\ / / \\ \\   / |_   _|_   _|
| |_ | | | | | | |  \\| | | | | |_) \\ V /   \\ \\ / /  | |   | |  
|  _|| |_| | |_| | |\\  | |_| |  _ < | |     \\ V /   | |   | |  
|_|   \\___/ \\___/|_| \\_|____/|_| \\_\\|_|      \\_/    |_|   |_|  
===============================================================`;

  /**
   * Define the allowed ActiveEffect application modes.
   * @remarks
   * Other arbitrary mode numbers can be used by systems and modules to identify special behaviors and are ignored
   * @enum {number}
   */
  const ACTIVE_EFFECT_MODES = {
    /**
     * Used to denote that the handling of the effect is programmatically provided by a system or module.
     */
    CUSTOM: 0,

    /**
     * Multiplies a numeric base value by the numeric effect value
     * @example
     * 2 (base value) * 3 (effect value) = 6 (derived value)
     */
    MULTIPLY: 1,

    /**
     * Adds a numeric base value to a numeric effect value, or concatenates strings
     * @example
     * 2 (base value) + 3 (effect value) = 5 (derived value)
     * @example
     * "Hello" (base value) + " World" (effect value) = "Hello World"
     */
    ADD: 2,

    /**
     * Keeps the lower value of the base value and the effect value
     * @example
     * 2 (base value), 0 (effect value) = 0 (derived value)
     * @example
     * 2 (base value), 3 (effect value) = 2 (derived value)
     */
    DOWNGRADE: 3,

    /**
     * Keeps the greater value of the base value and the effect value
     * @example
     * 2 (base value), 4 (effect value) = 4 (derived value)
     * @example
     * 2 (base value), 1 (effect value) = 2 (derived value)
     */
    UPGRADE: 4,

    /**
     * Directly replaces the base value with the effect value
     * @example
     * 2 (base value), 4 (effect value) = 4 (derived value)
     */
    OVERRIDE: 5
  };

  /**
   * Define the string name used for the base document type when specific sub-types are not defined by the system
   * @type {string}
   */
  const BASE_DOCUMENT_TYPE = "base";

  /**
   * Define the methods by which a Card can be drawn from a Cards stack
   * @enum {number}
   */
  const CARD_DRAW_MODES = {
    /**
     * Draw the first card from the stack
     * Synonymous with {@link CARD_DRAW_MODES.TOP}
     */
    FIRST: 0,

    /**
     * Draw the top card from the stack
     * Synonymous with {@link CARD_DRAW_MODES.FIRST}
     */
    TOP: 0,

    /**
     * Draw the last card from the stack
     * Synonymous with {@link CARD_DRAW_MODES.BOTTOM}
     */
    LAST: 1,

    /**
     * Draw the bottom card from the stack
     * Synonymous with {@link CARD_DRAW_MODES.LAST}
     */
    BOTTOM: 1,

    /**
     * Draw a random card from the stack
     */
    RANDOM: 2
  };

  /**
   * An enumeration of canvas performance modes.
   * @enum {number}
   */
  const CANVAS_PERFORMANCE_MODES = {
    LOW: 0,
    MED: 1,
    HIGH: 2,
    MAX: 3
  };

  /**
   * Valid Chat Message styles which affect how the message is presented in the chat log.
   * @enum {number}
   */
  const CHAT_MESSAGE_STYLES = {
    /**
     * An uncategorized chat message
     */
    OTHER: 0,

    /**
     * The message is spoken out of character (OOC).
     * OOC messages will be outlined by the player's color to make them more easily recognizable.
     */
    OOC: 1,

    /**
     * The message is spoken by an associated character.
     */
    IC: 2,

    /**
     * The message is an emote performed by the selected character.
     * Entering "/emote waves his hand." while controlling a character named Simon will send the message, "Simon waves his hand."
     */
    EMOTE: 3,
  };


  /**
   * Define the set of languages which have built-in support in the core software
   * @type {string[]}
   */
  const CORE_SUPPORTED_LANGUAGES = ["en"];

  /**
   * Configure the severity of compatibility warnings.
   * @enum {number}
   */
  const COMPATIBILITY_MODES = {
    /**
     * Nothing will be logged
     */
    SILENT: 0,

    /**
     * A message will be logged at the "warn" level
     */
    WARNING: 1,

    /**
     * A message will be logged at the "error" level
     */
    ERROR: 2,

    /**
     * An Error will be thrown
     */
    FAILURE: 3
  };

  /**
   * The lighting illumination levels which are supported.
   * @enum {number}
   */
  const LIGHTING_LEVELS = {
    DARKNESS: -2,
    HALFDARK: -1,
    UNLIT: 0,
    DIM: 1,
    BRIGHT: 2,
    BRIGHTEST: 3
  };

  /**
   * The CSS themes which are currently supported for the V11 Setup menu.
   * @enum {{id: string, label: string}}
   */
  const CSS_THEMES = Object.freeze({
    foundry: "THEME.foundry",
    fantasy: "THEME.fantasy",
    scifi: "THEME.scifi"
  });

  /**
   * The default artwork used for Token images if none is provided
   * @type {string}
   */
  const DEFAULT_TOKEN = 'icons/svg/mystery-man.svg';

  /**
   * The primary Document types.
   * @type {string[]}
   */
  const PRIMARY_DOCUMENT_TYPES = [
    "Actor",
    "Adventure",
    "Cards",
    "ChatMessage",
    "Combat",
    "FogExploration",
    "Folder",
    "Item",
    "JournalEntry",
    "Macro",
    "Playlist",
    "RollTable",
    "Scene",
    "Setting",
    "User"
  ];

  /**
   * The embedded Document types.
   * @type {Readonly<string[]>}
   */
  const EMBEDDED_DOCUMENT_TYPES = [
    "ActiveEffect",
    "ActorDelta",
    "AmbientLight",
    "AmbientSound",
    "Card",
    "Combatant",
    "Drawing",
    "Item",
    "JournalEntryPage",
    "MeasuredTemplate",
    "Note",
    "PlaylistSound",
    "Region",
    "RegionBehavior",
    "TableResult",
    "Tile",
    "Token",
    "Wall"
  ];

  /**
   * A listing of all valid Document types, both primary and embedded.
   * @type {Readonly<string[]>}
   */
  const ALL_DOCUMENT_TYPES = Array.from(new Set([
    ...PRIMARY_DOCUMENT_TYPES,
    ...EMBEDDED_DOCUMENT_TYPES
  ])).sort();

  /**
   * The allowed primary Document types which may exist within a World.
   * @type {string[]}
   */
  const WORLD_DOCUMENT_TYPES = [
    "Actor",
    "Cards",
    "ChatMessage",
    "Combat",
    "FogExploration",
    "Folder",
    "Item",
    "JournalEntry",
    "Macro",
    "Playlist",
    "RollTable",
    "Scene",
    "Setting",
    "User"
  ];

  /**
   * The allowed primary Document types which may exist within a Compendium pack.
   * @type {string[]}
   */
  const COMPENDIUM_DOCUMENT_TYPES = [
    "Actor",
    "Adventure",
    "Cards",
    "Item",
    "JournalEntry",
    "Macro",
    "Playlist",
    "RollTable",
    "Scene"
  ];

  /**
   * Define the allowed ownership levels for a Document.
   * Each level is assigned a value in ascending order.
   * Higher levels grant more permissions.
   * @enum {number}
   * @see https://foundryvtt.com/article/users/
   */
  const DOCUMENT_OWNERSHIP_LEVELS = {
    /**
     * The User inherits permissions from the parent Folder.
     */
    INHERIT: -1,

    /**
     * Restricts the associated Document so that it may not be seen by this User.
     */
    NONE: 0,

    /**
     * Allows the User to interact with the Document in basic ways, allowing them to see it in sidebars and see only limited aspects of its contents. The limits of this interaction are defined by the game system being used.
     */
    LIMITED: 1,

    /**
     * Allows the User to view this Document as if they were owner, but prevents them from making any changes to it.
     */
    OBSERVER: 2,

    /**
     * Allows the User to view and make changes to the Document as its owner. Owned documents cannot be deleted by anyone other than a gamemaster level User.
     */
    OWNER: 3
  };
  Object.freeze(DOCUMENT_OWNERSHIP_LEVELS);

  /**
   * Meta ownership levels that are used in the UI but never stored.
   * @enum {number}
   */
  const DOCUMENT_META_OWNERSHIP_LEVELS = {
    DEFAULT: -20,
    NOCHANGE: -10
  };
  Object.freeze(DOCUMENT_META_OWNERSHIP_LEVELS);

  /**
   * Define the allowed Document types which may be dynamically linked in chat
   * @type {string[]}
   */
  const DOCUMENT_LINK_TYPES = ["Actor", "Cards", "Item", "Scene", "JournalEntry", "Macro", "RollTable", "PlaylistSound"];

  /**
   * The supported dice roll visibility modes
   * @enum {string}
   * @see https://foundryvtt.com/article/dice/
   */
  const DICE_ROLL_MODES = {
    /**
     * This roll is visible to all players.
     */
    PUBLIC: "publicroll",

    /**
     * Rolls of this type are only visible to the player that rolled and any Game Master users.
     */
    PRIVATE: "gmroll",

    /**
     * A private dice roll only visible to Game Master users. The rolling player will not see the result of their own roll.
     */
    BLIND: "blindroll",

    /**
     * A private dice roll which is only visible to the user who rolled it.
     */
    SELF: "selfroll"
  };

  /**
   * The allowed fill types which a Drawing object may display
   * @enum {number}
   * @see https://foundryvtt.com/article/drawings/
   */
  const DRAWING_FILL_TYPES = {
    /**
     * The drawing is not filled
     */
    NONE: 0,

    /**
     * The drawing is filled with a solid color
     */
    SOLID: 1,

    /**
     * The drawing is filled with a tiled image pattern
     */
    PATTERN: 2
  };

  /**
   * Define the allowed Document types which Folders may contain
   * @type {string[]}
   */
  const FOLDER_DOCUMENT_TYPES = ["Actor", "Adventure", "Item", "Scene", "JournalEntry", "Playlist", "RollTable", "Cards", "Macro", "Compendium"];

  /**
   * The maximum allowed level of depth for Folder nesting
   * @type {number}
   */
  const FOLDER_MAX_DEPTH = 4;

  /**
   * A list of allowed game URL names
   * @type {string[]}
   */
  const GAME_VIEWS = ["game", "stream"];

  /**
   * The directions of movement.
   * @enum {number}
   */
  const MOVEMENT_DIRECTIONS = {
    UP: 0x1,
    DOWN: 0x2,
    LEFT: 0x4,
    RIGHT: 0x8,
    UP_LEFT: 0x1 | 0x4,
    UP_RIGHT: 0x1 | 0x8,
    DOWN_LEFT: 0x2 | 0x4,
    DOWN_RIGHT: 0x2 | 0x8
  };

  /**
   * The minimum allowed grid size which is supported by the software
   * @type {number}
   */
  const GRID_MIN_SIZE = 20;

  /**
   * The allowed Grid types which are supported by the software
   * @enum {number}
   * @see https://foundryvtt.com/article/scenes/
   */
  const GRID_TYPES = {
    /**
     * No fixed grid is used on this Scene allowing free-form point-to-point measurement without grid lines.
     */
    GRIDLESS: 0,

    /**
     * A square grid is used with width and height of each grid space equal to the chosen grid size.
     */
    SQUARE: 1,

    /**
     * A row-wise hexagon grid (pointy-topped) where odd-numbered rows are offset.
     */
    HEXODDR: 2,

    /**
     * A row-wise hexagon grid (pointy-topped) where even-numbered rows are offset.
     */
    HEXEVENR: 3,

    /**
     * A column-wise hexagon grid (flat-topped) where odd-numbered columns are offset.
     */
    HEXODDQ: 4,

    /**
     * A column-wise hexagon grid (flat-topped) where even-numbered columns are offset.
     */
    HEXEVENQ: 5
  };

  /**
   * The different rules to define and measure diagonal distance/cost in a square grid.
   * The description of each option refers to the distance/cost of moving diagonally relative to the distance/cost of a horizontal or vertical move.
   * @enum {number}
   */
  const GRID_DIAGONALS = {
    /**
     * The diagonal distance is 1. Diagonal movement costs the same as horizontal/vertical movement.
     */
    EQUIDISTANT: 0,

    /**
     * The diagonal distance is √2. Diagonal movement costs √2 times as much as horizontal/vertical movement.
     */
    EXACT: 1,

    /**
     * The diagonal distance is 1.5. Diagonal movement costs 1.5 times as much as horizontal/vertical movement.
     */
    APPROXIMATE: 2,

    /**
     * The diagonal distance is 2. Diagonal movement costs 2 times as much as horizontal/vertical movement.
     */
    RECTILINEAR: 3,

    /**
     * The diagonal distance alternates between 1 and 2 starting at 1.
     * The first diagonal movement costs the same as horizontal/vertical movement
     * The second diagonal movement costs 2 times as much as horizontal/vertical movement.
     * And so on...
     */
    ALTERNATING_1: 4,

    /**
     * The diagonal distance alternates between 2 and 1 starting at 2.
     * The first diagonal movement costs 2 times as much as horizontal/vertical movement.
     * The second diagonal movement costs the same as horizontal/vertical movement.
     * And so on...
     */
    ALTERNATING_2: 5,

    /**
     * The diagonal distance is ∞. Diagonal movement is not allowed/possible.
     */
    ILLEGAL: 6,
  };

  /**
   * The grid snapping modes.
   * @enum {number}
   */
  const GRID_SNAPPING_MODES = {
    /**
     * Nearest center point.
     */
    CENTER: 0x1,

    /**
     * Nearest edge midpoint.
     */
    EDGE_MIDPOINT: 0x2,

    /**
     * Nearest top-left vertex.
     */
    TOP_LEFT_VERTEX: 0x10,

    /**
     * Nearest top-right vertex.
     */
    TOP_RIGHT_VERTEX: 0x20,

    /**
     * Nearest bottom-left vertex.
     */
    BOTTOM_LEFT_VERTEX: 0x40,

    /**
     * Nearest bottom-right vertex.
     */
    BOTTOM_RIGHT_VERTEX: 0x80,

    /**
     * Nearest vertex.
     * Alias for `TOP_LEFT_VERTEX | TOP_RIGHT_VERTEX | BOTTOM_LEFT_VERTEX | BOTTOM_RIGHT_VERTEX`.
     */
    VERTEX: 0xF0,

    /**
     * Nearest top-left corner.
     */
    TOP_LEFT_CORNER: 0x100,

    /**
     * Nearest top-right corner.
     */
    TOP_RIGHT_CORNER: 0x200,

    /**
     * Nearest bottom-left corner.
     */
    BOTTOM_LEFT_CORNER: 0x400,

    /**
     * Nearest bottom-right corner.
     */
    BOTTOM_RIGHT_CORNER: 0x800,

    /**
     * Nearest corner.
     * Alias for `TOP_LEFT_CORNER | TOP_RIGHT_CORNER | BOTTOM_LEFT_CORNER | BOTTOM_RIGHT_CORNER`.
     */
    CORNER: 0xF00,

    /**
     * Nearest top side midpoint.
     */
    TOP_SIDE_MIDPOINT: 0x1000,

    /**
     * Nearest bottom side midpoint.
     */
    BOTTOM_SIDE_MIDPOINT: 0x2000,

    /**
     * Nearest left side midpoint.
     */
    LEFT_SIDE_MIDPOINT: 0x4000,

    /**
     * Nearest right side midpoint.
     */
    RIGHT_SIDE_MIDPOINT: 0x8000,

    /**
     * Nearest side midpoint.
     * Alias for `TOP_SIDE_MIDPOINT | BOTTOM_SIDE_MIDPOINT | LEFT_SIDE_MIDPOINT | RIGHT_SIDE_MIDPOINT`.
     */
    SIDE_MIDPOINT: 0xF000,
  };

  /**
   * A list of supported setup URL names
   * @type {string[]}
   */
  const SETUP_VIEWS = ["auth", "license", "setup", "players", "join", "update"];

  /**
   * An Array of valid MacroAction scope values
   * @type {string[]}
   */
  const MACRO_SCOPES = ["global", "actors", "actor"];

  /**
   * An enumeration of valid Macro types
   * @enum {string}
   * @see https://foundryvtt.com/article/macros/
   */
  const MACRO_TYPES = {
    /**
     * Complex and powerful macros which leverage the FVTT API through plain JavaScript to perform functions as simple or as advanced as you can imagine.
     */
    SCRIPT: "script",

    /**
     * Simple and easy to use, chat macros post pre-defined chat messages to the chat log when executed. All users can execute chat macros by default.
     */
    CHAT: "chat"
  };

  /**
   * The allowed channels for audio playback.
   * @enum {string}
   */
  const AUDIO_CHANNELS = {
    music: "AUDIO.CHANNELS.MUSIC.label",
    environment: "AUDIO.CHANNELS.ENVIRONMENT.label",
    interface: "AUDIO.CHANNELS.INTERFACE.label",
  };

  /**
   * The allowed playback modes for an audio Playlist
   * @enum {number}
   * @see https://foundryvtt.com/article/playlists/
   */
  const PLAYLIST_MODES = {
    /**
     * The playlist does not play on its own, only individual Sound tracks played as a soundboard.
     */
    DISABLED: -1,

    /**
     * The playlist plays sounds one at a time in sequence.
     */
    SEQUENTIAL: 0,

    /**
     * The playlist plays sounds one at a time in randomized order.
     */
    SHUFFLE: 1,

    /**
     * The playlist plays all contained sounds at the same time.
     */
    SIMULTANEOUS: 2
  };

  /**
   * The available sort modes for an audio Playlist.
   * @enum {string}
   * @see https://foundryvtt.com/article/playlists/
   */
  const PLAYLIST_SORT_MODES = {
    /**
     * Sort sounds alphabetically.
     * @defaultValue
     */
    ALPHABETICAL: "a",

    /**
     * Sort sounds by manual drag-and-drop.
     */
    MANUAL: "m"
  };

  /**
   * The available modes for searching within a DirectoryCollection
   * @type {{FULL: string, NAME: string}}
   */
  const DIRECTORY_SEARCH_MODES = {
    FULL: "full",
    NAME: "name"
  };

  /**
   * The allowed package types
   * @type {string[]}
   */
  const PACKAGE_TYPES = ["world", "system", "module"];

  /**
   * Encode the reasons why a package may be available or unavailable for use
   * @enum {number}
   */
  const PACKAGE_AVAILABILITY_CODES = {
    /**
     * Package availability could not be determined
     */
    UNKNOWN: 0,

    /**
     * The Package is verified to be compatible with the current core software build
     */
    VERIFIED: 1,

    /**
     * Package is available for use, but not verified for the current core software build
     */
    UNVERIFIED_BUILD: 2,

    /**
     * One or more installed system is incompatible with the Package.
     */
    UNVERIFIED_SYSTEM: 3,

    /**
     * Package is available for use, but not verified for the current core software generation
     */
    UNVERIFIED_GENERATION: 4,

    /**
     * The System that the Package relies on is not available
     */
    MISSING_SYSTEM: 5,

    /**
     * A dependency of the Package is not available
     */
    MISSING_DEPENDENCY: 6,

    /**
     * The Package is compatible with an older version of Foundry than the currently installed version
     */
    REQUIRES_CORE_DOWNGRADE: 7,

    /**
     * The Package is compatible with a newer version of Foundry than the currently installed version, and that version is Stable
     */
    REQUIRES_CORE_UPGRADE_STABLE: 8,

    /**
     * The Package is compatible with a newer version of Foundry than the currently installed version, and that version is not yet Stable
     */
    REQUIRES_CORE_UPGRADE_UNSTABLE: 9,

    /**
     * A required dependency is not compatible with the current version of Foundry
     */
    REQUIRES_DEPENDENCY_UPDATE: 10
  };

  /**
   * A safe password string which can be displayed
   * @type {string}
   */
  const PASSWORD_SAFE_STRING = "•".repeat(16);

  /**
   * The allowed software update channels
   * @enum {string}
   */
  const SOFTWARE_UPDATE_CHANNELS = {
    /**
     * The Stable release channel
     */
    stable: "SETUP.UpdateStable",

    /**
     * The User Testing release channel
     */
    testing: "SETUP.UpdateTesting",

    /**
     * The Development release channel
     */
    development: "SETUP.UpdateDevelopment",

    /**
     * The Prototype release channel
     */
    prototype: "SETUP.UpdatePrototype"
  };

  /**
   * The default sorting density for manually ordering child objects within a parent
   * @type {number}
   */
  const SORT_INTEGER_DENSITY = 100000;

  /**
   * The allowed types of a TableResult document
   * @enum {string}
   * @see https://foundryvtt.com/article/roll-tables/
   */
  const TABLE_RESULT_TYPES = {
    /**
     *  Plain text or HTML scripted entries which will be output to Chat.
     */
    TEXT: "text",

    /**
     * An in-World Document reference which will be linked to in the chat message.
     */
    DOCUMENT: "document",

    /**
     * A Compendium Pack reference which will be linked to in the chat message.
     */
    COMPENDIUM: "pack"
  };

  /**
   * The allowed formats of a Journal Entry Page.
   * @enum {number}
   * @see https://foundryvtt.com/article/journal/
   */
  const JOURNAL_ENTRY_PAGE_FORMATS = {
    /**
     * The page is formatted as HTML.
     */
    HTML: 1,

    /**
     * The page is formatted as Markdown.
     */
    MARKDOWN: 2,
  };

  /**
   * Define the valid anchor locations for a Tooltip displayed on a Placeable Object
   * @enum {number}
   * @see TooltipManager
   */
  const TEXT_ANCHOR_POINTS = {
    /**
     * Anchor the tooltip to the center of the element.
     */
    CENTER: 0,

    /**
     * Anchor the tooltip to the bottom of the element.
     */
    BOTTOM: 1,

    /**
     * Anchor the tooltip to the top of the element.
     */
    TOP: 2,

    /**
     * Anchor the tooltip to the left of the element.
     */
    LEFT: 3,

    /**
     * Anchor the tooltip to the right of the element.
     */
    RIGHT: 4
  };

  /**
   * Define the valid occlusion modes which a tile can use
   * @enum {number}
   * @see https://foundryvtt.com/article/tiles/
   */
  const OCCLUSION_MODES = {
    /**
     * Turns off occlusion, making the tile never fade while tokens are under it.
     */
    NONE: 0,

    /**
     * Causes the whole tile to fade when an actor token moves under it.
     * @defaultValue
     */
    FADE: 1,

    // ROOF: 2,  This mode is no longer supported so we don't use 2 for any other mode

    /**
     * Causes the tile to reveal the background in the vicinity of an actor token under it. The radius is determined by the token's size.
     */
    RADIAL: 3,

    /**
     * Causes the tile to be partially revealed based on the vision of the actor, which does not need to be under the tile to see what's beneath it.
     *
     * @remarks
     * This is useful for rooves on buildings where players could see through a window or door, viewing only a portion of what is obscured by the roof itself.
     */
    VISION: 4
  };

  /**
   * Alias for old tile occlusion modes definition
   */
  const TILE_OCCLUSION_MODES = OCCLUSION_MODES;

  /**
   * The occlusion modes that define the set of tokens that trigger occlusion.
   * @enum {number}
   */
  const TOKEN_OCCLUSION_MODES = {

    /**
     * Owned tokens that aren't hidden.
     */
    OWNED: 0x1,

    /**
     * Controlled tokens.
     */
    CONTROLLED: 0x2,

    /**
     * Hovered tokens that are visible.
     */
    HOVERED: 0x4,

    /**
     * Highlighted tokens that are visible.
     */
    HIGHLIGHTED: 0x8,

    /**
     * All visible tokens.
     */
    VISIBLE: 0x10
  };

  /**
   * Describe the various thresholds of token control upon which to show certain pieces of information
   * @enum {number}
   * @see https://foundryvtt.com/article/tokens/
   */
  const TOKEN_DISPLAY_MODES = {
    /**
     * No information is displayed.
     */
    NONE: 0,

    /**
     * Displayed when the token is controlled.
     */
    CONTROL: 10,

    /**
     * Displayed when hovered by a GM or a user who owns the actor.
     */
    OWNER_HOVER: 20,

    /**
     * Displayed when hovered by any user.
     */
    HOVER: 30,

    /**
     * Always displayed for a GM or for a user who owns the actor.
     */
    OWNER: 40,

    /**
     * Always displayed for everyone.
     */
    ALWAYS: 50
  };

  /**
   * The allowed Token disposition types
   * @enum {number}
   * @see https://foundryvtt.com/article/tokens/
   */
  const TOKEN_DISPOSITIONS = {
    /**
     * Displayed with a purple borders for owners and with no borders for others (and no pointer change).
     */
    SECRET: -2,

    /**
     * Displayed as an enemy with a red border.
     */
    HOSTILE: -1,

    /**
     * Displayed as neutral with a yellow border.
     */
    NEUTRAL: 0,

    /**
     * Displayed as an ally with a cyan border.
     */
    FRIENDLY: 1
  };

  /**
   * The possible shapes of Tokens in hexagonal grids.
   * @enum {number}
   */
  const TOKEN_HEXAGONAL_SHAPES = {

    /**
     * Ellipse (Variant 1)
     */
    ELLIPSE_1: 0,

    /**
     * Ellipse (Variant 2)
     */
    ELLIPSE_2: 1,

    /**
     * Trapezoid (Variant 1)
     */
    TRAPEZOID_1: 2,

    /**
     * Trapezoid (Variant 2)
     */
    TRAPEZOID_2: 3,

    /**
     * Rectangle (Variant 1)
     */
    RECTANGLE_1: 4,

    /**
     * Rectangle (Variant 2)
     */
    RECTANGLE_2: 5,
  };

  /**
   * Define the allowed User permission levels.
   * Each level is assigned a value in ascending order. Higher levels grant more permissions.
   * @enum {number}
   * @see https://foundryvtt.com/article/users/
   */
  const USER_ROLES = {
    /**
     * The User is blocked from taking actions in Foundry Virtual Tabletop.
     * You can use this role to temporarily or permanently ban a user from joining the game.
     */
    NONE: 0,

    /**
     * The User is able to join the game with permissions available to a standard player.
     * They cannot take some more advanced actions which require Trusted permissions, but they have the basic functionalities needed to operate in the virtual tabletop.
     */
    PLAYER: 1,

    /**
     * Similar to the Player role, except a Trusted User has the ability to perform some more advanced actions like create drawings, measured templates, or even to (optionally) upload media files to the server.
     */
    TRUSTED: 2,

    /**
     * A special User who has many of the same in-game controls as a Game Master User, but does not have the ability to perform administrative actions like changing User roles or modifying World-level settings.
     */
    ASSISTANT: 3,

    /**
     *  A special User who has administrative control over this specific World.
     *  Game Masters behave quite differently than Players in that they have the ability to see all Documents and Objects within the world as well as the capability to configure World settings.
     */
    GAMEMASTER: 4
  };

  /**
   * Invert the User Role mapping to recover role names from a role integer
   * @enum {string}
   * @see USER_ROLES
   */
  const USER_ROLE_NAMES = Object.entries(USER_ROLES).reduce((obj, r) => {
    obj[r[1]] = r[0];
    return obj;
  }, {});

  /**
   * An enumeration of the allowed types for a MeasuredTemplate embedded document
   * @enum {string}
   * @see https://foundryvtt.com/article/measurement/
   */
  const MEASURED_TEMPLATE_TYPES = {
    /**
     * Circular templates create a radius around the starting point.
     */
    CIRCLE: "circle",

    /**
     * Cones create an effect in the shape of a triangle or pizza slice from the starting point.
     */
    CONE: "cone",

    /**
     * A rectangle uses the origin point as one of the corners, treating the origin as being inside of the rectangle's area.
     */
    RECTANGLE: "rect",

    /**
     * A ray creates a single line that is one square in width and as long as you want it to be.
     */
    RAY: "ray"
  };

  /**
   * @typedef {Object} UserPermission
   * @property {string} label
   * @property {string} hint
   * @property {boolean} disableGM
   * @property {number} defaultRole
   */

  /**
   * Define the recognized User capabilities which individual Users or role levels may be permitted to perform
   * @type {Record<string, UserPermission>}
   */
  const USER_PERMISSIONS = {
    ACTOR_CREATE: {
      label: "PERMISSION.ActorCreate",
      hint: "PERMISSION.ActorCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    BROADCAST_AUDIO: {
      label: "PERMISSION.BroadcastAudio",
      hint: "PERMISSION.BroadcastAudioHint",
      disableGM: true,
      defaultRole: USER_ROLES.TRUSTED
    },
    BROADCAST_VIDEO: {
      label: "PERMISSION.BroadcastVideo",
      hint: "PERMISSION.BroadcastVideoHint",
      disableGM: true,
      defaultRole: USER_ROLES.TRUSTED
    },
    CARDS_CREATE: {
      label: "PERMISSION.CardsCreate",
      hint: "PERMISSION.CardsCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    DRAWING_CREATE: {
      label: "PERMISSION.DrawingCreate",
      hint: "PERMISSION.DrawingCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.TRUSTED
    },
    ITEM_CREATE: {
      label: "PERMISSION.ItemCreate",
      hint: "PERMISSION.ItemCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    FILES_BROWSE: {
      label: "PERMISSION.FilesBrowse",
      hint: "PERMISSION.FilesBrowseHint",
      disableGM: false,
      defaultRole: USER_ROLES.TRUSTED
    },
    FILES_UPLOAD: {
      label: "PERMISSION.FilesUpload",
      hint: "PERMISSION.FilesUploadHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    JOURNAL_CREATE: {
      label: "PERMISSION.JournalCreate",
      hint: "PERMISSION.JournalCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.TRUSTED
    },
    MACRO_SCRIPT: {
      label: "PERMISSION.MacroScript",
      hint: "PERMISSION.MacroScriptHint",
      disableGM: false,
      defaultRole: USER_ROLES.PLAYER
    },
    MANUAL_ROLLS: {
      label: "PERMISSION.ManualRolls",
      hint: "PERMISSION.ManualRollsHint",
      disableGM: true,
      defaultRole: USER_ROLES.TRUSTED
    },
    MESSAGE_WHISPER: {
      label: "PERMISSION.MessageWhisper",
      hint: "PERMISSION.MessageWhisperHint",
      disableGM: false,
      defaultRole: USER_ROLES.PLAYER
    },
    NOTE_CREATE: {
      label: "PERMISSION.NoteCreate",
      hint: "PERMISSION.NoteCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.TRUSTED
    },
    PING_CANVAS: {
      label: "PERMISSION.PingCanvas",
      hint: "PERMISSION.PingCanvasHint",
      disableGM: true,
      defaultRole: USER_ROLES.PLAYER
    },
    PLAYLIST_CREATE: {
      label: "PERMISSION.PlaylistCreate",
      hint: "PERMISSION.PlaylistCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    SETTINGS_MODIFY: {
      label: "PERMISSION.SettingsModify",
      hint: "PERMISSION.SettingsModifyHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    SHOW_CURSOR: {
      label: "PERMISSION.ShowCursor",
      hint: "PERMISSION.ShowCursorHint",
      disableGM: true,
      defaultRole: USER_ROLES.PLAYER
    },
    SHOW_RULER: {
      label: "PERMISSION.ShowRuler",
      hint: "PERMISSION.ShowRulerHint",
      disableGM: true,
      defaultRole: USER_ROLES.PLAYER
    },
    TEMPLATE_CREATE: {
      label: "PERMISSION.TemplateCreate",
      hint: "PERMISSION.TemplateCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.PLAYER
    },
    TOKEN_CREATE: {
      label: "PERMISSION.TokenCreate",
      hint: "PERMISSION.TokenCreateHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    TOKEN_DELETE: {
      label: "PERMISSION.TokenDelete",
      hint: "PERMISSION.TokenDeleteHint",
      disableGM: false,
      defaultRole: USER_ROLES.ASSISTANT
    },
    TOKEN_CONFIGURE: {
      label: "PERMISSION.TokenConfigure",
      hint: "PERMISSION.TokenConfigureHint",
      disableGM: false,
      defaultRole: USER_ROLES.TRUSTED
    },
    WALL_DOORS: {
      label: "PERMISSION.WallDoors",
      hint: "PERMISSION.WallDoorsHint",
      disableGM: false,
      defaultRole: USER_ROLES.PLAYER
    }
  };

  /**
   * The allowed directions of effect that a Wall can have
   * @enum {number}
   * @see https://foundryvtt.com/article/walls/
   */
  const WALL_DIRECTIONS = {
    /**
     * The wall collides from both directions.
     */
    BOTH: 0,

    /**
     * The wall collides only when a ray strikes its left side.
     */
    LEFT: 1,

    /**
     * The wall collides only when a ray strikes its right side.
     */
    RIGHT: 2
  };

  /**
   * The allowed door types which a Wall may contain
   * @enum {number}
   * @see https://foundryvtt.com/article/walls/
   */
  const WALL_DOOR_TYPES = {
    /**
     * The wall does not contain a door.
     */
    NONE: 0,

    /**
     *  The wall contains a regular door.
     */
    DOOR: 1,

    /**
     * The wall contains a secret door.
     */
    SECRET: 2
  };

  /**
   * The allowed door states which may describe a Wall that contains a door
   * @enum {number}
   * @see https://foundryvtt.com/article/walls/
   */
  const WALL_DOOR_STATES = {
    /**
     * The door is closed.
     */
    CLOSED: 0,

    /**
     * The door is open.
     */
    OPEN: 1,

    /**
     * The door is closed and locked.
     */
    LOCKED: 2
  };

  /**
   * The possible ways to interact with a door
   * @enum {string[]}
   */
  const WALL_DOOR_INTERACTIONS = ["open", "close", "lock", "unlock", "test"];

  /**
   * The wall properties which restrict the way interaction occurs with a specific wall
   * @type {string[]}
   */
  const WALL_RESTRICTION_TYPES = ["light", "sight", "sound", "move"];

  /**
   * The types of sensory collision which a Wall may impose
   * @enum {number}
   * @see https://foundryvtt.com/article/walls/
   */
  const WALL_SENSE_TYPES = {
    /**
     * Senses do not collide with this wall.
     */
    NONE: 0,

    /**
     * Senses collide with this wall.
     */
    LIMITED: 10,

    /**
     * Senses collide with the second intersection, bypassing the first.
     */
    NORMAL: 20,

    /**
     * Senses bypass the wall within a certain proximity threshold.
     */
    PROXIMITY: 30,

    /**
     * Senses bypass the wall outside a certain proximity threshold.
     */
    DISTANCE: 40
  };

  /**
   * The types of movement collision which a Wall may impose
   * @enum {number}
   * @see https://foundryvtt.com/article/walls/
   */
  const WALL_MOVEMENT_TYPES = {
    /**
     * Movement does not collide with this wall.
     */
    NONE: WALL_SENSE_TYPES.NONE,

    /**
     * Movement collides with this wall.
     */
    NORMAL: WALL_SENSE_TYPES.NORMAL
  };

  /**
   * The possible precedence values a Keybinding might run in
   * @enum {number}
   * @see https://foundryvtt.com/article/keybinds/
   */
  const KEYBINDING_PRECEDENCE = {
    /**
     * Runs in the first group along with other PRIORITY keybindings.
     */
    PRIORITY: 0,

    /**
     * Runs after the PRIORITY group along with other NORMAL keybindings.
     */
    NORMAL: 1,

    /**
     * Runs in the last group along with other DEFERRED keybindings.
     */
    DEFERRED: 2
  };

  /**
   * The allowed set of HTML template extensions
   * @type {string[]}
   */
  const HTML_FILE_EXTENSIONS = ["html", "handlebars", "hbs"];

  /**
   * The supported file extensions for image-type files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const IMAGE_FILE_EXTENSIONS = {
    apng: "image/apng",
    avif: "image/avif",
    bmp: "image/bmp",
    gif: "image/gif",
    jpeg: "image/jpeg",
    jpg: "image/jpeg",
    png: "image/png",
    svg: "image/svg+xml",
    tiff: "image/tiff",
    webp: "image/webp"
  };

  /**
   * The supported file extensions for video-type files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const VIDEO_FILE_EXTENSIONS = {
    m4v: "video/mp4",
    mp4: "video/mp4",
    ogv: "video/ogg",
    webm: "video/webm"
  };

  /**
   * The supported file extensions for audio-type files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const AUDIO_FILE_EXTENSIONS = {
    aac: "audio/aac",
    flac: "audio/flac",
    m4a: "audio/mp4",
    mid: "audio/midi",
    mp3: "audio/mpeg",
    ogg: "audio/ogg",
    opus: "audio/opus",
    wav: "audio/wav",
    webm: "audio/webm"
  };

  /**
   * The supported file extensions for text files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const TEXT_FILE_EXTENSIONS = {
    csv: "text/csv",
    json: "application/json",
    md: "text/markdown",
    pdf: "application/pdf",
    tsv: "text/tab-separated-values",
    txt: "text/plain",
    xml: "application/xml",
    yml: "application/yaml",
    yaml: "application/yaml"
  };

  /**
   * Supported file extensions for font files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const FONT_FILE_EXTENSIONS = {
    ttf: "font/ttf",
    otf: "font/otf",
    woff: "font/woff",
    woff2: "font/woff2"
  };

  /**
   * Supported file extensions for 3D files, and their corresponding mime types.
   * @type {Record<string, string>}
   */
  const GRAPHICS_FILE_EXTENSIONS = {
    fbx: "application/octet-stream",
    glb: "model/gltf-binary",
    gltf: "model/gltf+json",
    mtl: "model/mtl",
    obj: "model/obj",
    stl: "model/stl",
    usdz: "model/vnd.usdz+zip"
  };

  /**
   * A consolidated mapping of all extensions permitted for upload.
   * @type {Record<string, string>}
   */
  const UPLOADABLE_FILE_EXTENSIONS = {
    ...IMAGE_FILE_EXTENSIONS,
    ...VIDEO_FILE_EXTENSIONS,
    ...AUDIO_FILE_EXTENSIONS,
    ...TEXT_FILE_EXTENSIONS,
    ...FONT_FILE_EXTENSIONS,
    ...GRAPHICS_FILE_EXTENSIONS
  };

  /**
   * A list of MIME types which are treated as uploaded "media", which are allowed to overwrite existing files.
   * Any non-media MIME type is not allowed to replace an existing file.
   * @type {string[]}
   */
  const MEDIA_MIME_TYPES = Object.values(UPLOADABLE_FILE_EXTENSIONS);

  /**
   * An enumeration of file type categories which can be selected
   * @enum {Record<string, string>}
   */
  const FILE_CATEGORIES = {
    HTML: HTML_FILE_EXTENSIONS,
    IMAGE: IMAGE_FILE_EXTENSIONS,
    VIDEO: VIDEO_FILE_EXTENSIONS,
    AUDIO: AUDIO_FILE_EXTENSIONS,
    TEXT: TEXT_FILE_EXTENSIONS,
    FONT: FONT_FILE_EXTENSIONS,
    GRAPHICS: GRAPHICS_FILE_EXTENSIONS,
    MEDIA: MEDIA_MIME_TYPES,
  };

  /**
   * A font weight to name mapping.
   * @enum {number}
   */
  const FONT_WEIGHTS = {
    Thin: 100,
    ExtraLight: 200,
    Light: 300,
    Regular: 400,
    Medium: 500,
    SemiBold: 600,
    Bold: 700,
    ExtraBold: 800,
    Black: 900
  };

  /**
   * Stores shared commonly used timeouts, measured in MS
   * @enum {number}
   */
  const TIMEOUTS = {
    /**
     * The default timeout for interacting with the foundryvtt.com API.
     */
    FOUNDRY_WEBSITE: 10000,

    /**
     * The specific timeout for loading the list of packages from the foundryvtt.com API.
     */
    PACKAGE_REPOSITORY: 5000,

    /**
     * The specific timeout for the IP address lookup service.
     */
    IP_DISCOVERY: 5000
  };

  /**
   * A subset of Compendium types which require a specific system to be designated
   * @type {string[]}
   */
  const SYSTEM_SPECIFIC_COMPENDIUM_TYPES = ["Actor", "Item"];

  /**
   * The configured showdown bi-directional HTML <-> Markdown converter options.
   * @type {Record<string, boolean>}
   */
  const SHOWDOWN_OPTIONS = {
    disableForced4SpacesIndentedSublists: true,
    noHeaderId: true,
    parseImgDimensions: true,
    strikethrough: true,
    tables: true,
    tablesHeaderId: true
  };

  /**
   * The list of allowed attributes in HTML elements.
   * @type {Record<string, string[]>}
   */
  const ALLOWED_HTML_ATTRIBUTES = Object.freeze({
    "*": Object.freeze([
      "class", "data-*", "id", "title", "style", "draggable", "aria-*", "tabindex", "dir", "hidden", "inert", "role",
      "is", "lang", "popover"
    ]),
    a: Object.freeze(["href", "name", "target", "rel"]),
    area: Object.freeze(["alt", "coords", "href", "rel", "shape", "target"]),
    audio: Object.freeze(["controls", "loop", "muted", "src", "autoplay"]),
    blockquote: Object.freeze(["cite"]),
    button: Object.freeze(["disabled", "name", "type", "value"]),
    col: Object.freeze(["span"]),
    colgroup: Object.freeze(["span"]),
    details: Object.freeze(["open"]),
    fieldset: Object.freeze(["disabled"]),
    form: Object.freeze(["name"]),
    iframe: Object.freeze(["src", "srcdoc", "name", "height", "width", "loading", "sandbox"]),
    img: Object.freeze(["height", "src", "width", "usemap", "sizes", "srcset", "alt"]),
    input: Object.freeze([
      "checked", "disabled", "name", "value", "placeholder", "type", "alt", "height", "list",
      "max", "min", "placeholder", "readonly", "size", "src", "step", "width"
    ]),
    label: Object.freeze(["for"]),
    li: Object.freeze(["value"]),
    map: Object.freeze(["name"]),
    meter: Object.freeze(["value", "min", "max", "low", "high", "optimum"]),
    ol: Object.freeze(["reversed", "start", "type"]),
    optgroup: Object.freeze(["disabled", "label"]),
    option: Object.freeze(["disabled", "selected", "label", "value"]),
    progress: Object.freeze(["max", "value"]),
    select: Object.freeze(["name", "disabled", "multiple", "size"]),
    source: Object.freeze(["media", "sizes", "src", "srcset", "type"]),
    table: Object.freeze(["border"]),
    td: Object.freeze(["colspan", "headers", "rowspan"]),
    textarea: Object.freeze(["rows", "cols", "disabled", "name", "readonly", "wrap"]),
    time: Object.freeze(["datetime"]),
    th: Object.freeze(["abbr", "colspan", "headers", "rowspan", "scope", "sorted"]),
    track: Object.freeze(["default", "kind", "label", "src", "srclang"]),
    video: Object.freeze(["controls", "height", "width", "loop", "muted", "poster", "src", "autoplay"])
  });

  /**
   * The list of trusted iframe domains.
   * @type {string[]}
   */
  const TRUSTED_IFRAME_DOMAINS = Object.freeze(["google.com", "youtube.com"]);

  /**
   * Available themes for the world join page.
   * @enum {string}
   */
  const WORLD_JOIN_THEMES = {
    default: "WORLD.JoinThemeDefault",
    minimal: "WORLD.JoinThemeMinimal"
  };

  /**
   * Setup page package progress protocol.
   * @type {{ACTIONS: Record<string, string>, STEPS: Record<string, string>}}
   */
  const SETUP_PACKAGE_PROGRESS = {
    ACTIONS: {
      CREATE_BACKUP: "createBackup",
      RESTORE_BACKUP: "restoreBackup",
      DELETE_BACKUP: "deleteBackup",
      CREATE_SNAPSHOT: "createSnapshot",
      RESTORE_SNAPSHOT: "restoreSnapshot",
      DELETE_SNAPSHOT: "deleteSnapshot",
      INSTALL_PKG: "installPackage",
      LAUNCH_WORLD: "launchWorld",
      UPDATE_CORE: "updateCore",
      UPDATE_DOWNLOAD: "updateDownload"
    },
    STEPS: {
      ARCHIVE: "archive",
      CHECK_DISK_SPACE: "checkDiskSpace",
      CONNECT_WORLD: "connectWorld",
      MIGRATE_WORLD: "migrateWorld",
      CONNECT_PKG: "connectPackage",
      MIGRATE_PKG: "migratePackage",
      MIGRATE_CORE: "migrateCore",
      MIGRATE_SYSTEM: "migrateSystem",
      DOWNLOAD: "download",
      EXTRACT: "extract",
      INSTALL: "install",
      CLEANUP: "cleanup",
      COMPLETE: "complete",
      DELETE: "delete",
      ERROR: "error",
      VEND: "vend",
      SNAPSHOT_MODULES: "snapshotModules",
      SNAPSHOT_SYSTEMS: "snapshotSystems",
      SNAPSHOT_WORLDS: "snapshotWorlds"
    }
  };

  /**
   * The combat announcements.
   * @type {string[]}
   */
  const COMBAT_ANNOUNCEMENTS = ["startEncounter", "nextUp", "yourTurn"];

  /**
   * The fit modes of {@link foundry.data.TextureData#fit}.
   * @type {string[]}
   */
  const TEXTURE_DATA_FIT_MODES = ["fill", "contain", "cover", "width", "height"];

  /**
   * The maximum depth to recurse to when embedding enriched text.
   * @type {number}
   */
  const TEXT_ENRICH_EMBED_MAX_DEPTH = 5;

  /**
   * The Region events that are supported by core.
   * @enum {string}
   */
  const REGION_EVENTS = {

    /**
     * Triggered when the shapes or bottom/top elevation of the Region are changed.
     */
    REGION_BOUNDARY: "regionBoundary",

    /**
     * Triggered when the behavior is enabled/disabled or the Scene its Region is in is viewed/unviewed.
     */
    BEHAVIOR_STATUS: "behaviorStatus",

    /**
     * Triggered when a Token enters a Region.
     */
    TOKEN_ENTER: "tokenEnter",

    /**
     * Triggered when a Token exists a Region.
     */
    TOKEN_EXIT: "tokenExit",

    /**
     * Triggered when a Token is about to move into, out of, through, or within a Region.
     */
    TOKEN_PRE_MOVE: "tokenPreMove",

    /**
     * Triggered when a Token moves into, out of, through, or within a Region.
     */
    TOKEN_MOVE: "tokenMove",

    /**
     * Triggered when a Token moves into a Region.
     */
    TOKEN_MOVE_IN: "tokenMoveIn",

    /**
     * Triggered when a Token moves out of a Region.
     */
    TOKEN_MOVE_OUT: "tokenMoveOut",

    /**
     * Triggered when a Token starts its Combat turn in a Region.
     */
    TOKEN_TURN_START: "tokenTurnStart",

    /**
     * Triggered when a Token ends its Combat turn in a Region.
     */
    TOKEN_TURN_END: "tokenTurnEnd",

    /**
     * Triggered when a Token starts the Combat round in a Region.
     */
    TOKEN_ROUND_START: "tokenRoundStart",

    /**
     * Triggered when a Token ends the Combat round in a Region.
     */
    TOKEN_ROUND_END: "tokenRoundEnd"
  };

  /**
   * The possible visibility state of Region.
   * @enum {string}
   */
  const REGION_VISIBILITY = {

    /**
     * Only visible on the RegionLayer.
     */
    LAYER: 0,

    /**
     * Only visible to Gamemasters.
     */
    GAMEMASTER: 1,

    /**
     * Visible to anyone.
     */
    ALWAYS: 2
  };

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  const CHAT_MESSAGE_TYPES = new Proxy(CHAT_MESSAGE_STYLES, {
    get(target, prop, receiver) {
      const msg = "CONST.CHAT_MESSAGE_TYPES is deprecated in favor of CONST.CHAT_MESSAGE_STYLES because the " +
        "ChatMessage#type field has been renamed to ChatMessage#style";
      foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
      return Reflect.get(...arguments);
    }
  });

  // Deprecated chat message styles
  Object.defineProperties(CHAT_MESSAGE_STYLES, {
    /**
     * @deprecated since v12
     * @ignore
     */
    ROLL: {
      get() {
        foundry.utils.logCompatibilityWarning("CONST.CHAT_MESSAGE_STYLES.ROLL is deprecated in favor of defining " +
          "rolls directly in ChatMessage#rolls", {since: 12, until: 14, once: true});
        return 0;
      }
    },
    /**
     * @deprecated since v12
     * @ignore
     */
    WHISPER: {
      get() {
        foundry.utils.logCompatibilityWarning("CONST.CHAT_MESSAGE_STYLES.WHISPER is deprecated in favor of defining " +
          "whisper recipients directly in ChatMessage#whisper", {since: 12, until: 14, once: true});
        return 0;
      }
    }
  });

  /**
   * @deprecated since v12
   * @ignore
   */
  const _DOCUMENT_TYPES = Object.freeze(WORLD_DOCUMENT_TYPES.filter(t => {
    const excluded = ["FogExploration", "Setting"];
    return !excluded.includes(t);
  }));

  /**
   * @deprecated since v12
   * @ignore
   */
  const DOCUMENT_TYPES = new Proxy(_DOCUMENT_TYPES, {
    get(target, prop, receiver) {
      const msg = "CONST.DOCUMENT_TYPES is deprecated in favor of either CONST.WORLD_DOCUMENT_TYPES or "
        + "CONST.COMPENDIUM_DOCUMENT_TYPES.";
      foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
      return Reflect.get(...arguments);
    }
  });

  var CONST$1 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ACTIVE_EFFECT_MODES: ACTIVE_EFFECT_MODES,
    ALLOWED_HTML_ATTRIBUTES: ALLOWED_HTML_ATTRIBUTES,
    ALL_DOCUMENT_TYPES: ALL_DOCUMENT_TYPES,
    ASCII: ASCII,
    AUDIO_CHANNELS: AUDIO_CHANNELS,
    AUDIO_FILE_EXTENSIONS: AUDIO_FILE_EXTENSIONS,
    BASE_DOCUMENT_TYPE: BASE_DOCUMENT_TYPE,
    CANVAS_PERFORMANCE_MODES: CANVAS_PERFORMANCE_MODES,
    CARD_DRAW_MODES: CARD_DRAW_MODES,
    CHAT_MESSAGE_STYLES: CHAT_MESSAGE_STYLES,
    CHAT_MESSAGE_TYPES: CHAT_MESSAGE_TYPES,
    COMBAT_ANNOUNCEMENTS: COMBAT_ANNOUNCEMENTS,
    COMPATIBILITY_MODES: COMPATIBILITY_MODES,
    COMPENDIUM_DOCUMENT_TYPES: COMPENDIUM_DOCUMENT_TYPES,
    CORE_SUPPORTED_LANGUAGES: CORE_SUPPORTED_LANGUAGES,
    CSS_THEMES: CSS_THEMES,
    DEFAULT_TOKEN: DEFAULT_TOKEN,
    DICE_ROLL_MODES: DICE_ROLL_MODES,
    DIRECTORY_SEARCH_MODES: DIRECTORY_SEARCH_MODES,
    DOCUMENT_LINK_TYPES: DOCUMENT_LINK_TYPES,
    DOCUMENT_META_OWNERSHIP_LEVELS: DOCUMENT_META_OWNERSHIP_LEVELS,
    DOCUMENT_OWNERSHIP_LEVELS: DOCUMENT_OWNERSHIP_LEVELS,
    DOCUMENT_TYPES: DOCUMENT_TYPES,
    DRAWING_FILL_TYPES: DRAWING_FILL_TYPES,
    EMBEDDED_DOCUMENT_TYPES: EMBEDDED_DOCUMENT_TYPES,
    FILE_CATEGORIES: FILE_CATEGORIES,
    FOLDER_DOCUMENT_TYPES: FOLDER_DOCUMENT_TYPES,
    FOLDER_MAX_DEPTH: FOLDER_MAX_DEPTH,
    FONT_FILE_EXTENSIONS: FONT_FILE_EXTENSIONS,
    FONT_WEIGHTS: FONT_WEIGHTS,
    GAME_VIEWS: GAME_VIEWS,
    GRAPHICS_FILE_EXTENSIONS: GRAPHICS_FILE_EXTENSIONS,
    GRID_DIAGONALS: GRID_DIAGONALS,
    GRID_MIN_SIZE: GRID_MIN_SIZE,
    GRID_SNAPPING_MODES: GRID_SNAPPING_MODES,
    GRID_TYPES: GRID_TYPES,
    HTML_FILE_EXTENSIONS: HTML_FILE_EXTENSIONS,
    IMAGE_FILE_EXTENSIONS: IMAGE_FILE_EXTENSIONS,
    JOURNAL_ENTRY_PAGE_FORMATS: JOURNAL_ENTRY_PAGE_FORMATS,
    KEYBINDING_PRECEDENCE: KEYBINDING_PRECEDENCE,
    LIGHTING_LEVELS: LIGHTING_LEVELS,
    MACRO_SCOPES: MACRO_SCOPES,
    MACRO_TYPES: MACRO_TYPES,
    MEASURED_TEMPLATE_TYPES: MEASURED_TEMPLATE_TYPES,
    MEDIA_MIME_TYPES: MEDIA_MIME_TYPES,
    MOVEMENT_DIRECTIONS: MOVEMENT_DIRECTIONS,
    OCCLUSION_MODES: OCCLUSION_MODES,
    PACKAGE_AVAILABILITY_CODES: PACKAGE_AVAILABILITY_CODES,
    PACKAGE_TYPES: PACKAGE_TYPES,
    PASSWORD_SAFE_STRING: PASSWORD_SAFE_STRING,
    PLAYLIST_MODES: PLAYLIST_MODES,
    PLAYLIST_SORT_MODES: PLAYLIST_SORT_MODES,
    PRIMARY_DOCUMENT_TYPES: PRIMARY_DOCUMENT_TYPES,
    REGION_EVENTS: REGION_EVENTS,
    REGION_VISIBILITY: REGION_VISIBILITY,
    SETUP_PACKAGE_PROGRESS: SETUP_PACKAGE_PROGRESS,
    SETUP_VIEWS: SETUP_VIEWS,
    SHOWDOWN_OPTIONS: SHOWDOWN_OPTIONS,
    SOFTWARE_UPDATE_CHANNELS: SOFTWARE_UPDATE_CHANNELS,
    SORT_INTEGER_DENSITY: SORT_INTEGER_DENSITY,
    SYSTEM_SPECIFIC_COMPENDIUM_TYPES: SYSTEM_SPECIFIC_COMPENDIUM_TYPES,
    TABLE_RESULT_TYPES: TABLE_RESULT_TYPES,
    TEXTURE_DATA_FIT_MODES: TEXTURE_DATA_FIT_MODES,
    TEXT_ANCHOR_POINTS: TEXT_ANCHOR_POINTS,
    TEXT_ENRICH_EMBED_MAX_DEPTH: TEXT_ENRICH_EMBED_MAX_DEPTH,
    TEXT_FILE_EXTENSIONS: TEXT_FILE_EXTENSIONS,
    TILE_OCCLUSION_MODES: TILE_OCCLUSION_MODES,
    TIMEOUTS: TIMEOUTS,
    TOKEN_DISPLAY_MODES: TOKEN_DISPLAY_MODES,
    TOKEN_DISPOSITIONS: TOKEN_DISPOSITIONS,
    TOKEN_HEXAGONAL_SHAPES: TOKEN_HEXAGONAL_SHAPES,
    TOKEN_OCCLUSION_MODES: TOKEN_OCCLUSION_MODES,
    TRUSTED_IFRAME_DOMAINS: TRUSTED_IFRAME_DOMAINS,
    UPLOADABLE_FILE_EXTENSIONS: UPLOADABLE_FILE_EXTENSIONS,
    USER_PERMISSIONS: USER_PERMISSIONS,
    USER_ROLES: USER_ROLES,
    USER_ROLE_NAMES: USER_ROLE_NAMES,
    VIDEO_FILE_EXTENSIONS: VIDEO_FILE_EXTENSIONS,
    VTT: VTT,
    WALL_DIRECTIONS: WALL_DIRECTIONS,
    WALL_DOOR_INTERACTIONS: WALL_DOOR_INTERACTIONS,
    WALL_DOOR_STATES: WALL_DOOR_STATES,
    WALL_DOOR_TYPES: WALL_DOOR_TYPES,
    WALL_MOVEMENT_TYPES: WALL_MOVEMENT_TYPES,
    WALL_RESTRICTION_TYPES: WALL_RESTRICTION_TYPES,
    WALL_SENSE_TYPES: WALL_SENSE_TYPES,
    WEBSITE_API_URL: WEBSITE_API_URL,
    WEBSITE_URL: WEBSITE_URL,
    WORLD_DOCUMENT_TYPES: WORLD_DOCUMENT_TYPES,
    WORLD_JOIN_THEMES: WORLD_JOIN_THEMES,
    vtt: vtt$1
  });

  /** @module helpers */

  /**
   * Benchmark the performance of a function, calling it a requested number of iterations.
   * @param {Function} func       The function to benchmark
   * @param {number} iterations   The number of iterations to test
   * @param {...any} args         Additional arguments passed to the benchmarked function
   */
  async function benchmark(func, iterations, ...args) {
    const start = performance.now();
    for ( let i=0; i<iterations; i++ ) {
      await func(...args, i);
    }
    const end = performance.now();
    const t = Math.round((end - start) * 100) / 100;
    const name = func.name ?? "Evaluated Function";
    console.log(`${name} | ${iterations} iterations | ${t}ms | ${t / iterations}ms per`);
  }

  /* -------------------------------------------- */

  /**
   * A debugging function to test latency or timeouts by forcibly locking the thread for an amount of time.
   * @param {number} ms        A number of milliseconds to lock
   * @returns {Promise<void>}
   */
  async function threadLock(ms, debug=false) {
    const t0 = performance.now();
    let d = 0;
    while ( d < ms ) {
      d = performance.now() - t0;
      if ( debug && (d % 1000 === 0) ) {
        console.debug(`Thread lock for ${d / 1000} of ${ms / 1000} seconds`);
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Wrap a callback in a debounced timeout.
   * Delay execution of the callback function until the function has not been called for delay milliseconds
   * @param {Function} callback       A function to execute once the debounced threshold has been passed
   * @param {number} delay            An amount of time in milliseconds to delay
   * @return {Function}               A wrapped function which can be called to debounce execution
   */
  function debounce(callback, delay) {
    let timeoutId;
    return function(...args) {
      clearTimeout(timeoutId);
      timeoutId = setTimeout(() => {
        callback.apply(this, args);
      }, delay);
    }
  }

  /* -------------------------------------------- */

  /**
   * Wrap a callback in a throttled timeout.
   * Delay execution of the callback function when the last time the function was called was delay milliseconds ago
   * @param {Function} callback       A function to execute once the throttled threshold has been passed
   * @param {number} delay            A maximum amount of time in milliseconds between to execution
   * @return {Function}               A wrapped function which can be called to throttle execution
   */
  function throttle(callback, delay) {
    let pending;
    let lastTime = -delay;
    return function(...args) {
      if ( pending ) {
        pending.thisArg = this;
        pending.args = args;
        return;
      }
      pending = {thisArg: this, args};
      setTimeout(() => {
        const {thisArg, args} = pending;
        pending = null;
        callback.apply(thisArg, args);
        lastTime = performance.now();
      }, Math.max(delay - (performance.now() - lastTime), 0));
    }
  }

  /* -------------------------------------------- */

  /**
   * A utility function to reload the page with a debounce.
   * @callback debouncedReload
   */
  const debouncedReload = debounce( () => window.location.reload(), 250);

  /* -------------------------------------------- */

  /**
   * Quickly clone a simple piece of data, returning a copy which can be mutated safely.
   * This method DOES support recursive data structures containing inner objects or arrays.
   * This method DOES NOT support advanced object types like Set, Map, or other specialized classes.
   * @param {*} original                     Some sort of data
   * @param {object} [options]               Options to configure the behaviour of deepClone
   * @param {boolean} [options.strict=false]  Throw an Error if deepClone is unable to clone something instead of
   *                                          returning the original
   * @param {number} [options._d]             An internal depth tracker
   * @return {*}                             The clone of that data
   */
  function deepClone(original, {strict=false, _d=0}={}) {
    if ( _d > 100 ) {
      throw new Error("Maximum depth exceeded. Be sure your object does not contain cyclical data structures.");
    }
    _d++;

    // Simple types
    if ( (typeof original !== "object") || (original === null) ) return original;

    // Arrays
    if ( original instanceof Array ) return original.map(o => deepClone(o, {strict, _d}));

    // Dates
    if ( original instanceof Date ) return new Date(original);

    // Unsupported advanced objects
    if ( original.constructor && (original.constructor !== Object) ) {
      if ( strict ) throw new Error("deepClone cannot clone advanced objects");
      return original;
    }

    // Other objects
    const clone = {};
    for ( let k of Object.keys(original) ) {
      clone[k] = deepClone(original[k], {strict, _d});
    }
    return clone;
  }

  /* -------------------------------------------- */

  /**
   * Deeply difference an object against some other, returning the update keys and values.
   * @param {object} original       An object comparing data against which to compare
   * @param {object} other          An object containing potentially different data
   * @param {object} [options={}]   Additional options which configure the diff operation
   * @param {boolean} [options.inner=false]  Only recognize differences in other for keys which also exist in original
   * @param {boolean} [options.deletionKeys=false] Apply special logic to deletion keys. They will only be kept if the
   *                                               original object has a corresponding key that could be deleted.
   * @param {number} [options._d]           An internal depth tracker
   * @return {object}               An object of the data in other which differs from that in original
   */
  function diffObject(original, other, {inner=false, deletionKeys=false, _d=0}={}) {
    if ( _d > 100 ) {
      throw new Error("Maximum depth exceeded. Be careful that your object does not contain a cyclical data structure.")
    }
    _d++;

    function _difference(v0, v1) {

      // Eliminate differences in types
      let t0 = getType(v0);
      let t1 = getType(v1);
      if ( t0 !== t1 ) return [true, v1];

      // null and undefined
      if ( ["null", "undefined"].includes(t0) ) return [v0 !== v1, v1];

      // If the prototype explicitly exposes an equality-testing method, use it
      if ( v0?.equals instanceof Function ) return [!v0.equals(v1), v1];

      // Recursively diff objects
      if ( t0 === "Object" ) {
        if ( isEmpty$1(v1) ) return [false, {}];
        if ( isEmpty$1(v0) ) return [true, v1];
        let d = diffObject(v0, v1, {inner, deletionKeys, _d});
        return [!isEmpty$1(d), d];
      }

      // Differences in primitives
      return [v0.valueOf() !== v1.valueOf(), v1];
    }

    // Recursively call the _difference function
    return Object.keys(other).reduce((obj, key) => {
      const isDeletionKey = key.startsWith("-=");
      if ( isDeletionKey && deletionKeys ) {
        const otherKey = key.substring(2);
        if ( otherKey in original ) obj[key] = other[key];
        return obj;
      }
      if ( inner && !(key in original) ) return obj;
      let [isDifferent, difference] = _difference(original[key], other[key]);
      if ( isDifferent ) obj[key] = difference;
      return obj;
    }, {});
  }

  /* -------------------------------------------- */

  /**
   * Test if two objects contain the same enumerable keys and values.
   * @param {object} a  The first object.
   * @param {object} b  The second object.
   * @returns {boolean}
   */
  function objectsEqual(a, b) {
    if ( (a == null) || (b == null) ) return a === b;
    if ( (getType(a) !== "Object") || (getType(b) !== "Object") ) return a === b;
    if ( Object.keys(a).length !== Object.keys(b).length ) return false;
    return Object.entries(a).every(([k, v0]) => {
      const v1 = b[k];
      const t0 = getType(v0);
      const t1 = getType(v1);
      if ( t0 !== t1 ) return false;
      if ( v0?.equals instanceof Function ) return v0.equals(v1);
      if ( t0 === "Object" ) return objectsEqual(v0, v1);
      return v0 === v1;
    });
  }

  /* -------------------------------------------- */

  /**
   * A cheap data duplication trick which is relatively robust.
   * For a subset of cases the deepClone function will offer better performance.
   * @param {Object} original   Some sort of data
   */
  function duplicate(original) {
    return JSON.parse(JSON.stringify(original));
  }

  /* -------------------------------------------- */

  /**
   * Test whether some class is a subclass of a parent.
   * Returns true if the classes are identical.
   * @param {Function} cls        The class to test
   * @param {Function} parent     Some other class which may be a parent
   * @returns {boolean}           Is the class a subclass of the parent?
   */
  function isSubclass(cls, parent) {
    if ( typeof cls !== "function" ) return false;
    if ( cls === parent ) return true;
    return parent.isPrototypeOf(cls);
  }

  /* -------------------------------------------- */

  /**
   * Search up the prototype chain and return the class that defines the given property.
   * @param {Object|Constructor} obj    A class instance or class definition which contains a property.
   *                                    If a class instance is passed the property is treated as an instance attribute.
   *                                    If a class constructor is passed the property is treated as a static attribute.
   * @param {string} property           The property name
   * @returns {Constructor}             The class that defines the property
   */
  function getDefiningClass(obj, property) {
    const isStatic = obj.hasOwnProperty("prototype");
    let target = isStatic ? obj : Object.getPrototypeOf(obj);
    while ( target ) {
      if ( target.hasOwnProperty(property) ) return isStatic ? target : target.constructor;
      target = Object.getPrototypeOf(target);
    }
  }

  /* -------------------------------------------- */

  /**
   * Encode a url-like string by replacing any characters which need encoding
   * To reverse this encoding, the native decodeURIComponent can be used on the whole encoded string, without adjustment.
   * @param {string} path     A fully-qualified URL or url component (like a relative path)
   * @return {string}         An encoded URL string
   */
  function encodeURL(path) {

    // Determine whether the path is a well-formed URL
    const url = URL.parseSafe(path);

    // If URL, remove the initial protocol
    if ( url ) path = path.replace(url.protocol, "");

    // Split and encode each URL part
    path = path.split("/").map(p => encodeURIComponent(p).replace(/'/g, "%27")).join("/");

    // Return the encoded URL
    return url ? url.protocol + path : path;
  }

  /* -------------------------------------------- */

  /**
   * Expand a flattened object to be a standard nested Object by converting all dot-notation keys to inner objects.
   * Only simple objects will be expanded. Other Object types like class instances will be retained as-is.
   * @param {object} obj      The object to expand
   * @return {object}         An expanded object
   */
  function expandObject(obj) {
    function _expand(value, depth) {
      if ( depth > 32 ) throw new Error("Maximum object expansion depth exceeded");
      if ( !value ) return value;
      if ( Array.isArray(value) ) return value.map(v => _expand(v, depth+1)); // Map arrays
      if ( value.constructor?.name !== "Object" ) return value;               // Return advanced objects directly
      const expanded = {};                                                    // Expand simple objects
      for ( let [k, v] of Object.entries(value) ) {
        setProperty(expanded, k, _expand(v, depth+1));
      }
      return expanded;
    }
    return _expand(obj, 0);
  }

  /* -------------------------------------------- */

  /**
   * Filter the contents of some source object using the structure of a template object.
   * Only keys which exist in the template are preserved in the source object.
   *
   * @param {object} source           An object which contains the data you wish to filter
   * @param {object} template         An object which contains the structure you wish to preserve
   * @param {object} [options={}]     Additional options which customize the filtration
   * @param {boolean} [options.deletionKeys=false]    Whether to keep deletion keys
   * @param {boolean} [options.templateValues=false]  Instead of keeping values from the source, instead draw values from the template
   *
   * @example Filter an object
   * ```js
   * const source = {foo: {number: 1, name: "Tim", topping: "olives"}, bar: "baz"};
   * const template = {foo: {number: 0, name: "Mit", style: "bold"}, other: 72};
   * filterObject(source, template); // {foo: {number: 1, name: "Tim"}};
   * filterObject(source, template, {templateValues: true}); // {foo: {number: 0, name: "Mit"}};
   * ```
   */
  function filterObject(source, template, {deletionKeys=false, templateValues=false}={}) {

    // Validate input
    const ts = getType(source);
    const tt = getType(template);
    if ( (ts !== "Object") || (tt !== "Object")) throw new Error("One of source or template are not Objects!");

    // Define recursive filtering function
    const _filter = function(s, t, filtered) {
      for ( let [k, v] of Object.entries(s) ) {
        let has = t.hasOwnProperty(k);
        let x = t[k];

        // Case 1 - inner object
        if ( has && (getType(v) === "Object") && (getType(x) === "Object") ) {
          filtered[k] = _filter(v, x, {});
        }

        // Case 2 - inner key
        else if ( has ) {
          filtered[k] = templateValues ? x : v;
        }

        // Case 3 - special key
        else if ( deletionKeys && k.startsWith("-=") ) {
          filtered[k] = v;
        }
      }
      return filtered;
    };

    // Begin filtering at the outer-most layer
    return _filter(source, template, {});
  }

  /* -------------------------------------------- */

  /**
   * Flatten a possibly multi-dimensional object to a one-dimensional one by converting all nested keys to dot notation
   * @param {object} obj        The object to flatten
   * @param {number} [_d=0]     Track the recursion depth to prevent overflow
   * @return {object}           A flattened object
   */
  function flattenObject(obj, _d=0) {
    const flat = {};
    if ( _d > 100 ) {
      throw new Error("Maximum depth exceeded");
    }
    for ( let [k, v] of Object.entries(obj) ) {
      let t = getType(v);
      if ( t === "Object" ) {
        if ( isEmpty$1(v) ) flat[k] = v;
        let inner = flattenObject(v, _d+1);
        for ( let [ik, iv] of Object.entries(inner) ) {
          flat[`${k}.${ik}`] = iv;
        }
      }
      else flat[k] = v;
    }
    return flat;
  }

  /* -------------------------------------------- */

  /**
   * Obtain references to the parent classes of a certain class.
   * @param {Function} cls            An class definition
   * @return {Array<typeof Object>}   An array of parent classes which the provided class extends
   */
  function getParentClasses(cls) {
    if ( typeof cls !== "function" ) {
      throw new Error("The provided class is not a type of Function");
    }
    const parents = [];
    let parent = Object.getPrototypeOf(cls);
    while ( parent ) {
      parents.push(parent);
      parent = Object.getPrototypeOf(parent);
    }
    return parents.slice(0, -2)
  }

  /* -------------------------------------------- */

  /**
   * Get the URL route for a certain path which includes a path prefix, if one is set
   * @param {string} path             The Foundry URL path
   * @param {string|null} [prefix]    A path prefix to apply
   * @returns {string}                The absolute URL path
   */
  function getRoute(path, {prefix}={}) {
    prefix = prefix === undefined ? globalThis.ROUTE_PREFIX : prefix || null;
    path = path.replace(/(^[\/]+)|([\/]+$)/g, ""); // Strip leading and trailing slashes
    let paths = [""];
    if ( prefix ) paths.push(prefix);
    paths = paths.concat([path.replace(/(^\/)|(\/$)/g, "")]);
    return paths.join("/");
  }

  /* -------------------------------------------- */

  /**
   * Learn the underlying data type of some variable. Supported identifiable types include:
   * undefined, null, number, string, boolean, function, Array, Set, Map, Promise, Error,
   * HTMLElement (client side only), Object (catchall for other object types)
   * @param {*} variable  A provided variable
   * @return {string}     The named type of the token
   */
  function getType(variable) {

    // Primitive types, handled with simple typeof check
    const typeOf = typeof variable;
    if ( typeOf !== "object" ) return typeOf;

    // Special cases of object
    if ( variable === null ) return "null";
    if ( !variable.constructor ) return "Object"; // Object with the null prototype.
    if ( variable.constructor.name === "Object" ) return "Object";  // simple objects

    // Match prototype instances
    const prototypes = [
      [Array, "Array"],
      [Set, "Set"],
      [Map, "Map"],
      [Promise, "Promise"],
      [Error, "Error"],
      [Color$1, "number"]
    ];
    if ( "HTMLElement" in globalThis ) prototypes.push([globalThis.HTMLElement, "HTMLElement"]);
    for ( const [cls, type] of prototypes ) {
      if ( variable instanceof cls ) return type;
    }

    // Unknown Object type
    return "Object";
  }

  /* -------------------------------------------- */

  /**
   * A helper function which tests whether an object has a property or nested property given a string key.
   * The method also supports arrays if the provided key is an integer index of the array.
   * The string key supports the notation a.b.c which would return true if object[a][b][c] exists
   * @param {object} object   The object to traverse
   * @param {string} key      An object property with notation a.b.c
   * @returns {boolean}       An indicator for whether the property exists
   */
  function hasProperty(object, key) {
    if ( !key || !object ) return false;
    if ( key in object ) return true;
    let target = object;
    for ( let p of key.split('.') ) {
      if ( !target || (typeof target !== "object") ) return false;
      if ( p in target ) target = target[p];
      else return false;
    }
    return true;
  }

  /* -------------------------------------------- */

  /**
   * A helper function which searches through an object to retrieve a value by a string key.
   * The method also supports arrays if the provided key is an integer index of the array.
   * The string key supports the notation a.b.c which would return object[a][b][c]
   * @param {object} object   The object to traverse
   * @param {string} key      An object property with notation a.b.c
   * @return {*}              The value of the found property
   */
  function getProperty(object, key) {
    if ( !key || !object ) return undefined;
    if ( key in object ) return object[key];
    let target = object;
    for ( let p of key.split('.') ) {
      if ( !target || (typeof target !== "object") ) return undefined;
      if ( p in target ) target = target[p];
      else return undefined;
    }
    return target;
  }

  /* -------------------------------------------- */

  /**
   * A helper function which searches through an object to assign a value using a string key
   * This string key supports the notation a.b.c which would target object[a][b][c]
   * @param {object} object   The object to update
   * @param {string} key      The string key
   * @param {*} value         The value to be assigned
   * @return {boolean}        Whether the value was changed from its previous value
   */
  function setProperty(object, key, value) {
    if ( !key ) return false;

    // Convert the key to an object reference if it contains dot notation
    let target = object;
    if ( key.indexOf('.') !== -1 ) {
      let parts = key.split('.');
      key = parts.pop();
      target = parts.reduce((o, i) => {
        if ( !o.hasOwnProperty(i) ) o[i] = {};
        return o[i];
      }, object);
    }

    // Update the target
    if ( !(key in target) || (target[key] !== value) ) {
      target[key] = value;
      return true;
    }
    return false;
  }

  /* -------------------------------------------- */

  /**
   * Invert an object by assigning its values as keys and its keys as values.
   * @param {object} obj    The original object to invert
   * @returns {object}      The inverted object with keys and values swapped
   */
  function invertObject(obj) {
    const inverted = {};
    for ( let [k, v] of Object.entries(obj) ) {
      if ( v in inverted ) throw new Error("The values of the provided object must be unique in order to invert it.");
      inverted[v] = k;
    }
    return inverted;
  }

  /* -------------------------------------------- */

  /**
   * Return whether a target version (v1) is more advanced than some other reference version (v0).
   * Supports either numeric or string version comparison with version parts separated by periods.
   * @param {number|string} v1    The target version
   * @param {number|string} v0    The reference version
   * @return {boolean}            Is v1 a more advanced version than v0?
   */
  function isNewerVersion(v1, v0) {

    // Handle numeric versions
    if ( (typeof v1 === "number") && (typeof v0 === "number") ) return v1 > v0;

    // Handle string parts
    let v1Parts = String(v1).split(".");
    let v0Parts = String(v0).split(".");

    // Iterate over version parts
    for ( let [i, p1] of v1Parts.entries() ) {
      let p0 = v0Parts[i];

      // If the prior version doesn't have a part, v1 wins
      if ( p0 === undefined ) return true;

      // If both parts are numbers, use numeric comparison to avoid cases like "12" < "5"
      if ( Number.isNumeric(p0) && Number.isNumeric(p1) ) {
        if ( Number(p1) !== Number(p0) ) return Number(p1) > Number(p0);
      }

      // Otherwise, compare as strings
      if ( p1 !== p0 ) return p1 > p0;
    }

    // If there are additional parts to v0, it is not newer
    if ( v0Parts.length > v1Parts.length ) return false;

    // If we have not returned false by now, it's either newer or the same
    return !v1Parts.equals(v0Parts);
  }

  /* -------------------------------------------- */

  /**
   * Test whether a value is empty-like; either undefined or a content-less object.
   * @param {*} value       The value to test
   * @returns {boolean}     Is the value empty-like?
   */
  function isEmpty$1(value) {
    const t = getType(value);
    switch ( t ) {
      case "undefined":
        return true;
      case "null":
        return true;
      case "Array":
        return !value.length;
      case "Object":
        return !Object.keys(value).length;
      case "Set":
      case "Map":
        return !value.size;
      default:
        return false;
    }
  }

  /* -------------------------------------------- */

  /**
   * Update a source object by replacing its keys and values with those from a target object.
   *
   * @param {object} original                           The initial object which should be updated with values from the
   *                                                    target
   * @param {object} [other={}]                         A new object whose values should replace those in the source
   * @param {object} [options={}]                       Additional options which configure the merge
   * @param {boolean} [options.insertKeys=true]         Control whether to insert new top-level objects into the resulting
   *                                                    structure which do not previously exist in the original object.
   * @param {boolean} [options.insertValues=true]       Control whether to insert new nested values into child objects in
   *                                                    the resulting structure which did not previously exist in the
   *                                                    original object.
   * @param {boolean} [options.overwrite=true]          Control whether to replace existing values in the source, or only
   *                                                    merge values which do not already exist in the original object.
   * @param {boolean} [options.recursive=true]          Control whether to merge inner-objects recursively (if true), or
   *                                                    whether to simply replace inner objects with a provided new value.
   * @param {boolean} [options.inplace=true]            Control whether to apply updates to the original object in-place
   *                                                    (if true), otherwise the original object is duplicated and the
   *                                                    copy is merged.
   * @param {boolean} [options.enforceTypes=false]      Control whether strict type checking requires that the value of a
   *                                                    key in the other object must match the data type in the original
   *                                                    data to be merged.
   * @param {boolean} [options.performDeletions=false]  Control whether to perform deletions on the original object if
   *                                                    deletion keys are present in the other object.
   * @param {number} [_d=0]                             A privately used parameter to track recursion depth.
   * @returns {object}                                  The original source object including updated, inserted, or
   *                                                    overwritten records.
   *
   * @example Control how new keys and values are added
   * ```js
   * mergeObject({k1: "v1"}, {k2: "v2"}, {insertKeys: false}); // {k1: "v1"}
   * mergeObject({k1: "v1"}, {k2: "v2"}, {insertKeys: true});  // {k1: "v1", k2: "v2"}
   * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {insertValues: false}); // {k1: {i1: "v1"}}
   * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {insertValues: true}); // {k1: {i1: "v1", i2: "v2"}}
   * ```
   *
   * @example Control how existing data is overwritten
   * ```js
   * mergeObject({k1: "v1"}, {k1: "v2"}, {overwrite: true}); // {k1: "v2"}
   * mergeObject({k1: "v1"}, {k1: "v2"}, {overwrite: false}); // {k1: "v1"}
   * ```
   *
   * @example Control whether merges are performed recursively
   * ```js
   * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {recursive: false}); // {k1: {i2: "v2"}}
   * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {recursive: true}); // {k1: {i1: "v1", i2: "v2"}}
   * ```
   *
   * @example Deleting an existing object key
   * ```js
   * mergeObject({k1: "v1", k2: "v2"}, {"-=k1": null}, {performDeletions: true});   // {k2: "v2"}
   * ```
   */
  function mergeObject(original, other={}, {
      insertKeys=true, insertValues=true, overwrite=true, recursive=true, inplace=true, enforceTypes=false,
      performDeletions=false
    }={}, _d=0) {
    other = other || {};
    if (!(original instanceof Object) || !(other instanceof Object)) {
      throw new Error("One of original or other are not Objects!");
    }
    const options = {insertKeys, insertValues, overwrite, recursive, inplace, enforceTypes, performDeletions};

    // Special handling at depth 0
    if ( _d === 0 ) {
      if ( Object.keys(other).some(k => /\./.test(k)) ) other = expandObject(other);
      if ( Object.keys(original).some(k => /\./.test(k)) ) {
        const expanded = expandObject(original);
        if ( inplace ) {
          Object.keys(original).forEach(k => delete original[k]);
          Object.assign(original, expanded);
        }
        else original = expanded;
      }
      else if ( !inplace ) original = deepClone(original);
    }

    // Iterate over the other object
    for ( let k of Object.keys(other) ) {
      const v = other[k];
      if ( original.hasOwnProperty(k) ) _mergeUpdate(original, k, v, options, _d+1);
      else _mergeInsert(original, k, v, options, _d+1);
    }
    return original;
  }

  /**
   * A helper function for merging objects when the target key does not exist in the original
   * @private
   */
  function _mergeInsert(original, k, v, {insertKeys, insertValues, performDeletions}={}, _d) {
    // Delete a key
    if ( k.startsWith("-=") && performDeletions ) {
      delete original[k.slice(2)];
      return;
    }

    const canInsert = ((_d <= 1) && insertKeys) || ((_d > 1) && insertValues);
    if ( !canInsert ) return;

    // Recursively create simple objects
    if ( v?.constructor === Object ) {
      original[k] = mergeObject({}, v, {insertKeys: true, inplace: true, performDeletions});
      return;
    }

    // Insert a key
    original[k] = v;
  }

  /**
   * A helper function for merging objects when the target key exists in the original
   * @private
   */
  function _mergeUpdate(original, k, v, {
      insertKeys, insertValues, enforceTypes, overwrite, recursive, performDeletions
    }={}, _d) {
    const x = original[k];
    const tv = getType(v);
    const tx = getType(x);

    // Recursively merge an inner object
    if ( (tv === "Object") && (tx === "Object") && recursive) {
      return mergeObject(x, v, {
        insertKeys, insertValues, overwrite, enforceTypes, performDeletions,
        inplace: true
      }, _d);
    }

    // Overwrite an existing value
    if ( overwrite ) {
      if ( (tx !== "undefined") && (tv !== tx) && enforceTypes ) {
        throw new Error(`Mismatched data types encountered during object merge.`);
      }
      original[k] = v;
    }
  }

  /* -------------------------------------------- */

  /**
   * Parse an S3 key to learn the bucket and the key prefix used for the request.
   * @param {string} key  A fully qualified key name or prefix path.
   * @returns {{bucket: string|null, keyPrefix: string}}
   */
  function parseS3URL(key) {
    const url = URL.parseSafe(key);
    if ( url ) return {
      bucket: url.host.split(".").shift(),
      keyPrefix: url.pathname.slice(1)
    };
    return {
      bucket: null,
      keyPrefix: ""
    };
  }

  /* -------------------------------------------- */

  /**
   * Generate a random alphanumeric string ID of a given requested length using `crypto.getRandomValues()`.
   * @param {number} length    The length of the random string to generate, which must be at most 16384.
   * @return {string}          A string containing random letters (A-Z, a-z) and numbers (0-9).
   */
  function randomID(length=16) {
    const chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
    const cutoff = 0x100000000 - (0x100000000 % chars.length);
    const random = new Uint32Array(length);
    do {
      crypto.getRandomValues(random);
    } while ( random.some(x => x >= cutoff) );
    let id = "";
    for ( let i = 0; i < length; i++ ) id += chars[random[i] % chars.length];
    return id;
  }

  /* -------------------------------------------- */

  /**
   * Express a timestamp as a relative string
   * @param {Date|string} timeStamp   A timestamp string or Date object to be formatted as a relative time
   * @return {string}                 A string expression for the relative time
   */
  function timeSince(timeStamp) {
    timeStamp = new Date(timeStamp);
    const now = new Date();
    const secondsPast = (now - timeStamp) / 1000;
    let since = "";

    // Format the time
    if (secondsPast < 60) {
      since = secondsPast;
      if ( since < 1 ) return game.i18n.localize("TIME.Now");
      else since = Math.round(since) + game.i18n.localize("TIME.SecondsAbbreviation");
    }
    else if (secondsPast < 3600) since = Math.round(secondsPast / 60) + game.i18n.localize("TIME.MinutesAbbreviation");
    else if (secondsPast <= 86400) since = Math.round(secondsPast / 3600) + game.i18n.localize("TIME.HoursAbbreviation");
    else {
      const hours = Math.round(secondsPast / 3600);
      const days = Math.floor(hours / 24);
      since = `${days}${game.i18n.localize("TIME.DaysAbbreviation")} ${hours % 24}${game.i18n.localize("TIME.HoursAbbreviation")}`;
    }

    // Return the string
    return game.i18n.format("TIME.Since", {since: since});
  }

  /* -------------------------------------------- */

  /**
   * Format a file size to an appropriate order of magnitude.
   * @param {number} size  The size in bytes.
   * @param {object} [options]
   * @param {number} [options.decimalPlaces=2]  The number of decimal places to round to.
   * @param {2|10} [options.base=10]            The base to use. In base 10 a kilobyte is 1000 bytes. In base 2 it is
   *                                            1024 bytes.
   * @returns {string}
   */
  function formatFileSize(size, { decimalPlaces=2, base=10 }={}) {
    const units = ["B", "kB", "MB", "GB", "TB"];
    const divisor = base === 2 ? 1024 : 1000;
    let iterations = 0;
    while ( (iterations < units.length) && (size > divisor) ) {
      size /= divisor;
      iterations++;
    }
    return `${size.toFixed(decimalPlaces)} ${units[iterations]}`;
  }

  /* -------------------------------------------- */

  /**
   * @typedef {object} ResolvedUUID
   * @property {string} uuid                      The original UUID.
   * @property {string} [type]                    The type of Document referenced. Legacy compendium UUIDs will not
   *                                              populate this field if the compendium is not active in the World.
   * @property {string} id                        The ID of the Document referenced.
   * @property {string} [primaryType]             The primary Document type of this UUID. Only present if the Document
   *                                              is embedded.
   * @property {string} [primaryId]               The primary Document ID of this UUID. Only present if the Document
   *                                              is embedded.
   * @property {DocumentCollection} [collection]  The collection that the primary Document belongs to.
   * @property {string[]} embedded                Additional Embedded Document parts.
   * @property {Document} [doc]                   An already-resolved parent Document.
   * @property {string} [documentType]            Either the document type or the parent type. Retained for backwards
   *                                              compatibility.
   * @property {string} [documentId]              Either the document id or the parent id. Retained for backwards
   *                                              compatibility.
   */

  /**
   * Parse a UUID into its constituent parts, identifying the type and ID of the referenced document.
   * The ResolvedUUID result also identifies a "primary" document which is a root-level document either in the game
   * World or in a Compendium pack which is a parent of the referenced document.
   * @param {string} uuid                  The UUID to parse.
   * @param {object} [options]             Options to configure parsing behavior.
   * @param {foundry.abstract.Document} [options.relative]  A document to resolve relative UUIDs against.
   * @returns {ResolvedUUID}               Returns the Collection, Document Type, and Document ID to resolve the parent
   *                                       document, as well as the remaining Embedded Document parts, if any.
   * @throws {Error}                       An error if the provided uuid string is incorrectly structured
   */
  function parseUuid(uuid, {relative}={}) {
    if ( !uuid ) throw new Error("A uuid string is required");
    const packs = game.packs;

    // Relative UUID
    if ( uuid.startsWith(".") && relative ) return _resolveRelativeUuid(uuid, relative);

    // Split UUID parts
    const parts = uuid.split(".");

    // Check for redirects.
    if ( game.compendiumUUIDRedirects ) {
      const node = game.compendiumUUIDRedirects.nodeAtPrefix(parts, { hasLeaves: true });
      const [redirect] = node?.[foundry.utils.StringTree.leaves];
      if ( redirect?.length ) parts.splice(0, redirect.length, ...redirect);
    }

    let id;
    let type;
    let primaryId;
    let primaryType;
    let collection;

    // Compendium Documents.
    if ( parts[0] === "Compendium" ) {
      const [, scope, packName] = parts.splice(0, 3);
      collection = packs.get(`${scope}.${packName}`);

      // Re-interpret legacy compendium UUIDs which did not explicitly include their parent document type
      if ( !(COMPENDIUM_DOCUMENT_TYPES.includes(parts[0]) || (parts[0] === "Folder")) ) {
        const type = collection?.documentName;
        parts.unshift(type);
        if ( type ) uuid = ["Compendium", scope, packName, ...parts].filterJoin(".");
      }
      [primaryType, primaryId] = parts.splice(0, 2);
    }

    // World Documents
    else {
      [primaryType, primaryId] = parts.splice(0, 2);
      collection = globalThis.db?.[primaryType] ?? CONFIG[primaryType]?.collection?.instance;
    }

    // Embedded Documents
    if ( parts.length ) {
      if ( parts.length % 2 ) throw new Error("Invalid number of embedded UUID parts");
      id = parts.at(-1);
      type = parts.at(-2);
    }

    // Primary Documents
    else {
      id = primaryId;
      type = primaryType;
      primaryId = primaryType = undefined;
    }

    // Return resolved UUID
    return {uuid, type, id, collection, embedded: parts, primaryType, primaryId,
      documentType: primaryType ?? type, documentId: primaryId ?? id};
  }

  /* -------------------------------------------- */

  /**
   * Resolve a UUID relative to another document.
   * The general-purpose algorithm for resolving relative UUIDs is as follows:
   * 1. If the number of parts is odd, remove the first part and resolve it against the current document and update the
   *    current document.
   * 2. If the number of parts is even, resolve embedded documents against the current document.
   * @param {string} uuid        The UUID to resolve.
   * @param {foundry.abstract.Document} relative  The document to resolve against.
   * @returns {ResolvedUUID}     A resolved UUID object
   * @private
   */
  function _resolveRelativeUuid(uuid, relative) {
    if ( !(relative instanceof foundry.abstract.Document) ) {
      throw new Error("A relative Document instance must be provided to _resolveRelativeUuid");
    }
    uuid = uuid.substring(1);
    const parts = uuid.split(".");
    if ( !parts.length ) throw new Error("Invalid relative UUID");
    let id;
    let type;
    let root;
    let primaryType;
    let primaryId;
    let collection;

    // Identify the root document and its collection
    const getRoot = (doc) => {
      if ( doc.parent ) parts.unshift(doc.documentName, doc.id);
      return doc.parent ? getRoot(doc.parent) : doc;
    };

    // Even-numbered parts include an explicit child document type
    if ( (parts.length % 2) === 0 ) {
      root = getRoot(relative);
      id = parts.at(-1);
      type = parts.at(-2);
      primaryType = root.documentName;
      primaryId = root.id;
      uuid = [primaryType, primaryId, ...parts].join(".");
    }

    // Relative Embedded Document
    else if ( relative.parent ) {
      root = getRoot(relative.parent);
      id = parts.at(-1);
      type = relative.documentName;
      parts.unshift(type);
      primaryType = root.documentName;
      primaryId = root.id;
      uuid = [primaryType, primaryId, ...parts].join(".");
    }

    // Relative Document
    else {
      root = relative;
      id = parts.pop();
      type = relative.documentName;
      uuid = [type, id].join(".");
    }

    // Recreate fully-qualified UUID and return the resolved result
    collection = root.pack ? root.compendium : root.collection;
    if ( root.pack ) uuid = `Compendium.${root.pack}.${uuid}`;
    return {uuid, type, id, collection, primaryType, primaryId, embedded: parts,
      documentType: primaryType ?? type, documentId: primaryId ?? id};
  }

  /**
   * Flatten nested arrays by concatenating their contents
   * @returns {any[]}    An array containing the concatenated inner values
   */
  function deepFlatten() {
    return this.reduce((acc, val) => Array.isArray(val) ? acc.concat(val.deepFlatten()) : acc.concat(val), []);
  }

  /**
   * Test element-wise equality of the values of this array against the values of another array
   * @param {any[]} other   Some other array against which to test equality
   * @returns {boolean}     Are the two arrays element-wise equal?
   */
  function equals$1(other) {
    if ( !(other instanceof Array) || (other.length !== this.length) ) return false;
    return this.every((v0, i) => {
      const v1 = other[i];
      const t0 = getType(v0);
      const t1 = getType(v1);
      if ( t0 !== t1 ) return false;
      if ( v0?.equals instanceof Function ) return v0.equals(v1);
      if ( t0 === "Object" ) return objectsEqual(v0, v1);
      return v0 === v1;
    });
  }

  /**
   * Partition an original array into two children array based on a logical test
   * Elements which test as false go into the first result while elements testing as true appear in the second
   * @param rule {Function}
   * @returns {Array}    An Array of length two whose elements are the partitioned pieces of the original
   */
  function partition(rule) {
    return this.reduce((acc, val) => {
      let test = rule(val);
      acc[Number(test)].push(val);
      return acc;
    }, [[], []]);
  }

  /**
   * Join an Array using a string separator, first filtering out any parts which return a false-y value
   * @param {string} sep    The separator string
   * @returns {string}      The joined string, filtered of any false values
   */
  function filterJoin(sep) {
    return this.filter(p => !!p).join(sep);
  }

  /**
   * Find an element within the Array and remove it from the array
   * @param {Function} find   A function to use as input to findIndex
   * @param {*} [replace]     A replacement for the spliced element
   * @returns {*|null}        The replacement element, the removed element, or null if no element was found.
   */
  function findSplice(find, replace) {
    const idx = this.findIndex(find);
    if ( idx === -1 ) return null;
    if ( replace !== undefined ) {
      this.splice(idx, 1, replace);
      return replace;
    } else {
      const item = this[idx];
      this.splice(idx, 1);
      return item;
    }
  }

  /**
   * Create and initialize an array of length n with integers from 0 to n-1
   * @memberof Array
   * @param {number} n        The desired array length
   * @param {number} [min=0]  A desired minimum number from which the created array starts
   * @returns {number[]}      An array of integers from min to min+n
   */
  function fromRange(n, min=0) {
    return Array.from({length: n}, (v, i) => i + min);
  }

  // Define primitives on the Array prototype
  Object.defineProperties(Array.prototype, {
    deepFlatten: {value: deepFlatten},
    equals: {value: equals$1},
    filterJoin: {value: filterJoin},
    findSplice: {value: findSplice},
    partition: {value: partition}
  });
  Object.defineProperties(Array,{
    fromRange: {value: fromRange}
  });

  /**
   * Test whether a Date instance is valid.
   * A valid date returns a number for its timestamp, and NaN otherwise.
   * NaN is never equal to itself.
   * @returns {boolean}
   */
  function isValid() {
    return this.getTime() === this.getTime();
  }

  /**
   * Return a standard YYYY-MM-DD string for the Date instance.
   * @returns {string}    The date in YYYY-MM-DD format
   */
  function toDateInputString() {
    const yyyy = this.getFullYear();
    const mm = (this.getMonth() + 1).paddedString(2);
    const dd = this.getDate().paddedString(2);
    return `${yyyy}-${mm}-${dd}`;
  }

  /**
   * Return a standard H:M:S.Z string for the Date instance.
   * @returns {string}    The time in H:M:S format
   */
  function toTimeInputString() {
    return this.toTimeString().split(" ")[0];
  }

  // Define primitives on the Date prototype
  Object.defineProperties(Date.prototype, {
    isValid: {value: isValid},
    toDateInputString: {value: toDateInputString},
    toTimeInputString: {value: toTimeInputString}
  });

  /**
   * √3
   * @type {number}
   */
  const SQRT3 = 1.7320508075688772;

  /**
   * √⅓
   * @type {number}
   */
  const SQRT1_3 = 0.5773502691896257;

  /**
   * Bound a number between some minimum and maximum value, inclusively.
   * @param {number} num    The current value
   * @param {number} min    The minimum allowed value
   * @param {number} max    The maximum allowed value
   * @return {number}       The clamped number
   * @memberof Math
   */
  function clamp(num, min, max) {
    return Math.min(Math.max(num, min), max);
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  function clamped(num, min, max) {
    const msg = "Math.clamped is deprecated in favor of Math.clamp.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return clamp(num, min, max);
  }

  /**
   * Linear interpolation function
   * @param {number} a   An initial value when weight is 0.
   * @param {number} b   A terminal value when weight is 1.
   * @param {number} w   A weight between 0 and 1.
   * @return {number}    The interpolated value between a and b with weight w.
   */
  function mix(a, b, w) {
    return a * (1 - w) + b * w;
  }

  /**
   * Transform an angle in degrees to be bounded within the domain [0, 360)
   * @param {number} degrees  An angle in degrees
   * @returns {number}        The same angle on the range [0, 360)
   */
  function normalizeDegrees(degrees, base) {
    const d = degrees % 360;
    if ( base !== undefined ) {
      const msg = "Math.normalizeDegrees(degrees, base) is deprecated.";
      foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
      if ( base === 360 ) return d <= 0 ? d + 360 : d;
    }
    return d < 0 ? d + 360 : d;
  }

  /**
   * Transform an angle in radians to be bounded within the domain [-PI, PI]
   * @param {number} radians  An angle in degrees
   * @return {number}         The same angle on the range [-PI, PI]
   */
  function normalizeRadians(radians) {
    const pi = Math.PI;
    const pi2 = pi * 2;
    return radians - (pi2 * Math.floor((radians + pi) / pi2));
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  function roundDecimals(number, places) {
    const msg = "Math.roundDecimals is deprecated.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    places = Math.max(Math.trunc(places), 0);
    let scl = Math.pow(10, places);
    return Math.round(number * scl) / scl;
  }

  /**
   * Transform an angle in radians to a number in degrees
   * @param {number} angle    An angle in radians
   * @return {number}         An angle in degrees
   */
  function toDegrees(angle) {
    return angle * (180 / Math.PI);
  }

  /**
   * Transform an angle in degrees to an angle in radians
   * @param {number} angle    An angle in degrees
   * @return {number}         An angle in radians
   */
  function toRadians(angle) {
    return angle * (Math.PI / 180);
  }

  /**
   * Returns the value of the oscillation between `a` and `b` at time `t`.
   * @param {number} a                              The minimium value of the oscillation
   * @param {number} b                              The maximum value of the oscillation
   * @param {number} t                              The time
   * @param {number} [p=1]                          The period (must be nonzero)
   * @param {(x: number) => number} [f=Math.cos]    The periodic function (its period must be 2π)
   * @returns {number}                              `((b - a) * (f(2π * t / p) + 1) / 2) + a`
   */
  function oscillation(a, b, t, p=1, f=Math.cos) {
    return ((b - a) * (f((2 * Math.PI * t) / p) + 1) / 2) + a;
  }

  // Define properties on the Math environment
  Object.defineProperties(Math, {
    SQRT3: {value: SQRT3},
    SQRT1_3: {value: SQRT1_3},
    clamp: {
      value: clamp,
      configurable: true,
      writable: true
    },
    clamped: {
      value: clamped,
      configurable: true,
      writable: true
    },
    mix: {
      value: mix,
      configurable: true,
      writable: true
    },
    normalizeDegrees: {
      value: normalizeDegrees,
      configurable: true,
      writable: true
    },
    normalizeRadians: {
      value: normalizeRadians,
      configurable: true,
      writable: true
    },
    roundDecimals: {
      value: roundDecimals,
      configurable: true,
      writable: true
    },
    toDegrees: {
      value: toDegrees,
      configurable: true,
      writable: true
    },
    toRadians: {
      value: toRadians,
      configurable: true,
      writable: true
    },
    oscillation: {
      value: oscillation,
      configurable: true,
      writable: true
    }
  });

  /**
   * Test for near-equivalence of two numbers within some permitted epsilon
   * @param {number} n      Some other number
   * @param {number} e      Some permitted epsilon, by default 1e-8
   * @returns {boolean}     Are the numbers almost equal?
   */
  function almostEqual(n, e=1e-8) {
    return Math.abs(this - n) < e;
  }

  /**
   * Transform a number to an ordinal string representation. i.e.
   * 1 => 1st
   * 2 => 2nd
   * 3 => 3rd
   * @returns {string}
   */
  function ordinalString() {
    const s = ["th","st","nd","rd"];
    const v = this % 100;
    return this + (s[(v-20)%10]||s[v]||s[0]);
  }

  /**
   * Return a string front-padded by zeroes to reach a certain number of numeral characters
   * @param {number} digits     The number of characters desired
   * @returns {string}          The zero-padded number
   */
  function paddedString(digits) {
    return this.toString().padStart(digits, "0");
  }

  /**
   * Return a string prefaced by the sign of the number (+) or (-)
   * @returns {string}          The signed number as a string
   */
  function signedString() {
    return (( this < 0 ) ? "" : "+") + this;
  }

  /**
   * Round a number to the closest number which is a multiple of the provided interval.
   * This is a convenience function intended to humanize issues of floating point precision.
   * The interval is treated as a standard string representation to determine the amount of decimal truncation applied.
   * @param {number} interval       The interval to round the number to the nearest multiple of
   * @param {string} [method=round] The rounding method in: round, ceil, floor
   * @returns {number}              The rounded number
   *
   * @example Round a number to the nearest step interval
   * ```js
   * let n = 17.18;
   * n.toNearest(5); // 15
   * n.toNearest(10); // 20
   * n.toNearest(10, "floor"); // 10
   * n.toNearest(10, "ceil"); // 20
   * n.toNearest(0.25); // 17.25
   * ```
   */
  function toNearest(interval=1, method="round") {
    if ( interval < 0 ) throw new Error(`Number#toNearest interval must be positive`);
    const float = Math[method](this / interval) * interval;
    const trunc = Number.isInteger(interval) ? 0 : String(interval).length - 2;
    return Number(float.toFixed(trunc));
  }

  /**
   * A faster numeric between check which avoids type coercion to the Number object.
   * Since this avoids coercion, if non-numbers are passed in unpredictable results will occur. Use with caution.
   * @param {number} a            The lower-bound
   * @param {number} b            The upper-bound
   * @param {boolean} inclusive   Include the bounding values as a true result?
   * @return {boolean}            Is the number between the two bounds?
   */
  function between(a, b, inclusive=true) {
    const min = Math.min(a, b);
    const max = Math.max(a, b);
    return inclusive ? (this >= min) && (this <= max) : (this > min) && (this < max);
  }

  /**
   * @see Number#between
   * @ignore
   */
  Number.between = function(num, a, b, inclusive=true) {
    let min = Math.min(a, b);
    let max = Math.max(a, b);
    return inclusive ? (num >= min) && (num <= max) : (num > min) && (num < max);
  };

  /**
   * Test whether a value is numeric.
   * This is the highest performing algorithm currently available, per https://jsperf.com/isnan-vs-typeof/5
   * @memberof Number
   * @param {*} n       A value to test
   * @return {boolean}  Is it a number?
   */
  function isNumeric(n) {
    if ( n instanceof Array ) return false;
    else if ( [null, ""].includes(n) ) return false;
    return +n === +n;
  }

  /**
   * Attempt to create a number from a user-provided string.
   * @memberof Number
   * @param {string|number} n   The value to convert; typically a string, but may already be a number.
   * @return {number}           The number that the string represents, or NaN if no number could be determined.
   */
  function fromString(n) {
    if ( typeof n === "number" ) return n;
    if ( (typeof n !== "string") || !n.length ) return NaN;
    n = n.replace(/\s+/g, "");
    return Number(n);
  }

  // Define properties on the Number environment
  Object.defineProperties(Number.prototype, {
    almostEqual: {value: almostEqual},
    between: {value: between},
    ordinalString: {value: ordinalString},
    paddedString: {value: paddedString},
    signedString: {value: signedString},
    toNearest: {value: toNearest}
  });
  Object.defineProperties(Number, {
    isNumeric: {value: isNumeric},
    fromString: {value: fromString}
  });

  /**
   * Return the difference of two sets.
   * @param {Set} other       Some other set to compare against
   * @returns {Set}           The difference defined as objects in this which are not present in other
   */
  function difference(other) {
    if ( !(other instanceof Set) ) throw new Error("Some other Set instance must be provided.");
    const difference = new Set();
    for ( const element of this ) {
      if ( !other.has(element) ) difference.add(element);
    }
    return difference;
  }

  /**
   * Return the symmetric difference of two sets.
   * @param {Set} other  Another set.
   * @returns {Set}      The set of elements that exist in this or other, but not both.
   */
  function symmetricDifference(other) {
    if ( !(other instanceof Set) ) throw new Error("Some other Set instance must be provided.");
    const difference = new Set(this);
    for ( const element of other ) {
      if ( difference.has(element) ) difference.delete(element);
      else difference.add(element);
    }
    return difference
  }

  /**
   * Test whether this set is equal to some other set.
   * Sets are equal if they share the same members, independent of order
   * @param {Set} other       Some other set to compare against
   * @returns {boolean}       Are the sets equal?
   */
  function equals(other) {
    if ( !(other instanceof Set ) ) return false;
    if ( other.size !== this.size ) return false;
    for ( let element of this ) {
      if ( !other.has(element) ) return false;
    }
    return true;
  }

  /**
   * Return the first value from the set.
   * @returns {*}             The first element in the set, or undefined
   */
  function first() {
    return this.values().next().value;
  }

  /**
   * Return the intersection of two sets.
   * @param {Set} other       Some other set to compare against
   * @returns {Set}           The intersection of both sets
   */
  function intersection(other) {
    const n = new Set();
    for ( let element of this ) {
      if ( other.has(element) ) n.add(element);
    }
    return n;
  }

  /**
   * Test whether this set has an intersection with another set.
   * @param {Set} other       Another set to compare against
   * @returns {boolean}       Do the sets intersect?
   */
  function intersects(other) {
    for ( let element of this ) {
      if ( other.has(element) ) return true;
    }
    return false;
  }

  /**
   * Return the union of two sets.
   * @param {Set} other  The other set.
   * @returns {Set}
   */
  function union(other) {
    if ( !(other instanceof Set) ) throw new Error("Some other Set instance must be provided.");
    const union = new Set(this);
    for ( const element of other ) union.add(element);
    return union;
  }

  /**
   * Test whether this set is a subset of some other set.
   * A set is a subset if all its members are also present in the other set.
   * @param {Set} other       Some other set that may be a subset of this one
   * @returns {boolean}       Is the other set a subset of this one?
   */
  function isSubset(other) {
    if ( !(other instanceof Set ) ) return false;
    if ( other.size < this.size ) return false;
    for ( let element of this ) {
      if ( !other.has(element) ) return false;
    }
    return true;
  }

  /**
   * Convert a set to a JSON object by mapping its contents to an array
   * @returns {Array}           The set elements as an array.
   */
  function toObject() {
    return Array.from(this);
  }

  /**
   * Test whether every element in this Set satisfies a certain test criterion.
   * @see Array#every
   * @param {function(*,number,Set): boolean} test   The test criterion to apply. Positional arguments are the value,
   * the index of iteration, and the set being tested.
   * @returns {boolean}  Does every element in the set satisfy the test criterion?
   */
  function every(test) {
    let i = 0;
    for ( const v of this ) {
      if ( !test(v, i, this) ) return false;
      i++;
    }
    return true;
  }

  /**
   * Filter this set to create a subset of elements which satisfy a certain test criterion.
   * @see Array#filter
   * @param {function(*,number,Set): boolean} test  The test criterion to apply. Positional arguments are the value,
   * the index of iteration, and the set being filtered.
   * @returns {Set}  A new Set containing only elements which satisfy the test criterion.
   */
  function filter(test) {
    const filtered = new Set();
    let i = 0;
    for ( const v of this ) {
      if ( test(v, i, this) ) filtered.add(v);
      i++;
    }
    return filtered;
  }

  /**
   * Find the first element in this set which satisfies a certain test criterion.
   * @see Array#find
   * @param {function(*,number,Set): boolean} test  The test criterion to apply. Positional arguments are the value,
   * the index of iteration, and the set being searched.
   * @returns {*|undefined}  The first element in the set which satisfies the test criterion, or undefined.
   */
  function find(test) {
    let i = 0;
    for ( const v of this ) {
      if ( test(v, i, this) ) return v;
      i++;
    }
    return undefined;
  }

  /**
   * Create a new Set where every element is modified by a provided transformation function.
   * @see Array#map
   * @param {function(*,number,Set): boolean} transform  The transformation function to apply.Positional arguments are
   * the value, the index of iteration, and the set being transformed.
   * @returns {Set}  A new Set of equal size containing transformed elements.
   */
  function map(transform) {
    const mapped = new Set();
    let i = 0;
    for ( const v of this ) {
      mapped.add(transform(v, i, this));
      i++;
    }
    if ( mapped.size !== this.size ) {
      throw new Error("The Set#map operation illegally modified the size of the set");
    }
    return mapped;
  }

  /**
   * Create a new Set with elements that are filtered and transformed by a provided reducer function.
   * @see Array#reduce
   * @param {function(*,*,number,Set): *} reducer  A reducer function applied to each value. Positional
   * arguments are the accumulator, the value, the index of iteration, and the set being reduced.
   * @param {*} accumulator       The initial value of the returned accumulator.
   * @returns {*}                 The final value of the accumulator.
   */
  function reduce(reducer, accumulator) {
    let i = 0;
    for ( const v of this ) {
      accumulator = reducer(accumulator, v, i, this);
      i++;
    }
    return accumulator;
  }

  /**
   * Test whether any element in this Set satisfies a certain test criterion.
   * @see Array#some
   * @param {function(*,number,Set): boolean} test   The test criterion to apply. Positional arguments are the value,
   * the index of iteration, and the set being tested.
   * @returns {boolean}         Does any element in the set satisfy the test criterion?
   */
  function some(test) {
    let i = 0;
    for ( const v of this ) {
      if ( test(v, i, this) ) return true;
      i++;
    }
    return false;
  }

  // Assign primitives to Set prototype
  Object.defineProperties(Set.prototype, {
    difference: {value: difference},
    symmetricDifference: {value: symmetricDifference},
    equals: {value: equals},
    every: {value: every},
    filter: {value: filter},
    find: {value: find},
    first: {value: first},
    intersection: {value: intersection},
    intersects: {value: intersects},
    union: {value: union},
    isSubset: {value: isSubset},
    map: {value: map},
    reduce: {value: reduce},
    some: {value: some},
    toObject: {value: toObject}
  });

  /**
   * Capitalize a string, transforming it's first character to a capital letter.
   * @returns {string}
   */
  function capitalize() {
    if ( !this.length ) return this;
    return this.charAt(0).toUpperCase() + this.slice(1);
  }

  /**
   * Compare this string (x) with the other string (y) by comparing each character's Unicode code point value.
   * Returns a negative Number if x < y, a positive Number if x > y, or a zero otherwise.
   * This is the same comparision function that used by Array#sort if the compare function argument is omitted.
   * The result is host/locale-independent.
   * @param {string} other    The other string to compare this string to.
   * @returns {number}
   */
  function compare(other) {
    return this < other ? -1 : this > other ? 1 : 0;
  }

  /**
   * Convert a string to Title Case where the first letter of each word is capitalized.
   * @returns {string}
   */
  function titleCase() {
    if (!this.length) return this;
    return this.toLowerCase().split(' ').reduce((parts, word) => {
      if ( !word ) return parts;
      const title = word.replace(word[0], word[0].toUpperCase());
      parts.push(title);
      return parts;
    }, []).join(' ');
  }

  /**
   * Strip any script tags which were included within a provided string.
   * @returns {string}
   */
  function stripScripts() {
    let el = document.createElement("div");
    el.innerHTML = this;
    for ( let s of el.getElementsByTagName("script") ) {
      s.parentNode.removeChild(s);
    }
    return el.innerHTML;
  }

  /**
   * Map characters to lower case ASCII
   * @type {Record<string, string>}
   */
  const CHAR_MAP = JSON.parse('{"$":"dollar","%":"percent","&":"and","<":"less",">":"greater","|":"or","¢":"cent","£":"pound","¤":"currency","¥":"yen","©":"(c)","ª":"a","®":"(r)","º":"o","À":"A","Á":"A","Â":"A","Ã":"A","Ä":"A","Å":"A","Æ":"AE","Ç":"C","È":"E","É":"E","Ê":"E","Ë":"E","Ì":"I","Í":"I","Î":"I","Ï":"I","Ð":"D","Ñ":"N","Ò":"O","Ó":"O","Ô":"O","Õ":"O","Ö":"O","Ø":"O","Ù":"U","Ú":"U","Û":"U","Ü":"U","Ý":"Y","Þ":"TH","ß":"ss","à":"a","á":"a","â":"a","ã":"a","ä":"a","å":"a","æ":"ae","ç":"c","è":"e","é":"e","ê":"e","ë":"e","ì":"i","í":"i","î":"i","ï":"i","ð":"d","ñ":"n","ò":"o","ó":"o","ô":"o","õ":"o","ö":"o","ø":"o","ù":"u","ú":"u","û":"u","ü":"u","ý":"y","þ":"th","ÿ":"y","Ā":"A","ā":"a","Ă":"A","ă":"a","Ą":"A","ą":"a","Ć":"C","ć":"c","Č":"C","č":"c","Ď":"D","ď":"d","Đ":"DJ","đ":"dj","Ē":"E","ē":"e","Ė":"E","ė":"e","Ę":"e","ę":"e","Ě":"E","ě":"e","Ğ":"G","ğ":"g","Ģ":"G","ģ":"g","Ĩ":"I","ĩ":"i","Ī":"i","ī":"i","Į":"I","į":"i","İ":"I","ı":"i","Ķ":"k","ķ":"k","Ļ":"L","ļ":"l","Ľ":"L","ľ":"l","Ł":"L","ł":"l","Ń":"N","ń":"n","Ņ":"N","ņ":"n","Ň":"N","ň":"n","Ő":"O","ő":"o","Œ":"OE","œ":"oe","Ŕ":"R","ŕ":"r","Ř":"R","ř":"r","Ś":"S","ś":"s","Ş":"S","ş":"s","Š":"S","š":"s","Ţ":"T","ţ":"t","Ť":"T","ť":"t","Ũ":"U","ũ":"u","Ū":"u","ū":"u","Ů":"U","ů":"u","Ű":"U","ű":"u","Ų":"U","ų":"u","Ŵ":"W","ŵ":"w","Ŷ":"Y","ŷ":"y","Ÿ":"Y","Ź":"Z","ź":"z","Ż":"Z","ż":"z","Ž":"Z","ž":"z","ƒ":"f","Ơ":"O","ơ":"o","Ư":"U","ư":"u","ǈ":"LJ","ǉ":"lj","ǋ":"NJ","ǌ":"nj","Ș":"S","ș":"s","Ț":"T","ț":"t","˚":"o","Ά":"A","Έ":"E","Ή":"H","Ί":"I","Ό":"O","Ύ":"Y","Ώ":"W","ΐ":"i","Α":"A","Β":"B","Γ":"G","Δ":"D","Ε":"E","Ζ":"Z","Η":"H","Θ":"8","Ι":"I","Κ":"K","Λ":"L","Μ":"M","Ν":"N","Ξ":"3","Ο":"O","Π":"P","Ρ":"R","Σ":"S","Τ":"T","Υ":"Y","Φ":"F","Χ":"X","Ψ":"PS","Ω":"W","Ϊ":"I","Ϋ":"Y","ά":"a","έ":"e","ή":"h","ί":"i","ΰ":"y","α":"a","β":"b","γ":"g","δ":"d","ε":"e","ζ":"z","η":"h","θ":"8","ι":"i","κ":"k","λ":"l","μ":"m","ν":"n","ξ":"3","ο":"o","π":"p","ρ":"r","ς":"s","σ":"s","τ":"t","υ":"y","φ":"f","χ":"x","ψ":"ps","ω":"w","ϊ":"i","ϋ":"y","ό":"o","ύ":"y","ώ":"w","Ё":"Yo","Ђ":"DJ","Є":"Ye","І":"I","Ї":"Yi","Ј":"J","Љ":"LJ","Њ":"NJ","Ћ":"C","Џ":"DZ","А":"A","Б":"B","В":"V","Г":"G","Д":"D","Е":"E","Ж":"Zh","З":"Z","И":"I","Й":"J","К":"K","Л":"L","М":"M","Н":"N","О":"O","П":"P","Р":"R","С":"S","Т":"T","У":"U","Ф":"F","Х":"H","Ц":"C","Ч":"Ch","Ш":"Sh","Щ":"Sh","Ъ":"U","Ы":"Y","Ь":"","Э":"E","Ю":"Yu","Я":"Ya","а":"a","б":"b","в":"v","г":"g","д":"d","е":"e","ж":"zh","з":"z","и":"i","й":"j","к":"k","л":"l","м":"m","н":"n","о":"o","п":"p","р":"r","с":"s","т":"t","у":"u","ф":"f","х":"h","ц":"c","ч":"ch","ш":"sh","щ":"sh","ъ":"u","ы":"y","ь":"","э":"e","ю":"yu","я":"ya","ё":"yo","ђ":"dj","є":"ye","і":"i","ї":"yi","ј":"j","љ":"lj","њ":"nj","ћ":"c","ѝ":"u","џ":"dz","Ґ":"G","ґ":"g","Ғ":"GH","ғ":"gh","Қ":"KH","қ":"kh","Ң":"NG","ң":"ng","Ү":"UE","ү":"ue","Ұ":"U","ұ":"u","Һ":"H","һ":"h","Ә":"AE","ә":"ae","Ө":"OE","ө":"oe","฿":"baht","ა":"a","ბ":"b","გ":"g","დ":"d","ე":"e","ვ":"v","ზ":"z","თ":"t","ი":"i","კ":"k","ლ":"l","მ":"m","ნ":"n","ო":"o","პ":"p","ჟ":"zh","რ":"r","ს":"s","ტ":"t","უ":"u","ფ":"f","ქ":"k","ღ":"gh","ყ":"q","შ":"sh","ჩ":"ch","ც":"ts","ძ":"dz","წ":"ts","ჭ":"ch","ხ":"kh","ჯ":"j","ჰ":"h","Ẁ":"W","ẁ":"w","Ẃ":"W","ẃ":"w","Ẅ":"W","ẅ":"w","ẞ":"SS","Ạ":"A","ạ":"a","Ả":"A","ả":"a","Ấ":"A","ấ":"a","Ầ":"A","ầ":"a","Ẩ":"A","ẩ":"a","Ẫ":"A","ẫ":"a","Ậ":"A","ậ":"a","Ắ":"A","ắ":"a","Ằ":"A","ằ":"a","Ẳ":"A","ẳ":"a","Ẵ":"A","ẵ":"a","Ặ":"A","ặ":"a","Ẹ":"E","ẹ":"e","Ẻ":"E","ẻ":"e","Ẽ":"E","ẽ":"e","Ế":"E","ế":"e","Ề":"E","ề":"e","Ể":"E","ể":"e","Ễ":"E","ễ":"e","Ệ":"E","ệ":"e","Ỉ":"I","ỉ":"i","Ị":"I","ị":"i","Ọ":"O","ọ":"o","Ỏ":"O","ỏ":"o","Ố":"O","ố":"o","Ồ":"O","ồ":"o","Ổ":"O","ổ":"o","Ỗ":"O","ỗ":"o","Ộ":"O","ộ":"o","Ớ":"O","ớ":"o","Ờ":"O","ờ":"o","Ở":"O","ở":"o","Ỡ":"O","ỡ":"o","Ợ":"O","ợ":"o","Ụ":"U","ụ":"u","Ủ":"U","ủ":"u","Ứ":"U","ứ":"u","Ừ":"U","ừ":"u","Ử":"U","ử":"u","Ữ":"U","ữ":"u","Ự":"U","ự":"u","Ỳ":"Y","ỳ":"y","Ỵ":"Y","ỵ":"y","Ỷ":"Y","ỷ":"y","Ỹ":"Y","ỹ":"y","‘":"\'","’":"\'","“":"\\\"","”":"\\\"","†":"+","•":"*","…":"...","₠":"ecu","₢":"cruzeiro","₣":"french franc","₤":"lira","₥":"mill","₦":"naira","₧":"peseta","₨":"rupee","₩":"won","₪":"new shequel","₫":"dong","€":"euro","₭":"kip","₮":"tugrik","₯":"drachma","₰":"penny","₱":"peso","₲":"guarani","₳":"austral","₴":"hryvnia","₵":"cedi","₸":"kazakhstani tenge","₹":"indian rupee","₽":"russian ruble","₿":"bitcoin","℠":"sm","™":"tm","∂":"d","∆":"delta","∑":"sum","∞":"infinity","♥":"love","元":"yuan","円":"yen","﷼":"rial"}');

  /**
   * Transform any string into an url-viable slug string
   * @param {object} [options]      Optional arguments which customize how the slugify operation is performed
   * @param {string} [options.replacement="-"]  The replacement character to separate terms, default is '-'
   * @param {boolean} [options.strict=false]    Replace all non-alphanumeric characters, or allow them? Default false
   * @param {boolean} [options.lowercase=true]  Lowercase the string.
   * @returns {string}              The slugified input string
   */
  function slugify({replacement='-', strict=false, lowercase=true}={}) {
    let slug = this.split("").reduce((result, char) => result + (CHAR_MAP[char] || char), "").trim();
    if ( lowercase ) slug = slug.toLowerCase();

    // Convert any spaces to the replacement character and de-dupe
    slug = slug.replace(new RegExp('[\\s' + replacement + ']+', 'g'), replacement);

    // If we're being strict, replace anything that is not alphanumeric
    if ( strict ) slug = slug.replace(new RegExp('[^a-zA-Z0-9' + replacement + ']', 'g'), '');
    return slug;
  }

  // Define properties on the String environment
  Object.defineProperties(String.prototype, {
    capitalize: {value: capitalize},
    compare: {value: compare},
    titleCase: {value: titleCase},
    stripScripts: {value: stripScripts},
    slugify: {value: slugify}
  });

  /**
   * Escape a given input string, prefacing special characters with backslashes for use in a regular expression
   * @param {string} string     The un-escaped input string
   * @returns {string}          The escaped string, suitable for use in regular expression
   */
  function escape$1(string) {
    return string.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
  }

  // Define properties on the RegExp environment
  Object.defineProperties(RegExp, {
    escape: {value: escape$1}
  });

  /**
   * Attempt to parse a URL without throwing an error.
   * @param {string} url  The string to parse.
   * @returns {URL|null}  The parsed URL if successful, otherwise null.
   */
  function parseSafe(url) {
    try {
      return new URL(url);
    } catch (err) {}
    return null;
  }

  // Define properties on the URL environment
  Object.defineProperties(URL, {
    parseSafe: {value: parseSafe}
  });

  /**
   * @typedef {Object} DatabaseGetOperation
   * @property {Record<string, any>} query        A query object which identifies the set of Documents retrieved
   * @property {false} [broadcast]                Get requests are never broadcast
   * @property {boolean} [index]                  Return indices only instead of full Document records
   * @property {string[]} [indexFields]           An array of field identifiers which should be indexed
   * @property {string|null} [pack=null]          A compendium collection ID which contains the Documents
   * @property {foundry.abstract.Document|null} [parent=null] A parent Document within which Documents are embedded
   * @property {string} [parentUuid]              A parent Document UUID provided when the parent instance is unavailable
   */

  /**
   * @typedef {Object} DatabaseCreateOperation
   * @property {boolean} broadcast                Whether the database operation is broadcast to other connected clients
   * @property {object[]} data                    An array of data objects from which to create Documents
   * @property {boolean} [keepId=false]           Retain the _id values of provided data instead of generating new ids
   * @property {boolean} [keepEmbeddedIds=true]   Retain the _id values of embedded document data instead of generating
   *                                              new ids for each embedded document
   * @property {number} [modifiedTime]            The timestamp when the operation was performed
   * @property {boolean} [noHook=false]           Block the dispatch of hooks related to this operation
   * @property {boolean} [render=true]            Re-render Applications whose display depends on the created Documents
   * @property {boolean} [renderSheet=false]      Render the sheet Application for any created Documents
   * @property {foundry.abstract.Document|null} [parent=null] A parent Document within which Documents are embedded
   * @property {string|null} pack                 A compendium collection ID which contains the Documents
   * @property {string|null} [parentUuid]         A parent Document UUID provided when the parent instance is unavailable
   * @property {(string|object)[]} [_result]      An alias for 'data' used internally by the server-side backend
   */

  /**
   * @typedef {Object} DatabaseUpdateOperation
   * @property {boolean} broadcast                Whether the database operation is broadcast to other connected clients
   * @property {object[]} updates                 An array of data objects used to update existing Documents.
   *                                              Each update object must contain the _id of the target Document
   * @property {boolean} [diff=true]              Difference each update object against current Document data and only use
   *                                              differential data for the update operation
   * @property {number} [modifiedTime]            The timestamp when the operation was performed
   * @property {boolean} [recursive=true]         Merge objects recursively. If false, inner objects will be replaced
   *                                              explicitly. Use with caution!
   * @property {boolean} [render=true]            Re-render Applications whose display depends on the created Documents
   * @property {boolean} [noHook=false]           Block the dispatch of hooks related to this operation
   * @property {foundry.abstract.Document|null} [parent=null] A parent Document within which Documents are embedded
   * @property {string|null} pack                 A compendium collection ID which contains the Documents
   * @property {string|null} [parentUuid]         A parent Document UUID provided when the parent instance is unavailable
   * @property {(string|object)[]} [_result]      An alias for 'updates' used internally by the server-side backend
   *
   */

  /**
   * @typedef {Object} DatabaseDeleteOperation
   * @property {boolean} broadcast                Whether the database operation is broadcast to other connected clients
   * @property {string[]} ids                     An array of Document ids which should be deleted
   * @property {boolean} [deleteAll=false]        Delete all documents in the Collection, regardless of _id
   * @property {number} [modifiedTime]            The timestamp when the operation was performed
   * @property {boolean} [noHook=false]           Block the dispatch of hooks related to this operation
   * @property {boolean} [render=true]            Re-render Applications whose display depends on the deleted Documents
   * @property {foundry.abstract.Document|null} [parent=null] A parent Document within which Documents are embedded
   * @property {string|null} pack                 A compendium collection ID which contains the Documents
   * @property {string|null} [parentUuid]         A parent Document UUID provided when the parent instance is unavailable
   * @property {(string|object)[]} [_result]      An alias for 'ids' used internally by the server-side backend
   */

  /**
   * @typedef {"get"|"create"|"update"|"delete"} DatabaseAction
   */

  /**
   * @typedef {DatabaseGetOperation|DatabaseCreateOperation|DatabaseUpdateOperation|DatabaseDeleteOperation} DatabaseOperation
   */

  /**
   * @typedef {Object} DocumentSocketRequest
   * @property {string} type                      The type of Document being transacted
   * @property {DatabaseAction} action            The action of the request
   * @property {DatabaseOperation} operation      Operation parameters for the request
   * @property {string} userId                    The id of the requesting User
   * @property {boolean} broadcast                Should the response be broadcast to other connected clients?
   */

  var _types$4 = /*#__PURE__*/Object.freeze({
    __proto__: null
  });

  /** @module validators */

  /**
   * Test whether a string is a valid 16 character UID
   * @param {string} id
   * @return {boolean}
   */
  function isValidId(id) {
    return /^[a-zA-Z0-9]{16}$/.test(id);
  }

  /**
   * Test whether a file path has an extension in a list of provided extensions
   * @param {string} path
   * @param {string[]} extensions
   * @return {boolean}
   */
  function hasFileExtension(path, extensions) {
    const xts = extensions.map(ext => `\\.${ext}`).join("|");
    const rgx = new RegExp(`(${xts})(\\?.*)?$`, "i");
    return !!path && rgx.test(path);
  }

  /**
   * Test whether a string data blob contains base64 data, optionally of a specific type or types
   * @param {string} data       The candidate string data
   * @param {string[]} [types]  An array of allowed mime types to test
   * @return {boolean}
   */
  function isBase64Data(data, types) {
    if ( types === undefined ) return /^data:([a-z]+)\/([a-z0-9]+);base64,/.test(data);
    return types.some(type => data.startsWith(`data:${type};base64,`))
  }

  /**
   * Test whether an input represents a valid 6-character color string
   * @param {string} color      The input string to test
   * @return {boolean}          Is the string a valid color?
   */
  function isColorString(color) {
    return /^#[0-9A-Fa-f]{6}$/.test(color);
  }

  /**
   * Assert that the given value parses as a valid JSON string
   * @param {string} val        The value to test
   * @return {boolean}          Is the String valid JSON?
   */
  function isJSON(val) {
    try {
      JSON.parse(val);
      return true;
    } catch(err) {
      return false;
    }
  }

  var validators = /*#__PURE__*/Object.freeze({
    __proto__: null,
    hasFileExtension: hasFileExtension,
    isBase64Data: isBase64Data,
    isColorString: isColorString,
    isJSON: isJSON,
    isValidId: isValidId
  });

  /**
   * The messages that have been logged already and should not be logged again.
   * @type {Set<string>}
   */
  const loggedCompatibilityWarnings = new Set();

  /**
   * Log a compatibility warning which is filtered based on the client's defined compatibility settings.
   * @param {string} message            The original warning or error message
   * @param {object} [options={}]       Additional options which customize logging
   * @param {number} [options.mode]           A logging level in COMPATIBILITY_MODES which overrides the configured default
   * @param {number|string} [options.since]   A version identifier since which a change was made
   * @param {number|string} [options.until]   A version identifier until which a change remains supported
   * @param {string} [options.details]        Additional details to append to the logged message
   * @param {boolean} [options.stack=true]    Include the message stack trace
   * @param {boolean} [options.once=false]    Log this the message only once?
   * @throws                            An Error if the mode is ERROR
   */
  function logCompatibilityWarning(message, {mode, since, until, details, stack=true, once=false}={}) {

    // Determine the logging mode
    const modes = COMPATIBILITY_MODES;
    const compatibility = globalThis.CONFIG?.compatibility || {
      mode: modes.WARNING,
      includePatterns: [],
      excludePatterns: []
    };
    mode ??= compatibility.mode;
    if ( mode === modes.SILENT ) return;

    // Compose the message
    since = since ? `Deprecated since Version ${since}` : null;
    until = until ? `Backwards-compatible support will be removed in Version ${until}`: null;
    message = [message, since, until, details].filterJoin("\n");

    // Filter the message by its stack trace
    const error = new Error(message);
    if ( compatibility.includePatterns.length ) {
      if ( !compatibility.includePatterns.some(rgx => rgx.test(error.message) || rgx.test(error.stack)) ) return;
    }
    if ( compatibility.excludePatterns.length ) {
      if ( compatibility.excludePatterns.some(rgx => rgx.test(error.message) || rgx.test(error.stack)) ) return;
    }

    // Log the message
    const log = !(once && loggedCompatibilityWarnings.has(error.stack));
    switch ( mode ) {
      case modes.WARNING:
        if ( log ) globalThis.logger.warn(stack ? error : error.message);
        break;
      case modes.ERROR:
        if ( log ) globalThis.logger.error(stack ? error : error.message);
        break;
      case modes.FAILURE:
        throw error;
    }
    if ( log && once ) loggedCompatibilityWarnings.add(error.stack);
  }

  /**
   * A class responsible for recording information about a validation failure.
   */
  class DataModelValidationFailure {
    /**
     * @param {any} [invalidValue]       The value that failed validation for this field.
     * @param {any} [fallback]           The value it was replaced by, if any.
     * @param {boolean} [dropped=false]  Whether the value was dropped from some parent collection.
     * @param {string} [message]         The validation error message.
     * @param {boolean} [unresolved=false]     Whether this failure was unresolved
     */
    constructor({invalidValue, fallback, dropped=false, message, unresolved=false}={}) {
      this.invalidValue = invalidValue;
      this.fallback = fallback;
      this.dropped = dropped;
      this.message = message;
      this.unresolved = unresolved;
    }

    /**
     * The value that failed validation for this field.
     * @type {any}
     */
    invalidValue;

    /**
     * The value it was replaced by, if any.
     * @type {any}
     */
    fallback;

    /**
     * Whether the value was dropped from some parent collection.
     * @type {boolean}
     */
    dropped;

    /**
     * The validation error message.
     * @type {string}
     */
    message;

    /**
     * If this field contains other fields that are validated as part of its validation, their results are recorded here.
     * @type {Record<string, DataModelValidationFailure>}
     */
    fields = {};

    /**
     * @typedef {object} ElementValidationFailure
     * @property {string|number} id                    Either the element's index or some other identifier for it.
     * @property {string} [name]                       Optionally a user-friendly name for the element.
     * @property {DataModelValidationFailure} failure  The element's validation failure.
     */

    /**
     * If this field contains a list of elements that are validated as part of its validation, their results are recorded
     * here.
     * @type {ElementValidationFailure[]}
     */
    elements = [];

    /**
     * Record whether a validation failure is unresolved.
     * This reports as true if validation for this field or any hierarchically contained field is unresolved.
     * A failure is unresolved if the value was invalid and there was no valid fallback value available.
     * @type {boolean}
     */
    unresolved;

    /* -------------------------------------------- */

    /**
     * Return this validation failure as an Error object.
     * @returns {DataModelValidationError}
     */
    asError() {
      return new DataModelValidationError(this);
    }

    /* -------------------------------------------- */

    /**
     * Whether this failure contains other sub-failures.
     * @returns {boolean}
     */
    isEmpty() {
      return isEmpty$1(this.fields) && isEmpty$1(this.elements);
    }

    /* -------------------------------------------- */

    /**
     * Return the base properties of this failure, omitting any nested failures.
     * @returns {{invalidValue: any, fallback: any, dropped: boolean, message: string}}
     */
    toObject() {
      const {invalidValue, fallback, dropped, message} = this;
      return {invalidValue, fallback, dropped, message};
    }

    /* -------------------------------------------- */

    /**
     * Represent the DataModelValidationFailure as a string.
     * @returns {string}
     */
    toString() {
      return DataModelValidationFailure.#formatString(this);
    }

    /* -------------------------------------------- */

    /**
     * Format a DataModelValidationFailure instance as a string message.
     * @param {DataModelValidationFailure} failure    The failure instance
     * @param {number} _d                             An internal depth tracker
     * @returns {string}                              The formatted failure string
     */
    static #formatString(failure, _d=0) {
      let message = failure.message ?? "";
      _d++;
      if ( !isEmpty$1(failure.fields) ) {
        message += "\n";
        const messages = [];
        for ( const [name, subFailure] of Object.entries(failure.fields) ) {
          const subMessage = DataModelValidationFailure.#formatString(subFailure, _d);
          messages.push(`${" ".repeat(2 * _d)}${name}: ${subMessage}`);
        }
        message += messages.join("\n");
      }
      if ( !isEmpty$1(failure.elements) ) {
        message += "\n";
        const messages = [];
        for ( const element of failure.elements ) {
          const subMessage = DataModelValidationFailure.#formatString(element.failure, _d);
          messages.push(`${" ".repeat(2 * _d)}${element.id}: ${subMessage}`);
        }
        message += messages.join("\n");
      }
      return message;
    }
  }

  /* -------------------------------------------- */

  /**
   * A specialised Error to indicate a model validation failure.
   * @extends {Error}
   */
  class DataModelValidationError extends Error {
    /**
     * @param {DataModelValidationFailure|string} failure  The failure that triggered this error or an error message
     * @param {...any} [params]                            Additional Error constructor parameters
     */
    constructor(failure, ...params) {
      super(failure.toString(), ...params);
      if ( failure instanceof DataModelValidationFailure ) this.#failure = failure;
    }

    /**
     * The root validation failure that triggered this error.
     * @type {DataModelValidationFailure}
     */
    #failure;

    /* -------------------------------------------- */

    /**
     * Retrieve the root failure that caused this error, or a specific sub-failure via a path.
     * @param {string} [path]  The property path to the failure.
     * @returns {DataModelValidationFailure}
     *
     * @example Retrieving a failure.
     * ```js
     * const changes = {
     *   "foo.bar": "validValue",
     *   "foo.baz": "invalidValue"
     * };
     * try {
     *   doc.validate(expandObject(changes));
     * } catch ( err ) {
     *   const failure = err.getFailure("foo.baz");
     *   console.log(failure.invalidValue); // "invalidValue"
     * }
     * ```
     */
    getFailure(path) {
      if ( !this.#failure ) return;
      if ( !path ) return this.#failure;
      let failure = this.#failure;
      for ( const p of path.split(".") ) {
        if ( !failure ) return;
        if ( !isEmpty$1(failure.fields) ) failure = failure.fields[p];
        else if ( !isEmpty$1(failure.elements) ) failure = failure.elements.find(e => e.id?.toString() === p);
      }
      return failure;
    }

    /* -------------------------------------------- */

    /**
     * Retrieve a flattened object of all the properties that failed validation as part of this error.
     * @returns {Record<string, DataModelValidationFailure>}
     *
     * @example Removing invalid changes from an update delta.
     * ```js
     * const changes = {
     *   "foo.bar": "validValue",
     *   "foo.baz": "invalidValue"
     * };
     * try {
     *   doc.validate(expandObject(changes));
     * } catch ( err ) {
     *   const failures = err.getAllFailures();
     *   if ( failures ) {
     *     for ( const prop in failures ) delete changes[prop];
     *     doc.validate(expandObject(changes));
     *   }
     * }
     * ```
     */
    getAllFailures() {
      if ( !this.#failure || this.#failure.isEmpty() ) return;
      return DataModelValidationError.#aggregateFailures(this.#failure);
    }

    /* -------------------------------------------- */

    /**
     * Log the validation error as a table.
     */
    logAsTable() {
      const failures = this.getAllFailures();
      if ( isEmpty$1(failures) ) return;
      console.table(Object.entries(failures).reduce((table, [p, failure]) => {
        table[p] = failure.toObject();
        return table;
      }, {}));
    }

    /* -------------------------------------------- */

    /**
     * Generate a nested tree view of the error as an HTML string.
     * @returns {string}
     */
    asHTML() {
      const renderFailureNode = failure => {
        if ( failure.isEmpty() ) return `<li>${failure.message || ""}</li>`;
        const nodes = [];
        for ( const [field, subFailure] of Object.entries(failure.fields) ) {
          nodes.push(`<li><details><summary>${field}</summary><ul>${renderFailureNode(subFailure)}</ul></details></li>`);
        }
        for ( const element of failure.elements ) {
          const name = element.name || element.id;
          const html = `
          <li><details><summary>${name}</summary><ul>${renderFailureNode(element.failure)}</ul></details></li>
        `;
          nodes.push(html);
        }
        return nodes.join("");
      };
      return `<ul class="summary-tree">${renderFailureNode(this.#failure)}</ul>`;
    }

    /* -------------------------------------------- */

    /**
     * Collect nested failures into an aggregate object.
     * @param {DataModelValidationFailure} failure                               The failure.
     * @returns {DataModelValidationFailure|Record<string, DataModelValidationFailure>}  Returns the failure at the leaf of the
     *                                                                           tree, otherwise an object of
     *                                                                           sub-failures.
     */
    static #aggregateFailures(failure) {
      if ( failure.isEmpty() ) return failure;
      const failures = {};
      const recordSubFailures = (field, subFailures) => {
        if ( subFailures instanceof DataModelValidationFailure ) failures[field] = subFailures;
        else {
          for ( const [k, v] of Object.entries(subFailures) ) {
            failures[`${field}.${k}`] = v;
          }
        }
      };
      for ( const [field, subFailure] of Object.entries(failure.fields) ) {
        recordSubFailures(field, DataModelValidationError.#aggregateFailures(subFailure));
      }
      for ( const element of failure.elements ) {
        recordSubFailures(element.id, DataModelValidationError.#aggregateFailures(element.failure));
      }
      return failures;
    }
  }

  var validationFailure = /*#__PURE__*/Object.freeze({
    __proto__: null,
    DataModelValidationError: DataModelValidationError,
    DataModelValidationFailure: DataModelValidationFailure
  });

  /**
   * A reusable storage concept which blends the functionality of an Array with the efficient key-based lookup of a Map.
   * This concept is reused throughout Foundry VTT where a collection of uniquely identified elements is required.
   * @template {string} K
   * @template {*} V
   * @extends {Map<K, V>}
   */
  class Collection extends Map {
    constructor(entries) {
      super(entries);
    }

    /* -------------------------------------------- */

    /**
     * Then iterating over a Collection, we should iterate over its values instead of over its entries
     * @returns {IterableIterator<V>}
     */
    [Symbol.iterator]() {
      return this.values();
    }

    /* -------------------------------------------- */

    /**
     * Return an Array of all the entry values in the Collection
     * @type {V[]}
     */
    get contents() {
      return Array.from(this.values());
    }

    /* -------------------------------------------- */

    /**
     * Find an entry in the Map using a functional condition.
     * @see {Array#find}
     * @param {function(*,number,Collection): boolean} condition  The functional condition to test. Positional
     * arguments are the value, the index of iteration, and the collection being searched.
     * @return {*}  The value, if found, otherwise undefined
     *
     * @example Create a new Collection and reference its contents
     * ```js
     * let c = new Collection([["a", "A"], ["b", "B"], ["c", "C"]]);
     * c.get("a") === c.find(entry => entry === "A"); // true
     * ```
     */
    find(condition) {
      let i = 0;
      for ( let v of this.values() ) {
        if ( condition(v, i, this) ) return v;
        i++;
      }
      return undefined;
    }

    /* -------------------------------------------- */

    /**
     * Filter the Collection, returning an Array of entries which match a functional condition.
     * @see {Array#filter}
     * @param {function(*,number,Collection): boolean} condition  The functional condition to test. Positional
     * arguments are the value, the index of iteration, and the collection being filtered.
     * @return {Array<*>}           An Array of matched values
     *
     * @example Filter the Collection for specific entries
     * ```js
     * let c = new Collection([["a", "AA"], ["b", "AB"], ["c", "CC"]]);
     * let hasA = c.filters(entry => entry.slice(0) === "A");
     * ```
     */
    filter(condition) {
      const entries = [];
      let i = 0;
      for ( let v of this.values() ) {
        if ( condition(v, i , this) ) entries.push(v);
        i++;
      }
      return entries;
    }

    /* -------------------------------------------- */

    /**
     * Apply a function to each element of the collection
     * @see Array#forEach
     * @param {function(*): void} fn       A function to apply to each element
     *
     * @example Apply a function to each value in the collection
     * ```js
     * let c = new Collection([["a", {active: false}], ["b", {active: false}], ["c", {active: false}]]);
     * c.forEach(e => e.active = true);
     * ```
     */
    forEach(fn) {
      for ( let e of this.values() ) {
        fn(e);
      }
    }

    /* -------------------------------------------- */

    /**
     * Get an element from the Collection by its key.
     * @param {string} key      The key of the entry to retrieve
     * @param {object} [options]  Additional options that affect how entries are retrieved
     * @param {boolean} [options.strict=false] Throw an Error if the requested key does not exist. Default false.
     * @return {*|undefined}    The retrieved entry value, if the key exists, otherwise undefined
     *
     * @example Get an element from the Collection by key
     * ```js
     * let c = new Collection([["a", "Alfred"], ["b", "Bob"], ["c", "Cynthia"]]);
     * c.get("a"); // "Alfred"
     * c.get("d"); // undefined
     * c.get("d", {strict: true}); // throws Error
     * ```
     */
    get(key, {strict=false}={}) {
      const entry = super.get(key);
      if ( strict && (entry === undefined) ) {
        throw new Error(`The key ${key} does not exist in the ${this.constructor.name} Collection`);
      }
      return entry;
    }

    /* -------------------------------------------- */

    /**
     * Get an entry from the Collection by name.
     * Use of this method assumes that the objects stored in the collection have a "name" attribute.
     * @param {string} name       The name of the entry to retrieve
     * @param {object} [options]  Additional options that affect how entries are retrieved
     * @param {boolean} [options.strict=false] Throw an Error if the requested name does not exist. Default false.
     * @return {*}                The retrieved entry value, if one was found, otherwise undefined
     *
     * @example Get an element from the Collection by name (if applicable)
     * ```js
     * let c = new Collection([["a", "Alfred"], ["b", "Bob"], ["c", "Cynthia"]]);
     * c.getName("Alfred"); // "Alfred"
     * c.getName("D"); // undefined
     * c.getName("D", {strict: true}); // throws Error
     * ```
     */
    getName(name, {strict=false} = {}) {
      const entry = this.find(e => e.name === name);
      if ( strict && (entry === undefined) ) {
        throw new Error(`An entry with name ${name} does not exist in the collection`);
      }
      return entry ?? undefined;
    }

    /* -------------------------------------------- */

    /**
     * Transform each element of the Collection into a new form, returning an Array of transformed values
     * @param {function(*,number,Collection): *} transformer  A transformation function applied to each entry value.
     * Positional arguments are the value, the index of iteration, and the collection being mapped.
     * @return {Array<*>}  An Array of transformed values
     */
    map(transformer) {
      const transformed = [];
      let i = 0;
      for ( let v of this.values() ) {
        transformed.push(transformer(v, i, this));
        i++;
      }
      return transformed;
    }

    /* -------------------------------------------- */

    /**
     * Reduce the Collection by applying an evaluator function and accumulating entries
     * @see {Array#reduce}
     * @param {function(*,*,number,Collection): *} reducer  A reducer function applied to each entry value. Positional
     * arguments are the accumulator, the value, the index of iteration, and the collection being reduced.
     * @param {*} initial             An initial value which accumulates with each iteration
     * @return {*}                    The accumulated result
     *
     * @example Reduce a collection to an array of transformed values
     * ```js
     * let c = new Collection([["a", "A"], ["b", "B"], ["c", "C"]]);
     * let letters = c.reduce((s, l) => {
     *   return s + l;
     * }, ""); // "ABC"
     * ```
     */
    reduce(reducer, initial) {
      let accumulator = initial;
      let i = 0;
      for ( let v of this.values() ) {
        accumulator = reducer(accumulator, v, i, this);
        i++;
      }
      return accumulator;
    }

    /* -------------------------------------------- */

    /**
     * Test whether a condition is met by some entry in the Collection.
     * @see {Array#some}
     * @param {function(*,number,Collection): boolean} condition  The functional condition to test. Positional
     * arguments are the value, the index of iteration, and the collection being tested.
     * @return {boolean}  Was the test condition passed by at least one entry?
     */
    some(condition) {
      let i = 0;
      for ( let v of this.values() ) {
        const pass = condition(v, i, this);
        i++;
        if ( pass ) return true;
      }
      return false;
    }

    /* -------------------------------------------- */

    /**
     * Convert the Collection to a primitive array of its contents.
     * @returns {object[]}  An array of contained values
     */
    toJSON() {
      return this.map(e => e.toJSON ? e.toJSON() : e);
    }
  }

  /**
   * An extension of the Collection.
   * Used for the specific task of containing embedded Document instances within a parent Document.
   */
  class EmbeddedCollection extends Collection {
    /**
     * @param {string} name           The name of this collection in the parent Document.
     * @param {DataModel} parent      The parent DataModel instance to which this collection belongs.
     * @param {object[]} sourceArray  The source data array for the collection in the parent Document data.
     */
    constructor(name, parent, sourceArray) {
      if ( typeof name !== "string" ) throw new Error("The signature of EmbeddedCollection has changed in v11.");
      super();
      Object.defineProperties(this, {
        _source: {value: sourceArray, writable: false},
        documentClass: {value: parent.constructor.hierarchy[name].model, writable: false},
        name: {value: name, writable: false},
        model: {value: parent, writable: false}
      });
    }

    /**
     * The Document implementation used to construct instances within this collection.
     * @type {typeof foundry.abstract.Document}
     */
    documentClass;

    /**
     * The name of this collection in the parent Document.
     * @type {string}
     */
    name;

    /**
     * The parent DataModel to which this EmbeddedCollection instance belongs.
     * @type {DataModel}
     */
    model;

    /**
     * Has this embedded collection been initialized as a one-time workflow?
     * @type {boolean}
     * @protected
     */
    _initialized = false;

    /**
     * The source data array from which the embedded collection is created
     * @type {object[]}
     * @private
     */
    _source;

    /**
     * Record the set of document ids where the Document was not initialized because of invalid source data
     * @type {Set<string>}
     */
    invalidDocumentIds = new Set();

    /* -------------------------------------------- */

    /**
     * Instantiate a Document for inclusion in the Collection.
     * @param {object} data       The Document data.
     * @param {DocumentConstructionContext} [context]  Document creation context.
     * @returns {Document}
     */
    createDocument(data, context={}) {
      return new this.documentClass(data, {
        ...context,
        parent: this.model,
        parentCollection: this.name,
        pack: this.model.pack
      });
    }

    /* -------------------------------------------- */

    /**
     * Initialize the EmbeddedCollection object by constructing its contained Document instances
     * @param {DocumentConstructionContext} [options]  Initialization options.
     */
    initialize(options={}) {

      // Repeat initialization
      if ( this._initialized ) {
        for ( const doc of this ) doc._initialize(options);
        return;
      }

      // First-time initialization
      this.clear();
      for ( const d of this._source ) this._initializeDocument(d, options);
      this._initialized = true;
    }

    /* -------------------------------------------- */

    /**
     * Initialize an embedded document and store it in the collection.
     * @param {object} data                    The Document data.
     * @param {DocumentConstructionContext} [context]  Context to configure Document initialization.
     * @protected
     */
    _initializeDocument(data, context) {
      if ( !data._id ) data._id = randomID(16);
      let doc;
      try {
        doc = this.createDocument(data, context);
        super.set(doc.id, doc);
      } catch(err) {
        this._handleInvalidDocument(data._id, err, context);
      }
    }

    /* -------------------------------------------- */

    /**
     * Log warnings or errors when a Document is found to be invalid.
     * @param {string} id                      The invalid Document's ID.
     * @param {Error} err                      The validation error.
     * @param {object} [options]               Options to configure invalid Document handling.
     * @param {boolean} [options.strict=true]  Whether to throw an error or only log a warning.
     * @protected
     */
    _handleInvalidDocument(id, err, {strict=true}={}) {
      const docName = this.documentClass.documentName;
      const parent = this.model;
      this.invalidDocumentIds.add(id);

      // Wrap the error with more information
      const uuid = `${parent.uuid}.${docName}.${id}`;
      const msg = `Failed to initialize ${docName} [${uuid}]:\n${err.message}`;
      const error = new Error(msg, {cause: err});

      if ( strict ) globalThis.logger.error(error);
      else globalThis.logger.warn(error);
      if ( globalThis.Hooks && strict ) {
        Hooks.onError(`${this.constructor.name}#_initializeDocument`, error, {id, documentName: docName});
      }
    }

    /* -------------------------------------------- */

    /**
     * Get an element from the EmbeddedCollection by its ID.
     * @param {string} id                        The ID of the Embedded Document to retrieve.
     * @param {object} [options]                 Additional options to configure retrieval.
     * @param {boolean} [options.strict=false]   Throw an Error if the requested Embedded Document does not exist.
     * @param {boolean} [options.invalid=false]  Allow retrieving an invalid Embedded Document.
     * @returns {Document}
     * @throws If strict is true and the Embedded Document cannot be found.
     */
    get(id, {invalid=false, strict=false}={}) {
      let result = super.get(id);
      if ( !result && invalid ) result = this.getInvalid(id, { strict: false });
      if ( !result && strict ) throw new Error(`${this.constructor.documentName} id [${id}] does not exist in the `
        + `${this.constructor.name} collection.`);
      return result;
    }

    /* ---------------------------------------- */

    /**
     * Add an item to the collection.
     * @param {string} key                           The embedded Document ID.
     * @param {Document} value                       The embedded Document instance.
     * @param {object} [options]                     Additional options to the set operation.
     * @param {boolean} [options.modifySource=true]  Whether to modify the collection's source as part of the operation.
     * */
    set(key, value, {modifySource=true, ...options}={}) {
      if ( modifySource ) this._set(key, value, options);
      return super.set(key, value);
    }

    /* -------------------------------------------- */

    /**
     * Modify the underlying source array to include the Document.
     * @param {string} key      The Document ID key.
     * @param {Document} value  The Document.
     * @protected
     */
    _set(key, value) {
      if ( this.has(key) || this.invalidDocumentIds.has(key) ) this._source.findSplice(d => d._id === key, value._source);
      else this._source.push(value._source);
    }

    /* ---------------------------------------- */

    /**
     * @param {string} key                           The embedded Document ID.
     * @param {object} [options]                     Additional options to the delete operation.
     * @param {boolean} [options.modifySource=true]  Whether to modify the collection's source as part of the operation.
     * */
    delete(key, {modifySource=true, ...options}={}) {
      if ( modifySource ) this._delete(key, options);
      return super.delete(key);
    }

    /* -------------------------------------------- */

    /**
     * Remove the value from the underlying source array.
     * @param {string} key        The Document ID key.
     * @param {object} [options]  Additional options to configure deletion behavior.
     * @protected
     */
    _delete(key, options={}) {
      if ( this.has(key) || this.invalidDocumentIds.has(key) ) this._source.findSplice(d => d._id === key);
    }

    /* ---------------------------------------- */

    /**
     * Update an EmbeddedCollection using an array of provided document data.
     * @param {DataModel[]} changes         An array of provided Document data
     * @param {object} [options={}]         Additional options which modify how the collection is updated
     */
    update(changes, options={}) {
      const updated = new Set();

      // Create or update documents within the collection
      for ( let data of changes ) {
        if ( !data._id ) data._id = randomID(16);
        this._createOrUpdate(data, options);
        updated.add(data._id);
      }

      // If the update was not recursive, remove all non-updated documents
      if ( options.recursive === false ) {
        for ( const id of this._source.map(d => d._id) ) {
          if ( !updated.has(id) ) this.delete(id, options);
        }
      }
    }

    /* -------------------------------------------- */

    /**
     * Create or update an embedded Document in this collection.
     * @param {DataModel} data       The update delta.
     * @param {object} [options={}]  Additional options which modify how the collection is updated.
     * @protected
     */
    _createOrUpdate(data, options) {
      const current = this.get(data._id);
      if ( current ) current.updateSource(data, options);
      else {
        const doc = this.createDocument(data);
        this.set(doc.id, doc);
      }
    }

    /* ---------------------------------------- */

    /**
     * Obtain a temporary Document instance for a document id which currently has invalid source data.
     * @param {string} id                      A document ID with invalid source data.
     * @param {object} [options]               Additional options to configure retrieval.
     * @param {boolean} [options.strict=true]  Throw an Error if the requested ID is not in the set of invalid IDs for
     *                                         this collection.
     * @returns {Document}                     An in-memory instance for the invalid Document
     * @throws If strict is true and the requested ID is not in the set of invalid IDs for this collection.
     */
    getInvalid(id, {strict=true}={}) {
      if ( !this.invalidDocumentIds.has(id) ) {
        if ( strict ) throw new Error(`${this.constructor.documentName} id [${id}] is not in the set of invalid ids`);
        return;
      }
      const data = this._source.find(d => d._id === id);
      return this.documentClass.fromSource(foundry.utils.deepClone(data), {parent: this.model});
    }

    /* ---------------------------------------- */

    /**
     * Convert the EmbeddedCollection to an array of simple objects.
     * @param {boolean} [source=true]     Draw data for contained Documents from the underlying data source?
     * @returns {object[]}                The extracted array of primitive objects
     */
    toObject(source=true) {
      const arr = [];
      for ( let doc of this.values() ) {
        arr.push(doc.toObject(source));
      }
      return arr;
    }

    /* -------------------------------------------- */

    /**
     * Follow-up actions to take when a database operation modifies Documents in this EmbeddedCollection.
     * @param {DatabaseAction} action                   The database action performed
     * @param {foundry.abstract.Document[]} documents   The array of modified Documents
     * @param {any[]} result                            The result of the database operation
     * @param {DatabaseOperation} operation             Database operation details
     * @param {foundry.documents.BaseUser} user         The User who performed the operation
     * @internal
     */
    _onModifyContents(action, documents, result, operation, user) {}
  }

  /**
   * This class provides a {@link Collection} wrapper around a singleton embedded Document so that it can be interacted
   * with via a common interface.
   */
  class SingletonEmbeddedCollection extends EmbeddedCollection {
    /** @inheritdoc */
    set(key, value) {
      if ( this.size && !this.has(key) ) {
        const embeddedName = this.documentClass.documentName;
        const parentName = this.model.documentName;
        throw new Error(`Cannot create singleton embedded ${embeddedName} [${key}] in parent ${parentName} `
          + `[${this.model.id}] as it already has one assigned.`);
      }
      return super.set(key, value);
    }

    /* -------------------------------------------- */

    /** @override */
    _set(key, value) {
      this.model._source[this.name] = value?._source ?? null;
    }

    /* -------------------------------------------- */

    /** @override */
    _delete(key) {
      this.model._source[this.name] = null;
    }
  }

  /**
   * An embedded collection delta contains delta source objects that can be compared against other objects inside a base
   * embedded collection, and generate new embedded Documents by combining them.
   */
  class EmbeddedCollectionDelta extends EmbeddedCollection {
    /**
     * Maintain a list of IDs that are managed by this collection delta to distinguish from those IDs that are inherited
     * from the base collection.
     * @type {Set<string>}
     */
    #managedIds = new Set();

    /* -------------------------------------------- */

    /**
     * Maintain a list of IDs that are tombstone Documents.
     * @type {Set<string>}
     */
    #tombstones = new Set();

    /* -------------------------------------------- */

    /**
     * A convenience getter to return the corresponding base collection.
     * @type {EmbeddedCollection}
     */
    get baseCollection() {
      return this.model.getBaseCollection?.(this.name);
    }

    /* -------------------------------------------- */

    /**
     * A convenience getter to return the corresponding synthetic collection.
     * @type {EmbeddedCollection}
     */
    get syntheticCollection() {
      return this.model.syntheticActor?.getEmbeddedCollection(this.name);
    }

    /* -------------------------------------------- */

    /** @override */
    createDocument(data, context={}) {
      return new this.documentClass(data, {
        ...context,
        parent: this.model.syntheticActor ?? this.model,
        parentCollection: this.name,
        pack: this.model.pack
      });
    }

    /* -------------------------------------------- */

    /** @override */
    initialize({full=false, ...options} = {}) {
      // Repeat initialization.
      if ( this._initialized && !full ) return;

      // First-time initialization.
      this.clear();
      if ( !this.baseCollection ) return;

      // Initialize the deltas.
      for ( const d of this._source ) {
        if ( d._tombstone ) this.#tombstones.add(d._id);
        else this._initializeDocument(d, options);
        this.#managedIds.add(d._id);
      }

      // Include the Documents from the base collection.
      for ( const d of this.baseCollection._source ) {
        if ( this.has(d._id) || this.isTombstone(d._id) ) continue;
        this._initializeDocument(deepClone(d), options);
      }

      this._initialized = true;
    }

    /* -------------------------------------------- */

    /** @override */
    _initializeDocument(data, context) {
      if ( !data._id ) data._id = randomID(16);
      let doc;
      if ( this.syntheticCollection ) doc = this.syntheticCollection.get(data._id);
      else {
        try {
          doc = this.createDocument(data, context);
        } catch(err) {
          this._handleInvalidDocument(data._id, err, context);
        }
      }
      if ( doc ) super.set(doc.id, doc, {modifySource: false});
    }

    /* -------------------------------------------- */

    /** @override */
    _createOrUpdate(data, options) {
      if ( options.recursive === false ) {
        if ( data._tombstone ) return this.delete(data._id);
        else if ( this.isTombstone(data._id) ) return this.set(data._id, this.createDocument(data));
      }
      else if ( this.isTombstone(data._id) || data._tombstone ) return;
      let doc = this.get(data._id);
      if ( doc ) doc.updateSource(data, options);
      else doc = this.createDocument(data);
      this.set(doc.id, doc);
    }

    /* -------------------------------------------- */

    /**
     * Determine whether a given ID is managed directly by this collection delta or inherited from the base collection.
     * @param {string} key  The Document ID.
     * @returns {boolean}
     */
    manages(key) {
      return this.#managedIds.has(key);
    }

    /* -------------------------------------------- */

    /**
     * Determine whether a given ID exists as a tombstone Document in the collection delta.
     * @param {string} key  The Document ID.
     * @returns {boolean}
     */
    isTombstone(key) {
      return this.#tombstones.has(key);
    }

    /* -------------------------------------------- */

    /**
     * Restore a Document so that it is no longer managed by the collection delta and instead inherits from the base
     * Document.
     * @param {string} id            The Document ID.
     * @returns {Promise<Document>}  The restored Document.
     */
    async restoreDocument(id) {
      const docs = await this.restoreDocuments([id]);
      return docs.shift();
    }

    /* -------------------------------------------- */

    /**
     * Restore the given Documents so that they are no longer managed by the collection delta and instead inherit directly
     * from their counterparts in the base Actor.
     * @param {string[]} ids           The IDs of the Documents to restore.
     * @returns {Promise<Document[]>}  An array of updated Document instances.
     */
    async restoreDocuments(ids) {
      if ( !this.model.syntheticActor ) return [];
      const baseActor = this.model.parent.baseActor;
      const embeddedName = this.documentClass.documentName;
      const {deltas, tombstones} = ids.reduce((obj, id) => {
        if ( !this.manages(id) ) return obj;
        const doc = baseActor.getEmbeddedCollection(this.name).get(id);
        if ( this.isTombstone(id) ) obj.tombstones.push(doc.toObject());
        else obj.deltas.push(doc.toObject());
        return obj;
      }, {deltas: [], tombstones: []});

      // For the benefit of downstream CRUD workflows, we emulate events from the perspective of the synthetic Actor.
      // Restoring an Item to the version on the base Actor is equivalent to updating that Item on the synthetic Actor
      // with the version of the Item on the base Actor.
      // Restoring an Item that has been deleted on the synthetic Actor is equivalent to creating a new Item on the
      // synthetic Actor with the contents of the version on the base Actor.
      // On the ActorDelta, those Items are removed from this collection delta so that they are once again 'linked' to the
      // base Actor's Item, as though they had never been modified from the original in the first place.

      let updated = [];
      if ( deltas.length ) {
        updated = await this.model.syntheticActor.updateEmbeddedDocuments(embeddedName, deltas, {
          diff: false, recursive: false, restoreDelta: true
        });
      }

      let created = [];
      if ( tombstones.length ) {
        created = await this.model.syntheticActor.createEmbeddedDocuments(embeddedName, tombstones, {
          keepId: true, restoreDelta: true
        });
      }

      return updated.concat(created);
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    set(key, value, options={}) {
      super.set(key, value, options);
      this.syntheticCollection?.set(key, value, options);
    }

    /* -------------------------------------------- */

    /** @override */
    _set(key, value, {restoreDelta=false}={}) {
      if ( restoreDelta ) {
        this._source.findSplice(entry => entry._id === key);
        this.#managedIds.delete(key);
        this.#tombstones.delete(key);
        return;
      }

      if ( this.manages(key) ) this._source.findSplice(d => d._id === key, value._source);
      else this._source.push(value._source);
      this.#managedIds.add(key);
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    delete(key, options={}) {
      super.delete(key, options);
      this.syntheticCollection?.delete(key, options);
    }

    /* -------------------------------------------- */

    /** @override */
    _delete(key, {restoreDelta=false}={}) {
      if ( !this.baseCollection ) return;

      // Remove the document from this collection, if it exists.
      if ( this.manages(key) ) {
        this._source.findSplice(entry => entry._id === key);
        this.#managedIds.delete(key);
        this.#tombstones.delete(key);
      }

      // If the document exists in the base collection, push a tombstone in its place.
      if ( !restoreDelta && this.baseCollection.has(key) ) {
        this._source.push({_id: key, _tombstone: true});
        this.#managedIds.add(key);
        this.#tombstones.add(key);
      }
    }
  }

  /**
   * Determine the relative orientation of three points in two-dimensional space.
   * The result is also an approximation of twice the signed area of the triangle defined by the three points.
   * This method is fast - but not robust against issues of floating point precision. Best used with integer coordinates.
   * Adapted from https://github.com/mourner/robust-predicates.
   * @param {Point} a     An endpoint of segment AB, relative to which point C is tested
   * @param {Point} b     An endpoint of segment AB, relative to which point C is tested
   * @param {Point} c     A point that is tested relative to segment AB
   * @returns {number}    The relative orientation of points A, B, and C
   *                      A positive value if the points are in counter-clockwise order (C lies to the left of AB)
   *                      A negative value if the points are in clockwise order (C lies to the right of AB)
   *                      Zero if the points A, B, and C are collinear.
   */
  function orient2dFast(a, b, c) {
    return (a.y - c.y) * (b.x - c.x) - (a.x - c.x) * (b.y - c.y);
  }

  /* -------------------------------------------- */

  /**
   * Quickly test whether the line segment AB intersects with the line segment CD.
   * This method does not determine the point of intersection, for that use lineLineIntersection.
   * @param {Point} a                   The first endpoint of segment AB
   * @param {Point} b                   The second endpoint of segment AB
   * @param {Point} c                   The first endpoint of segment CD
   * @param {Point} d                   The second endpoint of segment CD
   * @returns {boolean}                 Do the line segments intersect?
   */
  function lineSegmentIntersects(a, b, c, d) {

    // First test the orientation of A and B with respect to CD to reject collinear cases
    const xa = foundry.utils.orient2dFast(a, b, c);
    const xb = foundry.utils.orient2dFast(a, b, d);
    if ( !xa && !xb ) return false;
    const xab = (xa * xb) <= 0;

    // Also require an intersection of CD with respect to AB
    const xcd = (foundry.utils.orient2dFast(c, d, a) * foundry.utils.orient2dFast(c, d, b)) <= 0;
    return xab && xcd;
  }

  /* -------------------------------------------- */

  /**
   * @typedef {Object}                  LineIntersection
   * @property {number} x               The x-coordinate of intersection
   * @property {number} y               The y-coordinate of intersection
   * @property {number} t0              The vector distance from A to B on segment AB
   * @property {number} [t1]            The vector distance from C to D on segment CD
   */

  /**
   * An internal helper method for computing the intersection between two infinite-length lines.
   * Adapted from http://paulbourke.net/geometry/pointlineplane/.
   * @param {Point} a                   The first endpoint of segment AB
   * @param {Point} b                   The second endpoint of segment AB
   * @param {Point} c                   The first endpoint of segment CD
   * @param {Point} d                   The second endpoint of segment CD
   * @param {object} [options]          Options which affect the intersection test
   * @param {boolean} [options.t1=false]    Return the optional vector distance from C to D on CD
   * @returns {LineIntersection|null}   An intersection point, or null if no intersection occurred
   */
  function lineLineIntersection(a, b, c, d, {t1=false}={}) {

    // If either line is length 0, they cannot intersect
    if (((a.x === b.x) && (a.y === b.y)) || ((c.x === d.x) && (c.y === d.y))) return null;

    // Check denominator - avoid parallel lines where d = 0
    const dnm = ((d.y - c.y) * (b.x - a.x) - (d.x - c.x) * (b.y - a.y));
    if (dnm === 0) return null;

    // Vector distances
    const t0 = ((d.x - c.x) * (a.y - c.y) - (d.y - c.y) * (a.x - c.x)) / dnm;
    t1 = t1 ? ((b.x - a.x) * (a.y - c.y) - (b.y - a.y) * (a.x - c.x)) / dnm : undefined;

    // Return the point of intersection
    return {
      x: a.x + t0 * (b.x - a.x),
      y: a.y + t0 * (b.y - a.y),
      t0: t0,
      t1: t1
    }
  }

  /* -------------------------------------------- */

  /**
   * An internal helper method for computing the intersection between two finite line segments.
   * Adapted from http://paulbourke.net/geometry/pointlineplane/
   * @param {Point} a                   The first endpoint of segment AB
   * @param {Point} b                   The second endpoint of segment AB
   * @param {Point} c                   The first endpoint of segment CD
   * @param {Point} d                   The second endpoint of segment CD
   * @param {number} [epsilon]          A small epsilon which defines a tolerance for near-equality
   * @returns {LineIntersection|null}   An intersection point, or null if no intersection occurred
   */
  function lineSegmentIntersection(a, b, c, d, epsilon=1e-8) {

    // If either line is length 0, they cannot intersect
    if (((a.x === b.x) && (a.y === b.y)) || ((c.x === d.x) && (c.y === d.y))) return null;

    // Check denominator - avoid parallel lines where d = 0
    const dnm = ((d.y - c.y) * (b.x - a.x) - (d.x - c.x) * (b.y - a.y));
    if (dnm === 0) return null;

    // Vector distance from a
    const t0 = ((d.x - c.x) * (a.y - c.y) - (d.y - c.y) * (a.x - c.x)) / dnm;
    if ( !Number.between(t0, 0-epsilon, 1+epsilon) ) return null;

    // Vector distance from c
    const t1 = ((b.x - a.x) * (a.y - c.y) - (b.y - a.y) * (a.x - c.x)) / dnm;
    if ( !Number.between(t1, 0-epsilon, 1+epsilon) ) return null;

    // Return the point of intersection and the vector distance from both line origins
    return {
      x: a.x + t0 * (b.x - a.x),
      y: a.y + t0 * (b.y - a.y),
      t0: Math.clamp(t0, 0, 1),
      t1: Math.clamp(t1, 0, 1)
    }
  }

  /* -------------------------------------------- */

  /**
   * @typedef {Object} LineCircleIntersection
   * @property {boolean} aInside        Is point A inside the circle?
   * @property {boolean} bInside        Is point B inside the circle?
   * @property {boolean} contained      Is the segment AB contained within the circle?
   * @property {boolean} outside        Is the segment AB fully outside the circle?
   * @property {boolean} tangent        Is the segment AB tangent to the circle?
   * @property {Point[]} intersections  Intersection points: zero, one, or two
   */

  /**
   * Determine the intersection between a line segment and a circle.
   * @param {Point} a                   The first vertex of the segment
   * @param {Point} b                   The second vertex of the segment
   * @param {Point} center              The center of the circle
   * @param {number} radius             The radius of the circle
   * @param {number} epsilon            A small tolerance for floating point precision
   * @returns {LineCircleIntersection}  The intersection of the segment AB with the circle
   */
  function lineCircleIntersection(a, b, center, radius, epsilon=1e-8) {
    const r2 = Math.pow(radius, 2);
    let intersections = [];

    // Test whether endpoint A is contained
    const ar2 = Math.pow(a.x - center.x, 2) + Math.pow(a.y - center.y, 2);
    const aInside = ar2 < r2 - epsilon;

    // Test whether endpoint B is contained
    const br2 = Math.pow(b.x - center.x, 2) + Math.pow(b.y - center.y, 2);
    const bInside = br2 < r2 - epsilon;

    // Find quadratic intersection points
    const contained = aInside && bInside;
    if ( !contained ) intersections = quadraticIntersection(a, b, center, radius, epsilon);

    // Return the intersection data
    return {
      aInside,
      bInside,
      contained,
      outside: !contained && !intersections.length,
      tangent: !aInside && !bInside && intersections.length === 1,
      intersections
    };
  }

  /* -------------------------------------------- */

  /**
   * Identify the point closest to C on segment AB
   * @param {Point} c     The reference point C
   * @param {Point} a     Point A on segment AB
   * @param {Point} b     Point B on segment AB
   * @returns {Point}     The closest point to C on segment AB
   */
  function closestPointToSegment(c, a, b) {
    const dx = b.x - a.x;
    const dy = b.y - a.y;
    if (( dx === 0 ) && ( dy === 0 )) {
      throw new Error("Zero-length segment AB not supported");
    }
    const u = (((c.x - a.x) * dx) + ((c.y - a.y) * dy)) / (dx * dx + dy * dy);
    if ( u < 0 ) return a;
    if ( u > 1 ) return b;
    else return {
      x: a.x + (u * dx),
      y: a.y + (u * dy)
    }
  }

  /* -------------------------------------------- */

  /**
   * Determine the points of intersection between a line segment (p0,p1) and a circle.
   * There will be zero, one, or two intersections
   * See https://math.stackexchange.com/a/311956.
   * @param {Point} p0            The initial point of the line segment
   * @param {Point} p1            The terminal point of the line segment
   * @param {Point} center        The center of the circle
   * @param {number} radius       The radius of the circle
   * @param {number} [epsilon=0]  A small tolerance for floating point precision
   */
  function quadraticIntersection(p0, p1, center, radius, epsilon=0) {
    const dx = p1.x - p0.x;
    const dy = p1.y - p0.y;

    // Quadratic terms where at^2 + bt + c = 0
    const a = Math.pow(dx, 2) + Math.pow(dy, 2);
    const b = (2 * dx * (p0.x - center.x)) + (2 * dy * (p0.y - center.y));
    const c = Math.pow(p0.x - center.x, 2) + Math.pow(p0.y - center.y, 2) - Math.pow(radius, 2);

    // Discriminant
    let disc2 = Math.pow(b, 2) - (4 * a * c);
    if ( disc2.almostEqual(0) ) disc2 = 0; // segment endpoint touches the circle; 1 intersection
    else if ( disc2 < 0 ) return []; // no intersections

    // Roots
    const disc = Math.sqrt(disc2);
    const t1 = (-b - disc) / (2 * a);

    // If t1 hits (between 0 and 1) it indicates an "entry"
    const intersections = [];
    if ( t1.between(0-epsilon, 1+epsilon) ) {
      intersections.push({
        x: p0.x + (dx * t1),
        y: p0.y + (dy * t1)
      });
    }
    if ( !disc2 ) return intersections; // 1 intersection

    // If t2 hits (between 0 and 1) it indicates an "exit"
    const t2 = (-b + disc) / (2 * a);
    if ( t2.between(0-epsilon, 1+epsilon) ) {
      intersections.push({
        x: p0.x + (dx * t2),
        y: p0.y + (dy * t2)
      });
    }
    return intersections;
  }

  /* -------------------------------------------- */

  /**
   * Calculate the centroid non-self-intersecting closed polygon.
   * See https://en.wikipedia.org/wiki/Centroid#Of_a_polygon.
   * @param {Point[]|number[]} points    The points of the polygon
   * @returns {Point}                    The centroid of the polygon
   */
  function polygonCentroid(points) {
    const n = points.length;
    if ( n === 0 ) return {x: 0, y: 0};
    let x = 0;
    let y = 0;
    let a = 0;
    if ( typeof points[0] === "number" ) {
      let x0 = points[n - 2];
      let y0 = points[n - 1];
      for ( let i = 0; i < n; i += 2 ) {
        const x1 = points[i];
        const y1 = points[i + 1];
        const z = (x0 * y1) - (x1 * y0);
        x += (x0 + x1) * z;
        y += (y0 + y1) * z;
        x0 = x1;
        y0 = y1;
        a += z;
      }
    } else {
      let {x: x0, y: y0} = points[n - 1];
      for ( let i = 0; i < n; i++ ) {
        const {x: x1, y: y1} = points[i];
        const z = (x0 * y1) - (x1 * y0);
        x += (x0 + x1) * z;
        y += (y0 + y1) * z;
        x0 = x1;
        y0 = y1;
        a += z;
      }
    }
    a *= 3;
    x /= a;
    y /= a;
    return {x, y};
  }

  /* -------------------------------------------- */

  /**
   * Test whether the circle given by the center and radius intersects the path (open or closed).
   * @param {Point[]|number[]} points    The points of the path
   * @param {boolean} close              If true, the edge from the last to the first point is tested
   * @param {Point} center               The center of the circle
   * @param {number} radius              The radius of the circle
   * @returns {boolean}                  Does the circle intersect the path?
   */
  function pathCircleIntersects(points, close, center, radius) {
    const n = points.length;
    if ( n === 0 ) return false;
    const {x: cx, y: cy} = center;
    const rr = radius * radius;
    let i;
    let x0;
    let y0;
    if ( typeof points[0] === "number" ) {
      if ( close ) {
        i = 0;
        x0 = points[n - 2];
        y0 = points[n - 1];
      } else {
        i = 2;
        x0 = points[0];
        y0 = points[1];
      }
      for ( ; i < n; i += 2 ) {
        const x1 = points[i];
        const y1 = points[i + 1];
        let dx = cx - x0;
        let dy = cy - y0;
        const nx = x1 - x0;
        const ny = y1 - y0;
        const t = Math.clamp(((dx * nx) + (dy * ny)) / ((nx * nx) + (ny * ny)), 0, 1);
        dx = (t * nx) - dx;
        dy = (t * ny) - dy;
        if ( (dx * dx) + (dy * dy) <= rr ) return true;
        x0 = x1;
        y0 = y1;
      }
    } else {
      if ( close ) {
        i = 0;
        ({x: x0, y: y0} = points[n - 1]);
      } else {
        i = 1;
        ({x: x0, y: y0} = points[0]);
      }
      for ( ; i < n; i++ ) {
        const {x: x2, y: y2} = points[i];
        let dx = cx - x0;
        let dy = cy - y0;
        const nx = x1 - x0;
        const ny = y1 - y0;
        const t = Math.clamp(((dx * nx) + (dy * ny)) / ((nx * nx) + (ny * ny)), 0, 1);
        dx = (t * nx) - dx;
        dy = (t * ny) - dy;
        if ( (dx * dx) + (dy * dy) <= rr ) return true;
        x0 = x2;
        y0 = y2;
      }
    }
    return false;
  }

  /* -------------------------------------------- */

  /**
   * Test whether two circles (with position and radius) intersect.
   * @param {number} x0    x center coordinate of circle A.
   * @param {number} y0    y center coordinate of circle A.
   * @param {number} r0    radius of circle A.
   * @param {number} x1    x center coordinate of circle B.
   * @param {number} y1    y center coordinate of circle B.
   * @param {number} r1    radius of circle B.
   * @returns {boolean}    True if the two circles intersect, false otherwise.
   */
  function circleCircleIntersects(x0, y0, r0, x1, y1, r1) {
    return Math.hypot(x0 - x1, y0 - y1) <= (r0 + r1);
  }

  /**
   * A wrapper method around `fetch` that attaches an AbortController signal to the `fetch` call for clean timeouts
   * @see https://developer.mozilla.org/en-US/docs/Web/API/AbortSignal#aborting_a_fetch_with_timeout_or_explicit_abort
   * @param {string} url            The URL to make the Request to
   * @param {Object} data           The data of the Request
   * @param {number|null} timeoutMs How long to wait for a Response before cleanly aborting.
   *                                If null, no timeout is applied
   * @param {function} onTimeout    A method to invoke if and when the timeout is reached
   * @return {Promise<Response>}
   * @throws {HttpError}
   */
  async function fetchWithTimeout(url, data = {}, {timeoutMs=30000, onTimeout = () => {}} = {}) {
    const controller = new AbortController();
    data.signal = controller.signal;
    let timedOut = false;
    const enforceTimeout = timeoutMs !== null;

    // Enforce a timeout
    let timeout;
    if ( enforceTimeout ) {
      timeout = setTimeout(() => {
        timedOut = true;
        controller.abort();
        onTimeout();
      }, timeoutMs);
    }

    // Attempt the request
    let response;
    try {
      response = await fetch(url, data);
    } catch(err) {
      if ( timedOut ) {
        const timeoutS = Math.round(timeoutMs / 1000);
        const msg = game.i18n
          ? game.i18n.format("SETUP.ErrorTimeout", { url, timeout: timeoutS })
          : `The request to ${url} timed out after ${timeoutS}s.`;
        throw new HttpError("Timed Out", 408, msg);
      }
      throw err;
    } finally {
      if ( enforceTimeout ) clearTimeout(timeout);
    }

    // Return the response
    if ( !response.ok && (response.type !== "opaqueredirect") ) {
      const responseBody = response.body ? await response.text() : "";
      throw new HttpError(response.statusText, response.status, responseBody);
    }
    return response;
  }

  /* ----------------------------------------- */

  /**
   * A small wrapper that automatically asks for JSON with a Timeout
   * @param {string} url          The URL to make the Request to
   * @param {Object} data         The data of the Request
   * @param {int} timeoutMs       How long to wait for a Response before cleanly aborting
   * @param {function} onTimeout  A method to invoke if and when the timeout is reached
   * @returns {Promise<*>}
   */
  async function fetchJsonWithTimeout(url, data = {}, {timeoutMs=30000, onTimeout = () => {}} = {}) {
    let response = await fetchWithTimeout(url, data, {timeoutMs: timeoutMs, onTimeout: onTimeout});
    return response.json();
  }

  /* ----------------------------------------- */

  /**
   * Represents an HTTP Error when a non-OK response is returned by Fetch
   * @extends {Error}
   */
  class HttpError extends Error {
    constructor(statusText, code, displayMessage="") {
      super(statusText);
      this.code = code;
      this.displayMessage = displayMessage;
    }

    /* -------------------------------------------- */

    /** @override */
    toString() {
      return this.displayMessage;
    }
  }

  /**
   * @typedef {import("../types.mjs").Constructor} Constructor
   */

  /**
   * @callback EmittedEventListener
   * @param {Event} event         The emitted event
   * @returns {any}
   */

  /**
   * Augment a base class with EventEmitter behavior.
   * @template {Constructor} BaseClass
   * @param {BaseClass} BaseClass         Some base class augmented with event emitter functionality
   */
  function EventEmitterMixin(BaseClass) {
    /**
     * A mixin class which implements the behavior of EventTarget.
     * This is useful in cases where a class wants EventTarget-like behavior but needs to extend some other class.
     * @see https://developer.mozilla.org/en-US/docs/Web/API/EventTarget
     */
    class EventEmitter extends BaseClass {

      /**
       * An array of event types which are valid for this class.
       * @type {string[]}
       */
      static emittedEvents = [];

      /**
       * A mapping of registered events.
       * @type {Record<string, Map<EmittedEventListener, {fn: EmittedEventListener, once: boolean}>>}
       */
      #events = {};

      /* -------------------------------------------- */

      /**
       * Add a new event listener for a certain type of event.
       * @see https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/addEventListener
       * @param {string} type                     The type of event being registered for
       * @param {EmittedEventListener} listener   The listener function called when the event occurs
       * @param {object} [options={}]             Options which configure the event listener
       * @param {boolean} [options.once=false]      Should the event only be responded to once and then removed
       */
      addEventListener(type, listener, {once = false} = {}) {
        if ( !this.constructor.emittedEvents.includes(type) ) {
          throw new Error(`"${type}" is not a supported event of the ${this.constructor.name} class`);
        }
        this.#events[type] ||= new Map();
        this.#events[type].set(listener, {fn: listener, once});
      }

      /* -------------------------------------------- */

      /**
       * Remove an event listener for a certain type of event.
       * @see https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/removeEventListener
       * @param {string} type                     The type of event being removed
       * @param {EmittedEventListener} listener   The listener function being removed
       */
      removeEventListener(type, listener) {
        this.#events[type]?.delete(listener);
      }

      /* -------------------------------------------- */

      /**
       * Dispatch an event on this target.
       * @see https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/dispatchEvent
       * @param {Event} event                     The Event to dispatch
       * @returns {boolean}                       Was default behavior for the event prevented?
       */
      dispatchEvent(event) {
        if ( !(event instanceof Event) ) {
          throw new Error("EventEmitter#dispatchEvent must be provided an Event instance");
        }
        if ( !this.constructor.emittedEvents.includes(event?.type) ) {
          throw new Error(`"${event.type}" is not a supported event of the ${this.constructor.name} class`);
        }
        const listeners = this.#events[event.type];
        if ( !listeners ) return true;

        // Extend and configure the Event
        Object.defineProperties(event, {
          target: {value: this},
          stopPropagation: {value: function() {
            event.propagationStopped = true;
            Event.prototype.stopPropagation.call(this);
          }},
          stopImmediatePropagation: {value: function() {
            event.propagationStopped = true;
            Event.prototype.stopImmediatePropagation.call(this);
          }}
        });

        // Call registered listeners
        for ( const listener of listeners.values() ) {
          listener.fn(event);
          if ( listener.once ) this.removeEventListener(event.type, listener.fn);
          if ( event.propagationStopped ) break;
        }
        return event.defaultPrevented;
      }
    }
    return EventEmitter;
  }

  /**
   * Stores a map of objects with weak references to the keys, allowing them to be garbage collected. Both keys and values
   * can be iterated over, unlike a WeakMap.
   */
  class IterableWeakMap extends WeakMap {
    /**
     * @typedef {object} IterableWeakMapHeldValue
     * @property {Set<WeakRef<any>>} set  The set to be cleaned.
     * @property {WeakRef<any>} ref       The ref to remove.
     */

    /**
     * @typedef {object} IterableWeakMapValue
     * @property {any} value         The value.
     * @property {WeakRef<any>} ref  The weak ref of the key.
     */

    /**
     * A set of weak refs to the map's keys, allowing enumeration.
     * @type {Set<WeakRef<any>>}
     */
    #refs = new Set();

    /**
     * A FinalizationRegistry instance to clean up the ref set when objects are garbage collected.
     * @type {FinalizationRegistry<IterableWeakMapHeldValue>}
     */
    #finalizer = new FinalizationRegistry(IterableWeakMap.#cleanup);

    /**
     * @param {Iterable<[any, any]>} [entries]  The initial entries.
     */
    constructor(entries=[]) {
      super();
      for ( const [key, value] of entries ) this.set(key, value);
    }

    /* -------------------------------------------- */

    /**
     * Clean up the corresponding ref in the set when its value is garbage collected.
     * @param {IterableWeakMapHeldValue} heldValue  The value held by the finalizer.
     */
    static #cleanup({ set, ref }) {
      set.delete(ref);
    }

    /* -------------------------------------------- */

    /**
     * Remove a key from the map.
     * @param {any} key  The key to remove.
     * @returns {boolean}
     */
    delete(key) {
      const entry = super.get(key);
      if ( !entry ) return false;
      super.delete(key);
      this.#refs.delete(entry.ref);
      this.#finalizer.unregister(key);
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Retrieve a value from the map.
     * @param {any} key  The value's key.
     * @returns {any}
     */
    get(key) {
      const entry = super.get(key);
      return entry && entry.value;
    }

    /* -------------------------------------------- */

    /**
     * Place a value in the map.
     * @param {any} key    The key.
     * @param {any} value  The value.
     * @returns {IterableWeakMap}
     */
    set(key, value) {
      const entry = super.get(key);
      if ( entry ) this.#refs.delete(entry.ref);
      const ref = new WeakRef(key);
      super.set(key, { value, ref });
      this.#refs.add(ref);
      this.#finalizer.register(key, { ref, set: this.#refs }, key);
      return this;
    }

    /* -------------------------------------------- */

    /**
     * Clear all values from the map.
     */
    clear() {
      for ( const ref of this.#refs ) {
        const key = ref.deref();
        if ( key ) this.delete(key);
        else this.#refs.delete(ref);
      }
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the entries.
     * @returns {Generator<[any, any], void, any>}
     */
    *[Symbol.iterator]() {
      for ( const ref of this.#refs ) {
        const key = ref.deref();
        if ( !key ) continue;
        const { value } = super.get(key);
        yield [key, value];
      }
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the entries.
     * @returns {Generator<[any, any], void, any>}
     */
    entries() {
      return this[Symbol.iterator]();
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the keys.
     * @returns {Generator<any, void, any>}
     */
    *keys() {
      for ( const [key] of this ) yield key;
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the values.
     * @returns {Generator<any, void, any>}
     */
    *values() {
      for ( const [, value] of this ) yield value;
    }
  }

  /**
   * Stores a set of objects with weak references to them, allowing them to be garbage collected. Can be iterated over,
   * unlike a WeakSet.
   */
  class IterableWeakSet extends WeakSet {
    /**
     * The backing iterable weak map.
     * @type {IterableWeakMap<any, any>}
     */
    #map = new IterableWeakMap();

    /**
     * @param {Iterable<any>} [entries]  The initial entries.
     */
    constructor(entries=[]) {
      super();
      for ( const entry of entries ) this.add(entry);
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the values.
     * @returns {Generator<any, void, any>}
     */
    [Symbol.iterator]() {
      return this.values();
    }

    /* -------------------------------------------- */

    /**
     * Add a value to the set.
     * @param {any} value  The value to add.
     * @returns {IterableWeakSet}
     */
    add(value) {
      this.#map.set(value, value);
      return this;
    }

    /* -------------------------------------------- */

    /**
     * Delete a value from the set.
     * @param {any} value  The value to delete.
     * @returns {boolean}
     */
    delete(value) {
      return this.#map.delete(value);
    }

    /* -------------------------------------------- */

    /**
     * Whether this set contains the given value.
     * @param {any} value  The value to test.
     * @returns {boolean}
     */
    has(value) {
      return this.#map.has(value);
    }

    /* -------------------------------------------- */

    /**
     * Enumerate the collection.
     * @returns {Generator<any, void, any>}
     */
    values() {
      return this.#map.values();
    }

    /* -------------------------------------------- */

    /**
     * Clear all values from the set.
     */
    clear() {
      this.#map.clear();
    }
  }

  /**
   * A simple Semaphore implementation which provides a limited queue for ensuring proper concurrency.
   * @param {number} [max=1]    The maximum number of tasks which are allowed concurrently.
   *
   * @example Using a Semaphore
   * ```js
   * // Some async function that takes time to execute
   * function fn(x) {
   *   return new Promise(resolve => {
   *     setTimeout(() => {
   *       console.log(x);
   *       resolve(x);
   *     }, 1000));
   *   }
   * };
   *
   * // Create a Semaphore and add many concurrent tasks
   * const semaphore = new Semaphore(1);
   * for ( let i of Array.fromRange(100) ) {
   *   semaphore.add(fn, i);
   * }
   * ```
   */
  class Semaphore {
    constructor(max=1) {

      /**
       * The maximum number of tasks which can be simultaneously attempted.
       * @type {number}
       */
      this.max = max;

      /**
       * A queue of pending function signatures
       * @type {Array<Array<Function|*>>}
       * @private
       */
      this._queue = [];

      /**
       * The number of tasks which are currently underway
       * @type {number}
       * @private
       */
      this._active = 0;
    }

    /**
     * The number of pending tasks remaining in the queue
     * @type {number}
     */
    get remaining() {
      return this._queue.length;
    }

    /**
     * The number of actively executing tasks
     * @type {number}
     */
    get active() {
      return this._active;
    }

    /**
     * Add a new tasks to the managed queue
     * @param {Function} fn     A callable function
     * @param {...*} [args]     Function arguments
     * @returns {Promise}       A promise that resolves once the added function is executed
     */
    add(fn, ...args) {
      return new Promise((resolve, reject) => {
        this._queue.push([fn, args, resolve, reject]);
        return this._try();
      });
    }

    /**
     * Abandon any tasks which have not yet concluded
     */
    clear() {
      this._queue = [];
    }

    /**
     * Attempt to perform a task from the queue.
     * If all workers are busy, do nothing.
     * If successful, try again.
     * @private
     */
    async _try() {
      if ( (this.active === this.max) || !this.remaining ) return false;

      // Obtain the next task from the queue
      const next = this._queue.shift();
      if ( !next ) return;
      this._active += 1;

      // Try and execute it, resolving its promise
      const [fn, args, resolve, reject] = next;
      try {
        const r = await fn(...args);
        resolve(r);
      }
      catch(err) {
        reject(err);
      }

      // Try the next function in the queue
      this._active -= 1;
      return this._try();
    }
  }

  /**
   * Create a new BitMask instance.
   * @param {Record<string, boolean>} [states=null] An object containing valid states and their corresponding initial boolean values (default is null).
   */
  class BitMask extends Number {
    constructor(states=null) {
      super();
      this.#generateValidStates(states);
      this.#generateEnum();
      this.#value = this.#computeValue(states);
    }

    /**
     * The real value behind the bitmask instance.
     * @type {number}
     */
    #value;

    /**
     * The structure of valid states and their associated values.
     * @type {Map<string, number>}
     */
    #validStates;

    /**
     * The enum associated with this structure.
     * @type {Record<string, string>}
     * @readonly
     */
    states;

    /* -------------------------------------------- */
    /*  Internals                                   */
    /* -------------------------------------------- */

    /**
     * Generates the valid states and their associated values.
     * @param {Record<string, boolean>} [states=null] The structure defining the valid states and their associated values.
     */
    #generateValidStates(states) {
      this.#validStates = new Map();
      let bitIndex = 0;
      for ( const state of Object.keys(states || {}) ) {
        if ( bitIndex >= 32 ) throw new Error("A bitmask can't handle more than 32 states");
        this.#validStates.set(state, 1 << bitIndex++);
      }
    }

    /* -------------------------------------------- */

    /**
     * Generates an enum based on the provided valid states.
     */
    #generateEnum() {
      this.states = {};
      for ( const state of this.#validStates.keys() ) this.states[state] = state;
      Object.freeze(this.states);
    }

    /* -------------------------------------------- */

    /**
     * Calculate the default value of the bitmask based on the initial states
     * @param {Record<string, boolean>} [initialStates={}] The structure defining the valid states and their associated values.
     * @returns {number}
     */
    #computeValue(initialStates={}) {
      let defaultValue = 0;
      for ( const state in initialStates ) {
        if ( !initialStates.hasOwnProperty(state) ) continue;
        this.#checkState(state);
        if ( initialStates[state] ) defaultValue |= this.#validStates.get(state);
      }
      return defaultValue;
    }

    /* -------------------------------------------- */

    /**
     * Checks a state and throws an error if it doesn't exist.
     * @param {string} state   Name of the state to check.
     */
    #checkState(state) {
      if ( !this.#validStates.has(state) ) {
        throw new Error(`${state} is an invalid state for this BitMask instance: ${this.toJSON()}`);
      }
    }

    /* -------------------------------------------- */
    /*  Properties                                  */
    /* -------------------------------------------- */

    /**
     * True if this bitmask is empty (no active states).
     * @type {boolean}
     */
    get isEmpty() {
      return this.#value === 0;
    }

    /* -------------------------------------------- */
    /*  Methods for Handling states                 */
    /* -------------------------------------------- */

    /**
     * Check if a specific state is active.
     * @param {string} state The state to check.
     * @returns {boolean} True if the state is active, false otherwise.
     */
    hasState(state) {
      return (this.#value & this.#validStates.get(state)) !== 0;
    }

    /* -------------------------------------------- */

    /**
     * Add a state to the bitmask.
     * @param {string} state The state to add.
     * @throws {Error} Throws an error if the provided state is not valid.
     */
    addState(state) {
      this.#checkState(state);
      this.#value |= this.#validStates.get(state);
    }

    /* -------------------------------------------- */

    /**
     * Remove a state from the bitmask.
     * @param {string} state The state to remove.
     * @throws {Error} Throws an error if the provided state is not valid.
     */
    removeState(state) {
      this.#checkState(state);
      this.#value &= ~this.#validStates.get(state);
    }

    /* -------------------------------------------- */

    /**
     * Toggle the state of a specific state in the bitmask.
     * @param {string} state The state to toggle.
     * @param {boolean} [enabled] Toggle on (true) or off (false)? If undefined, the state is switched automatically.
     * @throws {Error} Throws an error if the provided state is not valid.
     */
    toggleState(state, enabled) {
      this.#checkState(state);
      if ( enabled === undefined ) return (this.#value ^= this.#validStates.get(state));
      if ( enabled ) this.addState(state);
      else this.removeState(state);
    }

    /* -------------------------------------------- */

    /**
     * Clear the bitmask, setting all states to inactive.
     */
    clear() {
      this.#value = 0;
    }

    /* -------------------------------------------- */
    /*  bitmask representations                     */
    /* -------------------------------------------- */

    /**
     * Get the current value of the bitmask.
     * @returns {number} The current value of the bitmask.
     */
    valueOf() {
      return this.#value;
    }

    /* -------------------------------------------- */

    /**
     * Get a string representation of the bitmask in binary format.
     * @returns {string} The string representation of the bitmask.
     */
    toString() {
      return String(this.#value.toString(2)).padStart(this.#validStates.size, '0');
    }

    /* -------------------------------------------- */

    /**
     * Checks if two bitmasks structures are compatible (the same valid states).
     * @param {BitMask} otherBitMask The bitmask structure to compare with.
     * @returns {boolean} True if the two bitmasks have the same structure, false otherwise.
     */
    isCompatible(otherBitMask) {
      const states1 = Array.from(this.#validStates.keys()).sort().join(',');
      const states2 = Array.from(otherBitMask.#validStates.keys()).sort().join(',');
      return states1 === states2;
    }

    /* -------------------------------------------- */

    /**
     * Serializes the bitmask to a JSON string.
     * @returns {string} The JSON string representing the bitmask.
     */
    toJSON() {
      return JSON.stringify(this.toObject());
    }

    /* -------------------------------------------- */

    /**
     * Creates a new BitMask instance from a JSON string.
     * @param {string} jsonString The JSON string representing the bitmask.
     * @returns {BitMask} A new BitMask instance created from the JSON string.
     */
    static fromJSON(jsonString) {
      const data = JSON.parse(jsonString);
      return new BitMask(data);
    }

    /* -------------------------------------------- */

    /**
     * Convert value of this BitMask to object representation according to structure.
     * @returns {Object} The data represented by the bitmask.
     */
    toObject() {
      const result = {};
      for ( const [validState, value] of this.#validStates ) result[validState] = ((this.#value & value) !== 0);
      return result;
    }

    /* -------------------------------------------- */

    /**
     * Creates a clone of this BitMask instance.
     * @returns {BitMask} A new BitMask instance with the same value and valid states as this instance.
     */
    clone() {
      return new BitMask(this.toObject());
    }

    /* -------------------------------------------- */
    /*  Static Helpers                              */
    /* -------------------------------------------- */

    /**
     * Generates shader constants based on the provided states.
     * @param {string[]} states An array containing valid states.
     * @returns {string} Shader bit mask constants generated from the states.
     */
    static generateShaderBitMaskConstants(states) {
      let shaderConstants = '';
      let bitIndex = 0;
      for ( const state of states ) {
        shaderConstants += `const uint ${state.toUpperCase()} = 0x${(1 << bitIndex).toString(16).toUpperCase()}U;\n`;
        bitIndex++;
      }
      return shaderConstants;
    }
  }

  /**
   * A string tree node consists of zero-or-more string keys, and a leaves property that contains any objects that
   * terminate at the current node.
   * @typedef {object} StringTreeNode
   */

  /**
   * @callback StringTreeEntryFilter
   * @param {any} entry  The entry to filter.
   * @returns {boolean}  Whether the entry should be included in the result set.
   */

  /**
   * A data structure representing a tree of string nodes with arbitrary object leaves.
   */
  class StringTree {
    /**
     * The key symbol that stores the leaves of any given node.
     * @type {symbol}
     */
    static get leaves() {
      return StringTree.#leaves;
    }

    static #leaves = Symbol();

    /* -------------------------------------------- */

    /**
     * The tree's root.
     * @type {StringTreeNode}
     */
    #root = this.#createNode();

    /* -------------------------------------------- */

    /**
     * Create a new node.
     * @returns {StringTreeNode}
     */
    #createNode() {
      return { [StringTree.leaves]: [] };
    }

    /* -------------------------------------------- */

    /**
     * Insert an entry into the tree.
     * @param {string[]} strings  The string parents for the entry.
     * @param {any} entry         The entry to store.
     * @returns {StringTreeNode}  The node the entry was added to.
     */
    addLeaf(strings, entry) {
      let node = this.#root;
      for ( const string of strings ) {
        node[string] ??= this.#createNode();
        node = node[string];
      }

      // Once we've traversed the tree, we add our entry.
      node[StringTree.leaves].push(entry);
      return node;
    }

    /* -------------------------------------------- */

    /**
     * Traverse the tree along the given string path and return any entries reachable from the node.
     * @param {string[]} strings                               The string path to the desired node.
     * @param {object} [options]
     * @param {number} [options.limit]                         The maximum number of items to retrieve.
     * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
     * @returns {any[]}
     */
    lookup(strings, { limit, filterEntries }={}) {
      const entries = [];
      const node = this.nodeAtPrefix(strings);
      if ( !node ) return []; // No matching entries.
      const queue = [node];
      while ( queue.length ) {
        if ( limit && (entries.length >= limit) ) break;
        this._breadthFirstSearch(queue.shift(), entries, queue, { limit, filterEntries });
      }
      return entries;
    }

    /* -------------------------------------------- */

    /**
     * Returns the node at the given path through the tree.
     * @param {string[]} strings                    The string path to the desired node.
     * @param {object} [options]
     * @param {boolean} [options.hasLeaves=false]   Only return the most recently visited node that has leaves, otherwise
     *                                              return the exact node at the prefix, if it exists.
     * @returns {StringTreeNode|void}
     */
    nodeAtPrefix(strings, { hasLeaves=false }={}) {
      let node = this.#root;
      let withLeaves = node;
      for ( const string of strings ) {
        if ( !(string in node) ) return hasLeaves ? withLeaves : undefined;
        node = node[string];
        if ( node[StringTree.leaves].length ) withLeaves = node;
      }
      return hasLeaves ? withLeaves : node;
    }

    /* -------------------------------------------- */

    /**
     * Perform a breadth-first search starting from the given node and retrieving any entries reachable from that node,
     * until we reach the limit.
     * @param {StringTreeNode} node                            The starting node.
     * @param {any[]} entries                                  The accumulated entries.
     * @param {StringTreeNode[]} queue                         The working queue of nodes to search.
     * @param {object} [options]
     * @param {number} [options.limit]                         The maximum number of entries to retrieve before stopping.
     * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
     * @protected
     */
    _breadthFirstSearch(node, entries, queue, { limit, filterEntries }={}) {
      // Retrieve the entries at this node.
      let leaves = node[StringTree.leaves];
      if ( filterEntries instanceof Function ) leaves = leaves.filter(filterEntries);
      entries.push(...leaves);
      if ( limit && (entries.length >= limit) ) return;
      // Push this node's children onto the end of the queue.
      for ( const key of Object.keys(node) ) {
        if ( typeof key === "string" ) queue.push(node[key]);
      }
    }
  }

  /**
   * @typedef {import("./string-tree.mjs").StringTreeNode} StringTreeNode
   */

  /**
   * A leaf entry in the tree.
   * @typedef {object} WordTreeEntry
   * @property {Document|object} entry  An object that this entry represents.
   * @property {string} documentName    The document type.
   * @property {string} uuid            The document's UUID.
   * @property {string} [pack]          The pack ID.
   */

  /**
   * A data structure for quickly retrieving objects by a string prefix.
   * Note that this works well for languages with alphabets (latin, cyrillic, korean, etc.), but may need more nuanced
   * handling for languages that compose characters and letters.
   * @extends {StringTree}
   */
  class WordTree extends StringTree {
    /**
     * Insert an entry into the tree.
     * @param {string} string        The string key for the entry.
     * @param {WordTreeEntry} entry  The entry to store.
     * @returns {StringTreeNode}     The node the entry was added to.
     */
    addLeaf(string, entry) {
      string = string.toLocaleLowerCase(game.i18n.lang);
      return super.addLeaf(Array.from(string), entry);
    }

    /* -------------------------------------------- */

    /**
     * Return entries that match the given string prefix.
     * @param {string} prefix              The prefix.
     * @param {object} [options]           Additional options to configure behaviour.
     * @param {number} [options.limit=10]  The maximum number of items to retrieve. It is important to set this value as
     *                                     very short prefixes will naturally match large numbers of entries.
     * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
     * @returns {WordTreeEntry[]}          A number of entries that have the given prefix.
     */
    lookup(prefix, { limit=10, filterEntries }={}) {
      return super.lookup(prefix, { limit, filterEntries });
    }

    /* -------------------------------------------- */

    /**
     * Returns the node at the given prefix.
     * @param {string} prefix  The prefix.
     * @returns {StringTreeNode}
     */
    nodeAtPrefix(prefix) {
      prefix = prefix.toLocaleLowerCase(game.i18n.lang);
      return super.nodeAtPrefix(Array.from(prefix));
    }
  }

  /**
   * The constructor of an async function.
   * @type {typeof AsyncFunction}
   */
  const AsyncFunction = (async function() {}).constructor;

  var utils = /*#__PURE__*/Object.freeze({
    __proto__: null,
    AsyncFunction: AsyncFunction,
    BitMask: BitMask,
    Collection: Collection,
    Color: Color$1,
    EventEmitterMixin: EventEmitterMixin,
    HttpError: HttpError,
    IterableWeakMap: IterableWeakMap,
    IterableWeakSet: IterableWeakSet,
    Semaphore: Semaphore,
    StringTree: StringTree,
    WordTree: WordTree,
    benchmark: benchmark,
    circleCircleIntersects: circleCircleIntersects,
    closestPointToSegment: closestPointToSegment,
    debounce: debounce,
    debouncedReload: debouncedReload,
    deepClone: deepClone,
    diffObject: diffObject,
    duplicate: duplicate,
    encodeURL: encodeURL,
    expandObject: expandObject,
    fetchJsonWithTimeout: fetchJsonWithTimeout,
    fetchWithTimeout: fetchWithTimeout,
    filterObject: filterObject,
    flattenObject: flattenObject,
    formatFileSize: formatFileSize,
    getDefiningClass: getDefiningClass,
    getParentClasses: getParentClasses,
    getProperty: getProperty,
    getRoute: getRoute,
    getType: getType,
    hasProperty: hasProperty,
    invertObject: invertObject,
    isEmpty: isEmpty$1,
    isNewerVersion: isNewerVersion,
    isSubclass: isSubclass,
    lineCircleIntersection: lineCircleIntersection,
    lineLineIntersection: lineLineIntersection,
    lineSegmentIntersection: lineSegmentIntersection,
    lineSegmentIntersects: lineSegmentIntersects,
    logCompatibilityWarning: logCompatibilityWarning,
    mergeObject: mergeObject,
    objectsEqual: objectsEqual,
    orient2dFast: orient2dFast,
    parseS3URL: parseS3URL,
    parseUuid: parseUuid,
    pathCircleIntersects: pathCircleIntersects,
    polygonCentroid: polygonCentroid,
    quadraticIntersection: quadraticIntersection,
    randomID: randomID,
    setProperty: setProperty,
    threadLock: threadLock,
    throttle: throttle,
    timeSince: timeSince
  });

  /**
   * This module contains data field classes which are used to define a data schema.
   * A data field is responsible for cleaning, validation, and initialization of the value assigned to it.
   * Each data field extends the [DataField]{@link DataField} class to implement logic specific to its
   * contained data type.
   * @module fields
   */


  /* ---------------------------------------- */
  /*  Abstract Data Field                     */
  /* ---------------------------------------- */

  /**
   * @callback DataFieldValidator
   * A Custom DataField validator function.
   *
   * A boolean return value indicates that the value is valid (true) or invalid (false) with certainty. With an explicit
   * boolean return value no further validation functions will be evaluated.
   *
   * An undefined return indicates that the value may be valid but further validation functions should be performed,
   * if defined.
   *
   * An Error may be thrown which provides a custom error message explaining the reason the value is invalid.
   *
   * @param {any} value                     The value provided for validation
   * @param {DataFieldValidationOptions} options  Validation options
   * @returns {boolean|void}
   * @throws {Error}
   */

  /**
   * @typedef {Object} DataFieldOptions
   * @property {boolean} [required=false]   Is this field required to be populated?
   * @property {boolean} [nullable=false]   Can this field have null values?
   * @property {boolean} [gmOnly=false]     Can this field only be modified by a gamemaster or assistant gamemaster?
   * @property {Function|*} [initial]       The initial value of a field, or a function which assigns that initial value.
   * @property {string} [label]             A localizable label displayed on forms which render this field.
   * @property {string} [hint]              Localizable help text displayed on forms which render this field.
   * @property {DataFieldValidator} [validate] A custom data field validation function.
   * @property {string} [validationError]   A custom validation error string. When displayed will be prepended with the
   *                                        document name, field name, and candidate value. This error string is only
   *                                        used when the return type of the validate function is a boolean. If an Error
   *                                        is thrown in the validate function, the string message of that Error is used.
   */

  /**
   * @typedef {Object} DataFieldContext
   * @property {string} [name]               A field name to assign to the constructed field
   * @property {DataField} [parent]          Another data field which is a hierarchical parent of this one
   */

  /**
   * @typedef {object} DataFieldValidationOptions
   * @property {boolean} [partial]   Whether this is a partial schema validation, or a complete one.
   * @property {boolean} [fallback]  Whether to allow replacing invalid values with valid fallbacks.
   * @property {object} [source]     The full source object being evaluated.
   * @property {boolean} [dropInvalidEmbedded]  If true, invalid embedded documents will emit a warning and be placed in
   *                                            the invalidDocuments collection rather than causing the parent to be
   *                                            considered invalid.
   */

  /**
   * An abstract class that defines the base pattern for a data field within a data schema.
   * @abstract
   * @property {string} name                The name of this data field within the schema that contains it.
   * @mixes DataFieldOptions
   */
  class DataField {
    /**
     * @param {DataFieldOptions} [options]    Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(options={}, {name, parent}={}) {
      this.name = name;
      this.parent = parent;
      this.options = options;
      for ( let k in this.constructor._defaults ) {
        this[k] = k in this.options ? this.options[k] : this.constructor._defaults[k];
      }
    }

    /**
     * The field name of this DataField instance.
     * This is assigned by SchemaField#initialize.
     * @internal
     */
    name;

    /**
     * A reference to the parent schema to which this DataField belongs.
     * This is assigned by SchemaField#initialize.
     * @internal
     */
    parent;

    /**
     * The initially provided options which configure the data field
     * @type {DataFieldOptions}
     */
    options;

    /**
     * Whether this field defines part of a Document/Embedded Document hierarchy.
     * @type {boolean}
     */
    static hierarchical = false;

    /**
     * Does this field type contain other fields in a recursive structure?
     * Examples of recursive fields are SchemaField, ArrayField, or TypeDataField
     * Examples of non-recursive fields are StringField, NumberField, or ObjectField
     * @type {boolean}
     */
    static recursive = false;

    /**
     * Default parameters for this field type
     * @return {DataFieldOptions}
     * @protected
     */
    static get _defaults() {
      return {
        required: false,
        nullable: false,
        initial: undefined,
        readonly: false,
        gmOnly: false,
        label: "",
        hint: "",
        validationError: "is not a valid value"
      }
    }

    /**
     * A dot-separated string representation of the field path within the parent schema.
     * @type {string}
     */
    get fieldPath() {
      return [this.parent?.fieldPath, this.name].filterJoin(".");
    }

    /**
     * Apply a function to this DataField which propagates through recursively to any contained data schema.
     * @param {string|function} fn          The function to apply
     * @param {*} value                     The current value of this field
     * @param {object} [options={}]         Additional options passed to the applied function
     * @returns {object}                    The results object
     */
    apply(fn, value, options={}) {
      if ( typeof fn === "string" ) fn = this[fn];
      return fn.call(this, value, options);
    }

    /* -------------------------------------------- */
    /*  Field Cleaning                              */
    /* -------------------------------------------- */

    /**
     * Coerce source data to ensure that it conforms to the correct data type for the field.
     * Data coercion operations should be simple and synchronous as these are applied whenever a DataModel is constructed.
     * For one-off cleaning of user-provided input the sanitize method should be used.
     * @param {*} value           The initial value
     * @param {object} [options]  Additional options for how the field is cleaned
     * @param {boolean} [options.partial]   Whether to perform partial cleaning?
     * @param {object} [options.source]     The root data model being cleaned
     * @returns {*}               The cast value
     */
    clean(value, options={}) {

      // Permit explicitly null values for nullable fields
      if ( value === null ) {
        if ( this.nullable ) return value;
        value = undefined;
      }

      // Get an initial value for the field
      if ( value === undefined ) return this.getInitialValue(options.source);

      // Cast a provided value to the correct type
      value = this._cast(value);

      // Cleaning logic specific to the DataField.
      return this._cleanType(value, options);
    }

    /* -------------------------------------------- */

    /**
     * Apply any cleaning logic specific to this DataField type.
     * @param {*} value           The appropriately coerced value.
     * @param {object} [options]  Additional options for how the field is cleaned.
     * @returns {*}               The cleaned value.
     * @protected
     */
    _cleanType(value, options) {
      return value;
    }

    /* -------------------------------------------- */

    /**
     * Cast a non-default value to ensure it is the correct type for the field
     * @param {*} value       The provided non-default value
     * @returns {*}           The standardized value
     * @protected
     */
    _cast(value) {
      throw new Error(`Subclasses of DataField must implement the _cast method`);
    }

    /* -------------------------------------------- */

    /**
     * Attempt to retrieve a valid initial value for the DataField.
     * @param {object} data   The source data object for which an initial value is required
     * @returns {*}           A valid initial value
     * @throws                An error if there is no valid initial value defined
     */
    getInitialValue(data) {
      return this.initial instanceof Function ? this.initial(data) : this.initial;
    }

    /* -------------------------------------------- */
    /*  Field Validation                            */
    /* -------------------------------------------- */

    /**
     * Validate a candidate input for this field, ensuring it meets the field requirements.
     * A validation failure can be provided as a raised Error (with a string message), by returning false, or by returning
     * a DataModelValidationFailure instance.
     * A validator which returns true denotes that the result is certainly valid and further validations are unnecessary.
     * @param {*} value                                  The initial value
     * @param {DataFieldValidationOptions} [options={}]  Options which affect validation behavior
     * @returns {DataModelValidationFailure}             Returns a DataModelValidationFailure if a validation failure
     *                                                   occurred.
     */
    validate(value, options={}) {
      const validators = [this._validateSpecial, this._validateType];
      if ( this.options.validate ) validators.push(this.options.validate);
      try {
        for ( const validator of validators ) {
          const isValid = validator.call(this, value, options);
          if ( isValid === true ) return undefined;
          if ( isValid === false ) {
            return new DataModelValidationFailure({
              invalidValue: value,
              message: this.validationError,
              unresolved: true
            });
          }
          if ( isValid instanceof DataModelValidationFailure ) return isValid;
        }
      } catch(err) {
        return new DataModelValidationFailure({invalidValue: value, message: err.message, unresolved: true});
      }
    }

    /* -------------------------------------------- */

    /**
     * Special validation rules which supersede regular field validation.
     * This validator screens for certain values which are otherwise incompatible with this field like null or undefined.
     * @param {*} value               The candidate value
     * @returns {boolean|void}        A boolean to indicate with certainty whether the value is valid.
     *                                Otherwise, return void.
     * @throws                        May throw a specific error if the value is not valid
     * @protected
     */
    _validateSpecial(value) {

      // Allow null values for explicitly nullable fields
      if ( value === null ) {
        if ( this.nullable ) return true;
        else throw new Error("may not be null");
      }

      // Allow undefined if the field is not required
      if ( value === undefined ) {
        if ( this.required ) throw new Error("may not be undefined");
        else return true;
      }
    }

    /* -------------------------------------------- */

    /**
     * A default type-specific validator that can be overridden by child classes
     * @param {*} value                                    The candidate value
     * @param {DataFieldValidationOptions} [options={}]    Options which affect validation behavior
     * @returns {boolean|DataModelValidationFailure|void}  A boolean to indicate with certainty whether the value is
     *                                                     valid, or specific DataModelValidationFailure information,
     *                                                     otherwise void.
     * @throws                                             May throw a specific error if the value is not valid
     * @protected
     */
    _validateType(value, options={}) {}

    /* -------------------------------------------- */

    /**
     * Certain fields may declare joint data validation criteria.
     * This method will only be called if the field is designated as recursive.
     * @param {object} data       Candidate data for joint model validation
     * @param {object} options    Options which modify joint model validation
     * @throws  An error if joint model validation fails
     * @internal
     */
    _validateModel(data, options={}) {}

    /* -------------------------------------------- */
    /*  Initialization and Serialization            */
    /* -------------------------------------------- */

    /**
     * Initialize the original source data into a mutable copy for the DataModel instance.
     * @param {*} value                   The source value of the field
     * @param {Object} model              The DataModel instance that this field belongs to
     * @param {object} [options]          Initialization options
     * @returns {*}                       An initialized copy of the source data
     */
    initialize(value, model, options={}) {
      return value;
    }

    /**
     * Export the current value of the field into a serializable object.
     * @param {*} value                   The initialized value of the field
     * @returns {*}                       An exported representation of the field
     */
    toObject(value) {
      return value;
    }

    /**
     * Recursively traverse a schema and retrieve a field specification by a given path
     * @param {string[]} path             The field path as an array of strings
     * @internal
     */
    _getField(path) {
      return path.length ? undefined : this;
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /**
     * Does this form field class have defined form support?
     * @type {boolean}
     */
    static get hasFormSupport() {
      return this.prototype._toInput !== DataField.prototype._toInput;
    }

    /* -------------------------------------------- */

    /**
     * Render this DataField as an HTML element.
     * @param {FormInputConfig} config        Form element configuration parameters
     * @throws {Error}                        An Error if this DataField subclass does not support input rendering
     * @returns {HTMLElement|HTMLCollection}  A rendered HTMLElement for the field
     */
    toInput(config={}) {
      const inputConfig = {name: this.fieldPath, ...config};
      if ( inputConfig.input instanceof Function ) return config.input(this, inputConfig);
      return this._toInput(inputConfig);
    }

    /* -------------------------------------------- */

    /**
     * Render this DataField as an HTML element.
     * Subclasses should implement this method rather than the public toInput method which wraps it.
     * @param {FormInputConfig} config        Form element configuration parameters
     * @throws {Error}                        An Error if this DataField subclass does not support input rendering
     * @returns {HTMLElement|HTMLCollection}  A rendered HTMLElement for the field
     * @protected
     */
    _toInput(config) {
      throw new Error(`The ${this.constructor.name} class does not implement the _toInput method`);
    }

    /* -------------------------------------------- */

    /**
     * Render this DataField as a standardized form-group element.
     * @param {FormGroupConfig} groupConfig   Configuration options passed to the wrapping form-group
     * @param {FormInputConfig} inputConfig   Input element configuration options passed to DataField#toInput
     * @returns {HTMLDivElement}              The rendered form group element
     */
    toFormGroup(groupConfig={}, inputConfig={}) {
      if ( groupConfig.widget instanceof Function ) return groupConfig.widget(this, groupConfig, inputConfig);
      groupConfig.label ??= this.label ?? this.fieldPath;
      groupConfig.hint ??= this.hint;
      groupConfig.input ??= this.toInput(inputConfig);
      return foundry.applications.fields.createFormGroup(groupConfig);
    }

    /* -------------------------------------------- */
    /*  Active Effect Integration                   */
    /* -------------------------------------------- */

    /**
     * Apply an ActiveEffectChange to this field.
     * @param {*} value                  The field's current value.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The change to apply.
     * @returns {*}                      The updated value.
     */
    applyChange(value, model, change) {
      const delta = this._castChangeDelta(change.value);
      switch ( change.mode ) {
        case CONST.ACTIVE_EFFECT_MODES.ADD: return this._applyChangeAdd(value, delta, model, change);
        case CONST.ACTIVE_EFFECT_MODES.MULTIPLY: return this._applyChangeMultiply(value, delta, model, change);
        case CONST.ACTIVE_EFFECT_MODES.OVERRIDE: return this._applyChangeOverride(value, delta, model, change);
        case CONST.ACTIVE_EFFECT_MODES.UPGRADE: return this._applyChangeUpgrade(value, delta, model, change);
        case CONST.ACTIVE_EFFECT_MODES.DOWNGRADE: return this._applyChangeDowngrade(value, delta, model, change);
      }
      return this._applyChangeCustom(value, delta, model, change);
    }

    /* -------------------------------------------- */

    /**
     * Cast a change delta into an appropriate type to be applied to this field.
     * @param {*} delta  The change delta.
     * @returns {*}
     * @internal
     */
    _castChangeDelta(delta) {
      return this._cast(delta);
    }

    /* -------------------------------------------- */

    /**
     * Apply an ADD change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeAdd(value, delta, model, change) {
      return value + delta;
    }

    /* -------------------------------------------- */

    /**
     * Apply a MULTIPLY change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeMultiply(value, delta, model, change) {}

    /* -------------------------------------------- */

    /**
     * Apply an OVERRIDE change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeOverride(value, delta, model, change) {
      return delta;
    }

    /* -------------------------------------------- */

    /**
     * Apply an UPGRADE change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeUpgrade(value, delta, model, change) {}

    /* -------------------------------------------- */

    /**
     * Apply a DOWNGRADE change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeDowngrade(value, delta, model, change) {}

    /* -------------------------------------------- */

    /**
     * Apply a CUSTOM change to this field.
     * @param {*} value                  The field's current value.
     * @param {*} delta                  The change delta.
     * @param {DataModel} model          The model instance.
     * @param {EffectChangeData} change  The original change data.
     * @returns {*}                      The updated value.
     * @protected
     */
    _applyChangeCustom(value, delta, model, change) {
      const preHook = foundry.utils.getProperty(model, change.key);
      Hooks.call("applyActiveEffect", model, change, value, delta, {});
      const postHook = foundry.utils.getProperty(model, change.key);
      if ( postHook !== preHook ) return postHook;
    }
  }

  /* -------------------------------------------- */
  /*  Data Schema Field                           */
  /* -------------------------------------------- */

  /**
   * A special class of {@link DataField} which defines a data schema.
   */
  class SchemaField extends DataField {
    /**
     * @param {DataSchema} fields                 The contained field definitions
     * @param {DataFieldOptions} [options]        Options which configure the behavior of the field
     * @param {DataFieldContext} [context]        Additional context which describes the field
     */
    constructor(fields, options, context={}) {
      super(options, context);
      this.fields = this._initialize(fields);
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        initial() { return this.clean({}); }
      });
    }

    /** @override */
    static recursive = true;

    /* -------------------------------------------- */

    /**
     * The contained field definitions.
     * @type {DataSchema}
     */
    fields;

    /**
     * Any unknown keys encountered during the last cleaning.
     * @type {string[]}
     */
    unknownKeys;

    /* -------------------------------------------- */

    /**
     * Initialize and validate the structure of the provided field definitions.
     * @param {DataSchema} fields     The provided field definitions
     * @returns {DataSchema}          The validated schema
     * @protected
     */
    _initialize(fields) {
      if ( (typeof fields !== "object") ) {
        throw new Error("A DataSchema must be an object with string keys and DataField values.");
      }
      fields = {...fields};
      for ( const [name, field] of Object.entries(fields) ) {
        if ( !(field instanceof DataField) ) {
          throw new Error(`The "${name}" field is not an instance of the DataField class.`);
        }
        if ( field.parent !== undefined ) {
          throw new Error(`The "${field.fieldPath}" field already belongs to some other parent and may not be reused.`);
        }
        field.name = name;
        field.parent = this;
      }
      return fields;
    }

    /* -------------------------------------------- */
    /*  Schema Iteration                            */
    /* -------------------------------------------- */

    /**
     * Iterate over a SchemaField by iterating over its fields.
     * @type {Iterable<DataField>}
     */
    *[Symbol.iterator]() {
      for ( const field of Object.values(this.fields) ) {
        yield field;
      }
    }

    /**
     * An array of field names which are present in the schema.
     * @returns {string[]}
     */
    keys() {
      return Object.keys(this.fields);
    }

    /**
     * An array of DataField instances which are present in the schema.
     * @returns {DataField[]}
     */
    values() {
      return Object.values(this.fields);
    }

    /**
     * An array of [name, DataField] tuples which define the schema.
     * @returns {Array<[string, DataField]>}
     */
    entries() {
      return Object.entries(this.fields);
    }

    /**
     * Test whether a certain field name belongs to this schema definition.
     * @param {string} fieldName    The field name
     * @returns {boolean}           Does the named field exist in this schema?
     */
    has(fieldName) {
      return fieldName in this.fields;
    }

    /**
     * Get a DataField instance from the schema by name
     * @param {string} fieldName    The field name
     * @returns {DataField}         The DataField instance or undefined
     */
    get(fieldName) {
      return this.fields[fieldName];
    }

    /**
     * Traverse the schema, obtaining the DataField definition for a particular field.
     * @param {string[]|string} fieldName       A field path like ["abilities", "strength"] or "abilities.strength"
     * @returns {SchemaField|DataField}         The corresponding DataField definition for that field, or undefined
     */
    getField(fieldName) {
      let path;
      if ( typeof fieldName === "string" ) path = fieldName.split(".");
      else if ( Array.isArray(fieldName) ) path = fieldName.slice();
      else throw new Error("A field path must be an array of strings or a dot-delimited string");
      return this._getField(path);
    }

    /** @override */
    _getField(path) {
      if ( !path.length ) return this;
      const field = this.get(path.shift());
      return field?._getField(path);
    }

    /* -------------------------------------------- */
    /*  Data Field Methods                          */
    /* -------------------------------------------- */

    /** @override */
    _cast(value) {
      return typeof value === "object" ? value : {};
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    _cleanType(data, options={}) {
      options.source = options.source || data;

      // Clean each field which belongs to the schema
      for ( const [name, field] of this.entries() ) {
        if ( !(name in data) && options.partial ) continue;
        data[name] = field.clean(data[name], options);
      }

      // Delete any keys which do not
      this.unknownKeys = [];
      for ( const k of Object.keys(data) ) {
        if ( this.has(k) ) continue;
        this.unknownKeys.push(k);
        delete data[k];
      }
      return data;
    }

    /* -------------------------------------------- */

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      const data = {};
      for ( let [name, field] of this.entries() ) {
        const v = field.initialize(value[name], model, options);

        // Readonly fields
        if ( field.readonly ) {
          Object.defineProperty(data, name, {value: v, writable: false});
        }

        // Getter fields
        else if ( (typeof v === "function") && !v.prototype ) {
          Object.defineProperty(data, name, {get: v, set() {}, configurable: true});
        }

        // Writable fields
        else data[name] = v;
      }
      return data;
    }

    /* -------------------------------------------- */

    /** @override */
    _validateType(data, options={}) {
      if ( !(data instanceof Object) ) throw new Error("must be an object");
      options.source = options.source || data;
      const schemaFailure = new DataModelValidationFailure();
      for ( const [key, field] of this.entries() ) {
        if ( options.partial && !(key in data) ) continue;

        // Validate the field's current value
        const value = data[key];
        const failure = field.validate(value, options);

        // Failure may be permitted if fallback replacement is allowed
        if ( failure ) {
          schemaFailure.fields[field.name] = failure;

          // If the field internally applied fallback logic
          if ( !failure.unresolved ) continue;

          // If fallback is allowed at the schema level
          if ( options.fallback ) {
            const initial = field.getInitialValue(options.source);
            if ( field.validate(initial, {source: options.source}) === undefined ) {  // Ensure initial is valid
              data[key] = initial;
              failure.fallback = initial;
              failure.unresolved = false;
            }
            else failure.unresolved = schemaFailure.unresolved = true;
          }

          // Otherwise the field-level failure is unresolved
          else failure.unresolved = schemaFailure.unresolved = true;
        }
      }
      if ( !isEmpty$1(schemaFailure.fields) ) return schemaFailure;
    }

    /* ---------------------------------------- */

    /** @override */
    _validateModel(changes, options={}) {
      options.source = options.source || changes;
      if ( !changes ) return;
      for ( const [name, field] of this.entries() ) {
        const change = changes[name];  // May be nullish
        if ( change && field.constructor.recursive ) field._validateModel(change, options);
      }
    }

    /* -------------------------------------------- */

    /** @override */
    toObject(value) {
      if ( (value === undefined) || (value === null) ) return value;
      const data = {};
      for ( const [name, field] of this.entries() ) {
        data[name] = field.toObject(value[name]);
      }
      return data;
    }

    /* -------------------------------------------- */

    /** @override */
    apply(fn, data={}, options={}) {

      // Apply to this SchemaField
      const thisFn = typeof fn === "string" ? this[fn] : fn;
      thisFn?.call(this, data, options);

      // Recursively apply to inner fields
      const results = {};
      for ( const [key, field] of this.entries() ) {
        if ( options.partial && !(key in data) ) continue;
        const r = field.apply(fn, data[key], options);
        if ( !options.filter || !isEmpty$1(r) ) results[key] = r;
      }
      return results;
    }

    /* -------------------------------------------- */

    /**
     * Migrate this field's candidate source data.
     * @param {object} sourceData   Candidate source data of the root model
     * @param {any} fieldData       The value of this field within the source data
     */
    migrateSource(sourceData, fieldData) {
      for ( const [key, field] of this.entries() ) {
        const canMigrate = field.migrateSource instanceof Function;
        if ( canMigrate && fieldData[key] ) field.migrateSource(sourceData, fieldData[key]);
      }
    }
  }

  /* -------------------------------------------- */
  /*  Basic Field Types                           */
  /* -------------------------------------------- */

  /**
   * A subclass of [DataField]{@link DataField} which deals with boolean-typed data.
   */
  class BooleanField extends DataField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        initial: false
      });
    }

    /** @override */
    _cast(value) {
      if ( typeof value === "string" ) return value === "true";
      if ( typeof value === "object" ) return false;
      return Boolean(value);
    }

    /** @override */
    _validateType(value) {
      if (typeof value !== "boolean") throw new Error("must be a boolean");
    }

    /** @override */
    _toInput(config) {
      return foundry.applications.fields.createCheckboxInput(config);
    }

    /* -------------------------------------------- */
    /*  Active Effect Integration                   */
    /* -------------------------------------------- */

    /** @override */
    _applyChangeAdd(value, delta, model, change) {
      return value || delta;
    }

    /** @override */
    _applyChangeMultiply(value, delta, model, change) {
      return value && delta;
    }

    /** @override */
    _applyChangeUpgrade(value, delta, model, change) {
      return delta > value ? delta : value;
    }

    _applyChangeDowngrade(value, delta, model, change) {
      return delta < value ? delta : value;
    }
  }

  /* ---------------------------------------- */

  /**
   * @typedef {DataFieldOptions} NumberFieldOptions
   * @property {number} [min]               A minimum allowed value
   * @property {number} [max]               A maximum allowed value
   * @property {number} [step]              A permitted step size
   * @property {boolean} [integer=false]    Must the number be an integer?
   * @property {number} [positive=false]    Must the number be positive?
   * @property {number[]|object|function} [choices]  An array of values or an object of values/labels which represent
   *                                        allowed choices for the field. A function may be provided which dynamically
   *                                        returns the array of choices.
   */

  /**
   * A subclass of [DataField]{@link DataField} which deals with number-typed data.
   *
   * @property {number} min                 A minimum allowed value
   * @property {number} max                 A maximum allowed value
   * @property {number} step                A permitted step size
   * @property {boolean} integer=false      Must the number be an integer?
   * @property {number} positive=false      Must the number be positive?
   * @property {number[]|object|function} [choices]  An array of values or an object of values/labels which represent
   *                                        allowed choices for the field. A function may be provided which dynamically
   *                                        returns the array of choices.
   */
  class NumberField extends DataField {
    /**
     * @param {NumberFieldOptions} options  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]  Additional context which describes the field
     */
    constructor(options={}, context={}) {
      super(options, context);
      // If choices are provided, the field should not be null by default
      if ( this.choices ) {
        this.nullable = options.nullable ?? false;
      }
      if ( Number.isFinite(this.min) && Number.isFinite(this.max) && (this.min > this.max) ) {
        throw new Error("NumberField minimum constraint cannot exceed its maximum constraint");
      }
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        initial: null,
        nullable: true,
        min: undefined,
        max: undefined,
        step: undefined,
        integer: false,
        positive: false,
        choices: undefined
      });
    }

    /** @override */
    _cast(value) {
      return Number(value);
    }

    /** @inheritdoc */
    _cleanType(value, options) {
      value = super._cleanType(value, options);
      if ( typeof value !== "number" ) return value;
      if ( this.integer ) value = Math.round(value);
      if ( Number.isFinite(this.min) ) value = Math.max(value, this.min);
      if ( Number.isFinite(this.max) ) value = Math.min(value, this.max);
      if ( Number.isFinite(this.step) ) value = value.toNearest(this.step);
      return value;
    }

    /** @override */
    _validateType(value) {
      if ( typeof value !== "number" ) throw new Error("must be a number");
      if ( this.positive && (value <= 0) ) throw new Error("must be a positive number");
      if ( Number.isFinite(this.min) && (value < this.min) ) throw new Error(`must be at least ${this.min}`);
      if ( Number.isFinite(this.max) && (value > this.max) ) throw new Error(`must be at most ${this.max}`);
      if ( Number.isFinite(this.step) && (value.toNearest(this.step) !== value) ) {
        throw new Error(`must be an increment of ${this.step}`);
      }
      if ( this.choices && !this.#isValidChoice(value) ) throw new Error(`${value} is not a valid choice`);
      if ( this.integer ) {
        if ( !Number.isInteger(value) ) throw new Error("must be an integer");
      }
      else if ( !Number.isFinite(value) ) throw new Error("must be a finite number");
    }

    /**
     * Test whether a provided value is a valid choice from the allowed choice set
     * @param {number} value      The provided value
     * @returns {boolean}         Is the choice valid?
     */
    #isValidChoice(value) {
      let choices = this.choices;
      if ( choices instanceof Function ) choices = choices();
      if ( choices instanceof Array ) return choices.includes(value);
      return String(value) in choices;
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      config.min ??= this.min;
      config.max ??= this.max;
      config.step ??= this.step;
      if ( config.value === undefined ) config.value = this.getInitialValue({});
      if ( this.integer ) {
        if ( Number.isNumeric(config.value) ) config.value = Math.round(config.value);
        config.step ??= 1;
      }
      if ( this.positive && Number.isFinite(config.step) ) config.min ??= config.step;

      // Number Select
      config.choices ??= this.choices;
      if ( config.choices && !config.options ) {
        config.options = StringField._getChoices(config);
        delete config.valueAttr;
        delete config.labelAttr;
        config.dataset ||= {};
        config.dataset.dtype = "Number";
      }
      if ( config.options ) return foundry.applications.fields.createSelectInput(config);

      // Range Slider
      if ( ["min", "max", "step"].every(k => config[k] !== undefined) && (config.type !== "number") ) {
        return foundry.applications.elements.HTMLRangePickerElement.create(config);
      }

      // Number Input
      return foundry.applications.fields.createNumberInput(config);
    }

    /* -------------------------------------------- */
    /*  Active Effect Integration                   */
    /* -------------------------------------------- */

    /** @override */
    _applyChangeMultiply(value, delta, model, change) {
      return value * delta;
    }

    /** @override */
    _applyChangeUpgrade(value, delta, model, change) {
      return delta > value ? delta : value;
    }

    /** @override */
    _applyChangeDowngrade(value, delta, model, change) {
      return delta < value ? delta : value;
    }
  }

  /* ---------------------------------------- */

  /**
   * @typedef {Object} StringFieldParams
   * @property {boolean} [blank=true]       Is the string allowed to be blank (empty)?
   * @property {boolean} [trim=true]        Should any provided string be trimmed as part of cleaning?
   * @property {string[]|object|function} [choices]  An array of values or an object of values/labels which represent
   *                                        allowed choices for the field. A function may be provided which dynamically
   *                                        returns the array of choices.
   * @property {boolean} [textSearch=false] Is this string field a target for text search?
   * @typedef {DataFieldOptions&StringFieldParams} StringFieldOptions
   */

  /**
   * A subclass of {@link DataField} which deals with string-typed data.
   */
  class StringField extends DataField {
    /**
     * @param {StringFieldOptions} [options]  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(options={}, context={}) {
      super(options, context);

      // If choices are provided, the field should not be null or blank by default
      if ( this.choices ) {
        this.nullable = options.nullable ?? false;
        this.blank = options.blank ?? false;
      }
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        blank: true,
        trim: true,
        nullable: false,
        initial() {
          // The initial value depends on the field configuration
          if ( !this.required ) return undefined;
          else if ( this.blank ) return "";
          else if ( this.nullable ) return null;
          return undefined;
        },
        choices: undefined,
        textSearch: false
      });
    }

    /**
     * Is the string allowed to be blank (empty)?
     * @type {boolean}
     */
    blank = this.blank;

    /**
     * Should any provided string be trimmed as part of cleaning?
     * @type {boolean}
     */
    trim = this.trim;

    /**
     * An array of values or an object of values/labels which represent
     * allowed choices for the field. A function may be provided which dynamically
     * returns the array of choices.
     * @type {string[]|object|function}
     */
    choices = this.choices;

    /**
     * Is this string field a target for text search?
     * @type {boolean}
     */
    textSearch = this.textSearch;

    /** @inheritdoc */
    clean(value, options) {
      if ( (typeof value === "string") && this.trim ) value = value.trim(); // Trim input strings
      if ( value === "" ) {  // Permit empty strings for blank fields
        if ( this.blank ) return value;
        value = undefined;
      }
      return super.clean(value, options);
    }

    /** @override */
    _cast(value) {
      return String(value);
    }

    /** @inheritdoc */
    _validateSpecial(value) {
      if ( value === "" ) {
        if ( this.blank ) return true;
        else throw new Error("may not be a blank string");
      }
      return super._validateSpecial(value);
    }

    /** @override */
    _validateType(value) {
      if ( typeof value !== "string" ) throw new Error("must be a string");
      else if ( this.choices ) {
        if ( this._isValidChoice(value) ) return true;
        else throw new Error(`${value} is not a valid choice`);
      }
    }

    /**
     * Test whether a provided value is a valid choice from the allowed choice set
     * @param {string} value      The provided value
     * @returns {boolean}         Is the choice valid?
     * @protected
     */
    _isValidChoice(value) {
      let choices = this.choices;
      if ( choices instanceof Function ) choices = choices();
      if ( choices instanceof Array ) return choices.includes(value);
      return String(value) in choices;
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /**
     * Get a record of eligible choices for the field.
     * @param {object} [options]
     * @param {Record<any, any>|Array<any>} options.choices
     * @param {string} [options.labelAttr="label"]   The property in the choice object values to use as the option label.
     * @param {string} [options.valueAttr]
     * @param {boolean} [options.localize=false]     Pass each label through string localization?
     * @returns {FormSelectOption[]}
     * @internal
     */
    static _getChoices({choices, labelAttr="label", valueAttr, localize=false}={}) {
      if ( choices instanceof Function ) choices = choices();
      if ( typeof choices === "object" ) {
        choices = Object.entries(choices).reduce((arr, [value, label]) => {
          if ( typeof label !== "string" ) {
            if ( valueAttr && (valueAttr in label) ) value = label[valueAttr];
            label = label[labelAttr] ?? "undefined";
          }
          if ( localize ) label = game.i18n.localize(label);
          arr.push({value, label});
          return arr;
        }, []);
      }
      return choices;
    }

    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      if ( config.value === undefined ) config.value = this.getInitialValue({});
      config.choices ??= this.choices;
      if ( config.choices && !config.options ) {
        config.options = StringField._getChoices(config);
        delete config.choices;
        delete config.valueAttr;
        delete config.labelAttr;
        if ( this.blank || !this.required ) config.blank ??= "";
      }
      if ( config.options ) return foundry.applications.fields.createSelectInput(config);
      return foundry.applications.fields.createTextInput(config);
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [DataField]{@link DataField} which deals with object-typed data.
   */
  class ObjectField extends DataField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false
      });
    }

    /* -------------------------------------------- */

    /** @override */
    getInitialValue(data) {
      const initial = super.getInitialValue(data);
      if ( initial ) return initial;          // Explicit initial value defined by subclass
      if ( !this.required ) return undefined; // The ObjectField may be undefined
      if ( this.nullable ) return null;       // The ObjectField may be null
      return {};                              // Otherwise an empty object
    }

    /** @override */
    _cast(value) {
      return getType(value) === "Object" ? value : {};
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      return deepClone(value);
    }

    /** @override */
    toObject(value) {
      return deepClone(value);
    }

    /** @override */
    _validateType(value, options={}) {
      if ( getType(value) !== "Object" ) throw new Error("must be an object");
    }
  }

  /* -------------------------------------------- */

  /**
   * @typedef {DataFieldOptions} ArrayFieldOptions
   * @property {number} [min]          The minimum number of elements.
   * @property {number} [max]          The maximum number of elements.
   */

  /**
   * A subclass of [DataField]{@link DataField} which deals with array-typed data.
   * @property {number} min     The minimum number of elements.
   * @property {number} max     The maximum number of elements.
   */
  class ArrayField extends DataField {
    /**
     * @param {DataField} element            A DataField instance which defines the type of element contained in the Array
     * @param {ArrayFieldOptions} [options]  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]   Additional context which describes the field
     */
    constructor(element, options={}, context={}) {
      super(options, context);
      /**
       * The data type of each element in this array
       * @type {DataField}
       */
      this.element = this.constructor._validateElementType(element);
      if ( this.min > this.max ) throw new Error("ArrayField minimum length cannot exceed maximum length");
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        empty: true,
        exact: undefined,
        min: 0,
        max: Infinity,
        initial: () => []
      });
    }

    /** @override */
    static recursive = true;

    /* ---------------------------------------- */

    /**
     * Validate the contained element type of the ArrayField
     * @param {*} element       The type of Array element
     * @returns {*}             The validated element type
     * @throws                  An error if the element is not a valid type
     * @protected
     */
    static _validateElementType(element) {
      if ( !(element instanceof DataField) ) {
        throw new Error(`${this.name} must have a DataField as its contained element`);
      }
      return element;
    }

    /* ---------------------------------------- */

    /** @override */
    _validateModel(changes, options) {
      if ( !this.element.constructor.recursive ) return;
      for ( const element of changes ) {
        this.element._validateModel(element, options);
      }
    }

    /* ---------------------------------------- */

    /** @override */
    _cast(value) {
      const t = getType(value);
      if ( t === "Object" ) {
        const arr = [];
        for ( const [k, v] of Object.entries(value) ) {
          const i = Number(k);
          if ( Number.isInteger(i) && (i >= 0) ) arr[i] = v;
        }
        return arr;
      }
      else if ( t === "Set" ) return Array.from(value);
      return value instanceof Array ? value : [value];
    }

    /** @override */
    _cleanType(value, options) {
      // Force partial as false for array cleaning. Arrays are updated by replacing the entire array, so partial data
      // must be initialized.
      return value.map(v => this.element.clean(v, { ...options, partial: false }));
    }

    /** @override */
    _validateType(value, options={}) {
      if ( !(value instanceof Array) ) throw new Error("must be an Array");
      if ( value.length < this.min ) throw new Error(`cannot have fewer than ${this.min} elements`);
      if ( value.length > this.max ) throw new Error(`cannot have more than ${this.max} elements`);
      return this._validateElements(value, options);
    }

    /**
     * Validate every element of the ArrayField
     * @param {Array} value                         The array to validate
     * @param {DataFieldValidationOptions} options  Validation options
     * @returns {DataModelValidationFailure|void}   A validation failure if any of the elements failed validation,
     *                                              otherwise void.
     * @protected
     */
    _validateElements(value, options) {
      const arrayFailure = new DataModelValidationFailure();
      for ( let i=0; i<value.length; i++ ) {
        // Force partial as false for array validation. Arrays are updated by replacing the entire array, so there cannot
        // be partial data in the elements.
        const failure = this._validateElement(value[i], { ...options, partial: false });
        if ( failure ) {
          arrayFailure.elements.push({id: i, failure});
          arrayFailure.unresolved ||= failure.unresolved;
        }
      }
      if ( arrayFailure.elements.length ) return arrayFailure;
    }

    /**
     * Validate a single element of the ArrayField.
     * @param {*} value                       The value of the array element
     * @param {DataFieldValidationOptions} options  Validation options
     * @returns {DataModelValidationFailure}  A validation failure if the element failed validation
     * @protected
     */
    _validateElement(value, options) {
      return this.element.validate(value, options);
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      return value.map(v => this.element.initialize(v, model, options));
    }

    /** @override */
    toObject(value) {
      if ( !value ) return value;
      return value.map(v => this.element.toObject(v));
    }

    /** @override */
    apply(fn, value=[], options={}) {

      // Apply to this ArrayField
      const thisFn = typeof fn === "string" ? this[fn] : fn;
      thisFn?.call(this, value, options);

      // Recursively apply to array elements
      const results = [];
      if ( !value.length && options.initializeArrays ) value = [undefined];
      for ( const v of value ) {
        const r = this.element.apply(fn, v, options);
        if ( !options.filter || !isEmpty$1(r) ) results.push(r);
      }
      return results;
    }

    /** @override */
    _getField(path) {
      if ( !path.length ) return this;
      if ( path[0] === "element" ) path.shift();
      return this.element._getField(path);
    }

    /**
     * Migrate this field's candidate source data.
     * @param {object} sourceData   Candidate source data of the root model
     * @param {any} fieldData       The value of this field within the source data
     */
    migrateSource(sourceData, fieldData) {
      const canMigrate = this.element.migrateSource instanceof Function;
      if ( canMigrate && (fieldData instanceof Array) ) {
        for ( const entry of fieldData ) this.element.migrateSource(sourceData, entry);
      }
    }

    /* -------------------------------------------- */
    /*  Active Effect Integration                   */
    /* -------------------------------------------- */

    /** @override */
    _castChangeDelta(raw) {
      let delta;
      try {
        delta = JSON.parse(raw);
        delta = Array.isArray(delta) ? delta : [delta];
      } catch {
        delta = [raw];
      }
      return delta.map(value => this.element._castChangeDelta(value));
    }

    /** @override */
    _applyChangeAdd(value, delta, model, change) {
      value.push(...delta);
      return value;
    }
  }

  /* -------------------------------------------- */
  /*  Specialized Field Types                     */
  /* -------------------------------------------- */

  /**
   * A subclass of [ArrayField]{@link ArrayField} which supports a set of contained elements.
   * Elements in this set are treated as fungible and may be represented in any order or discarded if invalid.
   */
  class SetField extends ArrayField {

    /** @override */
    _validateElements(value, options) {
      const setFailure = new DataModelValidationFailure();
      for ( let i=value.length-1; i>=0; i-- ) {  // iterate backwards so we can splice as we go
        const failure = this._validateElement(value[i], options);
        if ( failure ) {
          setFailure.elements.unshift({id: i, failure});

          // The failure may have been internally resolved by fallback logic
          if ( !failure.unresolved && failure.fallback ) continue;

          // If fallback is allowed, remove invalid elements from the set
          if ( options.fallback ) {
            value.splice(i, 1);
            failure.dropped = true;
          }

          // Otherwise the set failure is unresolved
          else setFailure.unresolved = true;
        }
      }

      // Return a record of any failed set elements
      if ( setFailure.elements.length ) {
        if ( options.fallback && !setFailure.unresolved ) setFailure.fallback = value;
        return setFailure;
      }
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      return new Set(super.initialize(value, model, options));
    }

    /** @override */
    toObject(value) {
      if ( !value ) return value;
      return Array.from(value).map(v => this.element.toObject(v));
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      const e = this.element;

      // Document UUIDs
      if ( e instanceof DocumentUUIDField ) {
        Object.assign(config, {type: e.type, single: false});
        return foundry.applications.elements.HTMLDocumentTagsElement.create(config);
      }

      // Multi-Select Input
      if ( e.choices && !config.options ) {
        config.options = StringField._getChoices({choices: e.choices, ...config});
      }
      if ( config.options ) return foundry.applications.fields.createMultiSelectInput(config);

      // Arbitrary String Tags
      if ( e instanceof StringField ) return foundry.applications.elements.HTMLStringTagsElement.create(config);
      throw new Error(`SetField#toInput is not supported for a ${e.constructor.name} element type`);
    }

    /* -------------------------------------------- */
    /*  Active Effect Integration                   */
    /* -------------------------------------------- */

    /** @inheritDoc */
    _castChangeDelta(raw) {
      return new Set(super._castChangeDelta(raw));
    }

    /** @override */
    _applyChangeAdd(value, delta, model, change) {
      for ( const element of delta ) value.add(element);
      return value;
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [ObjectField]{@link ObjectField} which embeds some other DataModel definition as an inner object.
   */
  class EmbeddedDataField extends SchemaField {
    /**
     * @param {typeof DataModel} model          The class of DataModel which should be embedded in this field
     * @param {DataFieldOptions} [options]      Options which configure the behavior of the field
     * @param {DataFieldContext} [context]      Additional context which describes the field
     */
    constructor(model, options={}, context={}) {
      if ( !isSubclass(model, DataModel) ) {
        throw new Error("An EmbeddedDataField must specify a DataModel class as its type");
      }

      // Create an independent copy of the model schema
      const fields = model.defineSchema();
      super(fields, options, context);

      /**
       * The base DataModel definition which is contained in this field.
       * @type {typeof DataModel}
       */
      this.model = model;
    }

    /** @inheritdoc */
    clean(value, options) {
      return super.clean(value, {...options, source: value});
    }

    /** @inheritdoc */
    validate(value, options) {
      return super.validate(value, {...options, source: value});
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      const m = new this.model(value, {parent: model, ...options});
      Object.defineProperty(m, "schema", {value: this});
      return m;
    }

    /** @override */
    toObject(value) {
      if ( !value ) return value;
      return value.toObject(false);
    }

    /** @override */
    migrateSource(sourceData, fieldData) {
      if ( fieldData ) this.model.migrateDataSafe(fieldData);
    }

    /** @override */
    _validateModel(changes, options) {
      this.model.validateJoint(changes);
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [ArrayField]{@link ArrayField} which supports an embedded Document collection.
   * Invalid elements will be dropped from the collection during validation rather than failing for the field entirely.
   */
  class EmbeddedCollectionField extends ArrayField {
    /**
     * @param {typeof foundry.abstract.Document} element  The type of Document which belongs to this embedded collection
     * @param {DataFieldOptions} [options]  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]  Additional context which describes the field
     */
    constructor(element, options={}, context={}) {
      super(element, options, context);
      this.readonly = true; // Embedded collections are always immutable
    }

    /** @override */
    static _validateElementType(element) {
      if ( isSubclass(element, foundry.abstract.Document) ) return element;
      throw new Error("An EmbeddedCollectionField must specify a Document subclass as its type");
    }

    /**
     * The Collection implementation to use when initializing the collection.
     * @type {typeof EmbeddedCollection}
     */
    static get implementation() {
      return EmbeddedCollection;
    }

    /** @override */
    static hierarchical = true;

    /**
     * A reference to the DataModel subclass of the embedded document element
     * @type {typeof foundry.abstract.Document}
     */
    get model() {
      return this.element.implementation;
    }

    /**
     * The DataSchema of the contained Document model.
     * @type {SchemaField}
     */
    get schema() {
      return this.model.schema;
    }

    /** @inheritDoc */
    _cast(value) {
      if ( getType(value) !== "Map" ) return super._cast(value);
      const arr = [];
      for ( const [id, v] of value.entries() ) {
        if ( !("_id" in v) ) v._id = id;
        arr.push(v);
      }
      return super._cast(arr);
    }

    /** @override */
    _cleanType(value, options) {
      return value.map(v => this.schema.clean(v, {...options, source: v}));
    }

    /** @override */
    _validateElements(value, options) {
      const collectionFailure = new DataModelValidationFailure();
      for ( const v of value ) {
        const failure = this.schema.validate(v, {...options, source: v});
        if ( failure && !options.dropInvalidEmbedded ) {
          collectionFailure.elements.push({id: v._id, name: v.name, failure});
          collectionFailure.unresolved ||= failure.unresolved;
        }
      }
      if ( collectionFailure.elements.length ) return collectionFailure;
    }

    /** @override */
    initialize(value, model, options={}) {
      const collection = model.collections[this.name];
      collection.initialize(options);
      return collection;
    }

    /** @override */
    toObject(value) {
      return value.toObject(false);
    }

    /** @override */
    apply(fn, value=[], options={}) {

      // Apply to this EmbeddedCollectionField
      const thisFn = typeof fn === "string" ? this[fn] : fn;
      thisFn?.call(this, value, options);

      // Recursively apply to inner fields
      const results = [];
      if ( !value.length && options.initializeArrays ) value = [undefined];
      for ( const v of value ) {
        const r = this.schema.apply(fn, v, options);
        if ( !options.filter || !isEmpty$1(r) ) results.push(r);
      }
      return results;
    }

    /**
     * Migrate this field's candidate source data.
     * @param {object} sourceData   Candidate source data of the root model
     * @param {any} fieldData       The value of this field within the source data
     */
    migrateSource(sourceData, fieldData) {
      if ( fieldData instanceof Array ) {
        for ( const entry of fieldData ) this.model.migrateDataSafe(entry);
      }
    }

    /* -------------------------------------------- */
    /*  Embedded Document Operations                */
    /* -------------------------------------------- */

    /**
     * Return the embedded document(s) as a Collection.
     * @param {foundry.abstract.Document} parent  The parent document.
     * @returns {DocumentCollection}
     */
    getCollection(parent) {
      return parent[this.name];
    }
  }

  /* -------------------------------------------- */

  /**
   * A subclass of {@link EmbeddedCollectionField} which manages a collection of delta objects relative to another
   * collection.
   */
  class EmbeddedCollectionDeltaField extends EmbeddedCollectionField {
    /** @override */
    static get implementation() {
      return EmbeddedCollectionDelta;
    }

    /** @override */
    _cleanType(value, options) {
      return value.map(v => {
        if ( v._tombstone ) return foundry.data.TombstoneData.schema.clean(v, {...options, source: v});
        return this.schema.clean(v, {...options, source: v});
      });
    }

    /** @override */
    _validateElements(value, options) {
      const collectionFailure = new DataModelValidationFailure();
      for ( const v of value ) {
        const validationOptions = {...options, source: v};
        const failure = v._tombstone
          ? foundry.data.TombstoneData.schema.validate(v, validationOptions)
          : this.schema.validate(v, validationOptions);
        if ( failure && !options.dropInvalidEmbedded ) {
          collectionFailure.elements.push({id: v._id, name: v.name, failure});
          collectionFailure.unresolved ||= failure.unresolved;
        }
      }
      if ( collectionFailure.elements.length ) return collectionFailure;
    }
  }

  /* -------------------------------------------- */

  /**
   * A subclass of {@link EmbeddedDataField} which supports a single embedded Document.
   */
  class EmbeddedDocumentField extends EmbeddedDataField {
    /**
     * @param {typeof foundry.abstract.Document} model The type of Document which is embedded.
     * @param {DataFieldOptions} [options]  Options which configure the behavior of the field.
     * @param {DataFieldContext} [context]  Additional context which describes the field
     */
    constructor(model, options={}, context={}) {
      if ( !isSubclass(model, foundry.abstract.Document) ) {
        throw new Error("An EmbeddedDocumentField must specify a Document subclass as its type.");
      }
      super(model.implementation, options, context);
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        nullable: true
      });
    }

    /** @override */
    static hierarchical = true;

    /** @override */
    initialize(value, model, options={}) {
      if ( !value ) return value;
      if ( model[this.name] ) {
        model[this.name]._initialize(options);
        return model[this.name];
      }
      const m = new this.model(value, {...options, parent: model, parentCollection: this.name});
      Object.defineProperty(m, "schema", {value: this});
      return m;
    }

    /* -------------------------------------------- */
    /*  Embedded Document Operations                */
    /* -------------------------------------------- */

    /**
     * Return the embedded document(s) as a Collection.
     * @param {Document} parent  The parent document.
     * @returns {Collection<Document>}
     */
    getCollection(parent) {
      const collection = new SingletonEmbeddedCollection(this.name, parent, []);
      const doc = parent[this.name];
      if ( !doc ) return collection;
      collection.set(doc.id, doc);
      return collection;
    }
  }

  /* -------------------------------------------- */
  /*  Special Field Types                         */
  /* -------------------------------------------- */

  /**
   * A subclass of [StringField]{@link StringField} which provides the primary _id for a Document.
   * The field may be initially null, but it must be non-null when it is saved to the database.
   */
  class DocumentIdField extends StringField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        blank: false,
        nullable: true,
        initial: null,
        readonly: true,
        validationError: "is not a valid Document ID string"
      });
    }

    /** @override */
    _cast(value) {
      if ( value instanceof foundry.abstract.Document ) return value._id;
      else return String(value);
    }

    /** @override */
    _validateType(value) {
      if ( !isValidId(value) ) throw new Error("must be a valid 16-character alphanumeric ID");
    }
  }

  /* ---------------------------------------- */


  /**
   * @typedef {Object} DocumentUUIDFieldOptions
   * @property {string} [type]            A specific document type in CONST.ALL_DOCUMENT_TYPES required by this field
   * @property {boolean} [embedded]       Does this field require (or prohibit) embedded documents?
   */

  /**
   * A subclass of {@link StringField} which supports referencing some other Document by its UUID.
   * This field may not be blank, but may be null to indicate that no UUID is referenced.
   */
  class DocumentUUIDField extends StringField {
    /**
     * @param {StringFieldOptions & DocumentUUIDFieldOptions} [options] Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(options, context) {
      super(options, context);
    }

    /** @inheritdoc */
    static get _defaults() {
      return Object.assign(super._defaults, {
        required: true,
        blank: false,
        nullable: true,
        initial: null,
        type: undefined,
        embedded: undefined
      });
    }

    /** @override */
    _validateType(value) {
      const p = parseUuid(value);
      if ( this.type ) {
        if ( p.type !== this.type ) throw new Error(`Invalid document type "${p.type}" which must be a "${this.type}"`);
      }
      else if ( p.type && !ALL_DOCUMENT_TYPES.includes(p.type) ) throw new Error(`Invalid document type "${p.type}"`);
      if ( (this.embedded === true) && !p.embedded.length ) throw new Error("must be an embedded document");
      if ( (this.embedded === false) && p.embedded.length ) throw new Error("may not be an embedded document");
      if ( !isValidId(p.documentId) ) throw new Error(`Invalid document ID "${p.documentId}"`);
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      Object.assign(config, {type: this.type, single: true});
      return foundry.applications.elements.HTMLDocumentTagsElement.create(config);
    }
  }

  /* ---------------------------------------- */

  /**
   * A special class of [StringField]{@link StringField} field which references another DataModel by its id.
   * This field may also be null to indicate that no foreign model is linked.
   */
  class ForeignDocumentField extends DocumentIdField {
    /**
     * @param {typeof foundry.abstract.Document} model  The foreign DataModel class definition which this field links to
     * @param {StringFieldOptions} [options]    Options which configure the behavior of the field
     * @param {DataFieldContext} [context]      Additional context which describes the field
     */
    constructor(model, options={}, context={}) {
      super(options, context);
      if ( !isSubclass(model, DataModel) ) {
        throw new Error("A ForeignDocumentField must specify a DataModel subclass as its type");
      }
      /**
       * A reference to the model class which is stored in this field
       * @type {typeof foundry.abstract.Document}
       */
      this.model = model;
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        nullable: true,
        readonly: false,
        idOnly: false
      });
    }

    /** @override */
    _cast(value) {
      if ( typeof value === "string" ) return value;
      if ( (value instanceof this.model) ) return value._id;
      throw new Error(`The value provided to a ForeignDocumentField must be a ${this.model.name} instance.`);
    }

    /** @inheritdoc */
    initialize(value, model, options={}) {
      if ( this.idOnly ) return value;
      if ( model?.pack && !foundry.utils.isSubclass(this.model, foundry.documents.BaseFolder) ) return null;
      if ( !game.collections ) return value; // server-side
      return () => this.model?.get(value, {pack: model?.pack, ...options}) ?? null;
    }

    /** @inheritdoc */
    toObject(value) {
      return value?._id ?? value
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {

      // Prepare array of visible options
      const collection = game.collections.get(this.model.documentName);
      const current = collection.get(config.value);
      let hasCurrent = false;
      const options = collection.reduce((arr, doc) => {
        if ( !doc.visible ) return arr;
        if ( doc === current ) hasCurrent = true;
        arr.push({value: doc.id, label: doc.name});
        return arr;
      }, []);
      if ( current && !hasCurrent ) options.unshift({value: config.value, label: current.name});
      Object.assign(config, {options});

      // Allow blank
      if ( !this.required || this.nullable ) config.blank = "";

      // Create select input
      return foundry.applications.fields.createSelectInput(config);
    }
  }

  /* -------------------------------------------- */

  /**
   * A special [StringField]{@link StringField} which records a standardized CSS color string.
   */
  class ColorField extends StringField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        nullable: true,
        initial: null,
        blank: false,
        validationError: "is not a valid hexadecimal color string"
      });
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( (value === null) || (value === undefined) ) return value;
      return Color.from(value);
    }

    /** @override */
    getInitialValue(data) {
      const value = super.getInitialValue(data);
      if ( (value === undefined) || (value === null) || (value === "") ) return value;
      const color = Color.from(value);
      if ( !color.valid ) throw new Error("Invalid initial value for ColorField");
      return color.css;
    }

    /** @override */
    _cast(value) {
      if ( value === "" ) return value;
      return Color.from(value);
    }

    /** @override */
    _cleanType(value, options) {
      if ( value === "" ) return value;
      if ( value.valid ) return value.css;
      return this.getInitialValue(options.source);
    }

    /** @inheritdoc */
    _validateType(value, options) {
      const result = super._validateType(value, options);
      if ( result !== undefined ) return result;
      if ( !isColorString(value) ) throw new Error("must be a valid color string");
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      if ( (config.placeholder === undefined) && !this.nullable && !(this.initial instanceof Function) ) {
        config.placeholder = this.initial;
      }
      return foundry.applications.elements.HTMLColorPickerElement.create(config);
    }
  }

  /* -------------------------------------------- */

  /**
   * @typedef {StringFieldOptions} FilePathFieldOptions
   * @property {string[]} [categories]    A set of categories in CONST.FILE_CATEGORIES which this field supports
   * @property {boolean} [base64=false]   Is embedded base64 data supported in lieu of a file path?
   * @property {boolean} [wildcard=false] Does this file path field allow wildcard characters?
   * @property {object} [initial]         The initial values of the fields
   */

  /**
   * A special [StringField]{@link StringField} which records a file path or inline base64 data.
   * @property {string[]} categories      A set of categories in CONST.FILE_CATEGORIES which this field supports
   * @property {boolean} base64=false     Is embedded base64 data supported in lieu of a file path?
   * @property {boolean} wildcard=false   Does this file path field allow wildcard characters?
   */
  class FilePathField extends StringField {
    /**
     * @param {FilePathFieldOptions} [options]  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]      Additional context which describes the field
     */
    constructor(options={}, context={}) {
      super(options, context);
      if ( !this.categories.length || this.categories.some(c => !(c in FILE_CATEGORIES)) ) {
        throw new Error("The categories of a FilePathField must be keys in CONST.FILE_CATEGORIES");
      }
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        categories: [],
        base64: false,
        wildcard: false,
        nullable: true,
        blank: false,
        initial: null
      });
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    _validateType(value) {

      // Wildcard paths
      if ( this.wildcard && value.includes("*") ) return true;

      // Allowed extension or base64
      const isValid = this.categories.some(c => {
        const category = FILE_CATEGORIES[c];
        if ( hasFileExtension(value, Object.keys(category)) ) return true;
        /**
         * If the field contains base64 data, it is allowed (for now) regardless of the base64 setting for the field.
         * Eventually, this will become more strict and only be valid if base64 is configured as true for the field.
         * @deprecated since v10
         */
        return isBase64Data(value, Object.values(category));
      });

      // Throw an error for invalid paths
      if ( !isValid ) {
        let err = "does not have a valid file extension";
        if ( this.base64 ) err += " or provide valid base64 data";
        throw new Error(err);
      }
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      // FIXME: This logic is fragile and would require a mapping between CONST.FILE_CATEGORIES and FilePicker.TYPES
      config.type = this.categories.length === 1 ? this.categories[0].toLowerCase() : "any";
      return foundry.applications.elements.HTMLFilePickerElement.create(config);
    }
  }

  /* -------------------------------------------- */

  /**
   * A special {@link NumberField} which represents an angle of rotation in degrees between 0 and 360.
   * @property {boolean} normalize    Whether the angle should be normalized to [0,360) before being clamped to [0,360]. The default is true.
   */
  class AngleField extends NumberField {
    constructor(options={}, context={}) {
      super(options, context);
      if ( "base" in this.options ) this.base = this.options.base;
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        initial: 0,
        normalize: true,
        min: 0,
        max: 360,
        validationError: "is not a number between 0 and 360"
      });
    }

    /** @inheritdoc */
    _cast(value) {
      value = super._cast(value);
      if ( !this.normalize ) return value;
      value = Math.normalizeDegrees(value);
      /** @deprecated since v12 */
      if ( (this.#base === 360) && (value === 0) ) value = 360;
      return value;
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get base() {
      const msg = "The AngleField#base is deprecated in favor of AngleField#normalize.";
      foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
      return this.#base;
    }

    /**
     * @deprecated since v12
     * @ignore
     */
    set base(v) {
      const msg = "The AngleField#base is deprecated in favor of AngleField#normalize.";
      foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
      this.#base = v;
    }

    /**
     * @deprecated since v12
     * @ignore
     */
    #base = 0;
  }

  /* -------------------------------------------- */

  /**
   * A special [NumberField]{@link NumberField} represents a number between 0 and 1.
   */
  class AlphaField extends NumberField {
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        initial: 1,
        min: 0,
        max: 1,
        validationError: "is not a number between 0 and 1"
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * A special [NumberField]{@link NumberField} represents a number between 0 (inclusive) and 1 (exclusive).
   * Its values are normalized (modulo 1) to the range [0, 1) instead of being clamped.
   */
  class HueField extends NumberField {
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        initial: 0,
        min: 0,
        max: 1,
        validationError: "is not a number between 0 (inclusive) and 1 (exclusive)"
      });
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    _cast(value) {
      value = super._cast(value) % 1;
      if ( value < 0 ) value += 1;
      return value;
    }
  }

  /* -------------------------------------------- */

  /**
   * A special [ObjectField]{@link ObjectField} which captures a mapping of User IDs to Document permission levels.
   */
  class DocumentOwnershipField extends ObjectField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        initial: {"default": DOCUMENT_OWNERSHIP_LEVELS.NONE},
        validationError: "is not a mapping of user IDs and document permission levels"
      });
    }

    /** @override */
    _validateType(value) {
      for ( let [k, v] of Object.entries(value) ) {
        if ( k.startsWith("-=") ) return isValidId(k.slice(2)) && (v === null);   // Allow removals
        if ( (k !== "default") && !isValidId(k) ) return false;
        if ( !Object.values(DOCUMENT_OWNERSHIP_LEVELS).includes(v) ) return false;
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * A special [StringField]{@link StringField} which contains serialized JSON data.
   */
  class JSONField extends StringField {
    constructor(options, context) {
      super(options, context);
      this.blank = false;
      this.trim = false;
      this.choices = undefined;
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        blank: false,
        trim: false,
        initial: undefined,
        validationError: "is not a valid JSON string"
      });
    }

    /** @inheritdoc */
    clean(value, options) {
      if ( value === "" ) return '""';  // Special case for JSON fields
      return super.clean(value, options);
    }

    /** @override */
    _cast(value) {
      if ( (typeof value !== "string") || !isJSON(value) ) return JSON.stringify(value);
      return value;
    }

    /** @override */
    _validateType(value, options) {
      if ( (typeof value !== "string") || !isJSON(value) ) throw new Error("must be a serialized JSON string");
    }

    /** @override */
    initialize(value, model, options={}) {
      if ( (value === undefined) || (value === null) ) return value;
      return JSON.parse(value);
    }

    /** @override */
    toObject(value) {
      if ( (value === undefined) || (this.nullable && (value === null)) ) return value;
      return JSON.stringify(value);
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    _toInput(config) {
      if ( config.value !== "" ) config.value = JSON.stringify(config.value, null, 2);
      return foundry.applications.fields.createTextareaInput(config);
    }
  }

  /* -------------------------------------------- */

  /**
   * A special subclass of {@link DataField} which can contain any value of any type.
   * Any input is accepted and is treated as valid.
   * It is not recommended to use this class except for very specific circumstances.
   */
  class AnyField extends DataField {

    /** @override */
    _cast(value) {
      return value;
    }

    /** @override */
    _validateType(value) {
      return true;
    }
  }


  /* -------------------------------------------- */

  /**
   * A subclass of [StringField]{@link StringField} which contains a sanitized HTML string.
   * This class does not override any StringField behaviors, but is used by the server-side to identify fields which
   * require sanitization of user input.
   */
  class HTMLField extends StringField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        blank: true
      });
    }

    /** @override */
    toFormGroup(groupConfig={}, inputConfig) {
      groupConfig.stacked ??= true;
      return super.toFormGroup(groupConfig, inputConfig);
    }

    /** @override */
    _toInput(config) {
      return foundry.applications.elements.HTMLProseMirrorElement.create(config);
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of {@link NumberField} which is used for storing integer sort keys.
   */
  class IntegerSortField extends NumberField {
    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        integer: true,
        initial: 0,
        label: "FOLDER.DocumentSort",
        hint: "FOLDER.DocumentSortHint"
      });
    }
  }

  /* ---------------------------------------- */

  /**
   * @typedef {Object} DocumentStats
   * @property {string|null} coreVersion       The core version whose schema the Document data is in.
   *                                           It is NOT the version the Document was created or last modified in.
   * @property {string|null} systemId          The package name of the system the Document was created in.
   * @property {string|null} systemVersion     The version of the system the Document was created or last modified in.
   * @property {number|null} createdTime       A timestamp of when the Document was created.
   * @property {number|null} modifiedTime      A timestamp of when the Document was last modified.
   * @property {string|null} lastModifiedBy    The ID of the user who last modified the Document.
   * @property {string|null} compendiumSource  The UUID of the compendium Document this one was imported from.
   * @property {string|null} duplicateSource   The UUID of the Document this one is a duplicate of.
   */

  /**
   * A subclass of {@link SchemaField} which stores document metadata in the _stats field.
   * @mixes DocumentStats
   */
  class DocumentStatsField extends SchemaField {
    /**
     * @param {DataFieldOptions} [options]        Options which configure the behavior of the field
     * @param {DataFieldContext} [context]        Additional context which describes the field
     */
    constructor(options={}, context={}) {
      super({
        coreVersion: new StringField({required: true, blank: false, nullable: true, initial: () => game.release.version}),
        systemId: new StringField({required: true, blank: false, nullable: true, initial: () => game.system?.id ?? null}),
        systemVersion: new StringField({required: true, blank: false, nullable: true, initial: () => game.system?.version ?? null}),
        createdTime: new NumberField(),
        modifiedTime: new NumberField(),
        lastModifiedBy: new ForeignDocumentField(foundry.documents.BaseUser, {idOnly: true}),
        compendiumSource: new DocumentUUIDField(),
        duplicateSource: new DocumentUUIDField()
      }, options, context);
    }

    /**
     * All Document stats.
     * @type {string[]}
     */
    static fields = [
      "coreVersion", "systemId", "systemVersion", "createdTime", "modifiedTime", "lastModifiedBy", "compendiumSource",
      "duplicateSource"
    ];

    /**
     * These fields are managed by the server and are ignored if they appear in creation or update data.
     * @type {string[]}
     */
    static managedFields = ["coreVersion", "systemId", "systemVersion", "createdTime", "modifiedTime", "lastModifiedBy"];
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [StringField]{@link StringField} that is used specifically for the Document "type" field.
   */
  class DocumentTypeField extends StringField {
    /**
     * @param {typeof foundry.abstract.Document} documentClass  The base document class which belongs in this field
     * @param {StringFieldOptions} [options]  Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(documentClass, options={}, context={}) {
      options.choices = () => documentClass.TYPES;
      options.validationError = `is not a valid type for the ${documentClass.documentName} Document class`;
      super(options, context);
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        nullable: false,
        blank: false
      });
    }

    /** @override */
    _validateType(value, options) {
      if ( (typeof value !== "string") || !value ) throw new Error("must be a non-blank string");
      if ( this._isValidChoice(value) ) return true;
      // Allow unrecognized types if we are allowed to fallback (non-strict validation)
      if (options.fallback ) return true;
      throw new Error(`"${value}" ${this.options.validationError}`);
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [ObjectField]{@link ObjectField} which supports a type-specific data object.
   */
  class TypeDataField extends ObjectField {
    /**
     * @param {typeof foundry.abstract.Document} document  The base document class which belongs in this field
     * @param {DataFieldOptions} [options]    Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(document, options={}, context={}) {
      super(options, context);
      /**
       * The canonical document name of the document type which belongs in this field
       * @type {typeof foundry.abstract.Document}
       */
      this.document = document;
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {required: true});
    }

    /** @override */
    static recursive = true;

    /**
     * Return the package that provides the sub-type for the given model.
     * @param {DataModel} model       The model instance created for this sub-type.
     * @returns {System|Module|null}
     */
    static getModelProvider(model) {
      const document = model.parent;
      if ( !document ) return null;
      const documentClass = document.constructor;
      const documentName = documentClass.documentName;
      const type = document.type;

      // Unrecognized type
      if ( !documentClass.TYPES.includes(type) ) return null;

      // Core-defined sub-type
      const coreTypes = documentClass.metadata.coreTypes;
      if ( coreTypes.includes(type) ) return null;

      // System-defined sub-type
      const systemTypes = game.system.documentTypes[documentName];
      if ( systemTypes && (type in systemTypes) ) return game.system;

      // Module-defined sub-type
      const moduleId = type.substring(0, type.indexOf("."));
      return game.modules.get(moduleId) ?? null;
    }

    /**
     * A convenience accessor for the name of the document type associated with this TypeDataField
     * @type {string}
     */
    get documentName() {
      return this.document.documentName;
    }

    /**
     * Get the DataModel definition that should be used for this type of document.
     * @param {string} type              The Document instance type
     * @returns {typeof DataModel|null}  The DataModel class or null
     */
    getModelForType(type) {
      if ( !type ) return null;
      return globalThis.CONFIG?.[this.documentName]?.dataModels?.[type] ?? null;
    }

    /** @override */
    getInitialValue(data) {
      const cls = this.getModelForType(data.type);
      if ( cls ) return cls.cleanData();
      const template = game?.model[this.documentName]?.[data.type];
      if ( template ) return foundry.utils.deepClone(template);
      return {};
    }

    /** @override */
    _cleanType(value, options) {
      if ( !(typeof value === "object") ) value = {};

      // Use a defined DataModel
      const type = options.source?.type;
      const cls = this.getModelForType(type);
      if ( cls ) return cls.cleanData(value, {...options, source: value});
      if ( options.partial ) return value;

      // Use the defined template.json
      const template = this.getInitialValue(options.source);
      const insertKeys = (type === BASE_DOCUMENT_TYPE) || !game?.system?.strictDataCleaning;
      return mergeObject(template, value, {insertKeys, inplace: true});
    }

    /** @override */
    initialize(value, model, options={}) {
      const cls = this.getModelForType(model._source.type);
      if ( cls ) {
        const instance = new cls(value, {parent: model, ...options});
        if ( !("modelProvider" in instance) ) Object.defineProperty(instance, "modelProvider", {
          value: this.constructor.getModelProvider(instance),
          writable: false
        });
        return instance;
      }
      return deepClone(value);
    }

    /** @inheritdoc */
    _validateType(data, options={}) {
      const result = super._validateType(data, options);
      if ( result !== undefined ) return result;
      const cls = this.getModelForType(options.source?.type);
      const schema = cls?.schema;
      return schema?.validate(data, {...options, source: data});
    }

    /* ---------------------------------------- */

    /** @override */
    _validateModel(changes, options={}) {
      const cls = this.getModelForType(options.source?.type);
      return cls?.validateJoint(changes);
    }

    /* ---------------------------------------- */

    /** @override */
    toObject(value) {
      return value.toObject instanceof Function ? value.toObject(false) : deepClone(value);
    }

    /**
     * Migrate this field's candidate source data.
     * @param {object} sourceData   Candidate source data of the root model
     * @param {any} fieldData       The value of this field within the source data
     */
    migrateSource(sourceData, fieldData) {
      const cls = this.getModelForType(sourceData.type);
      if ( cls ) cls.migrateDataSafe(fieldData);
    }
  }

  /* ---------------------------------------- */

  /**
   * A subclass of [DataField]{@link DataField} which allows to typed schemas.
   */
  class TypedSchemaField extends DataField {
    /**
     * @param {{[type: string]: DataSchema|SchemaField|typeof DataModel}} types    The different types this field can represent.
     * @param {DataFieldOptions} [options]                                         Options which configure the behavior of the field
     * @param {DataFieldContext} [context]                                         Additional context which describes the field
     */
    constructor(types, options, context) {
      super(options, context);
      this.types = this.#configureTypes(types);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {required: true});
    }

    /* ---------------------------------------- */

    /**
    * The types of this field.
    * @type {{[type: string]: SchemaField}}
    */
    types;

    /* -------------------------------------------- */

    /**
     * Initialize and validate the structure of the provided type definitions.
     * @param {{[type: string]: DataSchema|SchemaField|typeof DataModel}} types    The provided field definitions
     * @returns {{[type: string]: SchemaField}}                                     The validated fields
     */
    #configureTypes(types) {
      if ( (typeof types !== "object") ) {
        throw new Error("A DataFields must be an object with string keys and DataField values.");
      }
      types = {...types};
      for ( let [type, field] of Object.entries(types) ) {
        if ( isSubclass(field, DataModel) ) field = new EmbeddedDataField(field);
        if ( field?.constructor?.name === "Object" ) {
          const schema = {...field};
          if ( !("type" in schema) ) {
            schema.type = new StringField({required: true, blank: false, initial: field,
              validate: value => value === type, validationError: `must be equal to "${type}"`});
          }
          field = new SchemaField(schema);
        }
        if ( !(field instanceof SchemaField)  ) {
          throw new Error(`The "${type}" field is not an instance of the SchemaField class or a subclass of DataModel.`);
        }
        if ( field.name !== undefined ) throw new Error(`The "${field.fieldPath}" field must not have a name.`);
        if ( field.parent !== undefined ) {
          throw new Error(`The "${field.fieldPath}" field already belongs to some other parent and may not be reused.`);
        }
        types[type] = field;
        field.parent = this;
        if ( !field.required ) throw new Error(`The "${field.fieldPath}" field must be required.`);
        if ( field.nullable ) throw new Error(`The "${field.fieldPath}" field must not be nullable.`);
        const typeField = field.fields.type;
        if ( !(typeField instanceof StringField) ) throw new Error(`The "${field.fieldPath}" field must have a "type" StringField.`);
        if ( !typeField.required ) throw new Error(`The "${typeField.fieldPath}" field must be required.`);
        if ( typeField.nullable ) throw new Error(`The "${typeField.fieldPath}" field must not be nullable.`);
        if ( typeField.blank ) throw new Error(`The "${typeField.fieldPath}" field must not be blank.`);
        if ( typeField.validate(type, {fallback: false}) !== undefined ) throw new Error(`"${type}" must be a valid type of "${typeField.fieldPath}".`);
      }
      return types;
    }

    /* ---------------------------------------- */

    /** @override */
    _getField(path) {
      if ( !path.length ) return this;
      return this.types[path.shift()]?._getField(path);
    }

    /* -------------------------------------------- */
    /*  Data Field Methods                          */
    /* -------------------------------------------- */

    /** @override */
    _cleanType(value, options) {
      const field = this.types[value?.type];
      if ( !field ) return value;
      return field.clean(value, options);
    }

    /* ---------------------------------------- */

    /** @override */
    _cast(value) {
      return typeof value === "object" ? value : {};
    }

    /* ---------------------------------------- */

    /** @override */
    _validateSpecial(value) {
      const result = super._validateSpecial(value);
      if ( result !== undefined ) return result;
      const field = this.types[value?.type];
      if ( !field ) throw new Error("does not have a valid type");
    }

    /* ---------------------------------------- */

    /** @override */
    _validateType(value, options) {
      return this.types[value.type].validate(value, options);
    }

    /* ---------------------------------------- */

    /** @override */
    initialize(value, model, options) {
      const field = this.types[value?.type];
      if ( !field ) return value;
      return field.initialize(value, model, options);
    }

    /* ---------------------------------------- */

    /** @override */
    toObject(value) {
      if ( !value ) return value;
      return this.types[value.type]?.toObject(value) ?? value;
    }

    /* -------------------------------------------- */

    /** @override */
    apply(fn, data, options) {

      // Apply to this TypedSchemaField
      const thisFn = typeof fn === "string" ? this[fn] : fn;
      thisFn?.call(this, data, options);

      // Apply to the inner typed field
      const typeField = this.types[data?.type];
      return typeField?.apply(fn, data, options);
    }

    /* -------------------------------------------- */

    /**
     * Migrate this field's candidate source data.
     * @param {object} sourceData   Candidate source data of the root model
     * @param {any} fieldData       The value of this field within the source data
     */
    migrateSource(sourceData, fieldData) {
      const field = this.types[fieldData?.type];
      const canMigrate = field?.migrateSource instanceof Function;
      if ( canMigrate ) field.migrateSource(sourceData, fieldData);
    }
  }

  /* ---------------------------------------- */
  /*  DEPRECATIONS                            */
  /* ---------------------------------------- */

  /**
   * @deprecated since v11
   * @see DataModelValidationError
   * @ignore
   */
  class ModelValidationError extends Error {
    constructor(errors) {
      logCompatibilityWarning(
        "ModelValidationError is deprecated. Please use DataModelValidationError instead.",
        {since: 11, until: 13});
      const message = ModelValidationError.formatErrors(errors);
      super(message);
      this.errors = errors;
    }

    /**
     * Collect all the errors into a single message for consumers who do not handle the ModelValidationError specially.
     * @param {Record<string, Error>|Error[]|string} errors   The raw error structure
     * @returns {string}                              A formatted error message
     */
    static formatErrors(errors) {
      if ( typeof errors === "string" ) return errors;
      const message = ["Model Validation Errors"];
      if ( errors instanceof Array ) message.push(...errors.map(e => e.message));
      else message.push(...Object.entries(errors).map(([k, e]) => `[${k}]: ${e.message}`));
      return message.join("\n");
    }
  }

  /* -------------------------------------------- */


  /**
   * @typedef {Object} JavaScriptFieldOptions
   * @property {boolean} [async=false]            Does the field allow async code?
   */

  /**
   * A subclass of {@link StringField} which contains JavaScript code.
   */
  class JavaScriptField extends StringField {
    /**
     * @param {StringFieldOptions & JavaScriptFieldOptions} [options] Options which configure the behavior of the field
     * @param {DataFieldContext} [context]    Additional context which describes the field
     */
    constructor(options, context) {
      super(options, context);
      this.choices = undefined;
    }

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        required: true,
        blank: true,
        nullable: false,
        async: false
      });
    }

    /** @inheritdoc */
    _validateType(value, options) {
      const result = super._validateType(value, options);
      if ( result !== undefined ) return result;
      try {
        new (this.async ? AsyncFunction : Function)(value);
      } catch(err) {
        const scope = this.async ? "an asynchronous" : "a synchronous";
        err.message = `must be valid JavaScript for ${scope} scope:\n${err.message}`;
        throw new Error(err);
      }
    }

    /* -------------------------------------------- */
    /*  Form Field Integration                      */
    /* -------------------------------------------- */

    /** @override */
    toFormGroup(groupConfig={}, inputConfig) {
      groupConfig.stacked ??= true;
      return super.toFormGroup(groupConfig, inputConfig);
    }

    /** @override */
    _toInput(config) {
      return foundry.applications.fields.createTextareaInput(config);
    }
  }

  var fields$1 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    AlphaField: AlphaField,
    AngleField: AngleField,
    AnyField: AnyField,
    ArrayField: ArrayField,
    BooleanField: BooleanField,
    ColorField: ColorField,
    DataField: DataField,
    DocumentIdField: DocumentIdField,
    DocumentOwnershipField: DocumentOwnershipField,
    DocumentStatsField: DocumentStatsField,
    DocumentTypeField: DocumentTypeField,
    DocumentUUIDField: DocumentUUIDField,
    EmbeddedCollectionDeltaField: EmbeddedCollectionDeltaField,
    EmbeddedCollectionField: EmbeddedCollectionField,
    EmbeddedDataField: EmbeddedDataField,
    EmbeddedDocumentField: EmbeddedDocumentField,
    FilePathField: FilePathField,
    ForeignDocumentField: ForeignDocumentField,
    HTMLField: HTMLField,
    HueField: HueField,
    IntegerSortField: IntegerSortField,
    JSONField: JSONField,
    JavaScriptField: JavaScriptField,
    ModelValidationError: ModelValidationError,
    NumberField: NumberField,
    ObjectField: ObjectField,
    SchemaField: SchemaField,
    SetField: SetField,
    StringField: StringField,
    TypeDataField: TypeDataField,
    TypedSchemaField: TypedSchemaField
  });

  /**
   * @typedef {Record<string, DataField>}  DataSchema
   */

  /**
   * @typedef {Object} DataValidationOptions
   * @property {boolean} [strict=true]     Throw an error if validation fails.
   * @property {boolean} [fallback=false]  Attempt to replace invalid values with valid defaults?
   * @property {boolean} [partial=false]   Allow partial source data, ignoring absent fields?
   * @property {boolean} [dropInvalidEmbedded=false]  If true, invalid embedded documents will emit a warning and be
   *                                                  placed in the invalidDocuments collection rather than causing the
   *                                                  parent to be considered invalid.
   */

  /**
   * The abstract base class which defines the data schema contained within a Document.
   * @param {object} [data={}]                    Initial data used to construct the data object. The provided object
   *                                              will be owned by the constructed model instance and may be mutated.
   * @param {DataValidationOptions} [options={}]  Options which affect DataModel construction
   * @param {Document} [options.parent]           A parent DataModel instance to which this DataModel belongs
   * @abstract
   */
  class DataModel {
    constructor(data={}, {parent=null, strict=true, ...options}={}) {

      // Parent model
      Object.defineProperty(this, "parent", {
        value: (() => {
          if ( parent === null ) return null;
          if ( parent instanceof DataModel ) return parent;
          throw new Error("The provided parent must be a DataModel instance");
        })(),
        writable: false,
        enumerable: false
      });

      // Source data
      Object.defineProperty(this, "_source", {
        value: this._initializeSource(data, {strict, ...options}),
        writable: false,
        enumerable: false
      });
      Object.seal(this._source);

      // Additional subclass configurations
      this._configure(options);

      // Data validation and initialization
      const fallback = options.fallback ?? !strict;
      const dropInvalidEmbedded = options.dropInvalidEmbedded ?? !strict;
      this.validate({strict, fallback, dropInvalidEmbedded, fields: true, joint: true});
      this._initialize({strict, ...options});
    }

    /**
     * Configure the data model instance before validation and initialization workflows are performed.
     * @protected
     */
    _configure(options={}) {}

    /* -------------------------------------------- */

    /**
     * The source data object for this DataModel instance.
     * Once constructed, the source object is sealed such that no keys may be added nor removed.
     * @type {object}
     */
    _source;

    /**
     * The defined and cached Data Schema for all instances of this DataModel.
     * @type {SchemaField}
     * @private
     */
    static _schema;

    /**
     * An immutable reverse-reference to a parent DataModel to which this model belongs.
     * @type {DataModel|null}
     */
    parent;

    /* ---------------------------------------- */
    /*  Data Schema                             */
    /* ---------------------------------------- */

    /**
     * Define the data schema for documents of this type.
     * The schema is populated the first time it is accessed and cached for future reuse.
     * @virtual
     * @returns {DataSchema}
     */
    static defineSchema() {
      throw new Error(`The ${this["name"]} subclass of DataModel must define its Document schema`);
    }

    /* ---------------------------------------- */

    /**
     * The Data Schema for all instances of this DataModel.
     * @type {SchemaField}
     */
    static get schema() {
      if ( this.hasOwnProperty("_schema") ) return this._schema;
      const schema = new SchemaField(Object.freeze(this.defineSchema()));
      Object.defineProperty(this, "_schema", {value: schema, writable: false});
      return schema;
    }

    /* ---------------------------------------- */

    /**
     * Define the data schema for this document instance.
     * @type {SchemaField}
     */
    get schema() {
      return this.constructor.schema;
    }

    /* ---------------------------------------- */

    /**
     * Is the current state of this DataModel invalid?
     * The model is invalid if there is any unresolved failure.
     * @type {boolean}
     */
    get invalid() {
      return Object.values(this.#validationFailures).some(f => f?.unresolved);
    }

    /**
     * An array of validation failure instances which may have occurred when this instance was last validated.
     * @type {{fields: DataModelValidationFailure|null, joint: DataModelValidationFailure|null}}
     */
    get validationFailures() {
      return this.#validationFailures;
    }

    #validationFailures = Object.seal({fields: null, joint: null });

    /**
     * A set of localization prefix paths which are used by this DataModel.
     * @type {string[]}
     */
    static LOCALIZATION_PREFIXES = [];

    /* ---------------------------------------- */
    /*  Data Cleaning Methods                   */
    /* ---------------------------------------- */

    /**
     * Initialize the source data for a new DataModel instance.
     * One-time migrations and initial cleaning operations are applied to the source data.
     * @param {object|DataModel} data   The candidate source data from which the model will be constructed
     * @param {object} [options]        Options provided to the model constructor
     * @returns {object}                Migrated and cleaned source data which will be stored to the model instance
     * @protected
     */
    _initializeSource(data, options={}) {
      if ( data instanceof DataModel ) data = data.toObject();
      const dt = getType(data);
      if ( dt !== "Object" ) {
        logger.error(`${this.constructor.name} was incorrectly constructed with a ${dt} instead of an object. 
      Attempting to fall back to default values.`);
        data = {};
      }
      data = this.constructor.migrateDataSafe(data);    // Migrate old data to the new format
      data = this.constructor.cleanData(data);          // Clean the data in the new format
      return this.constructor.shimData(data);           // Apply shims which preserve backwards compatibility
    }

    /* ---------------------------------------- */

    /**
     * Clean a data source object to conform to a specific provided schema.
     * @param {object} [source]         The source data object
     * @param {object} [options={}]     Additional options which are passed to field cleaning methods
     * @returns {object}                The cleaned source data
     */
    static cleanData(source={}, options={}) {
      return this.schema.clean(source, options);
    }

    /* ---------------------------------------- */
    /*  Data Initialization                     */
    /* ---------------------------------------- */

    /**
     * A generator that orders the DataFields in the DataSchema into an expected initialization order.
     * @returns {Generator<[string,DataField]>}
     * @protected
     */
    static *_initializationOrder() {
      for ( const entry of this.schema.entries() ) yield entry;
    }

    /* ---------------------------------------- */

    /**
     * Initialize the instance by copying data from the source object to instance attributes.
     * This mirrors the workflow of SchemaField#initialize but with some added functionality.
     * @param {object} [options]        Options provided to the model constructor
     * @protected
     */
    _initialize(options={}) {
      for ( let [name, field] of this.constructor._initializationOrder() ) {
        const sourceValue = this._source[name];

        // Field initialization
        const value = field.initialize(sourceValue, this, options);

        // Special handling for Document IDs.
        if ( (name === "_id") && (!Object.getOwnPropertyDescriptor(this, "_id") || (this._id === null)) ) {
          Object.defineProperty(this, name, {value, writable: false, configurable: true});
        }

        // Readonly fields
        else if ( field.readonly ) {
          if ( this[name] !== undefined ) continue;
          Object.defineProperty(this, name, {value, writable: false});
        }

        // Getter fields
        else if ( value instanceof Function ) {
          Object.defineProperty(this, name, {get: value, set() {}, configurable: true});
        }

        // Writable fields
        else this[name] = value;
      }
    }

    /* ---------------------------------------- */

    /**
     * Reset the state of this data instance back to mirror the contained source data, erasing any changes.
     */
    reset() {
      this._initialize();
    }

    /* ---------------------------------------- */

    /**
     * Clone a model, creating a new data model by combining current data with provided overrides.
     * @param {Object} [data={}]                    Additional data which overrides current document data at the time of creation
     * @param {object} [context={}]                 Context options passed to the data model constructor
     * @returns {Document|Promise<Document>}        The cloned Document instance
     */
    clone(data={}, context={}) {
      data = mergeObject(this.toObject(), data, {insertKeys: false, performDeletions: true, inplace: true});
      return new this.constructor(data, {parent: this.parent, ...context});
    }

    /* ---------------------------------------- */
    /*  Data Validation Methods                 */
    /* ---------------------------------------- */

    /**
     * Validate the data contained in the document to check for type and content
     * This function throws an error if data within the document is not valid
     *
     * @param {object} options                    Optional parameters which customize how validation occurs.
     * @param {object} [options.changes]          A specific set of proposed changes to validate, rather than the full
     *                                            source data of the model.
     * @param {boolean} [options.clean=false]     If changes are provided, attempt to clean the changes before validating
     *                                            them?
     * @param {boolean} [options.fallback=false]  Allow replacement of invalid values with valid defaults?
     * @param {boolean} [options.dropInvalidEmbedded=false]  If true, invalid embedded documents will emit a warning and
     *                                                       be placed in the invalidDocuments collection rather than
     *                                                       causing the parent to be considered invalid.
     * @param {boolean} [options.strict=true]     Throw if an invalid value is encountered, otherwise log a warning?
     * @param {boolean} [options.fields=true]     Perform validation on individual fields?
     * @param {boolean} [options.joint]           Perform joint validation on the full data model?
     *                                            Joint validation will be performed by default if no changes are passed.
     *                                            Joint validation will be disabled by default if changes are passed.
     *                                            Joint validation can be performed on a complete set of changes (for
     *                                            example testing a complete data model) by explicitly passing true.
     * @return {boolean}                          An indicator for whether the document contains valid data
     */
    validate({changes, clean=false, fallback=false, dropInvalidEmbedded=false, strict=true, fields=true, joint}={}) {
      const source = changes ?? this._source;
      this.#validationFailures.fields = this.#validationFailures.joint = null; // Remove any prior failures

      // Determine whether we are performing partial or joint validation
      const partial = !!changes;
      joint = joint ?? !changes;
      if ( partial && joint ) {
        throw new Error("It is not supported to perform joint data model validation with only a subset of changes");
      }

      // Optionally clean the data before validating
      if ( partial && clean ) this.constructor.cleanData(source, {partial});

      // Validate individual fields in the data or in a specific change-set, throwing errors if validation fails
      if ( fields ) {
        const failure = this.schema.validate(source, {partial, fallback, dropInvalidEmbedded});
        if ( failure ) {
          const id = this._source._id ? `[${this._source._id}] ` : "";
          failure.message = `${this.constructor.name} ${id}validation errors:`;
          this.#validationFailures.fields = failure;
          if ( strict && failure.unresolved ) throw failure.asError();
          else logger.warn(failure.asError());
        }
      }

      // Perform joint document-level validations which consider all fields together
      if ( joint ) {
        try {
          this.schema._validateModel(source);     // Validate inner models
          this.constructor.validateJoint(source); // Validate this model
        } catch (err) {
          const id = this._source._id ? `[${this._source._id}] ` : "";
          const message = [this.constructor.name, id, `Joint Validation Error:\n${err.message}`].filterJoin(" ");
          const failure = new DataModelValidationFailure({message, unresolved: true});
          this.#validationFailures.joint = failure;
          if ( strict ) throw failure.asError();
          else logger.warn(failure.asError());
        }
      }
      return !this.invalid;
    }

    /* ---------------------------------------- */

    /**
     * Evaluate joint validation rules which apply validation conditions across multiple fields of the model.
     * Field-specific validation rules should be defined as part of the DataSchema for the model.
     * This method allows for testing aggregate rules which impose requirements on the overall model.
     * @param {object} data     Candidate data for the model
     * @throws                  An error if a validation failure is detected
     */
    static validateJoint(data) {
      /**
       * @deprecated since v11
       * @ignore
       */
      if ( this.prototype._validateModel instanceof Function ) {
        const msg = `${this.name} defines ${this.name}.prototype._validateModel instance method which should now be`
                  + ` declared as ${this.name}.validateJoint static method.`;
        foundry.utils.logCompatibilityWarning(msg, {from: 11, until: 13});
        return this.prototype._validateModel.call(this, data);
      }
    }

    /* ---------------------------------------- */
    /*  Data Management                         */
    /* ---------------------------------------- */

    /**
     * Update the DataModel locally by applying an object of changes to its source data.
     * The provided changes are cleaned, validated, and stored to the source data object for this model.
     * The source data is then re-initialized to apply those changes to the prepared data.
     * The method returns an object of differential changes which modified the original data.
     *
     * @param {object} changes          New values which should be applied to the data model
     * @param {object} [options={}]     Options which determine how the new data is merged
     * @returns {object}                An object containing the changed keys and values
     */
    updateSource(changes={}, options={}) {
      const schema = this.schema;
      const source = this._source;
      const _diff = {};
      const _backup = {};
      const _collections = this.collections;
      const _singletons = this.singletons;

      // Expand the object, if dot-notation keys are provided
      if ( Object.keys(changes).some(k => /\./.test(k)) ) changes = expandObject(changes);

      // Clean and validate the provided changes, throwing an error if any change is invalid
      this.validate({changes, clean: true, fallback: options.fallback, strict: true, fields: true, joint: false});

      // Update the source data for all fields and validate the final combined model
      let error;
      try {
        DataModel.#updateData(schema, source, changes, {_backup, _collections, _singletons, _diff, ...options});
        this.validate({fields: this.invalid, joint: true, strict: true});
      } catch(err) {
        error = err;
      }

      // Restore the backup data
      if ( error || options.dryRun ) {
        mergeObject(this._source, _backup, { recursive: false });
        if ( error ) throw error;
      }

      // Initialize the updated data
      if ( !options.dryRun ) this._initialize();
      return _diff;
    }

    /* ---------------------------------------- */

    /**
     * Update the source data for a specific DataSchema.
     * This method assumes that both source and changes are valid objects.
     * @param {SchemaField} schema      The data schema to update
     * @param {object} source           Source data to be updated
     * @param {object} changes          Changes to apply to the source data
     * @param {object} [options={}]     Options which modify the update workflow
     * @returns {object}                The updated source data
     * @throws                          An error if the update operation was unsuccessful
     * @private
     */
    static #updateData(schema, source, changes, options) {
      const {_backup, _diff} = options;
      for ( let [name, value] of Object.entries(changes) ) {
        const field = schema.get(name);
        if ( !field ) continue;

        // Skip updates where the data is unchanged
        const prior = source[name];
        if ( (value?.equals instanceof Function) && value.equals(prior) ) continue;  // Arrays, Sets, etc...
        if ( (prior === value) ) continue; // Direct comparison
        _backup[name] = deepClone(prior);
        _diff[name] = value;

        // Field-specific updating logic
        this.#updateField(name, field, source, value, options);
      }
      return source;
    }

    /* ---------------------------------------- */

    /**
     * Update the source data for a specific DataField.
     * @param {string} name             The field name being updated
     * @param {DataField} field         The field definition being updated
     * @param {object} source           The source object being updated
     * @param {*} value                 The new value for the field
     * @param {object} options          Options which modify the update workflow
     * @throws                          An error if the new candidate value is invalid
     * @private
     */
    static #updateField(name, field, source, value, options) {
      const {dryRun, fallback, recursive, restoreDelta, _collections, _singletons, _diff, _backup} = options;
      let current = source?.[name];   // The current value may be null or undefined

      // Special Case: Update Embedded Collection
      if ( field instanceof EmbeddedCollectionField ) {
        _backup[name] = current;
        if ( !dryRun ) _collections[name].update(value, {fallback, recursive, restoreDelta});
        return;
      }

      // Special Case: Update Embedded Document
      if ( (field instanceof EmbeddedDocumentField) && _singletons[name] ) {
        _diff[name] = _singletons[name].updateSource(value ?? {}, {dryRun, fallback, recursive, restoreDelta});
        if ( isEmpty$1(_diff[name]) ) delete _diff[name];
        return;
      }

      // Special Case: Inner Data Schema
      let innerSchema;
      if ( (field instanceof SchemaField) || (field instanceof EmbeddedDataField) ) innerSchema = field;
      else if ( field instanceof TypeDataField ) {
        const cls = field.getModelForType(source.type);
        if ( cls ) {
          innerSchema = cls.schema;
          if ( dryRun ) {
            _backup[name] = current;
            current = deepClone(current);
          }
        }
      }
      if ( innerSchema && current && value ) {
        _diff[name] = {};
        const recursiveOptions = {fallback, recursive, _backup: current, _collections, _diff: _diff[name]};
        this.#updateData(innerSchema, current, value, recursiveOptions);
        if ( isEmpty$1(_diff[name]) ) delete _diff[name];
      }

      // Special Case: Object Field
      else if ( (field instanceof ObjectField) && current && value && (recursive !== false) ) {
        _diff[name] = diffObject(current, value);
        mergeObject(current, value, {insertKeys: true, insertValues: true, performDeletions: true});
        if ( isEmpty$1(_diff[name]) ) delete _diff[name];
      }

      // Standard Case: Update Directly
      else source[name] = value;
    }

    /* ---------------------------------------- */
    /*  Serialization and Storage               */
    /* ---------------------------------------- */

    /**
     * Copy and transform the DataModel into a plain object.
     * Draw the values of the extracted object from the data source (by default) otherwise from its transformed values.
     * @param {boolean} [source=true]     Draw values from the underlying data source rather than transformed values
     * @returns {object}                  The extracted primitive object
     */
    toObject(source=true) {
      if ( source ) return deepClone(this._source);

      // We have use the schema of the class instead of the schema of the instance to prevent an infinite recursion:
      // the EmbeddedDataField replaces the schema of its model instance with itself
      // and EmbeddedDataField#toObject calls DataModel#toObject.
      return this.constructor.schema.toObject(this);
    }

    /* ---------------------------------------- */

    /**
     * Extract the source data for the DataModel into a simple object format that can be serialized.
     * @returns {object}          The document source data expressed as a plain object
     */
    toJSON() {
      return this.toObject(true);
    }

    /* -------------------------------------------- */

    /**
     * Create a new instance of this DataModel from a source record.
     * The source is presumed to be trustworthy and is not strictly validated.
     * @param {object} source                    Initial document data which comes from a trusted source.
     * @param {DocumentConstructionContext & DataValidationOptions} [context]  Model construction context
     * @param {boolean} [context.strict=false]   Models created from trusted source data are validated non-strictly
     * @returns {DataModel}
     */
    static fromSource(source, {strict=false, ...context}={}) {
      return new this(source, {strict, ...context});
    }

    /* ---------------------------------------- */

    /**
     * Create a DataModel instance using a provided serialized JSON string.
     * @param {string} json       Serialized document data in string format
     * @returns {DataModel}       A constructed data model instance
     */
    static fromJSON(json) {
      return this.fromSource(JSON.parse(json))
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /**
     * Migrate candidate source data for this DataModel which may require initial cleaning or transformations.
     * @param {object} source           The candidate source data from which the model will be constructed
     * @returns {object}                Migrated source data, if necessary
     */
    static migrateData(source) {
      if ( !source ) return source;
      this.schema.migrateSource(source, source);
      return source;
    }

    /* ---------------------------------------- */

    /**
     * Wrap data migration in a try/catch which attempts it safely
     * @param {object} source           The candidate source data from which the model will be constructed
     * @returns {object}                Migrated source data, if necessary
     */
    static migrateDataSafe(source) {
      try {
        this.migrateData(source);
      } catch(err) {
        err.message = `Failed data migration for ${this.name}: ${err.message}`;
        logger.warn(err);
      }
      return source;
    }

    /* ---------------------------------------- */

    /**
     * Take data which conforms to the current data schema and add backwards-compatible accessors to it in order to
     * support older code which uses this data.
     * @param {object} data         Data which matches the current schema
     * @param {object} [options={}] Additional shimming options
     * @param {boolean} [options.embedded=true] Apply shims to embedded models?
     * @returns {object}            Data with added backwards-compatible properties
     */
    static shimData(data, {embedded=true}={}) {
      if ( Object.isSealed(data) ) return data;
      const schema = this.schema;
      if ( embedded ) {
        for ( const [name, value] of Object.entries(data) ) {
          const field = schema.get(name);
          if ( (field instanceof EmbeddedDataField) && !Object.isSealed(value) ) {
            data[name] = field.model.shimData(value || {});
          }
          else if ( field instanceof EmbeddedCollectionField ) {
            for ( const d of (value || []) ) {
              if ( !Object.isSealed(d) ) field.model.shimData(d);
            }
          }
        }
      }
      return data;
    }
  }

  /**
   * A specialized subclass of DataModel, intended to represent a Document's type-specific data.
   * Systems or Modules that provide DataModel implementations for sub-types of Documents (such as Actors or Items)
   * should subclass this class instead of the base DataModel class.
   *
   * @see {@link Document}
   * @extends {DataModel}
   * @abstract
   *
   * @example Registering a custom sub-type for a Module.
   *
   * **module.json**
   * ```json
   * {
   *   "id": "my-module",
   *   "esmodules": ["main.mjs"],
   *   "documentTypes": {
   *     "Actor": {
   *       "sidekick": {},
   *       "villain": {}
   *     },
   *     "JournalEntryPage": {
   *       "dossier": {},
   *       "quest": {
   *         "htmlFields": ["description"]
   *       }
   *     }
   *   }
   * }
   * ```
   *
   * **main.mjs**
   * ```js
   * Hooks.on("init", () => {
   *   Object.assign(CONFIG.Actor.dataModels, {
   *     "my-module.sidekick": SidekickModel,
   *     "my-module.villain": VillainModel
   *   });
   *   Object.assign(CONFIG.JournalEntryPage.dataModels, {
   *     "my-module.dossier": DossierModel,
   *     "my-module.quest": QuestModel
   *   });
   * });
   *
   * class QuestModel extends foundry.abstract.TypeDataModel {
   *   static defineSchema() {
   *     const fields = foundry.data.fields;
   *     return {
   *       description: new fields.HTMLField({required: false, blank: true, initial: ""}),
   *       steps: new fields.ArrayField(new fields.StringField())
   *     };
   *   }
   *
   *   prepareDerivedData() {
   *     this.totalSteps = this.steps.length;
   *   }
   * }
   * ```
   */
  class TypeDataModel extends DataModel {

    /** @inheritdoc */
    constructor(data={}, options={}) {
      super(data, options);

      /**
       * The package that is providing this DataModel for the given sub-type.
       * @type {System|Module|null}
       */
      Object.defineProperty(this, "modelProvider", {value: TypeDataField.getModelProvider(this), writable: false});
    }

    /**
     * A set of localization prefix paths which are used by this data model.
     * @type {string[]}
     */
    static LOCALIZATION_PREFIXES = [];

    /* ---------------------------------------- */

    /** @override */
    static get schema() {
      if ( this.hasOwnProperty("_schema") ) return this._schema;
      const schema = super.schema;
      schema.name = "system";
      return schema;
    }

    /* -------------------------------------------- */

    /**
     * Prepare data related to this DataModel itself, before any derived data is computed.
     *
     * Called before {@link ClientDocument#prepareBaseData} in {@link ClientDocument#prepareData}.
     */
    prepareBaseData() {}

    /* -------------------------------------------- */

    /**
     * Apply transformations of derivations to the values of the source data object.
     * Compute data fields whose values are not stored to the database.
     *
     * Called before {@link ClientDocument#prepareDerivedData} in {@link ClientDocument#prepareData}.
     */
    prepareDerivedData() {}

    /* -------------------------------------------- */

    /**
     * Convert this Document to some HTML display for embedding purposes.
     * @param {DocumentHTMLEmbedConfig} config  Configuration for embedding behavior.
     * @param {EnrichmentOptions} [options]     The original enrichment options for cases where the Document embed content
     *                                          also contains text that must be enriched.
     * @returns {Promise<HTMLElement|HTMLCollection|null>}
     */
    async toEmbed(config, options={}) {
      return null;
    }

    /* -------------------------------------------- */
    /*  Database Operations                         */
    /* -------------------------------------------- */

    /**
     * Called by {@link ClientDocument#_preCreate}.
     *
     * @param {object} data                         The initial data object provided to the document creation request
     * @param {object} options                      Additional options which modify the creation request
     * @param {documents.BaseUser} user             The User requesting the document creation
     * @returns {Promise<boolean|void>}             Return false to exclude this Document from the creation operation
     * @internal
     */
    async _preCreate(data, options, user) {}

    /* -------------------------------------------- */

    /**
     * Called by {@link ClientDocument#_onCreate}.
     *
     * @param {object} data                         The initial data object provided to the document creation request
     * @param {object} options                      Additional options which modify the creation request
     * @param {string} userId                       The id of the User requesting the document update
     * @protected
     * @internal
     */
    _onCreate(data, options, userId) {}

    /* -------------------------------------------- */

    /**
     * Called by {@link ClientDocument#_preUpdate}.
     *
     * @param {object} changes            The candidate changes to the Document
     * @param {object} options            Additional options which modify the update request
     * @param {documents.BaseUser} user   The User requesting the document update
     * @returns {Promise<boolean|void>}   A return value of false indicates the update operation should be cancelled.
     * @protected
     * @internal
     */
    async _preUpdate(changes, options, user) {}

    /* -------------------------------------------- */

    /**
     * Called by {@link ClientDocument#_onUpdate}.
     *
     * @param {object} changed            The differential data that was changed relative to the documents prior values
     * @param {object} options            Additional options which modify the update request
     * @param {string} userId             The id of the User requesting the document update
     * @protected
     * @internal
     */
    _onUpdate(changed, options, userId) {}

    /* -------------------------------------------- */


    /**
     * Called by {@link ClientDocument#_preDelete}.
     *
     * @param {object} options            Additional options which modify the deletion request
     * @param {documents.BaseUser} user   The User requesting the document deletion
     * @returns {Promise<boolean|void>}   A return value of false indicates the deletion operation should be cancelled.
     * @protected
     * @internal
     */
    async _preDelete(options, user) {}

    /* -------------------------------------------- */

    /**
     * Called by {@link ClientDocument#_onDelete}.
     *
     * @param {object} options            Additional options which modify the deletion request
     * @param {string} userId             The id of the User requesting the document update
     * @protected
     * @internal
     */
    _onDelete(options, userId) {}
  }

  /**
   * An extension of the base DataModel which defines a Document.
   * Documents are special in that they are persisted to the database and referenced by _id.
   * @memberof abstract
   * @abstract
   * @alias foundry.abstract.Document
   *
   * @param {object} data                           Initial data from which to construct the Document
   * @param {DocumentConstructionContext} context   Construction context options
   *
   * @property {string|null} _id                    The document identifier, unique within its Collection, or null if the
   *                                                Document has not yet been assigned an identifier
   * @property {string} [name]                      Documents typically have a human-readable name
   * @property {DataModel} [system]                 Certain document types may have a system data model which contains
   *                                                subtype-specific data defined by the game system or a module
   * @property {DocumentStats} [_stats]             Primary document types have a _stats object which provides metadata
   *                                                about their status
   * @property {Record<string, any>} flags          Documents each have an object of arbitrary flags which are used by
   *                                                systems or modules to store additional Document-specific data
   */
  class Document extends DataModel {

    /** @override */
    _configure({pack=null, parentCollection=null}={}) {
      /**
       * An immutable reverse-reference to the name of the collection that this Document exists in on its parent, if any.
       * @type {string|null}
       */
      Object.defineProperty(this, "parentCollection", {
        value: this._getParentCollection(parentCollection),
        writable: false
      });

      /**
       * An immutable reference to a containing Compendium collection to which this Document belongs.
       * @type {string|null}
       */
      Object.defineProperty(this, "pack", {
        value: (() => {
          if ( typeof pack === "string" ) return pack;
          if ( this.parent?.pack ) return this.parent.pack;
          if ( pack === null ) return null;
          throw new Error("The provided compendium pack ID must be a string");
        })(),
        writable: false
      });

      // Construct Embedded Collections
      const collections = {};
      for ( const [fieldName, field] of Object.entries(this.constructor.hierarchy) ) {
        if ( !field.constructor.implementation ) continue;
        const data = this._source[fieldName];
        const c = collections[fieldName] = new field.constructor.implementation(fieldName, this, data);
        Object.defineProperty(this, fieldName, {value: c, writable: false});
      }

      /**
       * A mapping of embedded Document collections which exist in this model.
       * @type {Record<string, EmbeddedCollection>}
       */
      Object.defineProperty(this, "collections", {value: Object.seal(collections), writable: false});
    }

    /* ---------------------------------------- */

    /**
     * Ensure that all Document classes share the same schema of their base declaration.
     * @type {SchemaField}
     * @override
     */
    static get schema() {
      if ( this._schema ) return this._schema;
      const base = this.baseDocument;
      if ( !base.hasOwnProperty("_schema") ) {
        const schema = new SchemaField(Object.freeze(base.defineSchema()));
        Object.defineProperty(base, "_schema", {value: schema, writable: false});
      }
      Object.defineProperty(this, "_schema", {value: base._schema, writable: false});
      return base._schema;
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    _initialize(options={}) {
      super._initialize(options);

      const singletons = {};
      for ( const [fieldName, field] of Object.entries(this.constructor.hierarchy) ) {
        if ( field instanceof foundry.data.fields.EmbeddedDocumentField ) {
          Object.defineProperty(singletons, fieldName, { get: () => this[fieldName] });
        }
      }

      /**
       * A mapping of singleton embedded Documents which exist in this model.
       * @type {Record<string, Document>}
       */
      Object.defineProperty(this, "singletons", {value: Object.seal(singletons), configurable: true});
    }

    /* -------------------------------------------- */

    /** @override */
    static *_initializationOrder() {
      const hierarchy = this.hierarchy;

      // Initialize non-hierarchical fields first
      for ( const [name, field] of this.schema.entries() ) {
        if ( name in hierarchy ) continue;
        yield [name, field];
      }

      // Initialize hierarchical fields last
      for ( const [name, field] of Object.entries(hierarchy) ) {
        yield [name, field];
      }
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /**
     * Default metadata which applies to each instance of this Document type.
     * @type {object}
     */
    static metadata = Object.freeze({
      name: "Document",
      collection: "documents",
      indexed: false,
      compendiumIndexFields: [],
      label: "DOCUMENT.Document",
      coreTypes: [BASE_DOCUMENT_TYPE],
      embedded: {},
      permissions: {
        create: "ASSISTANT",
        update: "ASSISTANT",
        delete: "ASSISTANT"
      },
      preserveOnImport: ["_id", "sort", "ownership"],
      /*
       * The metadata has to include the version of this Document schema, which needs to be increased
       * whenever the schema is changed such that Document data created before this version
       * would come out different if `fromSource(data).toObject()` was applied to it so that
       * we always vend data to client that is in the schema of the current core version.
       * The schema version needs to be bumped if
       *   - a field was added or removed,
       *   - the class/type of any field was changed,
       *   - the casting or cleaning behavior of any field class was changed,
       *   - the data model of an embedded data field was changed,
       *   - certain field properties are changed (e.g. required, nullable, blank, ...), or
       *   - there have been changes to cleanData or migrateData of the Document.
       *
       * Moreover, the schema version needs to be bumped if the sanitization behavior
       * of any field in the schema was changed.
       */
      schemaVersion: undefined
    });

    /* -------------------------------------------- */

    /**
     * The database backend used to execute operations and handle results.
     * @type {abstract.DatabaseBackend}
     */
    static get database() {
      return globalThis.CONFIG.DatabaseBackend;
    }

    /* -------------------------------------------- */

    /**
     * Return a reference to the configured subclass of this base Document type.
     * @type {typeof Document}
     */
    static get implementation() {
      return globalThis.CONFIG[this.documentName]?.documentClass || this;
    }

    /* -------------------------------------------- */

    /**
     * The base document definition that this document class extends from.
     * @type {typeof Document}
     */
    static get baseDocument() {
      let cls;
      let parent = this;
      while ( parent ) {
        cls = parent;
        parent = Object.getPrototypeOf(cls);
        if ( parent === Document ) return cls;
      }
      throw new Error(`Base Document class identification failed for "${this.documentName}"`);
    }

    /* -------------------------------------------- */

    /**
     * The named collection to which this Document belongs.
     * @type {string}
     */
    static get collectionName() {
      return this.metadata.collection;
    }
    get collectionName() {
      return this.constructor.collectionName;
    }

    /* -------------------------------------------- */

    /**
     * The canonical name of this Document type, for example "Actor".
     * @type {string}
     */
    static get documentName() {
      return this.metadata.name;
    }
    get documentName() {
      return this.constructor.documentName;
    }

    /* ---------------------------------------- */

    /**
     * The allowed types which may exist for this Document class.
     * @type {string[]}
     */
    static get TYPES() {
      return Object.keys(game.model[this.metadata.name]);
    }

    /* -------------------------------------------- */

    /**
     * Does this Document support additional subtypes?
     * @type {boolean}
     */
    static get hasTypeData() {
      return this.metadata.hasTypeData;
    }

    /* -------------------------------------------- */
    /*  Model Properties                            */
    /* -------------------------------------------- */

    /**
     * The Embedded Document hierarchy for this Document.
     * @returns {Readonly<Record<string, EmbeddedCollectionField|EmbeddedDocumentField>>}
     */
    static get hierarchy() {
      const hierarchy = {};
      for ( const [fieldName, field] of this.schema.entries() ) {
        if ( field.constructor.hierarchical ) hierarchy[fieldName] = field;
      }
      Object.defineProperty(this, "hierarchy", {value: Object.freeze(hierarchy), writable: false});
      return hierarchy;
    }

    /* -------------------------------------------- */

    /**
     * Identify the collection in a parent Document that this Document belongs to, if any.
     * @param {string|null} [parentCollection]  An explicitly provided parent collection name.
     * @returns {string|null}
     * @internal
     */
    _getParentCollection(parentCollection) {
      if ( !this.parent ) return null;
      if ( parentCollection ) return parentCollection;
      return this.parent.constructor.getCollectionName(this.documentName);
    }

    /**
     * The canonical identifier for this Document.
     * @type {string|null}
     */
    get id() {
      return this._id;
    }

    /**
     * Test whether this Document is embedded within a parent Document
     * @type {boolean}
     */
    get isEmbedded() {
      return !!(this.parent && this.parentCollection);
    }

    /* -------------------------------------------- */

    /**
     * A Universally Unique Identifier (uuid) for this Document instance.
     * @type {string}
     */
    get uuid() {
      let parts = [this.documentName, this.id];
      if ( this.parent ) parts = [this.parent.uuid].concat(parts);
      else if ( this.pack ) parts = ["Compendium", this.pack].concat(parts);
      return parts.join(".");
    }

    /* ---------------------------------------- */
    /*  Model Permissions                       */
    /* ---------------------------------------- */

    /**
     * Test whether a given User has a sufficient role in order to create Documents of this type in general.
     * @param {documents.BaseUser} user       The User being tested
     * @return {boolean}                      Does the User have a sufficient role to create?
     */
    static canUserCreate(user) {
      // TODO: https://github.com/foundryvtt/foundryvtt/issues/11280
      const perm = this.metadata.permissions.create;
      if ( perm instanceof Function ) {
        throw new Error('Document.canUserCreate is not supported for this document type. ' +
          'Use Document#canUserModify(user, "create") to test whether a user is permitted to create a ' +
          'specific document instead.');
      }
      return user.hasPermission(perm) || user.hasRole(perm, {exact: false});
    }

    /* ---------------------------------------- */

    /**
     * Get the explicit permission level that a User has over this Document, a value in CONST.DOCUMENT_OWNERSHIP_LEVELS.
     * This method returns the value recorded in Document ownership, regardless of the User's role.
     * To test whether a user has a certain capability over the document, testUserPermission should be used.
     * @param {documents.BaseUser} [user=game.user] The User being tested
     * @returns {number|null}               A numeric permission level from CONST.DOCUMENT_OWNERSHIP_LEVELS or null
     */
    getUserLevel(user) {
      user = user || game.user;

      // Compendium content uses role-based ownership
      if ( this.pack ) return this.compendium.getUserLevel(user);

      // World content uses granular per-User ownership
      const ownership = this["ownership"] || {};
      return ownership[user.id] ?? ownership.default ?? null;
    }

    /* ---------------------------------------- */

    /**
     * Test whether a certain User has a requested permission level (or greater) over the Document
     * @param {documents.BaseUser} user       The User being tested
     * @param {string|number} permission      The permission level from DOCUMENT_OWNERSHIP_LEVELS to test
     * @param {object} options                Additional options involved in the permission test
     * @param {boolean} [options.exact=false]     Require the exact permission level requested?
     * @return {boolean}                      Does the user have this permission level over the Document?
     */
    testUserPermission(user, permission, {exact=false}={}) {
      const perms = DOCUMENT_OWNERSHIP_LEVELS;

      let level;
      if ( user.isGM ) level = perms.OWNER;
      else if ( user.isBanned ) level = perms.NONE;
      else level = this.getUserLevel(user);

      const target = (typeof permission === "string") ? (perms[permission] ?? perms.OWNER) : permission;
      return exact ? level === target : level >= target;
    }

    /* ---------------------------------------- */

    /**
     * Test whether a given User has permission to perform some action on this Document
     * @param {documents.BaseUser} user   The User attempting modification
     * @param {string} action             The attempted action
     * @param {object} [data]             Data involved in the attempted action
     * @return {boolean}                  Does the User have permission?
     */
    canUserModify(user, action, data={}) {
      const permissions = this.constructor.metadata.permissions;
      const perm = permissions[action];

      // Specialized permission test function
      if ( perm instanceof Function ) return perm(user, this, data);

      // User-level permission
      else if ( perm in USER_PERMISSIONS ) return user.hasPermission(perm);

      // Document-level permission
      const isOwner = this.testUserPermission(user, "OWNER");
      const hasRole = (perm in USER_ROLES) && user.hasRole(perm);
      return isOwner || hasRole;
    }

    /* ---------------------------------------- */
    /*  Model Methods                           */
    /* ---------------------------------------- */

    /**
     * Clone a document, creating a new document by combining current data with provided overrides.
     * The cloned document is ephemeral and not yet saved to the database.
     * @param {Object} [data={}]                         Additional data which overrides current document data at the time
     *                                                   of creation
     * @param {DocumentConstructionContext} [context={}] Additional context options passed to the create method
     * @param {boolean} [context.save=false]             Save the clone to the World database?
     * @param {boolean} [context.keepId=false]           Keep the same ID of the original document
     * @param {boolean} [context.addSource=false]        Track the clone source.
     * @returns {Document|Promise<Document>}             The cloned Document instance
     */
    clone(data={}, {save=false, keepId=false, addSource=false, ...context}={}) {
      if ( !keepId ) data["-=_id"] = null;
      if ( addSource ) data["_stats.duplicateSource"] = this.uuid;
      context.parent = this.parent;
      context.pack = this.pack;
      context.strict = false;
      const doc = super.clone(data, context);
      return save ? this.constructor.create(doc, context) : doc;
    }

    /* -------------------------------------------- */

    /**
     * For Documents which include game system data, migrate the system data object to conform to its latest data model.
     * The data model is defined by the template.json specification included by the game system.
     * @returns {object}              The migrated system data object
     */
    migrateSystemData() {
      if ( !this.constructor.hasTypeData ) {
        throw new Error(`The ${this.documentName} Document does not include a TypeDataField.`);
      }
      if ( (this.system instanceof DataModel) && !(this.system.modelProvider instanceof System) ) {
        throw new Error(`The ${this.documentName} Document does not have system-provided package data.`);
      }
      const model = game.model[this.documentName]?.[this["type"]] || {};
      return mergeObject(model, this["system"], {
        insertKeys: false,
        insertValues: true,
        enforceTypes: false,
        overwrite: true,
        inplace: false
      });
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    toObject(source=true) {
      const data = super.toObject(source);
      return this.constructor.shimData(data);
    }

    /* -------------------------------------------- */
    /*  Database Operations                         */
    /* -------------------------------------------- */

    /**
     * Create multiple Documents using provided input data.
     * Data is provided as an array of objects where each individual object becomes one new Document.
     *
     * @param {Array<object|Document>} data  An array of data objects or existing Documents to persist.
     * @param {Partial<Omit<DatabaseCreateOperation, "data">>} [operation={}]  Parameters of the requested creation
     *                                  operation
     * @return {Promise<Document[]>}         An array of created Document instances
     *
     * @example Create a single Document
     * ```js
     * const data = [{name: "New Actor", type: "character", img: "path/to/profile.jpg"}];
     * const created = await Actor.createDocuments(data);
     * ```
     *
     * @example Create multiple Documents
     * ```js
     * const data = [{name: "Tim", type: "npc"], [{name: "Tom", type: "npc"}];
     * const created = await Actor.createDocuments(data);
     * ```
     *
     * @example Create multiple embedded Documents within a parent
     * ```js
     * const actor = game.actors.getName("Tim");
     * const data = [{name: "Sword", type: "weapon"}, {name: "Breastplate", type: "equipment"}];
     * const created = await Item.createDocuments(data, {parent: actor});
     * ```
     *
     * @example Create a Document within a Compendium pack
     * ```js
     * const data = [{name: "Compendium Actor", type: "character", img: "path/to/profile.jpg"}];
     * const created = await Actor.createDocuments(data, {pack: "mymodule.mypack"});
     * ```
     */
    static async createDocuments(data=[], operation={}) {
      if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
      operation.data = data;
      const created = await this.database.create(this.implementation, operation);

      /** @deprecated since v12 */
      if ( getDefiningClass(this, "_onCreateDocuments") !== Document ) {
        foundry.utils.logCompatibilityWarning("The Document._onCreateDocuments static method is deprecated in favor of "
          + "Document._onCreateOperation", {since: 12, until: 14});
        await this._onCreateDocuments(created, operation);
      }
      return created;
    }

    /* -------------------------------------------- */

    /**
     * Update multiple Document instances using provided differential data.
     * Data is provided as an array of objects where each individual object updates one existing Document.
     *
     * @param {object[]} updates          An array of differential data objects, each used to update a single Document
     * @param {Partial<Omit<DatabaseUpdateOperation, "updates">>} [operation={}] Parameters of the database update
     *                                    operation
     * @return {Promise<Document[]>}      An array of updated Document instances
     *
     * @example Update a single Document
     * ```js
     * const updates = [{_id: "12ekjf43kj2312ds", name: "Timothy"}];
     * const updated = await Actor.updateDocuments(updates);
     * ```
     *
     * @example Update multiple Documents
     * ```js
     * const updates = [{_id: "12ekjf43kj2312ds", name: "Timothy"}, {_id: "kj549dk48k34jk34", name: "Thomas"}]};
     * const updated = await Actor.updateDocuments(updates);
     * ```
     *
     * @example Update multiple embedded Documents within a parent
     * ```js
     * const actor = game.actors.getName("Timothy");
     * const updates = [{_id: sword.id, name: "Magic Sword"}, {_id: shield.id, name: "Magic Shield"}];
     * const updated = await Item.updateDocuments(updates, {parent: actor});
     * ```
     *
     * @example Update Documents within a Compendium pack
     * ```js
     * const actor = await pack.getDocument(documentId);
     * const updated = await Actor.updateDocuments([{_id: actor.id, name: "New Name"}], {pack: "mymodule.mypack"});
     * ```
     */
    static async updateDocuments(updates=[], operation={}) {
      if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
      operation.updates = updates;
      const updated = await this.database.update(this.implementation, operation);

      /** @deprecated since v12 */
      if ( getDefiningClass(this, "_onUpdateDocuments") !== Document ) {
        foundry.utils.logCompatibilityWarning("The Document._onUpdateDocuments static method is deprecated in favor of "
          + "Document._onUpdateOperation", {since: 12, until: 14});
        await this._onUpdateDocuments(updated, operation);
      }
      return updated;
    }

    /* -------------------------------------------- */

    /**
     * Delete one or multiple existing Documents using an array of provided ids.
     * Data is provided as an array of string ids for the documents to delete.
     *
     * @param {string[]} ids              An array of string ids for the documents to be deleted
     * @param {Partial<Omit<DatabaseDeleteOperation, "ids">>} [operation={}]  Parameters of the database deletion
     *                                    operation
     * @return {Promise<Document[]>}      An array of deleted Document instances
     *
     * @example Delete a single Document
     * ```js
     * const tim = game.actors.getName("Tim");
     * const deleted = await Actor.deleteDocuments([tim.id]);
     * ```
     *
     * @example Delete multiple Documents
     * ```js
     * const tim = game.actors.getName("Tim");
     * const tom = game.actors.getName("Tom");
     * const deleted = await Actor.deleteDocuments([tim.id, tom.id]);
     * ```
     *
     * @example Delete multiple embedded Documents within a parent
     * ```js
     * const tim = game.actors.getName("Tim");
     * const sword = tim.items.getName("Sword");
     * const shield = tim.items.getName("Shield");
     * const deleted = await Item.deleteDocuments([sword.id, shield.id], parent: actor});
     * ```
     *
     * @example Delete Documents within a Compendium pack
     * ```js
     * const actor = await pack.getDocument(documentId);
     * const deleted = await Actor.deleteDocuments([actor.id], {pack: "mymodule.mypack"});
     * ```
     */
    static async deleteDocuments(ids=[], operation={}) {
      if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
      operation.ids = ids;
      const deleted = await this.database.delete(this.implementation, operation);

      /** @deprecated since v12 */
      if ( getDefiningClass(this, "_onDeleteDocuments") !== Document ) {
        foundry.utils.logCompatibilityWarning("The Document._onDeleteDocuments static method is deprecated in favor of "
          + "Document._onDeleteOperation", {since: 12, until: 14});
        await this._onDeleteDocuments(deleted, operation);
      }
      return deleted;
    }

    /* -------------------------------------------- */

    /**
     * Create a new Document using provided input data, saving it to the database.
     * @see Document.createDocuments
     * @param {object|Document|(object|Document)[]} [data={}] Initial data used to create this Document, or a Document
     *                                                        instance to persist.
     * @param {Partial<Omit<DatabaseCreateOperation, "data">>} [operation={}]  Parameters of the creation operation
     * @returns {Promise<Document | Document[] | undefined>}        The created Document instance
     *
     * @example Create a World-level Item
     * ```js
     * const data = [{name: "Special Sword", type: "weapon"}];
     * const created = await Item.create(data);
     * ```
     *
     * @example Create an Actor-owned Item
     * ```js
     * const data = [{name: "Special Sword", type: "weapon"}];
     * const actor = game.actors.getName("My Hero");
     * const created = await Item.create(data, {parent: actor});
     * ```
     *
     * @example Create an Item in a Compendium pack
     * ```js
     * const data = [{name: "Special Sword", type: "weapon"}];
     * const created = await Item.create(data, {pack: "mymodule.mypack"});
     * ```
     */
    static async create(data, operation={}) {
      const createData = data instanceof Array ? data : [data];
      const created = await this.createDocuments(createData, operation);
      return data instanceof Array ? created : created.shift();
    }

    /* -------------------------------------------- */

    /**
     * Update this Document using incremental data, saving it to the database.
     * @see Document.updateDocuments
     * @param {object} [data={}]          Differential update data which modifies the existing values of this document
     * @param {Partial<Omit<DatabaseUpdateOperation, "updates">>} [operation={}]  Parameters of the update operation
     * @returns {Promise<Document>}       The updated Document instance
     */
    async update(data={}, operation={}) {
      data._id = this.id;
      operation.parent = this.parent;
      operation.pack = this.pack;
      const updates = await this.constructor.updateDocuments([data], operation);
      return updates.shift();
    }

    /* -------------------------------------------- */

    /**
     * Delete this Document, removing it from the database.
     * @see Document.deleteDocuments
     * @param {Partial<Omit<DatabaseDeleteOperation, "ids">>} [operation={}]  Parameters of the deletion operation
     * @returns {Promise<Document>}       The deleted Document instance
     */
    async delete(operation={}) {
      operation.parent = this.parent;
      operation.pack = this.pack;
      const deleted = await this.constructor.deleteDocuments([this.id], operation);
      return deleted.shift();
    }

    /* -------------------------------------------- */

    /**
     * Get a World-level Document of this type by its id.
     * @param {string} documentId         The Document ID
     * @param {DatabaseGetOperation} [operation={}] Parameters of the get operation
     * @returns {abstract.Document|null}  The retrieved Document, or null
     */
    static get(documentId, operation={}) {
      if ( !documentId ) return null;
      if ( operation.pack ) {
        const pack = game.packs.get(operation.pack);
        return pack?.index.get(documentId) || null;
      }
      else {
        const collection = game.collections?.get(this.documentName);
        return collection?.get(documentId) || null;
      }
    }

    /* -------------------------------------------- */
    /*  Embedded Operations                         */
    /* -------------------------------------------- */

    /**
     * A compatibility method that returns the appropriate name of an embedded collection within this Document.
     * @param {string} name    An existing collection name or a document name.
     * @returns {string|null}  The provided collection name if it exists, the first available collection for the
     *                         document name provided, or null if no appropriate embedded collection could be found.
     * @example Passing an existing collection name.
     * ```js
     * Actor.getCollectionName("items");
     * // returns "items"
     * ```
     *
     * @example Passing a document name.
     * ```js
     * Actor.getCollectionName("Item");
     * // returns "items"
     * ```
     */
    static getCollectionName(name) {
      if ( name in this.hierarchy ) return name;
      for ( const [collectionName, field] of Object.entries(this.hierarchy) ) {
        if ( field.model.documentName === name ) return collectionName;
      }
      return null;
    }

    /* -------------------------------------------- */

    /**
     * Obtain a reference to the Array of source data within the data object for a certain embedded Document name
     * @param {string} embeddedName   The name of the embedded Document type
     * @return {DocumentCollection}   The Collection instance of embedded Documents of the requested type
     */
    getEmbeddedCollection(embeddedName) {
      const collectionName = this.constructor.getCollectionName(embeddedName);
      if ( !collectionName ) {
        throw new Error(`${embeddedName} is not a valid embedded Document within the ${this.documentName} Document`);
      }
      const field = this.constructor.hierarchy[collectionName];
      return field.getCollection(this);
    }

    /* -------------------------------------------- */

    /**
     * Get an embedded document by its id from a named collection in the parent document.
     * @param {string} embeddedName              The name of the embedded Document type
     * @param {string} id                        The id of the child document to retrieve
     * @param {object} [options]                 Additional options which modify how embedded documents are retrieved
     * @param {boolean} [options.strict=false]   Throw an Error if the requested id does not exist. See Collection#get
     * @param {boolean} [options.invalid=false]  Allow retrieving an invalid Embedded Document.
     * @return {Document}                        The retrieved embedded Document instance, or undefined
     * @throws If the embedded collection does not exist, or if strict is true and the Embedded Document could not be
     *         found.
     */
    getEmbeddedDocument(embeddedName, id, {invalid=false, strict=false}={}) {
      const collection = this.getEmbeddedCollection(embeddedName);
      return collection.get(id, {invalid, strict});
    }

    /* -------------------------------------------- */

    /**
     * Create multiple embedded Document instances within this parent Document using provided input data.
     * @see Document.createDocuments
     * @param {string} embeddedName                     The name of the embedded Document type
     * @param {object[]} data                           An array of data objects used to create multiple documents
     * @param {DatabaseCreateOperation} [operation={}]  Parameters of the database creation workflow
     * @return {Promise<Document[]>}                    An array of created Document instances
     */
    async createEmbeddedDocuments(embeddedName, data=[], operation={}) {
      this.getEmbeddedCollection(embeddedName); // Validation only
      operation.parent = this;
      operation.pack = this.pack;
      const cls = getDocumentClass(embeddedName);
      return cls.createDocuments(data, operation);
    }

    /* -------------------------------------------- */

    /**
     * Update multiple embedded Document instances within a parent Document using provided differential data.
     * @see Document.updateDocuments
     * @param {string} embeddedName                     The name of the embedded Document type
     * @param {object[]} updates                        An array of differential data objects, each used to update a
     *                                                  single Document
     * @param {DatabaseUpdateOperation} [operation={}]  Parameters of the database update workflow
     * @return {Promise<Document[]>}                    An array of updated Document instances
     */
    async updateEmbeddedDocuments(embeddedName, updates=[], operation={}) {
      this.getEmbeddedCollection(embeddedName); // Validation only
      operation.parent = this;
      operation.pack = this.pack;
      const cls = getDocumentClass(embeddedName);
      return cls.updateDocuments(updates, operation);
    }

    /* -------------------------------------------- */

    /**
     * Delete multiple embedded Document instances within a parent Document using provided string ids.
     * @see Document.deleteDocuments
     * @param {string} embeddedName                     The name of the embedded Document type
     * @param {string[]} ids                            An array of string ids for each Document to be deleted
     * @param {DatabaseDeleteOperation} [operation={}]  Parameters of the database deletion workflow
     * @return {Promise<Document[]>}                    An array of deleted Document instances
     */
    async deleteEmbeddedDocuments(embeddedName, ids, operation={}) {
      this.getEmbeddedCollection(embeddedName); // Validation only
      operation.parent = this;
      operation.pack = this.pack;
      const cls = getDocumentClass(embeddedName);
      return cls.deleteDocuments(ids, operation);
    }

    /* -------------------------------------------- */

    /**
     * Iterate over all embedded Documents that are hierarchical children of this Document.
     * @param {string} [_parentPath]                      A parent field path already traversed
     * @returns {Generator<[string, Document]>}
     */
    * traverseEmbeddedDocuments(_parentPath) {
      for ( const [fieldName, field] of Object.entries(this.constructor.hierarchy) ) {
        let fieldPath = _parentPath ? `${_parentPath}.${fieldName}` : fieldName;

        // Singleton embedded document
        if ( field instanceof foundry.data.fields.EmbeddedDocumentField ) {
          const document = this[fieldName];
          if ( document ) {
            yield [fieldPath, document];
            yield* document.traverseEmbeddedDocuments(fieldPath);
          }
        }

        // Embedded document collection
        else if ( field instanceof foundry.data.fields.EmbeddedCollectionField ) {
          const collection = this[fieldName];
          const isDelta = field instanceof foundry.data.fields.EmbeddedCollectionDeltaField;
          for ( const document of collection.values() ) {
            if ( isDelta && !collection.manages(document.id) ) continue;
            yield [fieldPath, document];
            yield* document.traverseEmbeddedDocuments(fieldPath);
          }
        }
      }
    }

    /* -------------------------------------------- */
    /*  Flag Operations                             */
    /* -------------------------------------------- */

    /**
     * Get the value of a "flag" for this document
     * See the setFlag method for more details on flags
     *
     * @param {string} scope        The flag scope which namespaces the key
     * @param {string} key          The flag key
     * @return {*}                  The flag value
     */
    getFlag(scope, key) {
      const scopes = this.constructor.database.getFlagScopes();
      if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);

      /** @deprecated since v12 */
      if ( (scope === "core") && (key === "sourceId") ) {
        foundry.utils.logCompatibilityWarning("The core.sourceId flag has been deprecated. "
          + "Please use the _stats.compendiumSource property instead.", { since: 12, until: 14 });
        return this._stats?.compendiumSource;
      }

      if ( !this.flags || !(scope in this.flags) ) return undefined;
      return getProperty(this.flags?.[scope], key);
    }

    /* -------------------------------------------- */

    /**
     * Assign a "flag" to this document.
     * Flags represent key-value type data which can be used to store flexible or arbitrary data required by either
     * the core software, game systems, or user-created modules.
     *
     * Each flag should be set using a scope which provides a namespace for the flag to help prevent collisions.
     *
     * Flags set by the core software use the "core" scope.
     * Flags set by game systems or modules should use the canonical name attribute for the module
     * Flags set by an individual world should "world" as the scope.
     *
     * Flag values can assume almost any data type. Setting a flag value to null will delete that flag.
     *
     * @param {string} scope        The flag scope which namespaces the key
     * @param {string} key          The flag key
     * @param {*} value             The flag value
     * @return {Promise<Document>}  A Promise resolving to the updated document
     */
    async setFlag(scope, key, value) {
      const scopes = this.constructor.database.getFlagScopes();
      if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);
      return this.update({
        flags: {
          [scope]: {
            [key]: value
          }
        }
      });
    }

    /* -------------------------------------------- */

    /**
     * Remove a flag assigned to the document
     * @param {string} scope        The flag scope which namespaces the key
     * @param {string} key          The flag key
     * @return {Promise<Document>}  The updated document instance
     */
    async unsetFlag(scope, key) {
      const scopes = this.constructor.database.getFlagScopes();
      if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);
      const head = key.split(".");
      const tail = `-=${head.pop()}`;
      key = ["flags", scope, ...head, tail].join(".");
      return this.update({[key]: null});
    }

    /* -------------------------------------------- */
    /*  Database Creation Operations                */
    /* -------------------------------------------- */

    /**
     * Pre-process a creation operation for a single Document instance. Pre-operation events only occur for the client
     * which requested the operation.
     *
     * Modifications to the pending Document instance must be performed using {@link Document#updateSource}.
     *
     * @param {object} data                         The initial data object provided to the document creation request
     * @param {object} options                      Additional options which modify the creation request
     * @param {documents.BaseUser} user             The User requesting the document creation
     * @returns {Promise<boolean|void>}             Return false to exclude this Document from the creation operation
     * @internal
     */
    async _preCreate(data, options, user) {}

    /**
     * Post-process a creation operation for a single Document instance. Post-operation events occur for all connected
     * clients.
     *
     * @param {object} data                         The initial data object provided to the document creation request
     * @param {object} options                      Additional options which modify the creation request
     * @param {string} userId                       The id of the User requesting the document update
     * @internal
     */
    _onCreate(data, options, userId) {}

    /**
     * Pre-process a creation operation, potentially altering its instructions or input data. Pre-operation events only
     * occur for the client which requested the operation.
     *
     * This batch-wise workflow occurs after individual {@link Document#_preCreate} workflows and provides a final
     * pre-flight check before a database operation occurs.
     *
     * Modifications to pending documents must mutate the documents array or alter individual document instances using
     * {@link Document#updateSource}.
     *
     * @param {Document[]} documents                Pending document instances to be created
     * @param {DatabaseCreateOperation} operation   Parameters of the database creation operation
     * @param {documents.BaseUser} user             The User requesting the creation operation
     * @returns {Promise<boolean|void>}             Return false to cancel the creation operation entirely
     * @internal
     */
    static async _preCreateOperation(documents, operation, user) {}

    /**
     * Post-process a creation operation, reacting to database changes which have occurred. Post-operation events occur
     * for all connected clients.
     *
     * This batch-wise workflow occurs after individual {@link Document#_onCreate} workflows.
     *
     * @param {Document[]} documents                The Document instances which were created
     * @param {DatabaseCreateOperation} operation   Parameters of the database creation operation
     * @param {documents.BaseUser} user             The User who performed the creation operation
     * @returns {Promise<void>}
     * @internal
     */
    static async _onCreateOperation(documents, operation, user) {}

    /* -------------------------------------------- */
    /*  Database Update Operations                  */
    /* -------------------------------------------- */

    /**
     * Pre-process an update operation for a single Document instance. Pre-operation events only occur for the client
     * which requested the operation.
     *
     * @param {object} changes            The candidate changes to the Document
     * @param {object} options            Additional options which modify the update request
     * @param {documents.BaseUser} user   The User requesting the document update
     * @returns {Promise<boolean|void>}   A return value of false indicates the update operation should be cancelled.
     * @internal
     */
    async _preUpdate(changes, options, user) {}

    /**
     * Post-process an update operation for a single Document instance. Post-operation events occur for all connected
     * clients.
     *
     * @param {object} changed            The differential data that was changed relative to the documents prior values
     * @param {object} options            Additional options which modify the update request
     * @param {string} userId             The id of the User requesting the document update
     * @internal
     */
    _onUpdate(changed, options, userId) {}

    /**
     * Pre-process an update operation, potentially altering its instructions or input data. Pre-operation events only
     * occur for the client which requested the operation.
     *
     * This batch-wise workflow occurs after individual {@link Document#_preUpdate} workflows and provides a final
     * pre-flight check before a database operation occurs.
     *
     * Modifications to the requested updates are performed by mutating the data array of the operation.
     * {@link Document#updateSource}.
     *
     * @param {Document[]} documents                Document instances to be updated
     * @param {DatabaseUpdateOperation} operation   Parameters of the database update operation
     * @param {documents.BaseUser} user             The User requesting the update operation
     * @returns {Promise<boolean|void>}             Return false to cancel the update operation entirely
     * @internal
     */
    static async _preUpdateOperation(documents, operation, user) {}

    /**
     * Post-process an update operation, reacting to database changes which have occurred. Post-operation events occur
     * for all connected clients.
     *
     * This batch-wise workflow occurs after individual {@link Document#_onUpdate} workflows.
     *
     * @param {Document[]} documents                The Document instances which were updated
     * @param {DatabaseUpdateOperation} operation   Parameters of the database update operation
     * @param {documents.BaseUser} user             The User who performed the update operation
     * @returns {Promise<void>}
     * @internal
     */
    static async _onUpdateOperation(documents, operation, user) {}

    /* -------------------------------------------- */
    /*  Database Delete Operations                  */
    /* -------------------------------------------- */

    /**
     * Pre-process a deletion operation for a single Document instance. Pre-operation events only occur for the client
     * which requested the operation.
     *
     * @param {object} options            Additional options which modify the deletion request
     * @param {documents.BaseUser} user   The User requesting the document deletion
     * @returns {Promise<boolean|void>}   A return value of false indicates the deletion operation should be cancelled.
     * @internal
     */
    async _preDelete(options, user) {}

    /**
     * Post-process a deletion operation for a single Document instance. Post-operation events occur for all connected
     * clients.
     *
     * @param {object} options            Additional options which modify the deletion request
     * @param {string} userId             The id of the User requesting the document update
     * @internal
     */
    _onDelete(options, userId) {}

    /**
     * Pre-process a deletion operation, potentially altering its instructions or input data. Pre-operation events only
     * occur for the client which requested the operation.
     *
     * This batch-wise workflow occurs after individual {@link Document#_preDelete} workflows and provides a final
     * pre-flight check before a database operation occurs.
     *
     * Modifications to the requested deletions are performed by mutating the operation object.
     * {@link Document#updateSource}.
     *
     * @param {Document[]} documents                Document instances to be deleted
     * @param {DatabaseDeleteOperation} operation   Parameters of the database update operation
     * @param {documents.BaseUser} user             The User requesting the deletion operation
     * @returns {Promise<boolean|void>}             Return false to cancel the deletion operation entirely
     * @internal
     */
    static async _preDeleteOperation(documents, operation, user) {}

    /**
     * Post-process a deletion operation, reacting to database changes which have occurred. Post-operation events occur
     * for all connected clients.
     *
     * This batch-wise workflow occurs after individual {@link Document#_onDelete} workflows.
     *
     * @param {Document[]} documents                The Document instances which were deleted
     * @param {DatabaseDeleteOperation} operation   Parameters of the database deletion operation
     * @param {documents.BaseUser} user             The User who performed the deletion operation
     * @returns {Promise<void>}
     * @internal
     */
    static async _onDeleteOperation(documents, operation, user) {}

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /**
     * @deprecated since v10
     * @ignore
     */
    get data() {
      if ( this.constructor.schema.has("system") ) {
        throw new Error(`You are accessing the ${this.constructor.name} "data" field of which was deprecated in v10 and `
          + `replaced with "system". Continued usage of pre-v10 ".data" paths is no longer supported"`);
      }
    }

    /* -------------------------------------------- */


    /**
     * @deprecated since v11
     * @ignore
     */
    static get hasSystemData() {
      foundry.utils.logCompatibilityWarning(`You are accessing ${this.name}.hasSystemData which is deprecated. `
      + `Please use ${this.name}.hasTypeData instead.`, {since: 11, until: 13});
      return this.hasTypeData;
    }

    /* ---------------------------------------- */

    /**
     * A reusable helper for adding migration shims.
     * @protected
     * @ignore
     */
    static _addDataFieldShims(data, shims, options) {
      for ( const [oldKey, newKey] of Object.entries(shims) ) {
        this._addDataFieldShim(data, oldKey, newKey, options);
      }
    }

    /* ---------------------------------------- */

    /**
     * A reusable helper for adding a migration shim
     * @protected
     * @ignore
     */
    static _addDataFieldShim(data, oldKey, newKey, options={}) {
      if ( data.hasOwnProperty(oldKey) ) return;
      Object.defineProperty(data, oldKey, {
        get: () => {
          if ( options.warning ) logCompatibilityWarning(options.warning);
          else this._logDataFieldMigration(oldKey, newKey, options);
          return ("value" in options) ? options.value : getProperty(data, newKey);
        },
        set: value => {
          if ( newKey ) setProperty(data, newKey, value);
        },
        configurable: true,
        enumerable: false
      });
    }

    /* ---------------------------------------- */

    /**
     * Define a simple migration from one field name to another.
     * The value of the data can be transformed during the migration by an optional application function.
     * @param {object} data     The data object being migrated
     * @param {string} oldKey   The old field name
     * @param {string} newKey   The new field name
     * @param {function(data: object): any} [apply] An application function, otherwise the old value is applied
     * @returns {boolean}       Whether a migration was applied.
     * @internal
     */
    static _addDataFieldMigration(data, oldKey, newKey, apply) {
      if ( !hasProperty(data, newKey) && hasProperty(data, oldKey) ) {
        const prop = Object.getOwnPropertyDescriptor(data, oldKey);
        if ( prop && !prop.writable ) return false;
        setProperty(data, newKey, apply ? apply(data) : getProperty(data, oldKey));
        delete data[oldKey];
        return true;
      }
      return false;
    }

    /* ---------------------------------------- */

    /** @protected */
    static _logDataFieldMigration(oldKey, newKey, options={}) {
      const msg = `You are accessing ${this.name}#${oldKey} which has been migrated to ${this.name}#${newKey}`;
      return logCompatibilityWarning(msg, {...options})
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    static async _onCreateDocuments(documents, operation) {}

    /**
     * @deprecated since v12
     * @ignore
     */
    static async _onUpdateDocuments(documents, operation) {}

    /**
     * @deprecated since v12
     * @ignore
     */
    static async _onDeleteDocuments(documents, operation) {}
  }

  /**
   * @typedef {import("./_types.mjs").DatabaseAction} DatabaseAction
   * @typedef {import("./_types.mjs").DatabaseOperation} DatabaseOperation
   * @typedef {import("./_types.mjs").DocumentSocketRequest} DocumentSocketRequest
   */

  /**
   * The data structure of a modifyDocument socket response.
   * @alias foundry.abstract.DocumentSocketResponse
   */
  class DocumentSocketResponse {
    /**
     * Prepare a response for an incoming request.
     * @param {DocumentSocketRequest} request     The incoming request that is being responded to
     */
    constructor(request) {
      for ( const [k, v] of Object.entries(request) ) {
        if ( this.hasOwnProperty(k) ) this[k] = v;
      }
    }

    /**
     * The type of Document being transacted.
     * @type {string}
     */
    type;

    /**
     * The database action that was performed.
     * @type {DatabaseAction}
     */
    action;

    /**
     * Was this response broadcast to other connected clients?
     * @type {boolean}
     */
    broadcast;

    /**
     * The database operation that was requested.
     * @type {DatabaseOperation}
     */
    operation;

    /**
     * The identifier of the requesting user.
     * @type {string}
     */
    userId;

    /**
     * The result of the request. Present if successful
     * @type {object[]|string[]}
     */
    result;

    /**
     * An error that occurred. Present if unsuccessful
     * @type {Error}
     */
    error;
  }

  /**
   * @typedef {import("./_types.mjs").DatabaseGetOperation} DatabaseGetOperation
   * @typedef {import("./_types.mjs").DatabaseCreateOperation} DatabaseCreateOperation
   * @typedef {import("./_types.mjs").DatabaseUpdateOperation} DatabaseUpdateOperation
   * @typedef {import("./_types.mjs").DatabaseDeleteOperation} DatabaseDeleteOperation
   */

  /**
   * An abstract base class extended on both the client and server which defines how Documents are retrieved, created,
   * updated, and deleted.
   * @alias foundry.abstract.DatabaseBackend
   * @abstract
   */
  class DatabaseBackend {

    /* -------------------------------------------- */
    /*  Get Operations                              */
    /* -------------------------------------------- */

    /**
     * Retrieve Documents based on provided query parameters.
     * It recommended to use CompendiumCollection#getDocuments or CompendiumCollection#getIndex rather
     * than calling this method directly.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseGetOperation} operation          Parameters of the get operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]|object[]>}          An array of retrieved Document instances or index objects
     */
    async get(documentClass, operation, user) {
      operation = await this.#configureGet(operation);
      return this._getDocuments(documentClass, operation, user);
    }

    /* -------------------------------------------- */

    /**
     * Validate and configure the parameters of the get operation.
     * @param {DatabaseGetOperation} operation          The requested operation
     */
    async #configureGet(operation) {
      await this.#configureOperation(operation);
      operation.broadcast = false; // Get requests are never broadcast
      return operation;
    }

    /* -------------------------------------------- */

    /**
     * Retrieve Document instances using the specified operation parameters.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseGetOperation} operation          Parameters of the get operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]|object[]>}          An array of retrieved Document instances or index objects
     * @abstract
     * @internal
     * @ignore
     */
    async _getDocuments(documentClass, operation, user) {}

    /* -------------------------------------------- */
    /*  Create Operations                           */
    /* -------------------------------------------- */

    /**
     * Create new Documents using provided data and context.
     * It is recommended to use {@link Document.createDocuments} or {@link Document.create} rather than calling this
     * method directly.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseCreateOperation} operation       Parameters of the create operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of created Document instances
     */
    async create(documentClass, operation, user) {
      operation = await this.#configureCreate(operation);
      return this._createDocuments(documentClass, operation, user);
    }

    /* -------------------------------------------- */

    /**
     * Validate and configure the parameters of the create operation.
     * @param {DatabaseCreateOperation} operation       The requested operation
     */
    async #configureCreate(operation) {
      if ( !Array.isArray(operation.data) ) {
        throw new Error("The data provided to the DatabaseBackend#create operation must be an array of data objects");
      }
      await this.#configureOperation(operation);
      operation.render ??= true;
      operation.renderSheet ??= false;
      return operation;
    }

    /* -------------------------------------------- */

    /**
     * Create Document instances using provided data and operation parameters.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseCreateOperation} operation       Parameters of the create operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of created Document instances
     * @abstract
     * @internal
     * @ignore
     */
    async _createDocuments(documentClass, operation, user) {}

    /* -------------------------------------------- */
    /*  Update Operations                           */
    /* -------------------------------------------- */

    /**
     * Update Documents using provided data and context.
     * It is recommended to use {@link Document.updateDocuments} or {@link Document#update} rather than calling this
     * method directly.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseUpdateOperation} operation       Parameters of the update operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of updated Document instances
     */
    async update(documentClass, operation, user) {
      operation = await this.#configureUpdate(operation);
      return this._updateDocuments(documentClass, operation, user);
    }

    /* -------------------------------------------- */

    /**
     * Validate and configure the parameters of the update operation.
     * @param {DatabaseUpdateOperation} operation       The requested operation
     */
    async #configureUpdate(operation) {
      if ( !Array.isArray(operation.updates) ) {
        throw new Error("The updates provided to the DatabaseBackend#update operation must be an array of data objects");
      }
      await this.#configureOperation(operation);
      operation.diff ??= true;
      operation.recursive ??= true;
      operation.render ??= true;
      return operation;
    }

    /* -------------------------------------------- */

    /**
     * Update Document instances using provided data and operation parameters.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseUpdateOperation} operation       Parameters of the update operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of updated Document instances
     * @abstract
     * @internal
     * @ignore
     */
    async _updateDocuments(documentClass, operation, user) {}

    /* -------------------------------------------- */
    /*  Delete Operations                           */
    /* -------------------------------------------- */

    /**
     * Delete Documents using provided ids and context.
     * It is recommended to use {@link foundry.abstract.Document.deleteDocuments} or
     * {@link foundry.abstract.Document#delete} rather than calling this method directly.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseDeleteOperation} operation       Parameters of the delete operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of deleted Document instances
     */
    async delete(documentClass, operation, user) {
      operation = await this.#configureDelete(operation);
      return this._deleteDocuments(documentClass, operation, user);
    }

    /* -------------------------------------------- */

    /**
     * Validate and configure the parameters of the delete operation.
     * @param {DatabaseDeleteOperation} operation       The requested operation
     */
    async #configureDelete(operation) {
      if ( !Array.isArray(operation.ids) ) {
        throw new Error("The document ids provided to the DatabaseBackend#delete operation must be an array of strings");
      }
      await this.#configureOperation(operation);
      operation.deleteAll ??= false;
      operation.render ??= true;
      return operation;
    }

    /* -------------------------------------------- */

    /**
     * Delete Document instances using provided ids and operation parameters.
     * @param {typeof Document} documentClass           The Document class definition
     * @param {DatabaseDeleteOperation} operation       Parameters of the delete operation
     * @param {BaseUser} [user]                         The requesting User
     * @returns {Promise<Document[]>}                   An array of deleted Document instances
     * @abstract
     * @internal
     * @ignore
     */
    async _deleteDocuments(documentClass, operation, user) {}

    /* -------------------------------------------- */
    /*  Helper Methods                              */
    /* -------------------------------------------- */

    /**
     * Common database operation configuration steps.
     * @param {DatabaseOperation} operation           The requested operation
     * @returns {Promise<void>}
     */
    async #configureOperation(operation) {
      if ( operation.pack && !this.getCompendiumScopes().includes(operation.pack) ) {
        throw new Error(`Compendium pack "${operation.pack}" is not a valid Compendium identifier`);
      }
      operation.parent = await this._getParent(operation);
      operation.modifiedTime = Date.now();
    }

    /* -------------------------------------------- */

    /**
     * Get the parent Document (if any) associated with a request context.
     * @param {DatabaseOperation} operation           The requested database operation
     * @return {Promise<Document|null>}               The parent Document, or null
     * @internal
     * @ignore
     */
    async _getParent(operation) {
      if ( operation.parent && !(operation.parent instanceof Document) ) {
        throw new Error("A parent Document provided to the database operation must be a Document instance");
      }
      else if ( operation.parent ) return operation.parent;
      if ( operation.parentUuid ) return globalThis.fromUuid(operation.parentUuid, {invalid: true});
      return null;
    }

    /* -------------------------------------------- */

    /**
     * Describe the scopes which are suitable as the namespace for a flag key
     * @returns {string[]}
     */
    getFlagScopes() {}

    /* -------------------------------------------- */

    /**
     * Describe the scopes which are suitable as the namespace for a flag key
     * @returns {string[]}
     */
    getCompendiumScopes() {}

    /* -------------------------------------------- */

    /**
     * Log a database operations message.
     * @param {string} level      The logging level
     * @param {string} message    The message
     * @abstract
     * @protected
     */
    _log(level, message) {}

    /* -------------------------------------------- */

    /**
     * Log a database operation for an embedded document, capturing the action taken and relevant IDs
     * @param {string} action                       The action performed
     * @param {string} type                         The document type
     * @param {abstract.Document[]} documents       The documents modified
     * @param {string} [level=info]                 The logging level
     * @param {abstract.Document} [parent]          A parent document
     * @param {string} [pack]                       A compendium pack within which the operation occurred
     * @protected
     */
    _logOperation(action, type, documents, {parent, pack, level="info"}={}) {
      let msg = (documents.length === 1) ? `${action} ${type}` : `${action} ${documents.length} ${type} documents`;
      if (documents.length === 1) msg += ` with id [${documents[0].id}]`;
      else if (documents.length <= 5) msg += ` with ids: [${documents.map(d => d.id)}]`;
      msg += this.#logContext(parent, pack);
      this._log(level, msg);
    }

    /* -------------------------------------------- */

    /**
     * Construct a standardized error message given the context of an attempted operation
     * @returns {string}
     * @protected
     */
    _logError(user, action, subject, {parent, pack}={}) {
      if ( subject instanceof Document ) {
        subject = subject.id ? `${subject.documentName} [${subject.id}]` : `a new ${subject.documentName}`;
      }
      let msg = `User ${user.name} lacks permission to ${action} ${subject}`;
      return msg + this.#logContext(parent, pack);
    }

    /* -------------------------------------------- */

    /**
     * Determine a string suffix for a log message based on the parent and/or compendium context.
     * @param {Document|null} parent
     * @param {string|null} pack
     * @returns {string}
     */
    #logContext(parent, pack) {
      let context = "";
      if ( parent ) context += ` in parent ${parent.constructor.metadata.name} [${parent.id}]`;
      if ( pack ) context += ` in Compendium ${pack}`;
      return context;
    }
  }

  var abstract = /*#__PURE__*/Object.freeze({
    __proto__: null,
    DataModel: DataModel,
    DatabaseBackend: DatabaseBackend,
    Document: Document,
    DocumentSocketResponse: DocumentSocketResponse,
    EmbeddedCollection: EmbeddedCollection,
    EmbeddedCollectionDelta: EmbeddedCollectionDelta,
    SingletonEmbeddedCollection: SingletonEmbeddedCollection,
    TypeDataModel: TypeDataModel,
    types: _types$4
  });

  /**
   * @typedef {import("./_types.mjs").ActiveEffectData} ActiveEffectData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The ActiveEffect Document.
   * Defines the DataSchema and common behaviors for an ActiveEffect which are shared between both client and server.
   * @mixes {@link ActiveEffectData}
   */
  class BaseActiveEffect extends Document {
    /**
     * Construct an ActiveEffect document using provided data and context.
     * @param {Partial<ActiveEffectData>} data        Initial data from which to construct the ActiveEffect
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "ActiveEffect",
      collection: "effects",
      hasTypeData: true,
      label: "DOCUMENT.ActiveEffect",
      labelPlural: "DOCUMENT.ActiveEffects",
      schemaVersion: "12.324"
    }, {inplace: false}));

    /* -------------------------------------------- */

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "EFFECT.Name", textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"], label: "EFFECT.Image"}),
        type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
        system: new TypeDataField(this),
        changes: new ArrayField(new SchemaField({
          key: new StringField({required: true, label: "EFFECT.ChangeKey"}),
          value: new StringField({required: true, label: "EFFECT.ChangeValue"}),
          mode: new NumberField({integer: true, initial: ACTIVE_EFFECT_MODES.ADD,
            label: "EFFECT.ChangeMode"}),
          priority: new NumberField()
        })),
        disabled: new BooleanField(),
        duration: new SchemaField({
          startTime: new NumberField({initial: null, label: "EFFECT.StartTime"}),
          seconds: new NumberField({integer: true, min: 0, label: "EFFECT.DurationSecs"}),
          combat: new ForeignDocumentField(BaseCombat, {label: "EFFECT.Combat"}),
          rounds: new NumberField({integer: true, min: 0}),
          turns: new NumberField({integer: true, min: 0, label: "EFFECT.DurationTurns"}),
          startRound: new NumberField({integer: true, min: 0}),
          startTurn: new NumberField({integer: true, min: 0, label: "EFFECT.StartTurns"})
        }),
        description: new HTMLField({label: "EFFECT.Description", textSearch: true}),
        origin: new StringField({nullable: true, blank: false, initial: null, label: "EFFECT.Origin"}),
        tint: new ColorField({nullable: false, initial: "#ffffff", label: "EFFECT.Tint"}),
        transfer: new BooleanField({initial: true, label: "EFFECT.Transfer"}),
        statuses: new SetField(new StringField({required: true, blank: false})),
        sort: new IntegerSortField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    canUserModify(user, action, data={}) {
      if ( this.isEmbedded ) return this.parent.canUserModify(user, "update");
      return super.canUserModify(user, action, data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( this.isEmbedded ) return this.parent.testUserPermission(user, permission, {exact});
      return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */
    /*  Database Event Handlers                     */
    /* -------------------------------------------- */

    /** @inheritDoc */
    async _preCreate(data, options, user) {
      const allowed = await super._preCreate(data, options, user);
      if ( allowed === false ) return false;
      if ( this.parent instanceof BaseActor ) {
        this.updateSource({transfer: false});
      }
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(data) {
      /**
       * label -> name
       * @deprecated since v11
       */
      this._addDataFieldMigration(data, "label", "name", d => d.label || "Unnamed Effect");
      /**
       * icon -> img
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "icon", "img");
      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      this._addDataFieldShim(data, "label", "name", {since: 11, until: 13});
      this._addDataFieldShim(data, "icon", "img", {since: 12, until: 14});
      return super.shimData(data, options);
    }

    /* -------------------------------------------- */

    /**
     * @deprecated since v11
     * @ignore
     */
    get label() {
      this.constructor._logDataFieldMigration("label", "name", {since: 11, until: 13, once: true});
      return this.name;
    }

    /**
     * @deprecated since v11
     * @ignore
     */
    set label(value) {
      this.constructor._logDataFieldMigration("label", "name", {since: 11, until: 13, once: true});
      this.name = value;
    }

    /* -------------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get icon() {
      this.constructor._logDataFieldMigration("icon", "img", {since: 12, until: 14, once: true});
      return this.img;
    }

    /**
     * @deprecated since v12
     * @ignore
     */
    set icon(value) {
      this.constructor._logDataFieldMigration("icon", "img", {since: 12, until: 14, once: true});
      this.img = value;
    }
  }

  /**
   * The collection of data schema and document definitions for primary documents which are shared between the both the
   * client and the server.
   * @namespace data
   */


  /**
   * @typedef {import("./fields.mjs").DataFieldOptions} DataFieldOptions
   * @typedef {import("./fields.mjs").FilePathFieldOptions} FilePathFieldOptions
   */

  /**
   * @typedef {Object} LightAnimationData
   * @property {string} type          The animation type which is applied
   * @property {number} speed         The speed of the animation, a number between 0 and 10
   * @property {number} intensity     The intensity of the animation, a number between 1 and 10
   * @property {boolean} reverse      Reverse the direction of animation.
   */

  /**
   * A reusable document structure for the internal data used to render the appearance of a light source.
   * This is re-used by both the AmbientLightData and TokenData classes.
   * @extends DataModel
   * @memberof data
   *
   * @property {boolean} negative           Is this light source a negative source? (i.e. darkness source)
   * @property {number} alpha               An opacity for the emitted light, if any
   * @property {number} angle               The angle of emission for this point source
   * @property {number} bright              The allowed radius of bright vision or illumination
   * @property {number} color               A tint color for the emitted light, if any
   * @property {number} coloration          The coloration technique applied in the shader
   * @property {number} contrast            The amount of contrast this light applies to the background texture
   * @property {number} dim                 The allowed radius of dim vision or illumination
   * @property {number} attenuation         Fade the difference between bright, dim, and dark gradually?
   * @property {number} luminosity          The luminosity applied in the shader
   * @property {number} saturation          The amount of color saturation this light applies to the background texture
   * @property {number} shadows             The depth of shadows this light applies to the background texture
   * @property {LightAnimationData} animation  An animation configuration for the source
   * @property {{min: number, max: number}} darkness  A darkness range (min and max) for which the source should be active
   */
  class LightData extends DataModel {
    static defineSchema() {
      return {
        negative: new BooleanField(),
        priority: new NumberField({required: true, nullable: false, integer: true, initial: 0, min: 0}),
        alpha: new AlphaField({initial: 0.5}),
        angle: new AngleField({initial: 360, normalize: false}),
        bright: new NumberField({required: true,  nullable: false, initial: 0, min: 0, step: 0.01}),
        color: new ColorField({}),
        coloration: new NumberField({required: true, integer: true, initial: 1}),
        dim: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
        attenuation: new AlphaField({initial: 0.5}),
        luminosity: new NumberField({required: true, nullable: false, initial: 0.5, min: 0, max: 1}),
        saturation: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
        contrast: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
        shadows: new NumberField({required: true, nullable: false, initial: 0, min: 0, max: 1}),
        animation: new SchemaField({
          type: new StringField({nullable: true, blank: false, initial: null}),
          speed: new NumberField({required: true, nullable: false, integer: true, initial: 5, min: 0, max: 10,
            validationError: "Light animation speed must be an integer between 0 and 10"}),
          intensity: new NumberField({required: true, nullable: false, integer: true, initial: 5, min: 1, max: 10,
            validationError: "Light animation intensity must be an integer between 1 and 10"}),
          reverse: new BooleanField()
        }),
        darkness: new SchemaField({
          min: new AlphaField({initial: 0}),
          max: new AlphaField({initial: 1})
        }, {
          validate: d => (d.min ?? 0) <= (d.max ?? 1),
          validationError: "darkness.max may not be less than darkness.min"
        })
      }
    }

    /** @override */
    static LOCALIZATION_PREFIXES = ["LIGHT"];

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(data) {
      /**
       * Migration of negative luminosity
       * @deprecated since v12
       */
      const luminosity = data.luminosity;
      if ( luminosity < 0) {
        data.luminosity = 1 - luminosity;
        data.negative = true;
      }
      return super.migrateData(data);
    }
  }

  /* ---------------------------------------- */

  /**
   * A data model intended to be used as an inner EmbeddedDataField which defines a geometric shape.
   * @extends DataModel
   * @memberof data
   *
   * @property {string} type                The type of shape, a value in ShapeData.TYPES.
   *                                        For rectangles, the x/y coordinates are the top-left corner.
   *                                        For circles, the x/y coordinates are the center of the circle.
   *                                        For polygons, the x/y coordinates are the first point of the polygon.
   * @property {number} [width]             For rectangles, the pixel width of the shape.
   * @property {number} [height]            For rectangles, the pixel width of the shape.
   * @property {number} [radius]            For circles, the pixel radius of the shape.
   * @property {number[]} [points]          For polygons, the array of polygon coordinates which comprise the shape.
   */
  class ShapeData extends DataModel {
    static defineSchema() {
      return {
        type: new StringField({required: true, blank: false, choices: Object.values(this.TYPES), initial: "r"}),
        width: new NumberField({required: false, integer: true, min: 0}),
        height: new NumberField({required: false, integer: true, min: 0}),
        radius: new NumberField({required: false, integer: true, positive: true}),
        points: new ArrayField(new NumberField({nullable: false}))
      }
    }

    /**
     * The primitive shape types which are supported
     * @enum {string}
     */
    static TYPES = {
      RECTANGLE: "r",
      CIRCLE: "c",
      ELLIPSE: "e",
      POLYGON: "p"
    }
  }

  /* ---------------------------------------- */

  /**
   * A data model intended to be used as an inner EmbeddedDataField which defines a geometric shape.
   * @extends DataModel
   * @memberof data
   * @abstract
   *
   * @property {string} type                                          The type of shape, a value in BaseShapeData.TYPES.
   * @property {{bottom: number|null, top: number|null}} [elevation]  The bottom and top elevation of the shape.
   *                                                                  A value of null means -/+Infinity.
   * @property {boolean} [hole=false]                                 Is this shape a hole?
   */
  class BaseShapeData extends DataModel {

    /**
     * The possible shape types.
     * @type {Readonly<{
     *   rectangle: RectangleShapeData,
     *   circle: CircleShapeData,
     *   ellipse: EllipseShapeData,
     *   polygon: PolygonShapeData
     * }>}
     */
    static get TYPES() {
      return BaseShapeData.#TYPES ??= Object.freeze({
        [RectangleShapeData.TYPE]: RectangleShapeData,
        [CircleShapeData.TYPE]: CircleShapeData,
        [EllipseShapeData.TYPE]: EllipseShapeData,
        [PolygonShapeData.TYPE]: PolygonShapeData
      });
    }

    static #TYPES;

    /* -------------------------------------------- */

    /**
     * The type of this shape.
     * @type {string}
     */
    static TYPE = "";

    /* -------------------------------------------- */

    /** @override */
    static defineSchema() {
      return {
        type: new StringField({required: true, blank: false, initial: this.TYPE,
          validate: value => value === this.TYPE, validationError: `must be equal to "${this.TYPE}"`}),
        hole: new BooleanField()
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * The data model for a rectangular shape.
   * @extends DataModel
   * @memberof data
   *
   * @property {number} x               The top-left x-coordinate in pixels before rotation.
   * @property {number} y               The top-left y-coordinate in pixels before rotation.
   * @property {number} width           The width of the rectangle in pixels.
   * @property {number} height          The height of the rectangle in pixels.
   * @property {number} [rotation=0]    The rotation around the center of the rectangle in degrees.
   */
  class RectangleShapeData extends BaseShapeData {

    static {
      Object.defineProperty(this, "TYPE", {value: "rectangle"});
    }

    /** @inheritdoc */
    static defineSchema() {
      return Object.assign(super.defineSchema(), {
        x: new NumberField({required: true, nullable: false, initial: undefined}),
        y: new NumberField({required: true, nullable: false, initial: undefined}),
        width: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
        height: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
        rotation: new AngleField()
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * The data model for a circle shape.
   * @extends DataModel
   * @memberof data
   *
   * @property {number} x         The x-coordinate of the center point in pixels.
   * @property {number} y         The y-coordinate of the center point in pixels.
   * @property {number} radius    The radius of the circle in pixels.
   */
  class CircleShapeData extends BaseShapeData {

    static {
      Object.defineProperty(this, "TYPE", {value: "circle"});
    }

    /** @inheritdoc */
    static defineSchema() {
      return Object.assign(super.defineSchema(), {
        x: new NumberField({required: true, nullable: false, initial: undefined}),
        y: new NumberField({required: true, nullable: false, initial: undefined}),
        radius: new NumberField({required: true, nullable: false, initial: undefined, positive: true})
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * The data model for an ellipse shape.
   * @extends DataModel
   * @memberof data
   *
   * @property {number} x               The x-coordinate of the center point in pixels.
   * @property {number} y               The y-coordinate of the center point in pixels.
   * @property {number} radiusX         The x-radius of the circle in pixels.
   * @property {number} radiusY         The y-radius of the circle in pixels.
   * @property {number} [rotation=0]    The rotation around the center of the rectangle in degrees.
   */
  class EllipseShapeData extends BaseShapeData {

    static {
      Object.defineProperty(this, "TYPE", {value: "ellipse"});
    }

    /** @inheritdoc */
    static defineSchema() {
      return Object.assign(super.defineSchema(), {
        x: new NumberField({required: true, nullable: false, initial: undefined}),
        y: new NumberField({required: true, nullable: false, initial: undefined}),
        radiusX: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
        radiusY: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
        rotation: new AngleField()
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * The data model for a polygon shape.
   * @extends DataModel
   * @memberof data
   *
   * @property {number[]} points      The points of the polygon ([x0, y0, x1, y1, ...]).
   *                                  The polygon must not be self-intersecting.
   */
  class PolygonShapeData extends BaseShapeData {

    static {
      Object.defineProperty(this, "TYPE", {value: "polygon"});
    }

    /** @inheritdoc */
    static defineSchema() {
      return Object.assign(super.defineSchema(), {
        points: new ArrayField(new NumberField({required: true, nullable: false, initial: undefined}),
          {validate: value => {
            if ( value.length % 2 !== 0 ) throw new Error("must have an even length");
            if ( value.length < 6 ) throw new Error("must have at least 3 points");
          }}),
      });
    }
  }

  /* ---------------------------------------- */

  /**
   * A {@link fields.SchemaField} subclass used to represent texture data.
   * @property {string|null} src              The URL of the texture source.
   * @property {number} [anchorX=0]           The X coordinate of the texture anchor.
   * @property {number} [anchorY=0]           The Y coordinate of the texture anchor.
   * @property {number} [scaleX=1]            The scale of the texture in the X dimension.
   * @property {number} [scaleY=1]            The scale of the texture in the Y dimension.
   * @property {number} [offsetX=0]           The X offset of the texture with (0,0) in the top left.
   * @property {number} [offsetY=0]           The Y offset of the texture with (0,0) in the top left.
   * @property {number} [rotation=0]           An angle of rotation by which this texture is rotated around its center.
   * @property {string} [tint="#ffffff"]      The tint applied to the texture.
   * @property {number} [alphaThreshold=0]    Only pixels with an alpha value at or above this value are consider solid
   *                                          w.r.t. to occlusion testing and light/weather blocking.
   */
  class TextureData extends SchemaField {
    /**
     * @param {DataFieldOptions} options        Options which are forwarded to the SchemaField constructor
     * @param {FilePathFieldOptions} srcOptions Additional options for the src field
     */
    constructor(options={}, {categories=["IMAGE", "VIDEO"], initial={}, wildcard=false, label=""}={}) {
      /** @deprecated since v12 */
      if ( typeof initial === "string" ) {
        const msg = "Passing the initial value of the src field as a string is deprecated. Pass {src} instead.";
        logCompatibilityWarning(msg, {since: 12, until: 14});
        initial = {src: initial};
      }
      super({
        src: new FilePathField({categories, initial: initial.src ?? null, label, wildcard}),
        anchorX: new NumberField({nullable: false, initial: initial.anchorX ?? 0}),
        anchorY: new NumberField({nullable: false, initial: initial.anchorY ?? 0}),
        offsetX: new NumberField({nullable: false, integer: true, initial: initial.offsetX ?? 0}),
        offsetY: new NumberField({nullable: false, integer: true, initial: initial.offsetY ?? 0}),
        fit: new StringField({initial: initial.fit ?? "fill", choices: CONST.TEXTURE_DATA_FIT_MODES}),
        scaleX: new NumberField({nullable: false, initial: initial.scaleX ?? 1}),
        scaleY: new NumberField({nullable: false, initial: initial.scaleY ?? 1}),
        rotation: new AngleField({initial: initial.rotation ?? 0}),
        tint: new ColorField({nullable: false, initial: initial.tint ?? "#ffffff"}),
        alphaThreshold: new AlphaField({nullable: false, initial: initial.alphaThreshold ?? 0})
      }, options);
    }
  }

  /* ---------------------------------------- */

  /**
   * Extend the base TokenData to define a PrototypeToken which exists within a parent Actor.
   * @extends abstract.DataModel
   * @memberof data
   * @property {boolean} randomImg      Does the prototype token use a random wildcard image?
   * @alias {PrototypeToken}
   */
  class PrototypeToken extends DataModel {
    constructor(data={}, options={}) {
      super(data, options);
      Object.defineProperty(this, "apps", {value: {}});
    }

    /** @override */
    static defineSchema() {
      const schema = BaseToken.defineSchema();
      const excluded = ["_id", "actorId", "delta", "x", "y", "elevation", "sort", "hidden", "locked", "_regions"];
      for ( let x of excluded ) {
        delete schema[x];
      }
      schema.name.textSearch = schema.name.options.textSearch = false;
      schema.randomImg = new BooleanField();
      PrototypeToken.#applyDefaultTokenSettings(schema);
      return schema;
    }

    /** @override */
    static LOCALIZATION_PREFIXES = ["TOKEN"];

    /**
     * The Actor which owns this Prototype Token
     * @type {documents.BaseActor}
     */
    get actor() {
      return this.parent;
    }

    /** @inheritdoc */
    toObject(source=true) {
      const data = super.toObject(source);
      data["actorId"] = this.document?.id;
      return data;
    }

    /**
     * @see ClientDocument.database
     * @ignore
     */
    static get database() {
      return globalThis.CONFIG.DatabaseBackend;
    }

    /* -------------------------------------------- */

    /**
     * Apply configured default token settings to the schema.
     * @param {DataSchema} [schema]  The schema to apply the settings to.
     */
    static #applyDefaultTokenSettings(schema) {
      if ( typeof DefaultTokenConfig === "undefined" ) return;
      const settings = foundry.utils.flattenObject(game.settings.get("core", DefaultTokenConfig.SETTING) ?? {});
      for ( const [k, v] of Object.entries(settings) ) {
        const path = k.split(".");
        let field = schema[path.shift()];
        if ( path.length ) field = field._getField(path);
        if ( field ) field.initial = v;
      }
    }

    /* -------------------------------------------- */
    /*  Document Compatibility Methods              */
    /* -------------------------------------------- */

    /**
     * @see abstract.Document#update
     * @ignore
     */
    update(data, options) {
      return this.actor.update({prototypeToken: data}, options);
    }

    /* -------------------------------------------- */

    /**
     * @see abstract.Document#getFlag
     * @ignore
     */
    getFlag(...args) {
      return foundry.abstract.Document.prototype.getFlag.call(this, ...args);
    }

    /* -------------------------------------------- */

    /**
     * @see abstract.Document#getFlag
     * @ignore
     */
    setFlag(...args) {
      return foundry.abstract.Document.prototype.setFlag.call(this, ...args);
    }

    /* -------------------------------------------- */

    /**
     * @see abstract.Document#unsetFlag
     * @ignore
     */
    async unsetFlag(...args) {
      return foundry.abstract.Document.prototype.unsetFlag.call(this, ...args);
    }

    /* -------------------------------------------- */

    /**
     * @see abstract.Document#testUserPermission
     * @ignore
     */
    testUserPermission(user, permission, {exact=false}={}) {
      return this.actor.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */

    /**
     * @see documents.BaseActor#isOwner
     * @ignore
     */
    get isOwner() {
      return this.actor.isOwner;
    }
  }

  /* -------------------------------------------- */

  /**
   * A minimal data model used to represent a tombstone entry inside an EmbeddedCollectionDelta.
   * @see {EmbeddedCollectionDelta}
   * @extends DataModel
   * @memberof data
   *
   * @property {string} _id              The _id of the base Document that this tombstone represents.
   * @property {boolean} _tombstone      A property that identifies this entry as a tombstone.
   */
  class TombstoneData extends DataModel {
    /** @override */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        _tombstone: new BooleanField({initial: true, validate: v => v === true, validationError: "must be true"})
      };
    }
  }

  /**
   * @typedef {import("./_types.mjs").ActorData} ActorData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Actor Document.
   * Defines the DataSchema and common behaviors for an Actor which are shared between both client and server.
   * @mixes ActorData
   */
  class BaseActor extends Document {
    /**
     * Construct an Actor document using provided data and context.
     * @param {Partial<ActorData>} data               Initial data from which to construct the Actor
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Actor",
      collection: "actors",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "img", "type", "sort", "folder"],
      embedded: {ActiveEffect: "effects", Item: "items"},
      hasTypeData: true,
      label: "DOCUMENT.Actor",
      labelPlural: "DOCUMENT.Actors",
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /* ---------------------------------------- */

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"], initial: data => {
          return this.implementation.getDefaultArtwork(data).img;
        }}),
        type: new DocumentTypeField(this),
        system: new TypeDataField(this),
        prototypeToken: new EmbeddedDataField(PrototypeToken),
        items: new EmbeddedCollectionField(BaseItem),
        effects: new EmbeddedCollectionField(BaseActiveEffect),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      };
    }

    /* ---------------------------------------- */

    /**
     * The default icon used for newly created Actor documents.
     * @type {string}
     */
    static DEFAULT_ICON = DEFAULT_TOKEN;

    /* -------------------------------------------- */

    /**
     * Determine default artwork based on the provided actor data.
     * @param {ActorData} actorData                      The source actor data.
     * @returns {{img: string, texture: {src: string}}}  Candidate actor image and prototype token artwork.
     */
    static getDefaultArtwork(actorData) {
      return {
        img: this.DEFAULT_ICON,
        texture: {
          src: this.DEFAULT_ICON
        }
      };
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    _initializeSource(source, options) {
      source = super._initializeSource(source, options);
      source.prototypeToken.name = source.prototypeToken.name || source.name;
      source.prototypeToken.texture.src = source.prototypeToken.texture.src || source.img;
      return source;
    }

    /* -------------------------------------------- */

    /** @override */
    static canUserCreate(user) {
      return user.hasPermission("ACTOR_CREATE");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to create this actor?
     * @param {User} user  The user attempting the creation operation.
     * @param {Actor} doc  The Actor being created.
     */
    static #canCreate(user, doc) {
      if ( !user.hasPermission("ACTOR_CREATE") ) return false;      // User cannot create actors at all
      if ( doc._source.prototypeToken.randomImg && !user.hasPermission("FILES_BROWSE") ) return false;
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Is a user able to update an existing actor?
     * @param {User} user    The user attempting the update operation.
     * @param {Actor} doc    The Actor being updated.
     * @param {object} data  The update delta being applied.
     */
    static #canUpdate(user, doc, data) {
      if ( !doc.testUserPermission(user, "OWNER") ) return false; // Ownership is required.

      // Users can only enable token wildcard images if they have FILES_BROWSE permission.
      const tokenChange = data?.prototypeToken || {};
      const enablingRandomImage = tokenChange.randomImg === true;
      if ( enablingRandomImage ) return user.hasPermission("FILES_BROWSE");

      // Users can only change a token wildcard path if they have FILES_BROWSE permission.
      const randomImageEnabled = doc._source.prototypeToken.randomImg && (tokenChange.randomImg !== false);
      const changingRandomImage = ("img" in tokenChange) && randomImageEnabled;
      if ( changingRandomImage ) return user.hasPermission("FILES_BROWSE");
      return true;
    }

    /* ---------------------------------------- */

    /** @inheritDoc */
    async _preCreate(data, options, user) {
      const allowed = await super._preCreate(data, options, user);
      if ( allowed === false ) return false;
      if ( !this.prototypeToken.name ) this.prototypeToken.updateSource({name: this.name});
      if ( !this.prototypeToken.texture.src || (this.prototypeToken.texture.src === DEFAULT_TOKEN)) {
        const { texture } = this.constructor.getDefaultArtwork(this.toObject());
        this.prototypeToken.updateSource("img" in data ? { texture: { src: this.img } } : { texture });
      }
    }

    /* ---------------------------------------- */

    /** @inheritDoc */
    async _preUpdate(changed, options, user) {
      const allowed = await super._preUpdate(changed, options, user);
      if ( allowed === false ) return false;
      if ( changed.img && !getProperty(changed, "prototypeToken.texture.src") ) {
        const { texture } = this.constructor.getDefaultArtwork(foundry.utils.mergeObject(this.toObject(), changed));
        if ( !this.prototypeToken.texture.src || (this.prototypeToken.texture.src === texture?.src) ) {
          setProperty(changed, "prototypeToken.texture.src", changed.img);
        }
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").ActorDeltaData} ActorDeltaData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The ActorDelta Document.
   * Defines the DataSchema and common behaviors for an ActorDelta which are shared between both client and server.
   * ActorDeltas store a delta that can be applied to a particular Actor in order to produce a new Actor.
   * @mixes ActorDeltaData
   */
  class BaseActorDelta extends Document {
    /**
     * Construct an ActorDelta document using provided data and context.
     * @param {Partial<ActorDeltaData>} data         Initial data used to construct the ActorDelta.
     * @param {DocumentConstructionContext} context  Construction context options.
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "ActorDelta",
      collection: "delta",
      label: "DOCUMENT.ActorDelta",
      labelPlural: "DOCUMENT.ActorDeltas",
      isEmbedded: true,
      embedded: {
        Item: "items",
        ActiveEffect: "effects"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @override */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: false, nullable: true, initial: null}),
        type: new StringField({required: false, nullable: true, initial: null}),
        img: new FilePathField({categories: ["IMAGE"], nullable: true, initial: null, required: false}),
        system: new ObjectField(),
        items: new EmbeddedCollectionDeltaField(BaseItem),
        effects: new EmbeddedCollectionDeltaField(BaseActiveEffect),
        ownership: new DocumentOwnershipField({required: false, nullable: true, initial: null}),
        flags: new ObjectField()
      };
    }

    /* -------------------------------------------- */

    /** @override */
    canUserModify(user, action, data={}) {
      return this.parent.canUserModify(user, action, data);
    }

    /* -------------------------------------------- */

    /** @override */
    testUserPermission(user, permission, { exact=false }={}) {
      return this.parent.testUserPermission(user, permission, { exact });
    }

    /* -------------------------------------------- */
    /*  Methods                                     */
    /* -------------------------------------------- */

    /**
     * Retrieve the base actor's collection, if it exists.
     * @param {string} collectionName  The collection name.
     * @returns {Collection}
     */
    getBaseCollection(collectionName) {
      const baseActor = this.parent?.baseActor;
      return baseActor?.getEmbeddedCollection(collectionName);
    }

    /* -------------------------------------------- */

    /**
     * Apply an ActorDelta to an Actor and return the resultant synthetic Actor.
     * @param {ActorDelta} delta  The ActorDelta.
     * @param {Actor} baseActor   The base Actor.
     * @param {object} [context]  Context to supply to synthetic Actor instantiation.
     * @returns {Actor|null}
     */
    static applyDelta(delta, baseActor, context={}) {
      if ( !baseActor ) return null;
      if ( delta.parent?.isLinked ) return baseActor;

      // Get base actor data.
      const cls = game?.actors?.documentClass ?? db.Actor;
      const actorData = baseActor.toObject();
      const deltaData = delta.toObject();
      delete deltaData._id;

      // Merge embedded collections.
      BaseActorDelta.#mergeEmbeddedCollections(cls, actorData, deltaData);

      // Merge the rest of the delta.
      mergeObject(actorData, deltaData);
      return new cls(actorData, {parent: delta.parent, ...context});
    }

    /* -------------------------------------------- */

    /**
     * Merge delta Document embedded collections with the base Document.
     * @param {typeof Document} documentClass  The parent Document class.
     * @param {object} baseData                The base Document data.
     * @param {object} deltaData               The delta Document data.
     */
    static #mergeEmbeddedCollections(documentClass, baseData, deltaData) {
      for ( const collectionName of Object.keys(documentClass.hierarchy) ) {
        const baseCollection = baseData[collectionName];
        const deltaCollection = deltaData[collectionName];
        baseData[collectionName] = BaseActorDelta.#mergeEmbeddedCollection(baseCollection, deltaCollection);
        delete deltaData[collectionName];
      }
    }

    /* -------------------------------------------- */

    /**
     * Apply an embedded collection delta.
     * @param {object[]} base   The base embedded collection.
     * @param {object[]} delta  The delta embedded collection.
     * @returns {object[]}
     */
    static #mergeEmbeddedCollection(base=[], delta=[]) {
      const deltaIds = new Set();
      const records = [];
      for ( const record of delta ) {
        if ( !record._tombstone ) records.push(record);
        deltaIds.add(record._id);
      }
      for ( const record of base ) {
        if ( !deltaIds.has(record._id) ) records.push(record);
      }
      return records;
    }

    /* -------------------------------------------- */

    /** @override */
    static migrateData(source) {
      return BaseActor.migrateData(source);
    }

    /* -------------------------------------------- */
    /*  Serialization                               */
    /* -------------------------------------------- */

    /** @override */
    toObject(source=true) {
      const data = {};
      const value = source ? this._source : this;
      for ( const [name, field] of this.schema.entries() ) {
        const v = value[name];
        if ( !field.required && ((v === undefined) || (v === null)) ) continue; // Drop optional fields
        data[name] = source ? deepClone(value[name]) : field.toObject(value[name]);
      }
      return data;
    }
  }

  /**
   * @typedef {import("./_types.mjs").AdventureData} AdventureData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Adventure Document.
   * Defines the DataSchema and common behaviors for an Adventure which are shared between both client and server.
   * @mixes AdventureData
   */
  class BaseAdventure extends Document {
    /**
     * Construct an Adventure document using provided data and context.
     * @param {Partial<AdventureData>} data         Initial data used to construct the Adventure.
     * @param {DocumentConstructionContext} context  Construction context options.
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Adventure",
      collection: "adventures",
      compendiumIndexFields: ["_id", "name", "description", "img", "sort", "folder"],
      label: "DOCUMENT.Adventure",
      labelPlural: "DOCUMENT.Adventures",
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "ADVENTURE.Name", hint: "ADVENTURE.NameHint", textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"], label: "ADVENTURE.Image", hint: "ADVENTURE.ImageHint"}),
        caption: new HTMLField({label: "ADVENTURE.Caption", hint: "ADVENTURE.CaptionHint"}),
        description: new HTMLField({label: "ADVENTURE.Description", hint: "ADVENTURE.DescriptionHint", textSearch: true}),
        actors: new SetField(new EmbeddedDataField(BaseActor)),
        combats: new SetField(new EmbeddedDataField(BaseCombat)),
        items: new SetField(new EmbeddedDataField(BaseItem)),
        journal: new SetField(new EmbeddedDataField(BaseJournalEntry)),
        scenes: new SetField(new EmbeddedDataField(BaseScene)),
        tables: new SetField(new EmbeddedDataField(BaseRollTable)),
        macros: new SetField(new EmbeddedDataField(BaseMacro)),
        cards: new SetField(new EmbeddedDataField(BaseCards)),
        playlists: new SetField(new EmbeddedDataField(BasePlaylist)),
        folders: new SetField(new EmbeddedDataField(BaseFolder)),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      };
    }

    /* -------------------------------------------- */
    /*  Model Properties                            */
    /* -------------------------------------------- */

    /**
     * An array of the fields which provide imported content from the Adventure.
     * @type {Record<string, typeof Document>}
     */
    static get contentFields() {
      const content = {};
      for ( const field of this.schema ) {
        if ( field instanceof SetField ) content[field.name] = field.element.model.implementation;
      }
      return content;
    }

    /**
     * Provide a thumbnail image path used to represent the Adventure document.
     * @type {string}
     */
    get thumbnail() {
      return this.img;
    }
  }

  /**
   * @typedef {import("./_types.mjs").AmbientLightData} AmbientLightData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The AmbientLight Document.
   * Defines the DataSchema and common behaviors for an AmbientLight which are shared between both client and server.
   * @mixes AmbientLightData
   */
  class BaseAmbientLight extends Document {
    /**
     * Construct an AmbientLight document using provided data and context.
     * @param {Partial<AmbientLightData>} data        Initial data from which to construct the AmbientLight
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "AmbientLight",
      collection: "lights",
      label: "DOCUMENT.AmbientLight",
      labelPlural: "DOCUMENT.AmbientLights",
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        rotation: new AngleField(),
        walls: new BooleanField({initial: true}),
        vision: new BooleanField(),
        config: new EmbeddedDataField(LightData),
        hidden: new BooleanField(),
        flags: new ObjectField()
      }
    }

    /** @override */
    static LOCALIZATION_PREFIXES = ["AMBIENT_LIGHT"];
  }

  /**
   * @typedef {import("./_types.mjs").AmbientSoundData} AmbientSoundData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The AmbientSound Document.
   * Defines the DataSchema and common behaviors for an AmbientSound which are shared between both client and server.
   * @mixes AmbientSoundData
   */
  class BaseAmbientSound extends Document {
    /**
     * Construct an AmbientSound document using provided data and context.
     * @param {Partial<AmbientSoundData>} data        Initial data from which to construct the AmbientSound
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "AmbientSound",
      collection: "sounds",
      label: "DOCUMENT.AmbientSound",
      labelPlural: "DOCUMENT.AmbientSounds",
      isEmbedded: true,
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        radius: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
        path: new FilePathField({categories: ["AUDIO"]}),
        repeat: new BooleanField(),
        volume: new AlphaField({initial: 0.5, step: 0.01}),
        walls: new BooleanField({initial: true}),
        easing: new BooleanField({initial: true}),
        hidden: new BooleanField(),
        darkness: new SchemaField({
          min: new AlphaField({initial: 0}),
          max: new AlphaField({initial: 1})
        }),
        effects: new SchemaField({
          base: new SchemaField({
            type: new StringField(),
            intensity: new NumberField({required: true, integer: true, initial: 5, min: 1, max: 10, step: 1})
          }),
          muffled: new SchemaField({
            type: new StringField(),
            intensity: new NumberField({required: true, integer: true, initial: 5, min: 1, max: 10, step: 1})
          })
        }),
        flags: new ObjectField()
      }
    }

    /** @override */
    static LOCALIZATION_PREFIXES = ["AMBIENT_SOUND"];
  }

  /**
   * @typedef {import("./_types.mjs").CardData} CardData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Card Document.
   * Defines the DataSchema and common behaviors for a Card which are shared between both client and server.
   * @mixes CardData
   */
  class BaseCard extends Document {
    /**
     * Construct a Card document using provided data and context.
     * @param {Partial<CardData>} data                Initial data from which to construct the Card
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Card",
      collection: "cards",
      hasTypeData: true,
      indexed: true,
      label: "DOCUMENT.Card",
      labelPlural: "DOCUMENT.Cards",
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      compendiumIndexFields: ["name", "type", "suit", "sort"],
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "CARD.Name", textSearch: true}),
        description: new HTMLField({label: "CARD.Description"}),
        type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
        system: new TypeDataField(this),
        suit: new StringField({label: "CARD.Suit"}),
        value: new NumberField({label: "CARD.Value"}),
        back: new SchemaField({
          name: new StringField({label: "CARD.BackName"}),
          text: new HTMLField({label: "CARD.BackText"}),
          img: new FilePathField({categories: ["IMAGE", "VIDEO"], label: "CARD.BackImage"}),
        }),
        faces: new ArrayField(new SchemaField({
          name: new StringField({label: "CARD.FaceName"}),
          text: new HTMLField({label: "CARD.FaceText"}),
          img: new FilePathField({categories: ["IMAGE", "VIDEO"], initial: () => this.DEFAULT_ICON,
            label: "CARD.FaceImage"}),
        })),
        face: new NumberField({required: true, initial: null, integer: true, min: 0, label: "CARD.Face"}),
        drawn: new BooleanField({label: "CARD.Drawn"}),
        origin: new ForeignDocumentField(BaseCards),
        width: new NumberField({integer: true, positive: true, label: "Width"}),
        height: new NumberField({integer: true, positive: true, label: "Height"}),
        rotation: new AngleField({label: "Rotation"}),
        sort: new IntegerSortField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * The default icon used for a Card face that does not have a custom image set
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/card-joker.svg";

    /**
     * Is a User able to create a new Card within this parent?
     * @private
     */
    static #canCreate(user, doc, data) {
      if ( user.isGM ) return true;                             // GM users can always create
      if ( doc.parent.type !== "deck" ) return true;            // Users can pass cards to card hands or piles
      return doc.parent.canUserModify(user, "create", data);    // Otherwise require parent document permission
    }

    /**
     * Is a user able to update an existing Card?
     * @private
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                               // GM users can always update
      const wasDrawn = new Set(["drawn", "_id"]);                 // Users can draw cards from a deck
      if ( new Set(Object.keys(data)).equals(wasDrawn) ) return true;
      return doc.parent.canUserModify(user, "update", data);      // Otherwise require parent document permission
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( this.isEmbedded ) return this.parent.testUserPermission(user, permission, {exact});
      return super.testUserPermission(user, permission, {exact});
    }
  }

  /**
   * @typedef {import("./_types.mjs").CardsData} CardsData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Cards Document.
   * Defines the DataSchema and common behaviors for a Cards Document which are shared between both client and server.
   * @mixes CardsData
   */
  class BaseCards extends Document {
    /**
     * Construct a Cards document using provided data and context.
     * @param {Partial<CardsData>} data               Initial data from which to construct the Cards
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Cards",
      collection: "cards",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "description", "img", "type", "sort", "folder"],
      embedded: {Card: "cards"},
      hasTypeData: true,
      label: "DOCUMENT.Cards",
      labelPlural: "DOCUMENT.CardsPlural",
      permissions: {create: "CARDS_CREATE"},
      coreTypes: ["deck", "hand", "pile"],
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "CARDS.Name", textSearch: true}),
        type: new DocumentTypeField(this),
        description: new HTMLField({label: "CARDS.Description", textSearch: true}),
        img: new FilePathField({categories: ["IMAGE", "VIDEO"], initial: () => this.DEFAULT_ICON,
          label: "CARDS.Image"}),
        system: new TypeDataField(this),
        cards: new EmbeddedCollectionField(BaseCard),
        width: new NumberField({integer: true, positive: true, label: "Width"}),
        height: new NumberField({integer: true, positive: true, label: "Height"}),
        rotation: new AngleField({label: "Rotation"}),
        displayCount: new BooleanField(),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * The default icon used for a cards stack that does not have a custom image set
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/card-hand.svg";

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").ChatMessageData} ChatMessageData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The ChatMessage Document.
   * Defines the DataSchema and common behaviors for a ChatMessage which are shared between both client and server.
   * @mixes ChatMessageData
   */
  class BaseChatMessage extends Document {
    /**
     * Construct a Cards document using provided data and context.
     * @param {Partial<ChatMessageData>} data         Initial data from which to construct the ChatMessage
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "ChatMessage",
      collection: "messages",
      label: "DOCUMENT.ChatMessage",
      labelPlural: "DOCUMENT.ChatMessages",
      hasTypeData: true,
      isPrimary: true,
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
        system: new TypeDataField(this),
        style: new NumberField({required: true, choices: Object.values(CHAT_MESSAGE_STYLES),
          initial: CHAT_MESSAGE_STYLES.OTHER, validationError: "must be a value in CONST.CHAT_MESSAGE_STYLES"}),
        author: new ForeignDocumentField(BaseUser, {nullable: false, initial: () => game?.user?.id}),
        timestamp: new NumberField({required: true, nullable: false, initial: Date.now}),
        flavor: new HTMLField(),
        content: new HTMLField({textSearch: true}),
        speaker: new SchemaField({
          scene: new ForeignDocumentField(BaseScene, {idOnly: true}),
          actor: new ForeignDocumentField(BaseActor, {idOnly: true}),
          token: new ForeignDocumentField(BaseToken, {idOnly: true}),
          alias: new StringField()
        }),
        whisper: new ArrayField(new ForeignDocumentField(BaseUser, {idOnly: true})),
        blind: new BooleanField(),
        rolls: new ArrayField(new JSONField({validate: BaseChatMessage.#validateRoll})),
        sound: new FilePathField({categories: ["AUDIO"]}),
        emote: new BooleanField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      };
    }

    /**
     * Is a user able to create a new chat message?
     */
    static #canCreate(user, doc) {
      if ( user.isGM ) return true;
      if ( user.id !== doc._source.author ) return false; // You cannot impersonate a different user
      return user.hasRole("PLAYER");                      // Any player can create messages
    }

    /**
     * Is a user able to update an existing chat message?
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                       // GM users can do anything
      if ( user.id !== doc._source.author ) return false; // Otherwise, message authors
      if ( ("author" in data) && (data.author !== user.id) ) return false; // Message author is immutable
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Validate that Rolls belonging to the ChatMessage document are valid
     * @param {string} rollJSON     The serialized Roll data
     */
    static #validateRoll(rollJSON) {
      const roll = JSON.parse(rollJSON);
      if ( !roll.evaluated ) throw new Error(`Roll objects added to ChatMessage documents must be evaluated`);
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( !exact && (user.id === this._source.author) ) return true; // The user who created the chat message
      return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {
      /**
       * V12 migration from user to author
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "user", "author");
      BaseChatMessage.#migrateTypeToStyle(data);
      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /**
     * Migrate the type field to the style field in order to allow the type field to be used for system sub-types.
     * @param {Partial<ChatMessageData>} data
     */
    static #migrateTypeToStyle(data) {
      if ( (typeof data.type !== "number") || ("style" in data) ) return;
      // WHISPER, ROLL, and any other invalid style are redirected to OTHER
      data.style = Object.values(CHAT_MESSAGE_STYLES).includes(data.type) ? data.type : 0;
      data.type = BASE_DOCUMENT_TYPE;
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      this._addDataFieldShim(data, "user", "author", {since: 12, until: 14});
      return super.shimData(data, options);
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get user() {
      this.constructor._logDataFieldMigration("user", "author", {since: 12, until: 14});
      return this.author;
    }
  }

  /**
   * @typedef {import("./_types.mjs").CombatData} CombatData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Card Document.
   * Defines the DataSchema and common behaviors for a Combat which are shared between both client and server.
   * @mixes CombatData
   */
  class BaseCombat extends Document {
    /**
     * Construct a Combat document using provided data and context.
     * @param {Partial<CombatData>} data              Initial data from which to construct the Combat
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Combat",
      collection: "combats",
      label: "DOCUMENT.Combat",
      labelPlural: "DOCUMENT.Combats",
      embedded: {
        Combatant: "combatants"
      },
      hasTypeData: true,
      permissions: {
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /* -------------------------------------------- */

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
        system: new TypeDataField(this),
        scene: new ForeignDocumentField(BaseScene),
        combatants: new EmbeddedCollectionField(BaseCombatant),
        active: new BooleanField(),
        round: new NumberField({required: true, nullable: false, integer: true, min: 0, initial: 0,
          label: "COMBAT.Round"}),
        turn: new NumberField({required: true, integer: true, min: 0, initial: null, label: "COMBAT.Turn"}),
        sort: new IntegerSortField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */

    /**
     * Is a user able to update an existing Combat?
     * @protected
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                             // GM users can do anything
      const turnOnly = ["_id", "round", "turn", "combatants"];  // Players may only modify a subset of fields
      if ( Object.keys(data).some(k => !turnOnly.includes(k)) ) return false;
      if ( ("round" in data) && !doc._canChangeRound(user) ) return false;
      if ( ("turn" in data) && !doc._canChangeTurn(user) ) return false;
      if ( ("combatants" in data) && !doc.#canModifyCombatants(user, data.combatants) ) return false;
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Can a certain User change the Combat round?
     * @param {User} user     The user attempting to change the round
     * @returns {boolean}     Is the user allowed to change the round?
     * @protected
     */
    _canChangeRound(user) {
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Can a certain User change the Combat turn?
     * @param {User} user     The user attempting to change the turn
     * @returns {boolean}     Is the user allowed to change the turn?
     * @protected
     */
    _canChangeTurn(user) {
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Can a certain user make modifications to the array of Combatants?
     * @param {User} user     The user attempting to modify combatants
     * @param {Partial<CombatantData>[]} combatants   Proposed combatant changes
     * @returns {boolean}     Is the user allowed to make this change?
     */
    #canModifyCombatants(user, combatants) {
      for ( const {_id, ...change} of combatants ) {
        const c = this.combatants.get(_id);
        if ( !c ) return false;
        if ( !c.canUserModify(user, "update", change) ) return false;
      }
      return true;
    }

    /* -------------------------------------------- */
    /*  Event Handlers                              */
    /* -------------------------------------------- */

    /** @inheritDoc */
    async _preUpdate(changed, options, user) {
      const allowed = await super._preUpdate(changed, options, user);
      if ( allowed === false ) return false;
      // Don't allow linking to a Scene that doesn't contain all its Combatants
      if ( !("scene" in changed) ) return;
      const sceneId = this.schema.fields.scene.clean(changed.scene);
      if ( (sceneId !== null) && isValidId(sceneId)
        && this.combatants.some(c => c.sceneId && (c.sceneId !== sceneId)) ) {
        throw new Error("You cannot link the Combat to a Scene that doesn't contain all its Combatants.");
      }
    }
  }

  /**
   * @typedef {import("./_types.mjs").CombatantData} CombatantData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Combatant Document.
   * Defines the DataSchema and common behaviors for a Combatant which are shared between both client and server.
   * @mixes CombatantData
   */
  class BaseCombatant extends Document {
    /**
     * Construct a Combatant document using provided data and context.
     * @param {Partial<CombatantData>} data           Initial data from which to construct the Combatant
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Combatant",
      collection: "combatants",
      label: "DOCUMENT.Combatant",
      labelPlural: "DOCUMENT.Combatants",
      isEmbedded: true,
      hasTypeData: true,
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
        system: new TypeDataField(this),
        actorId: new ForeignDocumentField(BaseActor, {label: "COMBAT.CombatantActor", idOnly: true}),
        tokenId: new ForeignDocumentField(BaseToken, {label: "COMBAT.CombatantToken", idOnly: true}),
        sceneId: new ForeignDocumentField(BaseScene, {label: "COMBAT.CombatantScene", idOnly: true}),
        name: new StringField({label: "COMBAT.CombatantName", textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"], label: "COMBAT.CombatantImage"}),
        initiative: new NumberField({label: "COMBAT.CombatantInitiative"}),
        hidden: new BooleanField({label: "COMBAT.CombatantHidden"}),
        defeated: new BooleanField({label: "COMBAT.CombatantDefeated"}),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * Is a user able to update an existing Combatant?
     * @private
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true; // GM users can do anything
      if ( doc.actor && !doc.actor.canUserModify(user, "update", data) ) return false;
      const updateKeys = new Set(Object.keys(data));
      const allowedKeys = new Set(["_id", "initiative", "flags", "defeated"]);
      return updateKeys.isSubset(allowedKeys); // Players may only update initiative scores, flags, and the defeated state
    }

    /**
     * Is a user able to create this Combatant?
     * @private
     */
    static #canCreate(user, doc, data) {
      if ( user.isGM ) return true;
      if ( doc.actor ) return doc.actor.canUserModify(user, "update", data);
      return true;
    }

    /** @override */
    getUserLevel(user) {
      user = user || game.user;
      const {NONE, OWNER} = DOCUMENT_OWNERSHIP_LEVELS;
      if ( user.isGM ) return OWNER;
      return this.actor?.getUserLevel(user) ?? NONE;
    }
  }

  /**
   * @typedef {import("./_types.mjs").DrawingData} DrawingData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Drawing Document.
   * Defines the DataSchema and common behaviors for a Drawing which are shared between both client and server.
   * @mixes DrawingData
   */
  class BaseDrawing extends Document {
    /**
     * Construct a Drawing document using provided data and context.
     * @param {Partial<DrawingData>} data             Initial data from which to construct the Drawing
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* ---------------------------------------- */
    /*  Model Configuration                     */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Drawing",
      collection: "drawings",
      label: "DOCUMENT.Drawing",
      labelPlural: "DOCUMENT.Drawings",
      isEmbedded: true,
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /* ---------------------------------------- */

    /** @inheritDoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        author: new ForeignDocumentField(BaseUser, {nullable: false, initial: () => game.user?.id}),
        shape: new EmbeddedDataField(ShapeData),
        x: new NumberField({required: true, nullable: false, initial: 0, label: "XCoord"}),
        y: new NumberField({required: true, nullable: false, initial: 0, label: "YCoord"}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        rotation: new AngleField({label: "DRAWING.Rotation"}),
        bezierFactor: new AlphaField({initial: 0, label: "DRAWING.SmoothingFactor", max: 0.5,
          hint: "DRAWING.SmoothingFactorHint"}),
        fillType: new NumberField({required: true, nullable: false, initial: DRAWING_FILL_TYPES.NONE,
          choices: Object.values(DRAWING_FILL_TYPES), label: "DRAWING.FillTypes",
          validationError: "must be a value in CONST.DRAWING_FILL_TYPES"
        }),
        fillColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff", label: "DRAWING.FillColor"}),
        fillAlpha: new AlphaField({initial: 0.5, label: "DRAWING.FillOpacity"}),
        strokeWidth: new NumberField({nullable: false, integer: true, initial: 8, min: 0, label: "DRAWING.LineWidth"}),
        strokeColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff", label: "DRAWING.StrokeColor"}),
        strokeAlpha: new AlphaField({initial: 1, label: "DRAWING.LineOpacity"}),
        texture: new FilePathField({categories: ["IMAGE"], label: "DRAWING.FillTexture"}),
        text: new StringField({label: "DRAWING.TextLabel"}),
        fontFamily: new StringField({blank: false, label: "DRAWING.FontFamily",
          initial: () => globalThis.CONFIG?.defaultFontFamily || "Signika"}),
        fontSize: new NumberField({nullable: false, integer: true, min: 8, max: 256, initial: 48, label: "DRAWING.FontSize",
          validationError: "must be an integer between 8 and 256"}),
        textColor: new ColorField({nullable: false, initial: "#ffffff", label: "DRAWING.TextColor"}),
        textAlpha: new AlphaField({label: "DRAWING.TextOpacity"}),
        hidden: new BooleanField(),
        locked: new BooleanField(),
        interface: new BooleanField(),
        flags: new ObjectField()
      }
    }

    /* ---------------------------------------- */

    /**
     * Validate whether the drawing has some visible content (as required by validation).
     * @returns {boolean}
     */
    static #validateVisibleContent(data) {
      const hasText = (data.text !== "") && (data.textAlpha > 0);
      const hasFill = (data.fillType !== DRAWING_FILL_TYPES.NONE) && (data.fillAlpha > 0);
      const hasLine = (data.strokeWidth > 0) && (data.strokeAlpha > 0);
      return hasText || hasFill || hasLine;
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static validateJoint(data) {
      if ( !BaseDrawing.#validateVisibleContent(data) ) {
        throw new Error(game.i18n.localize("DRAWING.JointValidationError"));
      }
    }

    /* -------------------------------------------- */

    /** @override */
    static canUserCreate(user) {
      return user.hasPermission("DRAWING_CREATE");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to create a new Drawing?
     * @param {User} user            The user attempting the creation operation.
     * @param {BaseDrawing} doc      The Drawing being created.
     * @returns {boolean}
     */
    static #canCreate(user, doc) {
      if ( !user.isGM && (doc._source.author !== user.id) ) return false;
      return user.hasPermission("DRAWING_CREATE");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to update the Drawing document?
     */
    static #canUpdate(user, doc, data) {
      if ( !user.isGM && ("author" in data) && (data.author !== user.id) ) return false;
      return doc.testUserPermission(user, "OWNER");
    }

    /* ---------------------------------------- */
    /*  Model Methods                           */
    /* ---------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( !exact && (user.id === this._source.author) ) return true; // The user who created the drawing
      return super.testUserPermission(user, permission, {exact});
    }

    /* ---------------------------------------- */
    /*  Deprecations and Compatibility          */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {
      /**
       * V12 migration to elevation and sort fields
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "z", "elevation");
      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      this._addDataFieldShim(data, "z", "elevation", {since: 12, until: 14});
      return super.shimData(data, options);
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get z() {
      this.constructor._logDataFieldMigration("z", "elevation", {since: 12, until: 14});
      return this.elevation;
    }
  }

  /**
   * @typedef {import("./_types.mjs").FogExplorationData} FogExplorationData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The FogExploration Document.
   * Defines the DataSchema and common behaviors for a FogExploration which are shared between both client and server.
   * @mixes FogExplorationData
   */
  class BaseFogExploration extends Document {
    /**
     * Construct a FogExploration document using provided data and context.
     * @param {Partial<FogExplorationData>} data      Initial data from which to construct the FogExploration
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* ---------------------------------------- */
    /*  Model Configuration                     */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "FogExploration",
      collection: "fog",
      label: "DOCUMENT.FogExploration",
      labelPlural: "DOCUMENT.FogExplorations",
      isPrimary: true,
      permissions: {
        create: "PLAYER",
        update: this.#canModify,
        delete: this.#canModify
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        scene: new ForeignDocumentField(BaseScene, {initial: () => canvas?.scene?.id}),
        user: new ForeignDocumentField(BaseUser, {initial: () => game?.user?.id}),
        explored: new FilePathField({categories: ["IMAGE"], required: true, base64: true}),
        positions: new ObjectField(),
        timestamp: new NumberField({nullable: false, initial: Date.now}),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * Test whether a User can modify a FogExploration document.
     */
    static #canModify(user, doc) {
      return (user.id === doc._source.user) || user.hasRole("ASSISTANT");
    }

    /* ---------------------------------------- */
    /*  Database Event Handlers                 */
    /* ---------------------------------------- */

    /** @inheritDoc */
    async _preUpdate(changed, options, user) {
      const allowed = await super._preUpdate(changed, options, user);
      if ( allowed === false ) return false;
      changed.timestamp = Date.now();
    }
  }

  /**
   * @typedef {import("./_types.mjs").FolderData} FolderData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Folder Document.
   * Defines the DataSchema and common behaviors for a Folder which are shared between both client and server.
   * @mixes FolderData
   */
  class BaseFolder extends Document {
    /**
     * Construct a Folder document using provided data and context.
     * @param {Partial<FolderData>} data              Initial data from which to construct the Folder
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* ---------------------------------------- */
    /*  Model Configuration                     */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Folder",
      collection: "folders",
      label: "DOCUMENT.Folder",
      labelPlural: "DOCUMENT.Folders",
      coreTypes: FOLDER_DOCUMENT_TYPES,
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        type: new DocumentTypeField(this),
        description: new HTMLField({textSearch: true}),
        folder: new ForeignDocumentField(BaseFolder),
        sorting: new StringField({required: true, initial: "a", choices: this.SORTING_MODES}),
        sort: new IntegerSortField(),
        color: new ColorField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /** @inheritdoc */
    static validateJoint(data) {
      if ( (data.folder !== null) && (data.folder === data._id) ) {
        throw new Error("A Folder may not contain itself");
      }
    }

    /**
     * Allow folder sorting modes
     * @type {string[]}
     */
    static SORTING_MODES = ["a", "m"];

    /* -------------------------------------------- */

    /** @override */
    static get(documentId, options={}) {
      if ( !documentId ) return null;
      if ( !options.pack ) return super.get(documentId, options);
      const pack = game.packs.get(options.pack);
      if ( !pack ) {
        console.error(`The ${this.name} model references a non-existent pack ${options.pack}.`);
        return null;
      }
      return pack.folders.get(documentId);
    }
  }

  /**
   * @typedef {import("./_types.mjs").ItemData} ItemData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Item Document.
   * Defines the DataSchema and common behaviors for a Item which are shared between both client and server.
   * @mixes ItemData
   */
  class BaseItem extends Document {
    /**
     * Construct a Item document using provided data and context.
     * @param {Partial<ItemData>} data                Initial data from which to construct the Item
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Item",
      collection: "items",
      hasTypeData: true,
      indexed: true,
      compendiumIndexFields: ["_id", "name", "img", "type", "sort", "folder"],
      embedded: {ActiveEffect: "effects"},
      label: "DOCUMENT.Item",
      labelPlural: "DOCUMENT.Items",
      permissions: {create: "ITEM_CREATE"},
      schemaVersion: "12.324"
    }, {inplace: false}));

    /* ---------------------------------------- */

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        type: new DocumentTypeField(this),
        img: new FilePathField({categories: ["IMAGE"], initial: data => {
          return this.implementation.getDefaultArtwork(data).img;
        }}),
        system: new TypeDataField(this),
        effects: new EmbeddedCollectionField(BaseActiveEffect),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* ---------------------------------------- */

    /**
     * The default icon used for newly created Item documents
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/item-bag.svg";

    /* -------------------------------------------- */

    /**
     * Determine default artwork based on the provided item data.
     * @param {ItemData} itemData  The source item data.
     * @returns {{img: string}}    Candidate item image.
     */
    static getDefaultArtwork(itemData) {
      return { img: this.DEFAULT_ICON };
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    canUserModify(user, action, data={}) {
      if ( this.isEmbedded ) return this.parent.canUserModify(user, "update");
      return super.canUserModify(user, action, data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( this.isEmbedded ) return this.parent.testUserPermission(user, permission, {exact});
      return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").JournalEntryData} JournalEntryData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The JournalEntry Document.
   * Defines the DataSchema and common behaviors for a JournalEntry which are shared between both client and server.
   * @mixes JournalEntryData
   */
  class BaseJournalEntry extends Document {
    /**
     * Construct a JournalEntry document using provided data and context.
     * @param {Partial<JournalEntryData>} data        Initial data from which to construct the JournalEntry
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "JournalEntry",
      collection: "journal",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "sort", "folder"],
      embedded: {JournalEntryPage: "pages"},
      label: "DOCUMENT.JournalEntry",
      labelPlural: "DOCUMENT.JournalEntries",
      permissions: {
        create: "JOURNAL_CREATE"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        pages: new EmbeddedCollectionField(BaseJournalEntryPage),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").JournalEntryPageData} JournalEntryPageData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The JournalEntryPage Document.
   * Defines the DataSchema and common behaviors for a JournalEntryPage which are shared between both client and server.
   * @mixes JournalEntryPageData
   */
  class BaseJournalEntryPage extends Document {
    /**
     * Construct a JournalEntryPage document using provided data and context.
     * @param {Partial<JournalEntryPageData>} data    Initial data from which to construct the JournalEntryPage
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "JournalEntryPage",
      collection: "pages",
      hasTypeData: true,
      indexed: true,
      label: "DOCUMENT.JournalEntryPage",
      labelPlural: "DOCUMENT.JournalEntryPages",
      coreTypes: ["text", "image", "pdf", "video"],
      compendiumIndexFields: ["name", "type", "sort"],
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "JOURNALENTRYPAGE.PageTitle", textSearch: true}),
        type: new DocumentTypeField(this, {initial: "text"}),
        system: new TypeDataField(this),
        title: new SchemaField({
          show: new BooleanField({initial: true}),
          level: new NumberField({required: true, initial: 1, min: 1, max: 6, integer: true, nullable: false})
        }),
        image: new SchemaField({
          caption: new StringField({required: false, initial: undefined})
        }),
        text: new SchemaField({
          content: new HTMLField({required: false, initial: undefined, textSearch: true}),
          markdown: new StringField({required: false, initial: undefined}),
          format: new NumberField({label: "JOURNALENTRYPAGE.Format",
            initial: JOURNAL_ENTRY_PAGE_FORMATS.HTML, choices: Object.values(JOURNAL_ENTRY_PAGE_FORMATS)})
        }),
        video: new SchemaField({
          controls: new BooleanField({initial: true}),
          loop: new BooleanField({required: false, initial: undefined}),
          autoplay: new BooleanField({required: false, initial: undefined}),
          volume: new AlphaField({required: true, step: 0.01, initial: .5}),
          timestamp: new NumberField({required: false, min: 0, initial: undefined}),
          width: new NumberField({required: false, positive: true, integer: true, initial: undefined}),
          height: new NumberField({required: false, positive: true, integer: true, initial: undefined})
        }),
        src: new StringField({required: false, blank: false, nullable: true, initial: null,
          label: "JOURNALENTRYPAGE.Source"}),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField({initial: {default: DOCUMENT_OWNERSHIP_LEVELS.INHERIT}}),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      };
    }

    /** @inheritdoc */
    getUserLevel(user) {
      user = user || game.user;
      const ownership = this.ownership[user.id] ?? this.ownership.default;
      const inherited = ownership === DOCUMENT_OWNERSHIP_LEVELS.INHERIT;
      return inherited ? this.parent.getUserLevel(user) : ownership;
    }
  }

  /**
   * @typedef {import("./_types.mjs").MacroData} MacroData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Macro Document.
   * Defines the DataSchema and common behaviors for a Macro which are shared between both client and server.
   * @mixes MacroData
   */
  class BaseMacro extends Document {
    /**
     * Construct a Macro document using provided data and context.
     * @param {Partial<MacroData>} data               Initial data from which to construct the Macro
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Macro",
      collection: "macros",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "img", "sort", "folder"],
      label: "DOCUMENT.Macro",
      labelPlural: "DOCUMENT.Macros",
      coreTypes: Object.values(MACRO_TYPES),
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "Name", textSearch: true}),
        type: new DocumentTypeField(this, {initial: MACRO_TYPES.CHAT, label: "Type"}),
        author: new ForeignDocumentField(BaseUser, {initial: () => game?.user?.id}),
        img: new FilePathField({categories: ["IMAGE"], initial: () => this.DEFAULT_ICON, label: "Image"}),
        scope: new StringField({required: true, choices: MACRO_SCOPES, initial: MACRO_SCOPES[0],
          validationError: "must be a value in CONST.MACRO_SCOPES", label: "Scope"}),
        command: new StringField({required: true, blank: true, label: "Command"}),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * The default icon used for newly created Macro documents.
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/dice-target.svg";

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @override */
    static validateJoint(data) {
      if ( data.type !== MACRO_TYPES.SCRIPT ) return;
      const field = new JavaScriptField({ async: true });
      const failure = field.validate(data.command);
      if ( failure ) throw failure.asError();
    }

    /* -------------------------------------------- */

    /** @override */
    static canUserCreate(user) {
      return user.hasRole("PLAYER");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to create the Macro document?
     */
    static #canCreate(user, doc) {
      if ( !user.isGM && (doc._source.author !== user.id) ) return false;
      if ( (doc._source.type === "script") && !user.hasPermission("MACRO_SCRIPT") ) return false;
      return user.hasRole("PLAYER");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to update the Macro document?
     */
    static #canUpdate(user, doc, data) {
      if ( !user.isGM && ("author" in data) && (data.author !== user.id) ) return false;
      if ( !user.hasPermission("MACRO_SCRIPT") ) {
        if ( data.type === "script" ) return false;
        if ( (doc._source.type === "script") && ("command" in data) ) return false;
      }
      return doc.testUserPermission(user, "OWNER");
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( !exact && (user.id === this._source.author) ) return true; // Macro authors can edit
      return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */
    /*  Database Event Handlers                     */
    /* -------------------------------------------- */

    /** @inheritDoc */
    async _preCreate(data, options, user) {
      const allowed = await super._preCreate(data, options, user);
      if ( allowed === false ) return false;
      this.updateSource({author: user.id});
    }
  }

  /**
   * @typedef {import("./_types.mjs").MeasuredTemplateData} MeasuredTemplateData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The MeasuredTemplate Document.
   * Defines the DataSchema and common behaviors for a MeasuredTemplate which are shared between both client and server.
   * @mixes MeasuredTemplateData
   */
  class BaseMeasuredTemplate extends Document {
    /**
     * Construct a MeasuredTemplate document using provided data and context.
     * @param {Partial<MeasuredTemplateData>} data    Initial data from which to construct the MeasuredTemplate
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = mergeObject(super.metadata, {
      name: "MeasuredTemplate",
      collection: "templates",
      label: "DOCUMENT.MeasuredTemplate",
      labelPlural: "DOCUMENT.MeasuredTemplates",
      isEmbedded: true,
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false});

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        author: new ForeignDocumentField(BaseUser, {initial: () => game?.user?.id}),
        t: new StringField({required: true, choices: Object.values(MEASURED_TEMPLATE_TYPES), label: "Type",
          initial: MEASURED_TEMPLATE_TYPES.CIRCLE,
          validationError: "must be a value in CONST.MEASURED_TEMPLATE_TYPES",
        }),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "XCoord"}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "YCoord"}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        distance: new NumberField({required: true, nullable: false, initial: 0, min: 0, label: "Distance"}),
        direction: new AngleField({label: "Direction"}),
        angle: new AngleField({normalize: false, label: "Angle"}),
        width: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01, label: "Width"}),
        borderColor: new ColorField({nullable: false, initial: "#000000"}),
        fillColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff"}),
        texture: new FilePathField({categories: ["IMAGE", "VIDEO"]}),
        hidden: new BooleanField({label: "Hidden"}),
        flags: new ObjectField()
      }
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to create a new MeasuredTemplate?
     * @param {User} user                     The user attempting the creation operation.
     * @param {BaseMeasuredTemplate} doc      The MeasuredTemplate being created.
     * @returns {boolean}
     */
    static #canCreate(user, doc) {
      if ( !user.isGM && (doc._source.author !== user.id) ) return false;
      return user.hasPermission("TEMPLATE_CREATE");
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to update the MeasuredTemplate document?
     */
    static #canUpdate(user, doc, data) {
      if ( !user.isGM && ("author" in data) && (data.author !== user.id) ) return false;
      return doc.testUserPermission(user, "OWNER");
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( !exact && (user.id === this._source.author) ) return true; // The user who created the template
      return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {
      /**
       * V12 migration from user to author
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "user", "author");
      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      this._addDataFieldShim(data, "user", "author", {since: 12, until: 14});
      return super.shimData(data, options);
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get user() {
      this.constructor._logDataFieldMigration("user", "author", {since: 12, until: 14});
      return this.author;
    }
  }

  /**
   * @typedef {import("./_types.mjs").NoteData} NoteData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Note Document.
   * Defines the DataSchema and common behaviors for a Note which are shared between both client and server.
   * @mixes NoteData
   */
  class BaseNote extends Document {
    /**
     * Construct a Note document using provided data and context.
     * @param {Partial<NoteData>} data                Initial data from which to construct the Note
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Note",
      collection: "notes",
      label: "DOCUMENT.Note",
      labelPlural: "DOCUMENT.Notes",
      permissions: {
        create: "NOTE_CREATE"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        entryId: new ForeignDocumentField(BaseJournalEntry, {idOnly: true}),
        pageId: new ForeignDocumentField(BaseJournalEntryPage, {idOnly: true}),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "XCoord"}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "YCoord"}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        texture: new TextureData({}, {categories: ["IMAGE"],
          initial: {src: () => this.DEFAULT_ICON, anchorX: 0.5, anchorY: 0.5, fit: "contain"}, label: "NOTE.EntryIcon"}),
        iconSize: new NumberField({required: true, nullable: false, integer: true, min: 32, initial: 40,
          validationError: "must be an integer greater than 32", label: "NOTE.IconSize"}),
        text: new StringField({label: "NOTE.TextLabel", textSearch: true}),
        fontFamily: new StringField({required: true, label: "NOTE.FontFamily",
          initial: () => globalThis.CONFIG?.defaultFontFamily || "Signika"}),
        fontSize: new NumberField({required: true, integer: true, min: 8, max: 128, initial: 32,
          validationError: "must be an integer between 8 and 128", label: "NOTE.FontSize"}),
        textAnchor: new NumberField({required: true, choices: Object.values(TEXT_ANCHOR_POINTS),
          initial: TEXT_ANCHOR_POINTS.BOTTOM, label: "NOTE.AnchorPoint",
          validationError: "must be a value in CONST.TEXT_ANCHOR_POINTS"}),
        textColor: new ColorField({required: true, nullable: false, initial: "#ffffff", label: "NOTE.TextColor"}),
        global: new BooleanField(),
        flags: new ObjectField()
      }
    }

    /**
     * The default icon used for newly created Note documents.
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/book.svg";

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( user.isGM ) return true;                             // Game-masters always have control
      // Players can create and edit unlinked notes with the appropriate permission.
      if ( !this.entryId ) return user.hasPermission("NOTE_CREATE");
      if ( !this.entry ) return false;                          // Otherwise, permission comes through the JournalEntry
      return this.entry.testUserPermission(user, permission, {exact});
    }
  }

  /**
   * @typedef {import("./_types.mjs").PlaylistData} PlaylistData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Playlist Document.
   * Defines the DataSchema and common behaviors for a Playlist which are shared between both client and server.
   * @mixes PlaylistData
   */
  class BasePlaylist extends Document {
    /**
     * Construct a Playlist document using provided data and context.
     * @param {Partial<PlaylistData>} data            Initial data from which to construct the Playlist
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Playlist",
      collection: "playlists",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "description", "sort", "folder"],
      embedded: {PlaylistSound: "sounds"},
      label: "DOCUMENT.Playlist",
      labelPlural: "DOCUMENT.Playlists",
      permissions: {
        create: "PLAYLIST_CREATE"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        description: new StringField({textSearch: true}),
        sounds: new EmbeddedCollectionField(BasePlaylistSound),
        channel: new StringField({choices: AUDIO_CHANNELS, initial: "music", blank: false}),
        mode: new NumberField({required: true, choices: Object.values(PLAYLIST_MODES),
          initial: PLAYLIST_MODES.SEQUENTIAL, validationError: "must be a value in CONST.PLAYLIST_MODES"}),
        playing: new BooleanField(),
        fade: new NumberField({positive: true}),
        folder: new ForeignDocumentField(BaseFolder),
        sorting: new StringField({required: true, choices: Object.values(PLAYLIST_SORT_MODES),
          initial: PLAYLIST_SORT_MODES.ALPHABETICAL,
          validationError: "must be a value in CONST.PLAYLIST_SORTING_MODES"}),
        seed: new NumberField({integer: true, min: 0}),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");
      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").PlaylistSoundData} PlaylistSoundData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The PlaylistSound Document.
   * Defines the DataSchema and common behaviors for a PlaylistSound which are shared between both client and server.
   * @mixes PlaylistSoundData
   */
  class BasePlaylistSound extends Document {
    /**
     * Construct a PlaylistSound document using provided data and context.
     * @param {Partial<PlaylistSoundData>} data       Initial data from which to construct the PlaylistSound
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "PlaylistSound",
      collection: "sounds",
      indexed: true,
      label: "DOCUMENT.PlaylistSound",
      labelPlural: "DOCUMENT.PlaylistSounds",
      compendiumIndexFields: ["name", "sort"],
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        description: new StringField(),
        path: new FilePathField({categories: ["AUDIO"]}),
        channel: new StringField({choices: AUDIO_CHANNELS, initial: "music", blank: true}),
        playing: new BooleanField(),
        pausedTime: new NumberField({min: 0}),
        repeat: new BooleanField(),
        volume: new AlphaField({initial: 0.5, step: 0.01}),
        fade: new NumberField({integer: true, min: 0}),
        sort: new IntegerSortField(),
        flags: new ObjectField(),
      }
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact = false} = {}) {
      if ( this.isEmbedded ) return this.parent.testUserPermission(user, permission, {exact});
      return super.testUserPermission(user, permission, {exact});
    }
  }

  /**
   * @typedef {import("./_types.mjs").RollTableData} RollTableData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The RollTable Document.
   * Defines the DataSchema and common behaviors for a RollTable which are shared between both client and server.
   * @mixes RollTableData
   */
  class BaseRollTable extends Document {
    /**
     * Construct a RollTable document using provided data and context.
     * @param {Partial<RollTableData>} data           Initial data from which to construct the RollTable
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritDoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "RollTable",
      collection: "tables",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "description", "img", "sort", "folder"],
      embedded: {TableResult: "results"},
      label: "DOCUMENT.RollTable",
      labelPlural: "DOCUMENT.RollTables",
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritDoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"], initial: () => this.DEFAULT_ICON}),
        description: new HTMLField({textSearch: true}),
        results: new EmbeddedCollectionField(BaseTableResult),
        formula: new StringField(),
        replacement: new BooleanField({initial: true}),
        displayRoll: new BooleanField({initial: true}),
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /**
     * The default icon used for newly created Macro documents
     * @type {string}
     */
    static DEFAULT_ICON = "icons/svg/d20-grey.svg";

    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(source) {
      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(source);
    }
  }

  /**
   * @typedef {import("./_types.mjs").SceneData} SceneData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Scene Document.
   * Defines the DataSchema and common behaviors for a Scene which are shared between both client and server.
   * @mixes SceneData
   */
  class BaseScene extends Document {
    /**
     * Construct a Scene document using provided data and context.
     * @param {Partial<SceneData>} data               Initial data from which to construct the Scene
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Scene",
      collection: "scenes",
      indexed: true,
      compendiumIndexFields: ["_id", "name", "thumb", "sort", "folder"],
      embedded: {
        AmbientLight: "lights",
        AmbientSound: "sounds",
        Drawing: "drawings",
        MeasuredTemplate: "templates",
        Note: "notes",
        Region: "regions",
        Tile: "tiles",
        Token: "tokens",
        Wall: "walls"
      },
      label: "DOCUMENT.Scene",
      labelPlural: "DOCUMENT.Scenes",
      preserveOnImport: [...super.metadata.preserveOnImport, "active"],
      schemaVersion: "12.325"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      // Define reusable ambience schema for environment
      const environmentData = defaults => new SchemaField({
        hue: new HueField({required: true, initial: defaults.hue,
          label: "SCENES.ENVIRONMENT.Hue", hint: "SCENES.ENVIRONMENT.HueHint"}),
        intensity: new AlphaField({required: true, nullable: false, initial: defaults.intensity,
          label: "SCENES.ENVIRONMENT.Intensity", hint: "SCENES.ENVIRONMENT.IntensityHint"}),
        luminosity: new NumberField({required: true, nullable: false, initial: defaults.luminosity, min: -1, max: 1,
          label: "SCENES.ENVIRONMENT.Luminosity", hint: "SCENES.ENVIRONMENT.LuminosityHint"}),
        saturation: new NumberField({required: true, nullable: false, initial: defaults.saturation, min: -1, max: 1,
          label: "SCENES.ENVIRONMENT.Saturation", hint: "SCENES.ENVIRONMENT.SaturationHint"}),
        shadows: new NumberField({required: true, nullable: false, initial: defaults.shadows, min: 0, max: 1,
          label: "SCENES.ENVIRONMENT.Shadows", hint: "SCENES.ENVIRONMENT.ShadowsHint"})
      });
      // Reuse parts of the LightData schema for the global light
      const lightDataSchema = foundry.data.LightData.defineSchema();

      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),

        // Navigation
        active: new BooleanField(),
        navigation: new BooleanField({initial: true}),
        navOrder: new NumberField({required: true, nullable: false, integer: true, initial: 0}),
        navName: new HTMLField({textSearch: true}),

        // Canvas Dimensions
        background: new TextureData(),
        foreground: new FilePathField({categories: ["IMAGE", "VIDEO"]}),
        foregroundElevation: new NumberField({required: true, positive: true, integer: true}),
        thumb: new FilePathField({categories: ["IMAGE"]}),
        width: new NumberField({integer: true, positive: true, initial: 4000}),
        height: new NumberField({integer: true, positive: true, initial: 3000}),
        padding: new NumberField({required: true, nullable: false, min: 0, max: 0.5, step: 0.05, initial: 0.25}),
        initial: new SchemaField({
          x: new NumberField({integer: true, required: true}),
          y: new NumberField({integer: true, required: true}),
          scale: new NumberField({required: true, max: 3, positive: true, initial: 0.5})
        }),
        backgroundColor: new ColorField({nullable: false, initial: "#999999"}),

        // Grid Configuration
        grid: new SchemaField({
          type: new NumberField({required: true, choices: Object.values(GRID_TYPES),
            initial: () => game.system.grid.type, validationError: "must be a value in CONST.GRID_TYPES"}),
          size: new NumberField({required: true, nullable: false, integer: true, min: GRID_MIN_SIZE,
            initial: 100, validationError: `must be an integer number of pixels, ${GRID_MIN_SIZE} or greater`}),
          style: new StringField({required: true, blank: false, initial: "solidLines"}),
          thickness: new NumberField({required: true, nullable: false, positive: true, integer: true, initial: 1}),
          color: new ColorField({required: true, nullable: false, initial: "#000000"}),
          alpha: new AlphaField({initial: 0.2}),
          distance: new NumberField({required: true, nullable: false, positive: true,
            initial: () => game.system.grid.distance}),
          units: new StringField({required: true, initial: () => game.system.grid.units})
        }),

        // Vision Configuration
        tokenVision: new BooleanField({initial: true}),
        fog: new SchemaField({
          exploration: new BooleanField({initial: true}),
          reset: new NumberField({required: false, initial: undefined}),
          overlay: new FilePathField({categories: ["IMAGE", "VIDEO"]}),
          colors: new SchemaField({
            explored: new ColorField({label: "SCENES.FogExploredColor"}),
            unexplored: new ColorField({label: "SCENES.FogUnexploredColor"})
          })
        }),

        // Environment Configuration
        environment: new SchemaField({
          darknessLevel: new AlphaField({initial: 0}),
          darknessLock: new BooleanField({initial: false}),
          globalLight: new SchemaField({
            enabled: new BooleanField({required: true, initial: false}),
            alpha: lightDataSchema.alpha,
            bright: new BooleanField({required: true, initial: false}),
            color: lightDataSchema.color,
            coloration: lightDataSchema.coloration,
            luminosity: new NumberField({required: true, nullable: false, initial: 0, min: 0, max: 1}),
            saturation: lightDataSchema.saturation,
            contrast: lightDataSchema.contrast,
            shadows: lightDataSchema.shadows,
            darkness: lightDataSchema.darkness
          }),
          cycle: new BooleanField({initial: true}),
          base: environmentData({hue: 0, intensity: 0, luminosity: 0, saturation: 0, shadows: 0}),
          dark: environmentData({hue: 257/360, intensity: 0, luminosity: -0.25, saturation: 0, shadows: 0})
        }),

        // Embedded Collections
        drawings: new EmbeddedCollectionField(BaseDrawing),
        tokens: new EmbeddedCollectionField(BaseToken),
        lights: new EmbeddedCollectionField(BaseAmbientLight),
        notes: new EmbeddedCollectionField(BaseNote),
        sounds: new EmbeddedCollectionField(BaseAmbientSound),
        regions: new EmbeddedCollectionField(BaseRegion),
        templates: new EmbeddedCollectionField(BaseMeasuredTemplate),
        tiles: new EmbeddedCollectionField(BaseTile),
        walls: new EmbeddedCollectionField(BaseWall),

        // Linked Documents
        playlist: new ForeignDocumentField(BasePlaylist),
        playlistSound: new ForeignDocumentField(BasePlaylistSound, {idOnly: true}),
        journal: new ForeignDocumentField(BaseJournalEntry),
        journalEntryPage: new ForeignDocumentField(BaseJournalEntryPage, {idOnly: true}),
        weather: new StringField({required: true}),

        // Permissions
        folder: new ForeignDocumentField(BaseFolder),
        sort: new IntegerSortField(),
        ownership: new DocumentOwnershipField(),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /**
     * Static Initializer Block for deprecated properties.
     * @see [Static Initialization Blocks](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Classes/Static_initialization_blocks)
     */
    static {
      const migrations = {
        fogExploration: "fog.exploration",
        fogReset: "fog.reset",
        fogOverlay: "fog.overlay",
        fogExploredColor: "fog.colors.explored",
        fogUnexploredColor: "fog.colors.unexplored",
        globalLight: "environment.globalLight.enabled",
        globalLightThreshold: "environment.globalLight.darkness.max",
        darkness: "environment.darknessLevel"
      };
      Object.defineProperties(this.prototype, Object.fromEntries(
        Object.entries(migrations).map(([o, n]) => [o, {
          get() {
            this.constructor._logDataFieldMigration(o, n, {since: 12, until: 14});
            return foundry.utils.getProperty(this, n);
          },
          set(v) {
            this.constructor._logDataFieldMigration(o, n, {since: 12, until: 14});
            return foundry.utils.setProperty(this, n, v);
          },
          configurable: true
        }])));
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {
      /**
       * Migration to fog schema fields. Can be safely removed in V14+
       * @deprecated since v12
       */
      for ( const [oldKey, newKey] of Object.entries({
        "fogExploration": "fog.exploration",
        "fogReset": "fog.reset",
        "fogOverlay": "fog.overlay",
        "fogExploredColor": "fog.colors.explored",
        "fogUnexploredColor": "fog.colors.unexplored"
      }) ) this._addDataFieldMigration(data, oldKey, newKey);

      /**
       * Migration to global light embedded fields. Can be safely removed in V14+
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "globalLight", "environment.globalLight.enabled");
      this._addDataFieldMigration(data, "globalLightThreshold", "environment.globalLight.darkness.max",
        d => d.globalLightThreshold ?? 1);

      /**
       * Migration to environment darkness level. Can be safely removed in V14+
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "darkness", "environment.darknessLevel");

      /**
       * Migrate sourceId.
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "flags.core.sourceId", "_stats.compendiumSource");

      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      /** @deprecated since v12 */
      this._addDataFieldShims(data, {
        fogExploration: "fog.exploration",
        fogReset: "fog.reset",
        fogOverlay: "fog.overlay",
        fogExploredColor: "fog.colors.explored",
        fogUnexploredColor: "fog.colors.unexplored",
        globalLight: "environment.globalLight.enabled",
        globalLightThreshold: "environment.globalLight.darkness.max",
        darkness: "environment.darknessLevel"
      }, {since: 12, until: 14});
      return super.shimData(data, options);
    }
  }

  /**
   * @typedef {import("./_types.mjs").RegionData} RegionData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Region Document.
   * Defines the DataSchema and common behaviors for a Region which are shared between both client and server.
   * @mixes RegionData
   */
  class BaseRegion extends Document {
    /**
     * Construct a Region document using provided data and context.
     * @param {Partial<RegionData>} data         Initial data from which to construct the Region
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Region",
      collection: "regions",
      label: "DOCUMENT.Region",
      labelPlural: "DOCUMENT.Regions",
      isEmbedded: true,
      embedded: {
        RegionBehavior: "behaviors"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, label: "Name", textSearch: true}),
        color: new ColorField({required: true, nullable: false,
          initial: () => Color$1.fromHSV([Math.random(), 0.8, 0.8]).css,
          label: "REGION.FIELDS.color.label",
          hint: "REGION.FIELDS.color.hint"}),
        shapes: new ArrayField(new TypedSchemaField(BaseShapeData.TYPES),
          {label: "REGION.FIELDS.shapes.label", hint: "REGION.FIELDS.shapes.hint"}),
        elevation: new SchemaField({
          bottom: new NumberField({required: true,
            label: "REGION.FIELDS.elevation.FIELDS.bottom.label",
            hint: "REGION.FIELDS.elevation.FIELDS.bottom.hint"}), // null -> -Infinity
          top: new NumberField({required: true,
            label: "REGION.FIELDS.elevation.FIELDS.top.label",
            hint: "REGION.FIELDS.elevation.FIELDS.top.hint"}) // null -> +Infinity
        }, {
          label: "REGION.FIELDS.elevation.label",
          hint: "REGION.FIELDS.elevation.hint",
          validate: d => (d.bottom ?? -Infinity) <= (d.top ?? Infinity),
          validationError: "elevation.top may not be less than elevation.bottom"
        }),
        behaviors: new EmbeddedCollectionField(BaseRegionBehavior, {label: "REGION.FIELDS.behaviors.label",
          hint: "REGION.FIELDS.behaviors.hint"}),
        visibility: new NumberField({required: true,
          initial: CONST.REGION_VISIBILITY.LAYER,
          choices:  Object.fromEntries(Object.entries(CONST.REGION_VISIBILITY).map(([key, value]) =>
            [value, {label: `REGION.VISIBILITY.${key}.label`}])),
          label: "REGION.FIELDS.visibility.label",
          hint: "REGION.FIELDS.visibility.hint"}),
        locked: new BooleanField(),
        flags: new ObjectField()
      }
    };
  }

  /**
   * @typedef {import("./_types.mjs").RegionBehaviorData} RegionBehaviorData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The RegionBehavior Document.
   * Defines the DataSchema and common behaviors for a RegionBehavior which are shared between both client and server.
   * @mixes SceneRegionData
   */
  class BaseRegionBehavior extends Document {
    /**
     * Construct a RegionBehavior document using provided data and context.
     * @param {Partial<RegionBehaviorData>} data    Initial data from which to construct the RegionBehavior
     * @param {DocumentConstructionContext} context      Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "RegionBehavior",
      collection: "behaviors",
      label: "DOCUMENT.RegionBehavior",
      labelPlural: "DOCUMENT.RegionBehaviors",
      coreTypes: ["adjustDarknessLevel", "displayScrollingText", "executeMacro", "executeScript", "pauseGame", "suppressWeather", "teleportToken", "toggleBehavior"],
      hasTypeData: true,
      isEmbedded: true,
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: true, label: "Name", textSearch: true}),
        type: new DocumentTypeField(this),
        system: new TypeDataField(this),
        disabled: new BooleanField({label: "BEHAVIOR.FIELDS.disabled.label", hint: "BEHAVIOR.FIELDS.disabled.hint"}),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      };
    }

    /* -------------------------------------------- */

    /** @override */
    static canUserCreate(user) {
      return user.isGM;
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to create the RegionBehavior document?
     */
    static #canCreate(user, doc) {
      if ( (doc._source.type === "executeScript") && !user.hasPermission("MACRO_SCRIPT") ) return false;
      return user.isGM;
    }

    /* ---------------------------------------- */

    /**
     * Is a user able to update the RegionBehavior document?
     */
    static #canUpdate(user, doc, data) {
      if ( (((doc._source.type === "executeScript") && ("system" in data) && ("source" in data.system))
        || (data.type === "executeScript")) && !user.hasPermission("MACRO_SCRIPT") ) return false;
      return user.isGM;
    }
  }

  /**
   * @typedef {import("./_types.mjs").SettingData} SettingData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Setting Document.
   * Defines the DataSchema and common behaviors for a Setting which are shared between both client and server.
   * @mixes SettingData
   */
  class BaseSetting extends Document {
    /**
     * Construct a Setting document using provided data and context.
     * @param {Partial<SettingData>} data             Initial data from which to construct the Setting
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Setting",
      collection: "settings",
      label: "DOCUMENT.Setting",
      labelPlural: "DOCUMENT.Settings",
      permissions: {
        create: this.#canModify,
        update: this.#canModify,
        delete: this.#canModify
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        key: new StringField({required: true, nullable: false, blank: false,
          validate: k => k.split(".").length >= 2,
          validationError: "must have the format {scope}.{field}"}),
        value: new JSONField({required: true, nullable: true, initial: null}),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */

    /**
     * The settings that only full GMs can modify.
     * @type {string[]}
     */
    static #GAMEMASTER_ONLY_KEYS = ["core.permissions"];

    /* -------------------------------------------- */

    /**
     * The settings that assistant GMs can modify regardless of their permission.
     * @type {string[]}
     */
    static #ALLOWED_ASSISTANT_KEYS = ["core.time", "core.combatTrackerConfig", "core.sheetClasses", "core.scrollingStatusText",
      "core.tokenDragPreview", "core.adventureImports", "core.gridDiagonals", "core.gridTemplates", "core.coneTemplateType"];

    /* -------------------------------------------- */

    /** @override */
    static canUserCreate(user) {
      return user.hasPermission("SETTINGS_MODIFY");
    }

    /* -------------------------------------------- */

    /**
     * Define special rules which allow certain settings to be updated.
     * @protected
     */
    static #canModify(user, doc, data) {
      if ( BaseSetting.#GAMEMASTER_ONLY_KEYS.includes(doc._source.key)
        && (!("key" in data) || BaseSetting.#GAMEMASTER_ONLY_KEYS.includes(data.key)) ) return user.hasRole("GAMEMASTER");
      if ( user.hasPermission("SETTINGS_MODIFY") ) return true;
      if ( !user.isGM ) return false;
      return BaseSetting.#ALLOWED_ASSISTANT_KEYS.includes(doc._source.key)
        && (!("key" in data) || BaseSetting.#ALLOWED_ASSISTANT_KEYS.includes(data.key));
    }
  }

  /**
   * @typedef {import("./_types.mjs").TableResultData} TableResultData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The TableResult Document.
   * Defines the DataSchema and common behaviors for a TableResult which are shared between both client and server.
   * @mixes TableResultData
   */
  class BaseTableResult extends Document {
    /**
     * Construct a TableResult document using provided data and context.
     * @param {Partial<TableResultData>} data         Initial data from which to construct the TableResult
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "TableResult",
      collection: "results",
      label: "DOCUMENT.TableResult",
      labelPlural: "DOCUMENT.TableResults",
      coreTypes: Object.values(TABLE_RESULT_TYPES),
      permissions: {
        update: this.#canUpdate
      },
      compendiumIndexFields: ["type"],
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        type: new DocumentTypeField(this, {initial: TABLE_RESULT_TYPES.TEXT}),
        text: new HTMLField({textSearch: true}),
        img: new FilePathField({categories: ["IMAGE"]}),
        documentCollection: new StringField(),
        documentId: new ForeignDocumentField(Document, {idOnly: true}),
        weight: new NumberField({required: true, integer: true, positive: true, nullable: false, initial: 1}),
        range: new ArrayField(new NumberField({integer: true}), {
          validate: r => (r.length === 2) && (r[1] >= r[0]),
          validationError: "must be a length-2 array of ascending integers"
        }),
        drawn: new BooleanField(),
        flags: new ObjectField()
      }
    }

    /**
     * Is a user able to update an existing TableResult?
     * @private
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                               // GM users can do anything
      const wasDrawn = new Set(["drawn", "_id"]);                 // Users can update the drawn status of a result
      if ( new Set(Object.keys(data)).equals(wasDrawn) ) return true;
      return doc.parent.canUserModify(user, "update", data);      // Otherwise, go by parent document permission
    }

    /* -------------------------------------------- */
    /*  Model Methods                               */
    /* -------------------------------------------- */

    /** @inheritdoc */
    testUserPermission(user, permission, {exact=false}={}) {
      if ( this.isEmbedded ) return this.parent.testUserPermission(user, permission, {exact});
      return super.testUserPermission(user, permission, {exact});
    }

    /* ---------------------------------------- */
    /*  Deprecations and Compatibility          */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {

      /**
       * V12 migration of type from number to string.
       * @deprecated since v12
       */
      if ( typeof data.type === "number" ) {
        switch ( data.type ) {
          case 0: data.type = TABLE_RESULT_TYPES.TEXT; break;
          case 1: data.type = TABLE_RESULT_TYPES.DOCUMENT; break;
          case 2: data.type = TABLE_RESULT_TYPES.COMPENDIUM; break;
        }
      }
      return super.migrateData(data);
    }
  }

  /**
   * @typedef {import("./_types.mjs").TileData} TileData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Tile Document.
   * Defines the DataSchema and common behaviors for a Tile which are shared between both client and server.
   * @mixes TileData
   */
  class BaseTile extends Document {
    /**
     * Construct a Tile document using provided data and context.
     * @param {Partial<TileData>} data                Initial data from which to construct the Tile
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Tile",
      collection: "tiles",
      label: "DOCUMENT.Tile",
      labelPlural: "DOCUMENT.Tiles",
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        texture: new TextureData({}, {initial: {anchorX: 0.5, anchorY: 0.5, alphaThreshold: 0.75}}),
        width: new NumberField({required: true, min: 0, nullable: false, step: 0.1}),
        height: new NumberField({required: true, min: 0, nullable: false, step: 0.1}),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "XCoord"}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "YCoord"}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        rotation: new AngleField(),
        alpha: new AlphaField(),
        hidden: new BooleanField(),
        locked: new BooleanField(),
        restrictions: new SchemaField({
          light: new BooleanField(),
          weather: new BooleanField()
        }),
        occlusion: new SchemaField({
          mode: new NumberField({choices: Object.values(OCCLUSION_MODES),
            initial: OCCLUSION_MODES.NONE,
            validationError: "must be a value in CONST.TILE_OCCLUSION_MODES"}),
          alpha: new AlphaField({initial: 0})
        }),
        video: new SchemaField({
          loop: new BooleanField({initial: true}),
          autoplay: new BooleanField({initial: true}),
          volume: new AlphaField({initial: 0, step: 0.01})
        }),
        flags: new ObjectField()
      }
    }


    /* ---------------------------------------- */
    /*  Deprecations and Compatibility          */
    /* ---------------------------------------- */

    /** @inheritdoc */
    static migrateData(data) {
      /**
       * V12 migration to elevation and sort
       * @deprecated since v12
       */
      this._addDataFieldMigration(data, "z", "sort");

      /**
       * V12 migration from roof to restrictions.light and restrictions.weather
       * @deprecated since v12
       */
      if ( foundry.utils.hasProperty(data, "roof") ) {
        const value = foundry.utils.getProperty(data, "roof");
        if ( !foundry.utils.hasProperty(data, "restrictions.light") ) foundry.utils.setProperty(data, "restrictions.light", value);
        if ( !foundry.utils.hasProperty(data, "restrictions.weather") ) foundry.utils.setProperty(data, "restrictions.weather", value);
        delete data["roof"];
      }

      return super.migrateData(data);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      this._addDataFieldShim(data, "z", "sort", {since: 12, until: 14});
      return super.shimData(data, options);
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    set roof(enabled) {
      this.constructor._logDataFieldMigration("roof", "restrictions.{light|weather}", {since: 12, until: 14});
      this.restrictions.light = enabled;
      this.restrictions.weather = enabled;
    }

    /**
     * @deprecated since v12
     * @ignore
     */
    get roof() {
      this.constructor._logDataFieldMigration("roof", "restrictions.{light|weather}", {since: 12, until: 14});
      return this.restrictions.light && this.restrictions.weather;
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get z() {
      this.constructor._logDataFieldMigration("z", "sort", {since: 12, until: 14});
      return this.sort;
    }

    /* ---------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get overhead() {
      foundry.utils.logCompatibilityWarning(`${this.constructor.name}#overhead is deprecated.`, {since: 12, until: 14});
      return this.elevation >= this.parent?.foregroundElevation;
    }
  }

  /**
   * @typedef {import("./_types.mjs").TokenData} TokenData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Token Document.
   * Defines the DataSchema and common behaviors for a Token which are shared between both client and server.
   * @mixes TokenData
   */
  class BaseToken extends Document {
    /**
     * Construct a Token document using provided data and context.
     * @param {Partial<TokenData>} data               Initial data from which to construct the Token
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Token",
      collection: "tokens",
      label: "DOCUMENT.Token",
      labelPlural: "DOCUMENT.Tokens",
      isEmbedded: true,
      embedded: {
        ActorDelta: "delta"
      },
      permissions: {
        create: "TOKEN_CREATE",
        update: this.#canUpdate,
        delete: "TOKEN_DELETE"
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: true, textSearch: true}),
        displayName: new NumberField({required: true, initial: TOKEN_DISPLAY_MODES.NONE,
          choices: Object.values(TOKEN_DISPLAY_MODES),
          validationError: "must be a value in CONST.TOKEN_DISPLAY_MODES"
        }),
        actorId: new ForeignDocumentField(BaseActor, {idOnly: true}),
        actorLink: new BooleanField(),
        delta: new ActorDeltaField(BaseActorDelta),
        appendNumber: new BooleanField(),
        prependAdjective: new BooleanField(),
        width: new NumberField({nullable: false, positive: true, initial: 1, step: 0.5, label: "Width"}),
        height: new NumberField({nullable: false, positive: true, initial: 1, step: 0.5, label: "Height"}),
        texture: new TextureData({}, {initial: {src: () => this.DEFAULT_ICON, anchorX: 0.5, anchorY: 0.5, fit: "contain",
          alphaThreshold: 0.75}, wildcard: true}),
        hexagonalShape: new NumberField({initial: TOKEN_HEXAGONAL_SHAPES.ELLIPSE_1,
          choices: Object.values(TOKEN_HEXAGONAL_SHAPES)}),
        x: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "XCoord"}),
        y: new NumberField({required: true, integer: true, nullable: false, initial: 0, label: "YCoord"}),
        elevation: new NumberField({required: true, nullable: false, initial: 0}),
        sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
        locked: new BooleanField(),
        lockRotation: new BooleanField(),
        rotation: new AngleField(),
        alpha: new AlphaField(),
        hidden: new BooleanField(),
        disposition: new NumberField({required: true, choices: Object.values(TOKEN_DISPOSITIONS),
          initial: TOKEN_DISPOSITIONS.HOSTILE,
          validationError: "must be a value in CONST.TOKEN_DISPOSITIONS"
        }),
        displayBars: new NumberField({required: true, choices: Object.values(TOKEN_DISPLAY_MODES),
          initial: TOKEN_DISPLAY_MODES.NONE,
          validationError: "must be a value in CONST.TOKEN_DISPLAY_MODES"
        }),
        bar1: new SchemaField({
          attribute: new StringField({required: true, nullable: true, blank: false,
            initial: () => game?.system.primaryTokenAttribute || null})
        }),
        bar2: new SchemaField({
          attribute: new StringField({required: true, nullable: true, blank: false,
            initial: () => game?.system.secondaryTokenAttribute || null})
        }),
        light: new EmbeddedDataField(LightData),
        sight: new SchemaField({
          enabled: new BooleanField({initial: data => Number(data?.sight?.range) > 0}),
          range: new NumberField({required: true, nullable: true, min: 0, step: 0.01, initial: 0}),
          angle: new AngleField({initial: 360, normalize: false}),
          visionMode: new StringField({required: true, blank: false, initial: "basic",
            label: "TOKEN.VisionMode", hint: "TOKEN.VisionModeHint"}),
          color: new ColorField({label: "TOKEN.VisionColor"}),
          attenuation: new AlphaField({initial: 0.1, label: "TOKEN.VisionAttenuation", hint: "TOKEN.VisionAttenuationHint"}),
          brightness: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1,
            label: "TOKEN.VisionBrightness", hint: "TOKEN.VisionBrightnessHint"}),
          saturation: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1,
            label: "TOKEN.VisionSaturation", hint: "TOKEN.VisionSaturationHint"}),
          contrast: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1,
            label: "TOKEN.VisionContrast", hint: "TOKEN.VisionContrastHint"})
        }),
        detectionModes: new ArrayField(new SchemaField({
          id: new StringField(),
          enabled: new BooleanField({initial: true}),
          range: new NumberField({required: true, min: 0, step: 0.01})
        }), {
          validate: BaseToken.#validateDetectionModes
        }),
        occludable: new SchemaField({
          radius: new NumberField({nullable: false, min: 0, step: 0.01, initial: 0})
        }),
        ring: new SchemaField({
          enabled: new BooleanField(),
          colors: new SchemaField({
            ring: new ColorField(),
            background: new ColorField()
          }),
          effects: new NumberField({initial: 1, min: 0, max: 8388607, integer: true}),
          subject: new SchemaField({
            scale: new NumberField({initial: 1, min: 0.5}),
            texture: new FilePathField({categories: ["IMAGE"]})
          })
        }),
        /** @internal */
        _regions: new ArrayField(new ForeignDocumentField(BaseRegion, {idOnly: true})),
        flags: new ObjectField()
      }
    }

    /** @override */
    static LOCALIZATION_PREFIXES = ["TOKEN"];

    /* -------------------------------------------- */

    /**
     * Validate the structure of the detection modes array
     * @param {object[]} modes    Configured detection modes
     * @throws                    An error if the array is invalid
     */
    static #validateDetectionModes(modes) {
      const seen = new Set();
      for ( const mode of modes ) {
        if ( mode.id === "" ) continue;
        if ( seen.has(mode.id) ) {
          throw new Error(`may not have more than one configured detection mode of type "${mode.id}"`);
        }
        seen.add(mode.id);
      }
    }

    /* -------------------------------------------- */

    /**
     * The default icon used for newly created Token documents
     * @type {string}
     */
    static DEFAULT_ICON = DEFAULT_TOKEN;

    /**
     * Is a user able to update an existing Token?
     * @private
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                     // GM users can do anything
      if ( doc.actor ) {                                // You can update Tokens for Actors you control
        return doc.actor.canUserModify(user, "update", data);
      }
      return !!doc.actorId;                             // It would be good to harden this in the future
    }

    /** @override */
    testUserPermission(user, permission, {exact=false} = {}) {
      if ( this.actor ) return this.actor.testUserPermission(user, permission, {exact});
      else return super.testUserPermission(user, permission, {exact});
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    updateSource(changes={}, options={}) {
      const diff = super.updateSource(changes, options);

      // A copy of the source data is taken for the _backup in updateSource. When this backup is applied as part of a dry-
      // run, if a child singleton embedded document was updated, the reference to its source is broken. We restore it
      // here.
      if ( options.dryRun && ("delta" in changes) ) this._source.delta = this.delta._source;

      return diff;
    }

    /* -------------------------------------------- */

    /** @inheritdoc */
    toObject(source=true) {
      const obj = super.toObject(source);
      obj.delta = this.delta ? this.delta.toObject(source) : null;
      return obj;
    }

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /** @inheritDoc */
    static migrateData(data) {

      // Remember that any migrations defined here may also be required for the PrototypeToken model.

      /**
       * Migration of actorData field to ActorDelta document.
       * @deprecated since v11
       */
      if ( ("actorData" in data) && !("delta" in data) ) {
        data.delta = data.actorData;
        if ( "_id" in data ) data.delta._id = data._id;
      }
      return super.migrateData(data);
    }

    /* ----------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {

      // Remember that any shims defined here may also be required for the PrototypeToken model.

      this._addDataFieldShim(data, "actorData", "delta", {value: data.delta, since: 11, until: 13});
      this._addDataFieldShim(data, "effects", undefined, {value: [], since: 12, until: 14,
        warning: "TokenDocument#effects is deprecated in favor of using ActiveEffect"
          + " documents on the associated Actor"});
      this._addDataFieldShim(data, "overlayEffect", undefined, {value: "", since: 12, until: 14,
        warning: "TokenDocument#overlayEffect is deprecated in favor of using" +
          " ActiveEffect documents on the associated Actor"});
      return super.shimData(data, options);
    }

    /* -------------------------------------------- */

    /**
     * @deprecated since v12
     * @ignore
     */
    get effects() {
      foundry.utils.logCompatibilityWarning("TokenDocument#effects is deprecated in favor of using ActiveEffect"
        + " documents on the associated Actor", {since: 12, until: 14, once: true});
      return [];
    }

    /**
     * @deprecated since v12
     * @ignore
     */
    get overlayEffect() {
      foundry.utils.logCompatibilityWarning("TokenDocument#overlayEffect is deprecated in favor of using" +
        " ActiveEffect documents on the associated Actor", {since: 12, until: 14, once: true});
      return "";
    }
  }

  /* -------------------------------------------- */

  /**
   * A special subclass of EmbeddedDocumentField which allows construction of the ActorDelta to be lazily evaluated.
   */
  class ActorDeltaField extends EmbeddedDocumentField {
    /** @inheritdoc */
    initialize(value, model, options = {}) {
      if ( !value ) return value;
      const descriptor = Object.getOwnPropertyDescriptor(model, this.name);
      if ( (descriptor === undefined) || (!descriptor.get && !descriptor.value) ) {
        return () => {
          const m = new this.model(value, {...options, parent: model, parentCollection: this.name});
          Object.defineProperty(m, "schema", {value: this});
          Object.defineProperty(model, this.name, {
            value: m,
            configurable: true,
            writable: true
          });
          return m;
        };
      }
      else if ( descriptor.get instanceof Function ) return descriptor.get;
      model[this.name]._initialize(options);
      return model[this.name];
    }
  }

  /**
   * @typedef {import("./_types.mjs").UserData} UserData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The User Document.
   * Defines the DataSchema and common behaviors for a User which are shared between both client and server.
   * @mixes UserData
   */
  class BaseUser extends Document {
    /**
     * Construct a User document using provided data and context.
     * @param {Partial<UserData>} data                Initial data from which to construct the User
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "User",
      collection: "users",
      label: "DOCUMENT.User",
      labelPlural: "DOCUMENT.Users",
      permissions: {
        create: this.#canCreate,
        update: this.#canUpdate,
        delete: this.#canDelete
      },
      schemaVersion: "12.324",
    }, {inplace: false}));

    /** @override */
    static LOCALIZATION_PREFIXES = ["USER"];

    /* -------------------------------------------- */

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        name: new StringField({required: true, blank: false, textSearch: true}),
        role: new NumberField({required: true, choices: Object.values(USER_ROLES),
          initial: USER_ROLES.PLAYER, readonly: true}),
        password: new StringField({required: true, blank: true}),
        passwordSalt: new StringField(),
        avatar: new FilePathField({categories: ["IMAGE"]}),
        character: new ForeignDocumentField(BaseActor),
        color: new ColorField({required: true, nullable: false,
          initial: () => Color$1.fromHSV([Math.random(), 0.8, 0.8]).css
        }),
        pronouns: new StringField({required: true}),
        hotbar: new ObjectField({required: true, validate: BaseUser.#validateHotbar,
          validationError: "must be a mapping of slots to macro identifiers"}),
        permissions: new ObjectField({required: true, validate: BaseUser.#validatePermissions,
          validationError: "must be a mapping of permission names to booleans"}),
        flags: new ObjectField(),
        _stats: new DocumentStatsField()
      }
    }

    /* -------------------------------------------- */

    /**
     * Validate the structure of the User hotbar object
     * @param {object} bar      The attempted hotbar data
     * @return {boolean}
     * @private
     */
    static #validateHotbar(bar) {
      if ( typeof bar !== "object" ) return false;
      for ( let [k, v] of Object.entries(bar) ) {
        let slot = parseInt(k);
        if ( !slot || slot < 1 || slot > 50 ) return false;
        if ( !isValidId(v) ) return false;
      }
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Validate the structure of the User permissions object
     * @param {object} perms      The attempted permissions data
     * @return {boolean}
     */
    static #validatePermissions(perms) {
      for ( let [k, v] of Object.entries(perms) ) {
        if ( typeof k !== "string" ) return false;
        if ( k.startsWith("-=") ) {
          if ( v !== null ) return false;
        } else {
          if ( typeof v !== "boolean" ) return false;
        }
      }
      return true;
    }

    /* -------------------------------------------- */
    /*  Model Properties                            */
    /* -------------------------------------------- */

    /**
     * A convenience test for whether this User has the NONE role.
     * @type {boolean}
     */
    get isBanned() {
      return this.role === USER_ROLES.NONE;

    }

    /* -------------------------------------------- */

    /**
     * Test whether the User has a GAMEMASTER or ASSISTANT role in this World?
     * @type {boolean}
     */
    get isGM() {
      return this.hasRole(USER_ROLES.ASSISTANT);
    }

    /* -------------------------------------------- */

    /**
     * Test whether the User is able to perform a certain permission action.
     * The provided permission string may pertain to an explicit permission setting or a named user role.
     *
     * @param {string} action         The action to test
     * @return {boolean}              Does the user have the ability to perform this action?
     */
    can(action) {
      if ( action in USER_PERMISSIONS ) return this.hasPermission(action);
      return this.hasRole(action);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    getUserLevel(user) {
      return DOCUMENT_OWNERSHIP_LEVELS[user.id === this.id ? "OWNER" : "NONE"];
    }

    /* ---------------------------------------- */

    /**
     * Test whether the User has at least a specific permission
     * @param {string} permission    The permission name from USER_PERMISSIONS to test
     * @return {boolean}             Does the user have at least this permission
     */
    hasPermission(permission) {
      if ( this.isBanned ) return false;

      // CASE 1: The user has the permission set explicitly
      const explicit = this.permissions[permission];
      if (explicit !== undefined) return explicit;

      // CASE 2: Permission defined by the user's role
      const rolePerms = game.permissions[permission];
      return rolePerms ? rolePerms.includes(this.role) : false;
    }

    /* ----------------------------------------- */

    /**
     * Test whether the User has at least the permission level of a certain role
     * @param {string|number} role    The role name from USER_ROLES to test
     * @param {boolean} [exact]       Require the role match to be exact
     * @return {boolean}              Does the user have at this role level (or greater)?
     */
    hasRole(role, {exact = false} = {}) {
      const level = typeof role === "string" ? USER_ROLES[role] : role;
      if (level === undefined) return false;
      return exact ? this.role === level : this.role >= level;
    }

    /* ---------------------------------------- */
    /*  Model Permissions                       */
    /* ---------------------------------------- */

    /**
     * Is a user able to create an existing User?
     * @param {BaseUser} user    The user attempting the creation.
     * @param {BaseUser} doc     The User document being created.
     * @param {object} data      The supplied creation data.
     * @private
     */
    static #canCreate(user, doc, data) {
      if ( !user.isGM ) return false; // Only Assistants and above can create users.
      // Do not allow Assistants to create a new user with special permissions which might be greater than their own.
      if ( !isEmpty$1(doc.permissions) ) return user.hasRole(USER_ROLES.GAMEMASTER);
      return user.hasRole(doc.role);
    }

    /* -------------------------------------------- */

    /**
     * Is a user able to update an existing User?
     * @param {BaseUser} user    The user attempting the update.
     * @param {BaseUser} doc     The User document being updated.
     * @param {object} changes   Proposed changes.
     * @private
     */
    static #canUpdate(user, doc, changes) {
      const roles = USER_ROLES;
      if ( user.role === roles.GAMEMASTER ) return true; // Full GMs can do everything
      if ( user.role === roles.NONE ) return false; // Banned users can do nothing

      // Non-GMs cannot update certain fields.
      const restricted = ["permissions", "passwordSalt"];
      if ( user.role < roles.ASSISTANT ) restricted.push("name", "role");
      if ( doc.role === roles.GAMEMASTER ) restricted.push("password");
      if ( restricted.some(k => k in changes) ) return false;

      // Role changes may not escalate
      if ( ("role" in changes) && !user.hasRole(changes.role) ) return false;

      // Assistant GMs may modify other users. Players may only modify themselves
      return user.isGM || (user.id === doc.id);
    }

    /* -------------------------------------------- */

    /**
     * Is a user able to delete an existing User?
     * Only Assistants and Gamemasters can delete users, and only if the target user has a lesser or equal role.
     * @param {BaseUser} user   The user attempting the deletion.
     * @param {BaseUser} doc    The User document being deleted.
     * @private
     */
    static #canDelete(user, doc) {
      const role = Math.max(USER_ROLES.ASSISTANT, doc.role);
      return user.hasRole(role);
    }
  }

  /**
   * @typedef {import("./_types.mjs").WallData} WallData
   * @typedef {import("../types.mjs").DocumentConstructionContext} DocumentConstructionContext
   */

  /**
   * The Wall Document.
   * Defines the DataSchema and common behaviors for a Wall which are shared between both client and server.
   * @mixes WallData
   */
  class BaseWall extends Document {
    /**
     * Construct a Wall document using provided data and context.
     * @param {Partial<WallData>} data                Initial data from which to construct the Wall
     * @param {DocumentConstructionContext} context   Construction context options
     */
    constructor(data, context) {
      super(data, context);
    }

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /** @inheritdoc */
    static metadata = Object.freeze(mergeObject(super.metadata, {
      name: "Wall",
      collection: "walls",
      label: "DOCUMENT.Wall",
      labelPlural: "DOCUMENT.Walls",
      permissions: {
        update: this.#canUpdate
      },
      schemaVersion: "12.324"
    }, {inplace: false}));

    /** @inheritdoc */
    static defineSchema() {
      return {
        _id: new DocumentIdField(),
        c: new ArrayField(new NumberField({required: true, integer: true, nullable: false}), {
          validate: c => (c.length === 4),
          validationError: "must be a length-4 array of integer coordinates"}),
        light: new NumberField({required: true, choices: Object.values(WALL_SENSE_TYPES),
          initial: WALL_SENSE_TYPES.NORMAL,
          validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
        move: new NumberField({required: true, choices: Object.values(WALL_MOVEMENT_TYPES),
          initial: WALL_MOVEMENT_TYPES.NORMAL,
          validationError: "must be a value in CONST.WALL_MOVEMENT_TYPES"}),
        sight: new NumberField({required: true, choices: Object.values(WALL_SENSE_TYPES),
          initial: WALL_SENSE_TYPES.NORMAL,
          validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
        sound: new NumberField({required: true, choices: Object.values(WALL_SENSE_TYPES),
          initial: WALL_SENSE_TYPES.NORMAL,
          validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
        dir: new NumberField({required: true, choices: Object.values(WALL_DIRECTIONS),
          initial: WALL_DIRECTIONS.BOTH,
          validationError: "must be a value in CONST.WALL_DIRECTIONS"}),
        door: new NumberField({required: true, choices: Object.values(WALL_DOOR_TYPES),
          initial: WALL_DOOR_TYPES.NONE,
          validationError: "must be a value in CONST.WALL_DOOR_TYPES"}),
        ds: new NumberField({required: true, choices: Object.values(WALL_DOOR_STATES),
          initial: WALL_DOOR_STATES.CLOSED,
          validationError: "must be a value in CONST.WALL_DOOR_STATES"}),
        doorSound: new StringField({required: false, blank: true, initial: undefined}),
        threshold: new SchemaField({
          light: new NumberField({required: true, nullable: true, initial: null, positive: true}),
          sight: new NumberField({required: true, nullable: true, initial: null, positive: true}),
          sound: new NumberField({required: true, nullable: true, initial: null, positive: true}),
          attenuation: new BooleanField()
        }),
        flags: new ObjectField()
      };
    }

    /**
     * Is a user able to update an existing Wall?
     * @private
     */
    static #canUpdate(user, doc, data) {
      if ( user.isGM ) return true;                     // GM users can do anything
      const dsOnly = Object.keys(data).every(k => ["_id", "ds"].includes(k));
      if ( dsOnly && (doc.ds !== WALL_DOOR_STATES.LOCKED) && (data.ds !== WALL_DOOR_STATES.LOCKED) ) {
        return user.hasRole("PLAYER");                  // Players may open and close unlocked doors
      }
      return false;
    }
  }

  /** @module foundry.documents */

  var documents = /*#__PURE__*/Object.freeze({
    __proto__: null,
    BaseActiveEffect: BaseActiveEffect,
    BaseActor: BaseActor,
    BaseActorDelta: BaseActorDelta,
    BaseAdventure: BaseAdventure,
    BaseAmbientLight: BaseAmbientLight,
    BaseAmbientSound: BaseAmbientSound,
    BaseCard: BaseCard,
    BaseCards: BaseCards,
    BaseChatMessage: BaseChatMessage,
    BaseCombat: BaseCombat,
    BaseCombatant: BaseCombatant,
    BaseDrawing: BaseDrawing,
    BaseFogExploration: BaseFogExploration,
    BaseFolder: BaseFolder,
    BaseItem: BaseItem,
    BaseJournalEntry: BaseJournalEntry,
    BaseJournalEntryPage: BaseJournalEntryPage,
    BaseMacro: BaseMacro,
    BaseMeasuredTemplate: BaseMeasuredTemplate,
    BaseNote: BaseNote,
    BasePlaylist: BasePlaylist,
    BasePlaylistSound: BasePlaylistSound,
    BaseRegion: BaseRegion,
    BaseRegionBehavior: BaseRegionBehavior,
    BaseRollTable: BaseRollTable,
    BaseScene: BaseScene,
    BaseSetting: BaseSetting,
    BaseTableResult: BaseTableResult,
    BaseTile: BaseTile,
    BaseToken: BaseToken,
    BaseUser: BaseUser,
    BaseWall: BaseWall
  });

  /**
   * A custom SchemaField for defining package compatibility versions.
   * @property {string} minimum     The Package will not function before this version
   * @property {string} verified    Verified compatible up to this version
   * @property {string} maximum     The Package will not function after this version
   */
  class PackageCompatibility extends SchemaField {
    constructor(options) {
      super({
        minimum: new StringField({required: false, blank: false, initial: undefined}),
        verified: new StringField({required: false, blank: false, initial: undefined}),
        maximum: new StringField({required: false, blank: false, initial: undefined})
      }, options);
    }
  }

  /* -------------------------------------------- */

  /**
   * A custom SchemaField for defining package relationships.
   * @property {RelatedPackage[]} systems     Systems that this Package supports
   * @property {RelatedPackage[]} requires    Packages that are required for base functionality
   * @property {RelatedPackage[]} recommends  Packages that are recommended for optimal functionality
   */
  class PackageRelationships extends SchemaField {
    /** @inheritdoc */
    constructor(options) {
      super({
        systems: new PackageRelationshipField(new RelatedPackage({packageType: "system"})),
        requires: new PackageRelationshipField(new RelatedPackage()),
        recommends: new PackageRelationshipField(new RelatedPackage()),
        conflicts: new PackageRelationshipField(new RelatedPackage()),
        flags: new ObjectField()
      }, options);
    }
  }

  /* -------------------------------------------- */

  /**
   * A SetField with custom casting behavior.
   */
  class PackageRelationshipField extends SetField {
    /** @override */
    _cast(value) {
      return value instanceof Array ? value : [value];
    }
  }

  /* -------------------------------------------- */

  /**
   * A custom SchemaField for defining a related Package.
   * It may be required to be a specific type of package, by passing the packageType option to the constructor.
   */
  class RelatedPackage extends SchemaField {
    constructor({packageType, ...options}={}) {
      let typeOptions = {choices: PACKAGE_TYPES, initial:"module"};
      if ( packageType ) typeOptions = {choices: [packageType], initial: packageType};
      super({
        id: new StringField({required: true, blank: false}),
        type: new StringField(typeOptions),
        manifest: new StringField({required: false, blank: false, initial: undefined}),
        compatibility: new PackageCompatibility(),
        reason: new StringField({required: false, blank: false, initial: undefined})
      }, options);
    }
  }

  /* -------------------------------------------- */

  /**
   * A custom SchemaField for defining the folder structure of the included compendium packs.
   */
  class PackageCompendiumFolder extends SchemaField {
    constructor({depth=1, ...options}={}) {
      const schema = {
        name: new StringField({required: true, blank: false}),
        sorting: new StringField({required: false, blank: false, initial: undefined,
          choices: BaseFolder.SORTING_MODES}),
        color: new ColorField(),
        packs: new SetField(new StringField({required: true, blank: false}))
      };
      if ( depth < 4 ) schema.folders = new SetField(new PackageCompendiumFolder(
        {depth: depth+1, options}));
      super(schema, options);
    }
  }

  /* -------------------------------------------- */

  /**
   * A special ObjectField which captures a mapping of USER_ROLES to DOCUMENT_OWNERSHIP_LEVELS.
   */
  class CompendiumOwnershipField extends ObjectField {

    /** @inheritdoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        initial: {PLAYER: "OBSERVER", ASSISTANT: "OWNER"},
        validationError: "is not a mapping of USER_ROLES to DOCUMENT_OWNERSHIP_LEVELS"
      });
    }

    /** @override */
    _validateType(value, options) {
      for ( let [k, v] of Object.entries(value) ) {
        if ( !(k in USER_ROLES) ) throw new Error(`Compendium ownership key "${k}" is not a valid choice in USER_ROLES`);
        if ( !(v in DOCUMENT_OWNERSHIP_LEVELS) ) throw new Error(`Compendium ownership value "${v}" is not a valid 
      choice in DOCUMENT_OWNERSHIP_LEVELS`);
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * A special SetField which provides additional validation and initialization behavior specific to compendium packs.
   */
  class PackageCompendiumPacks extends SetField {

    /** @override */
    _cleanType(value, options) {
      return value.map(v => {
        v = this.element.clean(v, options);
        if ( v.path ) v.path = v.path.replace(/\.db$/, ""); // Strip old NEDB extensions
        else v.path = `packs/${v.name}`; // Auto-populate a default pack path
        return v;
      })
    }

    /* ---------------------------------------- */

    /** @override */
    initialize(value, model, options={}) {
      const packs = new Set();
      const packageName = model._source.id;
      for ( let v of value ) {
        try {
          const pack = this.element.initialize(v, model, options);
          pack.packageType = model.constructor.type;
          pack.packageName = packageName;
          pack.id = `${model.constructor.type === "world" ? "world" : packageName}.${pack.name}`;
          packs.add(pack);
        } catch(err) {
          logger.warn(err.message);
        }
      }
      return packs;
    }

    /* ---------------------------------------- */

    /**
     * Extend the logic for validating the complete set of packs to ensure uniqueness.
     * @inheritDoc
     */
    _validateElements(value, options) {
      const packNames = new Set();
      const duplicateNames = new Set();
      const packPaths = new Set();
      const duplicatePaths = new Set();
      for ( const pack of value ) {
        if ( packNames.has(pack.name) ) duplicateNames.add(pack.name);
        packNames.add(pack.name);
        if ( pack.path ) {
          if ( packPaths.has(pack.path) ) duplicatePaths.add(pack.path);
          packPaths.add(pack.path);
        }
      }
      return super._validateElements(value, {...options, duplicateNames, duplicatePaths});
    }

    /* ---------------------------------------- */

    /**
     * Validate each individual compendium pack, ensuring its name and path are unique.
     * @inheritDoc
     */
    _validateElement(value, {duplicateNames, duplicatePaths, ...options}={}) {
      if ( duplicateNames.has(value.name) ) {
        return new DataModelValidationFailure({
          invalidValue: value.name,
          message: `Duplicate Compendium name "${value.name}" already declared by some other pack`,
          unresolved: true
        });
      }
      if ( duplicatePaths.has(value.path) ) {
        return new DataModelValidationFailure({
          invalidValue: value.path,
          message: `Duplicate Compendium path "${value.path}" already declared by some other pack`,
          unresolved: true
        });
      }
      return this.element.validate(value, options);
    }
  }

  /* -------------------------------------------- */

  /**
   * The data schema used to define a Package manifest.
   * Specific types of packages extend this schema with additional fields.
   */
  class BasePackage extends DataModel {
    /**
     * @param {PackageManifestData} data  Source data for the package
     * @param {object} [options={}]       Options which affect DataModel construction
     */
    constructor(data, options={}) {
      const {availability, locked, exclusive, owned, tags, hasStorage} = data;
      super(data, options);

      /**
       * An availability code in PACKAGE_AVAILABILITY_CODES which defines whether this package can be used.
       * @type {number}
       */
      this.availability = availability ?? this.constructor.testAvailability(this);

      /**
       * A flag which tracks whether this package is currently locked.
       * @type {boolean}
       */
      this.locked = locked ?? false;

      /**
       * A flag which tracks whether this package is a free Exclusive pack
       * @type {boolean}
       */
      this.exclusive = exclusive ?? false;

      /**
       * A flag which tracks whether this package is owned, if it is protected.
       * @type {boolean|null}
       */
      this.owned = owned ?? false;

      /**
       * A set of Tags that indicate what kind of Package this is, provided by the Website
       * @type {string[]}
       */
      this.tags = tags ?? [];

      /**
       * A flag which tracks if this package has files stored in the persistent storage folder
       * @type {boolean}
       */
      this.hasStorage = hasStorage ?? false;
    }

    /**
     * Define the package type in CONST.PACKAGE_TYPES that this class represents.
     * Each BasePackage subclass must define this attribute.
     * @virtual
     * @type {string}
     */
    static type = "package";

    /**
     * The type of this package instance. A value in CONST.PACKAGE_TYPES.
     * @type {string}
     */
    get type() {
      return this.constructor.type;
    }

    /**
     * The canonical identifier for this package
     * @return {string}
     * @deprecated
     */
    get name() {
      logCompatibilityWarning("You are accessing BasePackage#name which is now deprecated in favor of id.",
        {since: 10, until: 13});
      return this.id;
    }

    /**
     * A flag which defines whether this package is unavailable to be used.
     * @type {boolean}
     */
    get unavailable() {
      return this.availability > PACKAGE_AVAILABILITY_CODES.UNVERIFIED_GENERATION;
    }

    /**
     * Is this Package incompatible with the currently installed core Foundry VTT software version?
     * @type {boolean}
     */
    get incompatibleWithCoreVersion() {
      return this.constructor.isIncompatibleWithCoreVersion(this.availability);
    }

    /**
     * Test if a given availability is incompatible with the core version.
     * @param {number} availability  The availability value to test.
     * @returns {boolean}
     */
    static isIncompatibleWithCoreVersion(availability) {
      const codes = CONST.PACKAGE_AVAILABILITY_CODES;
      return (availability >= codes.REQUIRES_CORE_DOWNGRADE) && (availability <= codes.REQUIRES_CORE_UPGRADE_UNSTABLE);
    }

    /**
     * The named collection to which this package type belongs
     * @type {string}
     */
    static get collection() {
      return `${this.type}s`;
    }

    /** @inheritDoc */
    static defineSchema() {
      const optionalString = {required: false, blank: false, initial: undefined};
      return {

        // Package metadata
        id: new StringField({required: true, blank: false, validate: this.validateId}),
        title: new StringField({required: true, blank: false}),
        description: new StringField({required: true}),
        authors: new SetField(new SchemaField({
          name: new StringField({required: true, blank: false}),
          email: new StringField(optionalString),
          url: new StringField(optionalString),
          discord: new StringField(optionalString),
          flags: new ObjectField(),
        })),
        url: new StringField(optionalString),
        license: new StringField(optionalString),
        readme: new StringField(optionalString),
        bugs: new StringField(optionalString),
        changelog: new StringField(optionalString),
        flags: new ObjectField(),
        media: new SetField(new SchemaField({
          type: new StringField(optionalString),
          url: new StringField(optionalString),
          caption: new StringField(optionalString),
          loop: new BooleanField({required: false, blank: false, initial: false}),
          thumbnail: new StringField(optionalString),
          flags: new ObjectField(),
        })),

        // Package versioning
        version: new StringField({required: true, blank: false, initial: "0"}),
        compatibility: new PackageCompatibility(),

        // Included content
        scripts: new SetField(new StringField({required: true, blank: false})),
        esmodules: new SetField(new StringField({required: true, blank: false})),
        styles: new SetField(new StringField({required: true, blank: false})),
        languages: new SetField(new SchemaField({
          lang: new StringField({required: true, blank: false, validate: Intl.getCanonicalLocales,
            validationError: "must be supported by the Intl.getCanonicalLocales function"
          }),
          name: new StringField({required: false}),
          path: new StringField({required: true, blank: false}),
          system: new StringField(optionalString),
          module: new StringField(optionalString),
          flags: new ObjectField(),
        })),
        packs: new PackageCompendiumPacks(new SchemaField({
          name: new StringField({required: true, blank: false, validate: this.validateId}),
          label: new StringField({required: true, blank: false}),
          banner: new StringField({...optionalString, nullable: true}),
          path: new StringField({required: false}),
          type: new StringField({required: true, blank: false, choices: COMPENDIUM_DOCUMENT_TYPES,
            validationError: "must be a value in CONST.COMPENDIUM_DOCUMENT_TYPES"}),
          system: new StringField(optionalString),
          ownership: new CompendiumOwnershipField(),
          flags: new ObjectField(),
        }, {validate: BasePackage.#validatePack})),
        packFolders: new SetField(new PackageCompendiumFolder()),

        // Package relationships
        relationships: new PackageRelationships(),
        socket: new BooleanField(),

        // Package downloading
        manifest: new StringField(),
        download: new StringField({required: false, blank: false, initial: undefined}),
        protected: new BooleanField(),
        exclusive: new BooleanField(),
        persistentStorage: new BooleanField(),
      }
    }

    /* -------------------------------------------- */

    /**
     * Check the given compatibility data against the current installation state and determine its availability.
     * @param {Partial<PackageManifestData>} data  The compatibility data to test.
     * @param {object} [options]
     * @param {ReleaseData} [options.release]      A specific software release for which to test availability.
     *                                             Tests against the current release by default.
     * @returns {number}
     */
    static testAvailability({ compatibility }, { release }={}) {
      release ??= globalThis.release ?? game.release;
      const codes = CONST.PACKAGE_AVAILABILITY_CODES;
      const {minimum, maximum, verified} = compatibility;
      const isGeneration = version => Number.isInteger(Number(version));

      // Require a certain minimum core version.
      if ( minimum && isNewerVersion(minimum, release.version) ) {
        const generation = Number(minimum.split(".").shift());
        const isStable = generation <= release.maxStableGeneration;
        const exists = generation <= release.maxGeneration;
        if ( isStable ) return codes.REQUIRES_CORE_UPGRADE_STABLE;
        return exists ? codes.REQUIRES_CORE_UPGRADE_UNSTABLE : codes.UNKNOWN;
      }

      // Require a certain maximum core version.
      if ( maximum ) {
        const compatible = isGeneration(maximum)
          ? release.generation <= Number(maximum)
          : !isNewerVersion(release.version, maximum);
        if ( !compatible ) return codes.REQUIRES_CORE_DOWNGRADE;
      }

      // Require a certain compatible core version.
      if ( verified ) {
        const compatible = isGeneration(verified)
          ? Number(verified) >= release.generation
          : !isNewerVersion(release.version, verified);
        const sameGeneration = release.generation === Number(verified.split(".").shift());
        if ( compatible ) return codes.VERIFIED;
        return sameGeneration ? codes.UNVERIFIED_BUILD : codes.UNVERIFIED_GENERATION;
      }

      // FIXME: Why do we not check if all of this package's dependencies are satisfied?
      // Proposal: Check all relationships.requires and set MISSING_DEPENDENCY if any dependencies are not VERIFIED,
      // UNVERIFIED_BUILD, or UNVERIFIED_GENERATION, or if they do not satisfy the given compatibility range for the
      // relationship.

      // No compatible version is specified.
      return codes.UNKNOWN;
    }

    /* -------------------------------------------- */

    /**
     * Test that the dependencies of a package are satisfied as compatible.
     * This method assumes that all packages in modulesCollection have already had their own availability tested.
     * @param {Collection<string,Module>} modulesCollection   A collection which defines the set of available modules
     * @returns {Promise<boolean>}                            Are all required dependencies satisfied?
     * @internal
     */
    async _testRequiredDependencies(modulesCollection) {
      const requirements = this.relationships.requires;
      for ( const {id, type, manifest, compatibility} of requirements ) {
        if ( type !== "module" ) continue; // Only test modules
        let pkg;

        // If the requirement specifies an explicit remote manifest URL, we need to load it
        if ( manifest ) {
          try {
            pkg = await this.constructor.fromRemoteManifest(manifest, {strict: true});
          } catch(err) {
            return false;
          }
        }

        // Otherwise the dependency must belong to the known modulesCollection
        else pkg = modulesCollection.get(id);
        if ( !pkg ) return false;

        // Ensure that the package matches the required compatibility range
        if ( !this.constructor.testDependencyCompatibility(compatibility, pkg) ) return false;

        // Test compatibility of the dependency
        if ( pkg.unavailable ) return false;
      }
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Test compatibility of a package's supported systems.
     * @param {Collection<string, System>} systemCollection  A collection which defines the set of available systems.
     * @returns {Promise<boolean>}                           True if all supported systems which are currently installed
     *                                                       are compatible or if the package has no supported systems.
     *                                                       Returns false otherwise, or if no supported systems are
     *                                                       installed.
     * @internal
     */
    async _testSupportedSystems(systemCollection) {
      const systems = this.relationships.systems;
      if ( !systems?.size ) return true;
      let supportedSystem = false;
      for ( const { id, compatibility } of systems ) {
        const pkg = systemCollection.get(id);
        if ( !pkg ) continue;
        if ( !this.constructor.testDependencyCompatibility(compatibility, pkg) || pkg.unavailable ) return false;
        supportedSystem = true;
      }
      return supportedSystem;
    }

    /* -------------------------------------------- */

    /**
     * Determine if a dependency is within the given compatibility range.
     * @param {PackageCompatibility} compatibility      The compatibility range declared for the dependency, if any
     * @param {BasePackage} dependency                  The known dependency package
     * @returns {boolean}                               Is the dependency compatible with the required range?
     */
    static testDependencyCompatibility(compatibility, dependency) {
      if ( !compatibility ) return true;
      const {minimum, maximum} = compatibility;
      if ( minimum && isNewerVersion(minimum, dependency.version) ) return false;
      if ( maximum && isNewerVersion(dependency.version, maximum) ) return false;
      return true;
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    static cleanData(source={}, { installed, ...options }={}) {

      // Auto-assign language name
      for ( let l of source.languages || [] ) {
        l.name = l.name ?? l.lang;
      }

      // Identify whether this package depends on a single game system
      let systemId = undefined;
      if ( this.type === "system" ) systemId = source.id;
      else if ( this.type === "world" ) systemId = source.system;
      else if ( source.relationships?.systems?.length === 1 ) systemId = source.relationships.systems[0].id;

      // Auto-configure some package data
      for ( const pack of source.packs || [] ) {
        if ( !pack.system && systemId ) pack.system = systemId; // System dependency
        if ( typeof pack.ownership === "string" ) pack.ownership = {PLAYER: pack.ownership};
      }

      /**
       * Clean unsupported non-module dependencies in requires or recommends.
       * @deprecated since v11
       */
      ["requires", "recommends"].forEach(rel => {
        const pkgs = source.relationships?.[rel];
        if ( !Array.isArray(pkgs) ) return;
        const clean = [];
        for ( const pkg of pkgs ) {
          if ( !pkg.type || (pkg.type === "module") ) clean.push(pkg);
        }
        const diff = pkgs.length - clean.length;
        if ( diff ) {
          source.relationships[rel] = clean;
          this._logWarning(
            source.id,
            `The ${this.type} "${source.id}" has a ${rel} relationship on a non-module, which is not supported.`,
            { since: 11, until: 13, stack: false, installed });
        }
      });
      return super.cleanData(source, options);
    }

    /* -------------------------------------------- */

    /**
     * Validate that a Package ID is allowed.
     * @param {string} id     The candidate ID
     * @throws                An error if the candidate ID is invalid
     */
    static validateId(id) {
      const allowed = /^[A-Za-z0-9-_]+$/;
      if ( !allowed.test(id) ) {
        throw new Error("Package and compendium pack IDs may only be alphanumeric with hyphens or underscores.");
      }
      const prohibited = /^(con|prn|aux|nul|com[0-9]|lpt[0-9])(\..*)?$/i;
      if ( prohibited.test(id) ) throw new Error(`The ID "${id}" uses an operating system prohibited value.`);
    }

    /* -------------------------------------------- */

    /**
     * Validate a single compendium pack object
     * @param {PackageCompendiumData} packData  Candidate compendium packs data
     * @throws                                  An error if the data is invalid
     */
    static #validatePack(packData) {
      if ( SYSTEM_SPECIFIC_COMPENDIUM_TYPES.includes(packData.type) && !packData.system ) {
        throw new Error(`The Compendium pack "${packData.name}" of the "${packData.type}" type must declare the "system"`
        + " upon which it depends.");
      }
    }

    /* -------------------------------------------- */

    /**
     * A wrapper around the default compatibility warning logger which handles some package-specific interactions.
     * @param {string} packageId            The package ID being logged
     * @param {string} message              The warning or error being logged
     * @param {object} options              Logging options passed to foundry.utils.logCompatibilityWarning
     * @param {object} [options.installed]  Is the package installed?
     * @internal
     */
    static _logWarning(packageId, message, { installed, ...options }={}) {
      logCompatibilityWarning(message, options);
      if ( installed ) globalThis.packages?.warnings?.add(packageId, {type: this.type, level: "warning", message});
    }

    /* -------------------------------------------- */

    /**
     * A set of package manifest keys that are migrated.
     * @type {Set<string>}
     */
    static migratedKeys = new Set([
      /** @deprecated since 10 until 13 */
      "name", "dependencies", "minimumCoreVersion", "compatibleCoreVersion"
    ]);

    /* -------------------------------------------- */

    /** @inheritdoc */
    static migrateData(data, { installed }={}) {
      this._migrateNameToId(data, {since: 10, until: 13, stack: false, installed});
      this._migrateDependenciesNameToId(data, {since: 10, until: 13, stack: false, installed});
      this._migrateToRelationships(data, {since: 10, until: 13, stack: false, installed});
      this._migrateCompatibility(data, {since: 10, until: 13, stack: false, installed});
      this._migrateMediaURL(data, {since: 11, until: 13, stack: false, installed});
      this._migrateOwnership(data, {since: 11, until: 13, stack: false, installed});
      this._migratePackIDs(data, {since: 12, until: 14, stack: false, installed});
      this._migratePackEntityToType(data, {since: 9, stack: false, installed});
      return super.migrateData(data);
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateNameToId(data, logOptions) {
      if ( data.name && !data.id ) {
        data.id = data.name;
        delete data.name;
        if ( this.type !== "world" ) {
          const warning = `The ${this.type} "${data.id}" is using "name" which is deprecated in favor of "id"`;
          this._logWarning(data.id, warning, logOptions);
        }
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateDependenciesNameToId(data, logOptions) {
      if ( data.relationships ) return;
      if ( data.dependencies ) {
        let hasDependencyName = false;
        for ( const dependency of data.dependencies ) {
          if ( dependency.name && !dependency.id ) {
            hasDependencyName = true;
            dependency.id = dependency.name;
            delete dependency.name;
          }
        }
        if ( hasDependencyName ) {
          const msg = `The ${this.type} "${data.id}" contains dependencies using "name" which is deprecated in favor of "id"`;
          this._logWarning(data.id, msg, logOptions);
        }
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateToRelationships(data, logOptions) {
      if ( data.relationships ) return;
      data.relationships = {
        requires: [],
        systems: []
      };

      // Dependencies -> Relationships.Requires
      if ( data.dependencies ) {
        for ( const d of data.dependencies ) {
          const relationship = {
            "id": d.id,
            "type": d.type,
            "manifest": d.manifest,
            "compatibility": {
              "compatible": d.version
            }
          };
          d.type === "system" ? data.relationships.systems.push(relationship) : data.relationships.requires.push(relationship);
        }
        const msg = `The ${this.type} "${data.id}" contains "dependencies" which is deprecated in favor of "relationships.requires"`;
        this._logWarning(data.id, msg, logOptions);
        delete data.dependencies;
      }

      // V9: system -> relationships.systems
      else if ( data.system && (this.type === "module") ) {
        data.system = data.system instanceof Array ? data.system : [data.system];
        const newSystems = data.system.map(id => ({id})).filter(s => !data.relationships.systems.find(x => x.id === s.id));
        data.relationships.systems = data.relationships.systems.concat(newSystems);
        const msg = `${this.type} "${data.id}" contains "system" which is deprecated in favor of "relationships.systems"`;
        this._logWarning(data.id, msg, logOptions);
        delete data.system;
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateCompatibility(data, logOptions) {
      if ( !data.compatibility && (data.minimumCoreVersion || data.compatibleCoreVersion) ) {
        this._logWarning(data.id, `The ${this.type} "${data.id}" is using the old flat core compatibility fields which `
          + `are deprecated in favor of the new "compatibility" object`,
          logOptions);
        data.compatibility = {
          minimum: data.minimumCoreVersion,
          verified: data.compatibleCoreVersion
        };
        delete data.minimumCoreVersion;
        delete data.compatibleCoreVersion;
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateMediaURL(data, logOptions) {
      if ( !data.media ) return;
      let hasMediaLink = false;
      for ( const media of data.media ) {
        if ( "link" in media ) {
          hasMediaLink = true;
          media.url = media.link;
          delete media.link;
        }
      }
      if ( hasMediaLink ) {
        const msg = `${this.type} "${data.id}" declares media.link which is unsupported, media.url should be used`;
        this._logWarning(data.id, msg, logOptions);
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migrateOwnership(data, logOptions) {
      if ( !data.packs ) return;
      let hasPrivatePack = false;
      for ( const pack of data.packs ) {
        if ( pack.private && !("ownership" in pack) ) {
          pack.ownership = {PLAYER: "LIMITED", ASSISTANT: "OWNER"};
          hasPrivatePack = true;
        }
        delete pack.private;
      }
      if ( hasPrivatePack ) {
        const msg = `${this.type} "${data.id}" uses pack.private which has been replaced with pack.ownership`;
        this._logWarning(data.id, msg, logOptions);
      }
      return data;
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migratePackIDs(data, logOptions) {
      if ( !data.packs ) return;
      for ( const pack of data.packs ) {
        const slugified = pack.name.replace(/[^A-Za-z0-9-_]/g, "");
        if ( pack.name !== slugified ) {
          const msg = `The ${this.type} "${data.id}" contains a pack with an invalid name "${pack.name}". `
            + "Pack names containing any character that is non-alphanumeric or an underscore will cease loading in "
            + "version 14 of the software.";
          pack.name = slugified;
          this._logWarning(data.id, msg, logOptions);
        }
      }
    }

    /* -------------------------------------------- */

    /** @internal */
    static _migratePackEntityToType(data, logOptions) {
      if ( !data.packs ) return;
      let hasPackEntity = false;
      for ( const pack of data.packs ) {
        if ( ("entity" in pack) && !("type" in pack) ) {
          pack.type = pack.entity;
          hasPackEntity = true;
        }
        delete pack.entity;
      }
      if ( hasPackEntity ) {
        const msg = `${this.type} "${data.id}" uses pack.entity which has been replaced with pack.type`;
        this._logWarning(data.id, msg, logOptions);
      }
    }

    /* -------------------------------------------- */

    /**
     * Retrieve the latest Package manifest from a provided remote location.
     * @param {string} manifestUrl        A remote manifest URL to load
     * @param {object} options            Additional options which affect package construction
     * @param {boolean} [options.strict=true]   Whether to construct the remote package strictly
     * @return {Promise<ServerPackage>}   A Promise which resolves to a constructed ServerPackage instance
     * @throws                            An error if the retrieved manifest data is invalid
     */
    static async fromRemoteManifest(manifestUrl, {strict=true}={}) {
      throw new Error("Not implemented");
    }
  }

  /**
   * The data schema used to define World manifest files.
   * Extends the basic PackageData schema with some additional world-specific fields.
   * @property {string} system            The game system name which this world relies upon
   * @property {string} coreVersion       The version of the core software for which this world has been migrated
   * @property {string} systemVersion     The version of the game system for which this world has been migrated
   * @property {string} [background]      A web URL or local file path which provides a background banner image
   * @property {string} [nextSession]     An ISO datetime string when the next game session is scheduled to occur
   * @property {boolean} [resetKeys]      Should user access keys be reset as part of the next launch?
   * @property {boolean} [safeMode]       Should the world launch in safe mode?
   * @property {string} [joinTheme]       The theme to use for this world's join page.
   */
  class BaseWorld extends BasePackage {

    /** @inheritDoc */
    static defineSchema() {
      return Object.assign({}, super.defineSchema(), {
        system: new StringField({required: true, blank: false}),
        background: new StringField({required: false, blank: false}),
        joinTheme: new StringField({
          required: false, initial: undefined, nullable: false, blank: false, choices: WORLD_JOIN_THEMES
        }),
        coreVersion: new StringField({required: true, blank: false}),
        systemVersion: new StringField({required: true, blank: false, initial: "0"}),
        lastPlayed: new StringField(),
        playtime: new NumberField({integer: true, min: 0, initial: 0}),
        nextSession: new StringField({blank: false, nullable: true, initial: null}),
        resetKeys: new BooleanField({required: false, initial: undefined}),
        safeMode: new BooleanField({required: false, initial: undefined}),
        version: new StringField({required: true, blank: false, nullable: true, initial: null})
      });
    }

    /** @override */
    static type = "world";

    /**
     * The default icon used for this type of Package.
     * @type {string}
     */
    static icon = "fa-globe-asia";

    /** @inheritDoc */
    static migrateData(data) {
      super.migrateData(data);

      // Legacy compatibility strings
      data.compatibility = data.compatibility || {};
      if ( data.compatibility.maximum === "1.0.0" ) data.compatibility.maximum = undefined;
      if ( data.coreVersion && !data.compatibility.verified ) {
        data.compatibility.minimum = data.compatibility.verified = data.coreVersion;
      }
      return data;
    }

    /* -------------------------------------------- */

    /**
     * Check the given compatibility data against the current installation state and determine its availability.
     * @param {Partial<PackageManifestData>} data  The compatibility data to test.
     * @param {object} [options]
     * @param {ReleaseData} [options.release]      A specific software release for which to test availability.
     *                                             Tests against the current release by default.
     * @param {Collection<string, Module>} [options.modules]  A specific collection of modules to test availability
     *                                                        against. Tests against the currently installed modules by
     *                                                        default.
     * @param {Collection<string, System>} [options.systems]  A specific collection of systems to test availability
     *                                                        against. Tests against the currently installed systems by
     *                                                        default.
     * @param {number} [options.systemAvailabilityThreshold]  Ignore the world's own core software compatibility and
     *                                                        instead defer entirely to the system's core software
     *                                                        compatibility, if the world's availability is less than
     *                                                        this.
     * @returns {number}
     */
    static testAvailability(data, { release, modules, systems, systemAvailabilityThreshold }={}) {
      systems ??= globalThis.packages?.System ?? game.systems;
      modules ??= globalThis.packages?.Module ?? game.modules;
      const { relationships } = data;
      const codes = CONST.PACKAGE_AVAILABILITY_CODES;
      systemAvailabilityThreshold ??= codes.UNKNOWN;

      // If the World itself is incompatible for some reason, report that directly.
      const wa = super.testAvailability(data, { release });
      if ( this.isIncompatibleWithCoreVersion(wa) ) return wa;

      // If the System is missing or incompatible, report that directly.
      const system = data.system instanceof foundry.packages.BaseSystem ? data.system : systems.get(data.system);
      if ( !system ) return codes.MISSING_SYSTEM;
      const sa = system.availability;
      // FIXME: Why do we only check if the system is incompatible with the core version or UNKNOWN?
      // Proposal: If the system is anything but VERIFIED, UNVERIFIED_BUILD, or UNVERIFIED_GENERATION, we should return
      // the system availability.
      if ( system.incompatibleWithCoreVersion || (sa === codes.UNKNOWN) ) return sa;

      // Test the availability of all required modules.
      const checkedModules = new Set();
      // TODO: We do not need to check system requirements here if the above proposal is implemented.
      const requirements = [...relationships.requires.values(), ...system.relationships.requires.values()];
      for ( const r of requirements ) {
        if ( (r.type !== "module") || checkedModules.has(r.id) ) continue;
        const module = modules.get(r.id);
        if ( !module ) return codes.MISSING_DEPENDENCY;
        // FIXME: Why do we only check if the module is incompatible with the core version?
        // Proposal: We should check the actual compatibility information for the relationship to ensure that the module
        // satisfies it.
        if ( module.incompatibleWithCoreVersion ) return codes.REQUIRES_DEPENDENCY_UPDATE;
        checkedModules.add(r.id);
      }

      // Inherit from the System availability in certain cases.
      if ( wa <= systemAvailabilityThreshold ) return sa;
      return wa;
    }
  }

  /**
   * @typedef {Record<string, Record<string, object>>} DocumentTypesConfiguration
   */

  /**
   * A special [ObjectField]{@link ObjectField} available to packages which configures any additional Document subtypes
   * provided by the package.
   */
  class AdditionalTypesField extends ObjectField {

    /** @inheritDoc */
    static get _defaults() {
      return mergeObject(super._defaults, {
        readonly: true,
        validationError: "is not a valid sub-types configuration"
      });
    }

    /* ----------------------------------------- */

    /** @inheritDoc */
    _validateType(value, options={}) {
      super._validateType(value, options);
      for ( const [documentName, subtypes] of Object.entries(value) ) {
        const cls = getDocumentClass(documentName);
        if ( !cls ) throw new Error(`${this.validationError}: '${documentName}' is not a valid Document type`);
        if ( !cls.hasTypeData ) {
          throw new Error(`${this.validationError}: ${documentName} Documents do not support sub-types`);
        }
        if ( getType(subtypes) !== "Object" ) throw new Error(`Malformed ${documentName} documentTypes declaration`);
        for ( const [type, config] of Object.entries(subtypes) ) this.#validateSubtype(cls, type, config);
      }
    }

    /* ----------------------------------------- */

    /**
     * Validate a single defined document subtype.
     * @param {typeof Document} documentClass       The document for which the subtype is being registered
     * @param {string} type                         The requested subtype name
     * @param {object} config                       The provided subtype configuration
     * @throws {Error}                              An error if the subtype is invalid or malformed
     */
    #validateSubtype(documentClass, type, config) {
      const dn = documentClass.documentName;
      if ( documentClass.metadata.coreTypes.includes(type) ) {
        throw new Error(`"${type}" is a reserved core type for the ${dn} document`);
      }
      if ( getType(config) !== "Object" ) {
        throw new Error(`Malformed "${type}" subtype declared for ${dn} documentTypes`);
      }
    }
  }

  /**
   * @typedef {import("./sub-types.mjs").DocumentTypesConfiguration} DocumentTypesConfiguration
   */

  /**
   * The data schema used to define System manifest files.
   * Extends the basic PackageData schema with some additional system-specific fields.
   * @property {DocumentTypesConfiguration} [documentTypes]  Additional document subtypes provided by this system.
   * @property {string} [background]        A web URL or local file path which provides a default background banner for
   *                                        worlds which are created using this system
   * @property {string} [initiative]        A default initiative formula used for this system
   * @property {number} [grid]              The default grid settings to use for Scenes in this system
   * @property {number} [grid.type]         A default grid type to use for Scenes in this system
   * @property {number} [grid.distance]     A default distance measurement to use for Scenes in this system
   * @property {string} [grid.units]        A default unit of measure to use for distance measurement in this system
   * @property {number} [grid.diagonals]    The default rule used by this system for diagonal measurement on square grids
   * @property {string} [primaryTokenAttribute] An Actor data attribute path to use for Token primary resource bars
   * @property {string} [secondaryTokenAttribute] An Actor data attribute path to use for Token secondary resource bars
   */
  class BaseSystem extends BasePackage {

    /** @inheritDoc */
    static defineSchema() {
      return Object.assign({}, super.defineSchema(), {
        documentTypes: new AdditionalTypesField(),
        background: new StringField({required: false, blank: false}),
        initiative: new StringField(),
        grid: new SchemaField({
          type: new NumberField({required: true, choices: Object.values(CONST.GRID_TYPES),
            initial: CONST.GRID_TYPES.SQUARE, validationError: "must be a value in CONST.GRID_TYPES"}),
          distance: new NumberField({required: true, nullable: false, positive: true, initial: 1}),
          units: new StringField({required: true}),
          diagonals: new NumberField({required: true, choices: Object.values(CONST.GRID_DIAGONALS),
            initial: CONST.GRID_DIAGONALS.EQUIDISTANT, validationError: "must be a value in CONST.GRID_DIAGONALS"}),
        }),
        primaryTokenAttribute: new StringField(),
        secondaryTokenAttribute: new StringField()
      });
    }

    /** @inheritdoc */
    static type = "system";

    /**
     * The default icon used for this type of Package.
     * @type {string}
     */
    static icon = "fa-dice";

    /**
     * Does the system template request strict type checking of data compared to template.json inferred types.
     * @type {boolean}
     */
    strictDataCleaning = false;

    /* -------------------------------------------- */
    /*  Deprecations and Compatibility              */
    /* -------------------------------------------- */

    /**
     * Static initializer block for deprecated properties.
     */
    static {
      /**
       * Shim grid distance and units.
       * @deprecated since v12
       */
      Object.defineProperties(this.prototype, Object.fromEntries(
        Object.entries({
          gridDistance: "grid.distance",
          gridUnits: "grid.units"
        }).map(([o, n]) => [o, {
          get() {
            const msg = `You are accessing BasePackage#${o} which has been migrated to BasePackage#${n}.`;
            foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
            return foundry.utils.getProperty(this, n);
          },
          set(v) {
            const msg = `You are accessing BasePackage#${o} which has been migrated to BasePackage#${n}.`;
            foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
            return foundry.utils.setProperty(this, n, v);
          },
          configurable: true
        }])
      ));
    }

    /* -------------------------------------------- */

    /** @override */
    static migratedKeys = (function() {
      return BasePackage.migratedKeys.union(new Set([
        /** @deprecated since 12 until 14 */
        "gridDistance", "gridUnits"
      ]));
    })();

    /* ---------------------------------------- */

    /** @inheritdoc */
    static migrateData(data, options) {
      /**
       * Migrate grid distance and units.
       * @deprecated since v12
       */
      for ( const [oldKey, [newKey, apply]] of Object.entries({
        gridDistance: ["grid.distance", d => Math.max(d.gridDistance || 0, 1)],
        gridUnits: ["grid.units", d => d.gridUnits || ""]
      })) {
        if ( (oldKey in data) && !foundry.utils.hasProperty(data, newKey) ) {
          foundry.utils.setProperty(data, newKey, apply(data));
          delete data[oldKey];
          const warning = `The ${this.type} "${data.id}" is using "${oldKey}" which is deprecated in favor of "${newKey}".`;
          this._logWarning(data.id, warning, {since: 12, until: 14, stack: false, installed: options.installed});
        }
      }
      return super.migrateData(data, options);
    }

    /* ---------------------------------------- */

    /** @inheritdoc */
    static shimData(data, options) {
      /**
       * Shim grid distance and units.
       * @deprecated since v12
       */
      for ( const [oldKey, newKey] of Object.entries({
        gridDistance: "grid.distance",
        gridUnits: "grid.units"
      })) {
        if ( !data.hasOwnProperty(oldKey) && foundry.utils.hasProperty(data, newKey) ) {
          Object.defineProperty(data, oldKey, {
            get: () => {
              const msg = `You are accessing BasePackage#${oldKey} which has been migrated to BasePackage#${newKey}.`;
              foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
              return foundry.utils.getProperty(data, newKey);
            },
            set: value => foundry.utils.setProperty(data, newKey, value),
            configurable: true
          });
        }
      }
      return super.shimData(data, options);
    }
  }

  /**
   * The data schema used to define Module manifest files.
   * Extends the basic PackageData schema with some additional module-specific fields.
   * @property {boolean} [coreTranslation]         Does this module provide a translation for the core software?
   * @property {boolean} [library]                 A library module provides no user-facing functionality and is solely
   *                                               for use by other modules. Loaded before any system or module scripts.
   * @property {Record<string, string[]>} [documentTypes]  Additional document subtypes provided by this module.
   */
  class BaseModule extends BasePackage {

    /** @inheritDoc */
    static defineSchema() {
      const parentSchema = super.defineSchema();
      return Object.assign({}, parentSchema, {
        coreTranslation: new BooleanField(),
        library: new BooleanField(),
        documentTypes: new AdditionalTypesField()
      });
    }

    /** @override */
    static type = "module";

    /**
     * The default icon used for this type of Package.
     * @type {string}
     */
    static icon = "fa-plug";
  }

  /** @module packages */


  /* ---------------------------------------- */
  /*  Type Definitions                        */
  /* ---------------------------------------- */

  /**
   * @typedef {Object} PackageAuthorData
   * @property {string} name        The author name
   * @property {string} [email]     The author email address
   * @property {string} [url]       A website url for the author
   * @property {string} [discord]   A Discord username for the author
   */

  /**
   * @typedef {Object} PackageCompendiumData
   * @property {string} name        The canonical compendium name. This should contain no spaces or special characters
   * @property {string} label       The human-readable compendium name
   * @property {string} path        The local relative path to the compendium source directory. The filename should match
   *                                the name attribute
   * @property {string} type        The specific document type that is contained within this compendium pack
   * @property {string} [system]    Denote that this compendium pack requires a specific game system to function properly
   */

  /**
   * @typedef {Object} PackageLanguageData
   * @property {string} lang        A string language code which is validated by Intl.getCanonicalLocales
   * @property {string} name        The human-readable language name
   * @property {string} path        The relative path to included JSON translation strings
   * @property {string} [system]    Only apply this set of translations when a specific system is being used
   * @property {string} [module]    Only apply this set of translations when a specific module is active
   */

  /**
   * @typedef {Object} RelatedPackage
   * @property {string} id                              The id of the related package
   * @property {string} type                            The type of the related package
   * @property {string} [manifest]                      An explicit manifest URL, otherwise learned from the Foundry web server
   * @property {PackageCompatibility} [compatibility]   The compatibility data with this related Package
   * @property {string} [reason]                        The reason for this relationship
   */

  /**
   * @typedef {Object} PackageManifestData
   * The data structure of a package manifest. This data structure is extended by BasePackage subclasses to add additional
   * type-specific fields.
   * [[include:full-manifest.md]]
   *
   * @property {string} id              The machine-readable unique package id, should be lower-case with no spaces or special characters
   * @property {string} title           The human-readable package title, containing spaces and special characters
   * @property {string} [description]   An optional package description, may contain HTML
   * @property {PackageAuthorData[]} [authors]  An array of author objects who are co-authors of this package. Preferred to the singular author field.
   * @property {string} [url]           A web url where more details about the package may be found
   * @property {string} [license]       A web url or relative file path where license details may be found
   * @property {string} [readme]        A web url or relative file path where readme instructions may be found
   * @property {string} [bugs]          A web url where bug reports may be submitted and tracked
   * @property {string} [changelog]     A web url where notes detailing package updates are available
   * @property {string} version         The current package version
   * @property {PackageCompatibility} [compatibility]  The compatibility of this version with the core Foundry software
   * @property {string[]} [scripts]     An array of urls or relative file paths for JavaScript files which should be included
   * @property {string[]} [esmodules]   An array of urls or relative file paths for ESModule files which should be included
   * @property {string[]} [styles]      An array of urls or relative file paths for CSS stylesheet files which should be included
   * @property {PackageLanguageData[]} [languages]  An array of language data objects which are included by this package
   * @property {PackageCompendiumData[]} [packs] An array of compendium packs which are included by this package
   * @property {PackageRelationships} [relationships] An organized object of relationships to other Packages
   * @property {boolean} [socket]       Whether to require a package-specific socket namespace for this package
   * @property {string} [manifest]      A publicly accessible web URL which provides the latest available package manifest file. Required in order to support module updates.
   * @property {string} [download]      A publicly accessible web URL where the source files for this package may be downloaded. Required in order to support module installation.
   * @property {boolean} [protected=false] Whether this package uses the protected content access system.
   */

  var packages = /*#__PURE__*/Object.freeze({
    __proto__: null,
    BaseModule: BaseModule,
    BasePackage: BasePackage,
    BaseSystem: BaseSystem,
    BaseWorld: BaseWorld,
    PackageCompatibility: PackageCompatibility,
    RelatedPackage: RelatedPackage
  });

  /** @namespace config */

  /**
   * A data model definition which describes the application configuration options.
   * These options are persisted in the user data Config folder in the options.json file.
   * The server-side software extends this class and provides additional validations and
   * @extends {DataModel}
   * @memberof config
   *
   * @property {string|null} adminPassword        The server administrator password (obscured)
   * @property {string|null} awsConfig            The relative path (to Config) of an AWS configuration file
   * @property {boolean} compressStatic           Whether to compress static files? True by default
   * @property {string} dataPath                  The absolute path of the user data directory (obscured)
   * @property {boolean} fullscreen               Whether the application should automatically start in fullscreen mode?
   * @property {string|null} hostname             A custom hostname applied to internet invitation addresses and URLs
   * @property {string} language                  The default language for the application
   * @property {string|null} localHostname        A custom hostname applied to local invitation addresses
   * @property {string|null} passwordSalt         A custom salt used for hashing user passwords (obscured)
   * @property {number} port                      The port on which the server is listening
   * @property {number} [protocol]                The Internet Protocol version to use, either 4 or 6.
   * @property {number} proxyPort                 An external-facing proxied port used for invitation addresses and URLs
   * @property {boolean} proxySSL                 Is the application running in SSL mode at a reverse-proxy level?
   * @property {string|null} routePrefix          A URL path part which prefixes normal application routing
   * @property {string|null} sslCert              The relative path (to Config) of a used SSL certificate
   * @property {string|null} sslKey               The relative path (to Config) of a used SSL key
   * @property {string} updateChannel             The current application update channel
   * @property {boolean} upnp                     Is UPNP activated?
   * @property {number} upnpLeaseDuration         The duration in seconds of a UPNP lease, if UPNP is active
   * @property {string} world                     A default world name which starts automatically on launch
   */
  class ApplicationConfiguration extends DataModel {
    static defineSchema() {
      return {
        adminPassword: new StringField({required: true, blank: false, nullable: true, initial: null,
          label: "SETUP.AdminPasswordLabel", hint: "SETUP.AdminPasswordHint"}),
        awsConfig: new StringField({label: "SETUP.AWSLabel", hint: "SETUP.AWSHint", blank: false, nullable: true,
          initial: null}),
        compressStatic: new BooleanField({initial: true, label: "SETUP.CompressStaticLabel",
          hint: "SETUP.CompressStaticHint"}),
        compressSocket: new BooleanField({initial: true, label: "SETUP.CompressSocketLabel",
          hint: "SETUP.CompressSocketHint"}),
        cssTheme: new StringField({blank: false, choices: CSS_THEMES, initial: "foundry",
          label: "SETUP.CSSTheme", hint: "SETUP.CSSThemeHint"}),
        dataPath: new StringField({label: "SETUP.DataPathLabel", hint: "SETUP.DataPathHint"}),
        deleteNEDB: new BooleanField({label: "SETUP.DeleteNEDBLabel", hint: "SETUP.DeleteNEDBHint"}),
        fullscreen: new BooleanField({initial: false}),
        hostname: new StringField({required: true, blank: false, nullable: true, initial: null}),
        hotReload: new BooleanField({initial: false, label: "SETUP.HotReloadLabel", hint: "SETUP.HotReloadHint"}),
        language: new StringField({required: true, blank: false, initial: "en.core",
          label: "SETUP.DefaultLanguageLabel", hint: "SETUP.DefaultLanguageHint"}),
        localHostname: new StringField({required: true, blank: false, nullable: true, initial: null}),
        passwordSalt: new StringField({required: true, blank: false, nullable: true, initial: null}),
        port: new NumberField({required: true, nullable: false, integer: true, initial: 30000,
          validate: this._validatePort, label: "SETUP.PortLabel", hint: "SETUP.PortHint"}),
        protocol: new NumberField({integer: true, choices: [4, 6], nullable: true}),
        proxyPort: new NumberField({required: true, nullable: true, integer: true, initial: null}),
        proxySSL: new BooleanField({initial: false}),
        routePrefix: new StringField({required: true, blank: false, nullable: true, initial: null}),
        sslCert: new StringField({label: "SETUP.SSLCertLabel", hint: "SETUP.SSLCertHint", blank: false,
          nullable: true, initial: null}),
        sslKey: new StringField({label: "SETUP.SSLKeyLabel", blank: false, nullable: true, initial: null}),
        telemetry: new BooleanField({required: false, initial: undefined, label: "SETUP.Telemetry",
          hint: "SETUP.TelemetryHint"}),
        updateChannel: new StringField({required: true, choices: SOFTWARE_UPDATE_CHANNELS, initial: "stable"}),
        upnp: new BooleanField({initial: true}),
        upnpLeaseDuration: new NumberField(),
        world: new StringField({required: true, blank: false, nullable: true, initial: null,
          label: "SETUP.WorldLabel", hint: "SETUP.WorldHint"}),
        noBackups: new BooleanField({required: false})
      }
    }

    /* ----------------------------------------- */

    /** @override */
    static migrateData(data) {

      // Backwards compatibility for -v9 update channels
      data.updateChannel = {
        "alpha": "prototype",
        "beta": "testing",
        "release": "stable"
      }[data.updateChannel] || data.updateChannel;

      // Backwards compatibility for awsConfig of true
      if ( data.awsConfig === true ) data.awsConfig = "";
      return data;
    }

    /* ----------------------------------------- */

    /**
     * Validate a port assignment.
     * @param {number} port     The requested port
     * @throws                  An error if the requested port is invalid
     * @private
     */
    static _validatePort(port) {
      if ( !Number.isNumeric(port) || ((port < 1024) && ![80, 443].includes(port)) || (port > 65535) ) {
        throw new Error(`The application port must be an integer, either 80, 443, or between 1024 and 65535`);
      }
    }
  }

  /* ----------------------------------------- */

  /**
   * A data object which represents the details of this Release of Foundry VTT
   * @extends {DataModel}
   * @memberof config
   *
   * @property {number} generation        The major generation of the Release
   * @property {number} [maxGeneration]   The maximum available generation of the software.
   * @property {number} [maxStableGeneration]  The maximum available stable generation of the software.
   * @property {string} channel           The channel the Release belongs to, such as "stable"
   * @property {string} suffix            An optional appended string display for the Release
   * @property {number} build             The internal build number for the Release
   * @property {number} time              When the Release was released
   * @property {number} [node_version]    The minimum required Node.js major version
   * @property {string} [notes]           Release notes for the update version
   * @property {string} [download]        A temporary download URL where this version may be obtained
   */
  class ReleaseData extends DataModel {
    /** @override */
    static defineSchema() {
      return {
        generation: new NumberField({required: true, nullable: false, integer: true, min: 1}),
        maxGeneration: new NumberField({
          required: false, nullable: false, integer: true, min: 1, initial: () => this.generation
        }),
        maxStableGeneration: new NumberField({
          required: false, nullable: false, integer: true, min: 1, initial: () => this.generation
        }),
        channel: new StringField({choices: SOFTWARE_UPDATE_CHANNELS, blank: false}),
        suffix: new StringField(),
        build: new NumberField({required: true, nullable: false, integer: true}),
        time: new NumberField({nullable: false, initial: Date.now}),
        node_version: new NumberField({required: true, nullable: false, integer: true, min: 10}),
        notes: new StringField(),
        download: new StringField()
      }
    }

    /* ----------------------------------------- */

    /**
     * A formatted string for shortened display, such as "Version 9"
     * @return {string}
     */
    get shortDisplay() {
      return `Version ${this.generation} Build ${this.build}`;
    }

    /**
     * A formatted string for general display, such as "V9 Prototype 1" or "Version 9"
     * @return {string}
     */
    get display() {
      return ["Version", this.generation, this.suffix].filterJoin(" ");
    }

    /**
     * A formatted string for Version compatibility checking, such as "9.150"
     * @return {string}
     */
    get version() {
      return `${this.generation}.${this.build}`;
    }

    /* ----------------------------------------- */

    /** @override */
    toString() {
      return this.shortDisplay;
    }

    /* ----------------------------------------- */

    /**
     * Is this ReleaseData object newer than some other version?
     * @param {string|ReleaseData} other        Some other version to compare against
     * @returns {boolean}                       Is this ReleaseData a newer version?
     */
    isNewer(other) {
      const version = other instanceof ReleaseData ? other.version : other;
      return isNewerVersion(this.version, version);
    }

    /* ----------------------------------------- */

    /**
     * Is this ReleaseData object a newer generation than some other version?
     * @param {string|ReleaseData} other        Some other version to compare against
     * @returns {boolean}                       Is this ReleaseData a newer generation?
     */
    isGenerationalChange(other) {
      if ( !other ) return true;
      let generation;
      if ( other instanceof ReleaseData ) generation = other.generation.toString();
      else {
        other = String(other);
        const parts = other.split(".");
        if ( parts[0] === "0" ) parts.shift();
        generation = parts[0];
      }
      return isNewerVersion(this.generation, generation);
    }
  }

  var config = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ApplicationConfiguration: ApplicationConfiguration,
    ReleaseData: ReleaseData
  });

  // ::- Persistent data structure representing an ordered mapping from
  // strings to values, with some convenient update methods.
  function OrderedMap(content) {
    this.content = content;
  }

  OrderedMap.prototype = {
    constructor: OrderedMap,

    find: function(key) {
      for (var i = 0; i < this.content.length; i += 2)
        if (this.content[i] === key) return i
      return -1
    },

    // :: (string) → ?any
    // Retrieve the value stored under `key`, or return undefined when
    // no such key exists.
    get: function(key) {
      var found = this.find(key);
      return found == -1 ? undefined : this.content[found + 1]
    },

    // :: (string, any, ?string) → OrderedMap
    // Create a new map by replacing the value of `key` with a new
    // value, or adding a binding to the end of the map. If `newKey` is
    // given, the key of the binding will be replaced with that key.
    update: function(key, value, newKey) {
      var self = newKey && newKey != key ? this.remove(newKey) : this;
      var found = self.find(key), content = self.content.slice();
      if (found == -1) {
        content.push(newKey || key, value);
      } else {
        content[found + 1] = value;
        if (newKey) content[found] = newKey;
      }
      return new OrderedMap(content)
    },

    // :: (string) → OrderedMap
    // Return a map with the given key removed, if it existed.
    remove: function(key) {
      var found = this.find(key);
      if (found == -1) return this
      var content = this.content.slice();
      content.splice(found, 2);
      return new OrderedMap(content)
    },

    // :: (string, any) → OrderedMap
    // Add a new key to the start of the map.
    addToStart: function(key, value) {
      return new OrderedMap([key, value].concat(this.remove(key).content))
    },

    // :: (string, any) → OrderedMap
    // Add a new key to the end of the map.
    addToEnd: function(key, value) {
      var content = this.remove(key).content.slice();
      content.push(key, value);
      return new OrderedMap(content)
    },

    // :: (string, string, any) → OrderedMap
    // Add a key after the given key. If `place` is not found, the new
    // key is added to the end.
    addBefore: function(place, key, value) {
      var without = this.remove(key), content = without.content.slice();
      var found = without.find(place);
      content.splice(found == -1 ? content.length : found, 0, key, value);
      return new OrderedMap(content)
    },

    // :: ((key: string, value: any))
    // Call the given function for each key/value pair in the map, in
    // order.
    forEach: function(f) {
      for (var i = 0; i < this.content.length; i += 2)
        f(this.content[i], this.content[i + 1]);
    },

    // :: (union<Object, OrderedMap>) → OrderedMap
    // Create a new map by prepending the keys in this map that don't
    // appear in `map` before the keys in `map`.
    prepend: function(map) {
      map = OrderedMap.from(map);
      if (!map.size) return this
      return new OrderedMap(map.content.concat(this.subtract(map).content))
    },

    // :: (union<Object, OrderedMap>) → OrderedMap
    // Create a new map by appending the keys in this map that don't
    // appear in `map` after the keys in `map`.
    append: function(map) {
      map = OrderedMap.from(map);
      if (!map.size) return this
      return new OrderedMap(this.subtract(map).content.concat(map.content))
    },

    // :: (union<Object, OrderedMap>) → OrderedMap
    // Create a map containing all the keys in this map that don't
    // appear in `map`.
    subtract: function(map) {
      var result = this;
      map = OrderedMap.from(map);
      for (var i = 0; i < map.content.length; i += 2)
        result = result.remove(map.content[i]);
      return result
    },

    // :: () → Object
    // Turn ordered map into a plain object.
    toObject: function() {
      var result = {};
      this.forEach(function(key, value) { result[key] = value; });
      return result
    },

    // :: number
    // The amount of keys in this map.
    get size() {
      return this.content.length >> 1
    }
  };

  // :: (?union<Object, OrderedMap>) → OrderedMap
  // Return a map with the given content. If null, create an empty
  // map. If given an ordered map, return that map itself. If given an
  // object, create a map from the object's properties.
  OrderedMap.from = function(value) {
    if (value instanceof OrderedMap) return value
    var content = [];
    if (value) for (var prop in value) content.push(prop, value[prop]);
    return new OrderedMap(content)
  };

  function findDiffStart(a, b, pos) {
      for (let i = 0;; i++) {
          if (i == a.childCount || i == b.childCount)
              return a.childCount == b.childCount ? null : pos;
          let childA = a.child(i), childB = b.child(i);
          if (childA == childB) {
              pos += childA.nodeSize;
              continue;
          }
          if (!childA.sameMarkup(childB))
              return pos;
          if (childA.isText && childA.text != childB.text) {
              for (let j = 0; childA.text[j] == childB.text[j]; j++)
                  pos++;
              return pos;
          }
          if (childA.content.size || childB.content.size) {
              let inner = findDiffStart(childA.content, childB.content, pos + 1);
              if (inner != null)
                  return inner;
          }
          pos += childA.nodeSize;
      }
  }
  function findDiffEnd(a, b, posA, posB) {
      for (let iA = a.childCount, iB = b.childCount;;) {
          if (iA == 0 || iB == 0)
              return iA == iB ? null : { a: posA, b: posB };
          let childA = a.child(--iA), childB = b.child(--iB), size = childA.nodeSize;
          if (childA == childB) {
              posA -= size;
              posB -= size;
              continue;
          }
          if (!childA.sameMarkup(childB))
              return { a: posA, b: posB };
          if (childA.isText && childA.text != childB.text) {
              let same = 0, minSize = Math.min(childA.text.length, childB.text.length);
              while (same < minSize && childA.text[childA.text.length - same - 1] == childB.text[childB.text.length - same - 1]) {
                  same++;
                  posA--;
                  posB--;
              }
              return { a: posA, b: posB };
          }
          if (childA.content.size || childB.content.size) {
              let inner = findDiffEnd(childA.content, childB.content, posA - 1, posB - 1);
              if (inner)
                  return inner;
          }
          posA -= size;
          posB -= size;
      }
  }

  /**
  A fragment represents a node's collection of child nodes.

  Like nodes, fragments are persistent data structures, and you
  should not mutate them or their content. Rather, you create new
  instances whenever needed. The API tries to make this easy.
  */
  class Fragment {
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      content, size) {
          this.content = content;
          this.size = size || 0;
          if (size == null)
              for (let i = 0; i < content.length; i++)
                  this.size += content[i].nodeSize;
      }
      /**
      Invoke a callback for all descendant nodes between the given two
      positions (relative to start of this fragment). Doesn't descend
      into a node when the callback returns `false`.
      */
      nodesBetween(from, to, f, nodeStart = 0, parent) {
          for (let i = 0, pos = 0; pos < to; i++) {
              let child = this.content[i], end = pos + child.nodeSize;
              if (end > from && f(child, nodeStart + pos, parent || null, i) !== false && child.content.size) {
                  let start = pos + 1;
                  child.nodesBetween(Math.max(0, from - start), Math.min(child.content.size, to - start), f, nodeStart + start);
              }
              pos = end;
          }
      }
      /**
      Call the given callback for every descendant node. `pos` will be
      relative to the start of the fragment. The callback may return
      `false` to prevent traversal of a given node's children.
      */
      descendants(f) {
          this.nodesBetween(0, this.size, f);
      }
      /**
      Extract the text between `from` and `to`. See the same method on
      [`Node`](https://prosemirror.net/docs/ref/#model.Node.textBetween).
      */
      textBetween(from, to, blockSeparator, leafText) {
          let text = "", first = true;
          this.nodesBetween(from, to, (node, pos) => {
              let nodeText = node.isText ? node.text.slice(Math.max(from, pos) - pos, to - pos)
                  : !node.isLeaf ? ""
                      : leafText ? (typeof leafText === "function" ? leafText(node) : leafText)
                          : node.type.spec.leafText ? node.type.spec.leafText(node)
                              : "";
              if (node.isBlock && (node.isLeaf && nodeText || node.isTextblock) && blockSeparator) {
                  if (first)
                      first = false;
                  else
                      text += blockSeparator;
              }
              text += nodeText;
          }, 0);
          return text;
      }
      /**
      Create a new fragment containing the combined content of this
      fragment and the other.
      */
      append(other) {
          if (!other.size)
              return this;
          if (!this.size)
              return other;
          let last = this.lastChild, first = other.firstChild, content = this.content.slice(), i = 0;
          if (last.isText && last.sameMarkup(first)) {
              content[content.length - 1] = last.withText(last.text + first.text);
              i = 1;
          }
          for (; i < other.content.length; i++)
              content.push(other.content[i]);
          return new Fragment(content, this.size + other.size);
      }
      /**
      Cut out the sub-fragment between the two given positions.
      */
      cut(from, to = this.size) {
          if (from == 0 && to == this.size)
              return this;
          let result = [], size = 0;
          if (to > from)
              for (let i = 0, pos = 0; pos < to; i++) {
                  let child = this.content[i], end = pos + child.nodeSize;
                  if (end > from) {
                      if (pos < from || end > to) {
                          if (child.isText)
                              child = child.cut(Math.max(0, from - pos), Math.min(child.text.length, to - pos));
                          else
                              child = child.cut(Math.max(0, from - pos - 1), Math.min(child.content.size, to - pos - 1));
                      }
                      result.push(child);
                      size += child.nodeSize;
                  }
                  pos = end;
              }
          return new Fragment(result, size);
      }
      /**
      @internal
      */
      cutByIndex(from, to) {
          if (from == to)
              return Fragment.empty;
          if (from == 0 && to == this.content.length)
              return this;
          return new Fragment(this.content.slice(from, to));
      }
      /**
      Create a new fragment in which the node at the given index is
      replaced by the given node.
      */
      replaceChild(index, node) {
          let current = this.content[index];
          if (current == node)
              return this;
          let copy = this.content.slice();
          let size = this.size + node.nodeSize - current.nodeSize;
          copy[index] = node;
          return new Fragment(copy, size);
      }
      /**
      Create a new fragment by prepending the given node to this
      fragment.
      */
      addToStart(node) {
          return new Fragment([node].concat(this.content), this.size + node.nodeSize);
      }
      /**
      Create a new fragment by appending the given node to this
      fragment.
      */
      addToEnd(node) {
          return new Fragment(this.content.concat(node), this.size + node.nodeSize);
      }
      /**
      Compare this fragment to another one.
      */
      eq(other) {
          if (this.content.length != other.content.length)
              return false;
          for (let i = 0; i < this.content.length; i++)
              if (!this.content[i].eq(other.content[i]))
                  return false;
          return true;
      }
      /**
      The first child of the fragment, or `null` if it is empty.
      */
      get firstChild() { return this.content.length ? this.content[0] : null; }
      /**
      The last child of the fragment, or `null` if it is empty.
      */
      get lastChild() { return this.content.length ? this.content[this.content.length - 1] : null; }
      /**
      The number of child nodes in this fragment.
      */
      get childCount() { return this.content.length; }
      /**
      Get the child node at the given index. Raise an error when the
      index is out of range.
      */
      child(index) {
          let found = this.content[index];
          if (!found)
              throw new RangeError("Index " + index + " out of range for " + this);
          return found;
      }
      /**
      Get the child node at the given index, if it exists.
      */
      maybeChild(index) {
          return this.content[index] || null;
      }
      /**
      Call `f` for every child node, passing the node, its offset
      into this parent node, and its index.
      */
      forEach(f) {
          for (let i = 0, p = 0; i < this.content.length; i++) {
              let child = this.content[i];
              f(child, p, i);
              p += child.nodeSize;
          }
      }
      /**
      Find the first position at which this fragment and another
      fragment differ, or `null` if they are the same.
      */
      findDiffStart(other, pos = 0) {
          return findDiffStart(this, other, pos);
      }
      /**
      Find the first position, searching from the end, at which this
      fragment and the given fragment differ, or `null` if they are
      the same. Since this position will not be the same in both
      nodes, an object with two separate positions is returned.
      */
      findDiffEnd(other, pos = this.size, otherPos = other.size) {
          return findDiffEnd(this, other, pos, otherPos);
      }
      /**
      Find the index and inner offset corresponding to a given relative
      position in this fragment. The result object will be reused
      (overwritten) the next time the function is called. (Not public.)
      */
      findIndex(pos, round = -1) {
          if (pos == 0)
              return retIndex(0, pos);
          if (pos == this.size)
              return retIndex(this.content.length, pos);
          if (pos > this.size || pos < 0)
              throw new RangeError(`Position ${pos} outside of fragment (${this})`);
          for (let i = 0, curPos = 0;; i++) {
              let cur = this.child(i), end = curPos + cur.nodeSize;
              if (end >= pos) {
                  if (end == pos || round > 0)
                      return retIndex(i + 1, end);
                  return retIndex(i, curPos);
              }
              curPos = end;
          }
      }
      /**
      Return a debugging string that describes this fragment.
      */
      toString() { return "<" + this.toStringInner() + ">"; }
      /**
      @internal
      */
      toStringInner() { return this.content.join(", "); }
      /**
      Create a JSON-serializeable representation of this fragment.
      */
      toJSON() {
          return this.content.length ? this.content.map(n => n.toJSON()) : null;
      }
      /**
      Deserialize a fragment from its JSON representation.
      */
      static fromJSON(schema, value) {
          if (!value)
              return Fragment.empty;
          if (!Array.isArray(value))
              throw new RangeError("Invalid input for Fragment.fromJSON");
          return new Fragment(value.map(schema.nodeFromJSON));
      }
      /**
      Build a fragment from an array of nodes. Ensures that adjacent
      text nodes with the same marks are joined together.
      */
      static fromArray(array) {
          if (!array.length)
              return Fragment.empty;
          let joined, size = 0;
          for (let i = 0; i < array.length; i++) {
              let node = array[i];
              size += node.nodeSize;
              if (i && node.isText && array[i - 1].sameMarkup(node)) {
                  if (!joined)
                      joined = array.slice(0, i);
                  joined[joined.length - 1] = node
                      .withText(joined[joined.length - 1].text + node.text);
              }
              else if (joined) {
                  joined.push(node);
              }
          }
          return new Fragment(joined || array, size);
      }
      /**
      Create a fragment from something that can be interpreted as a
      set of nodes. For `null`, it returns the empty fragment. For a
      fragment, the fragment itself. For a node or array of nodes, a
      fragment containing those nodes.
      */
      static from(nodes) {
          if (!nodes)
              return Fragment.empty;
          if (nodes instanceof Fragment)
              return nodes;
          if (Array.isArray(nodes))
              return this.fromArray(nodes);
          if (nodes.attrs)
              return new Fragment([nodes], nodes.nodeSize);
          throw new RangeError("Can not convert " + nodes + " to a Fragment" +
              (nodes.nodesBetween ? " (looks like multiple versions of prosemirror-model were loaded)" : ""));
      }
  }
  /**
  An empty fragment. Intended to be reused whenever a node doesn't
  contain anything (rather than allocating a new empty fragment for
  each leaf node).
  */
  Fragment.empty = new Fragment([], 0);
  const found = { index: 0, offset: 0 };
  function retIndex(index, offset) {
      found.index = index;
      found.offset = offset;
      return found;
  }

  function compareDeep(a, b) {
      if (a === b)
          return true;
      if (!(a && typeof a == "object") ||
          !(b && typeof b == "object"))
          return false;
      let array = Array.isArray(a);
      if (Array.isArray(b) != array)
          return false;
      if (array) {
          if (a.length != b.length)
              return false;
          for (let i = 0; i < a.length; i++)
              if (!compareDeep(a[i], b[i]))
                  return false;
      }
      else {
          for (let p in a)
              if (!(p in b) || !compareDeep(a[p], b[p]))
                  return false;
          for (let p in b)
              if (!(p in a))
                  return false;
      }
      return true;
  }

  /**
  A mark is a piece of information that can be attached to a node,
  such as it being emphasized, in code font, or a link. It has a
  type and optionally a set of attributes that provide further
  information (such as the target of the link). Marks are created
  through a `Schema`, which controls which types exist and which
  attributes they have.
  */
  class Mark {
      /**
      @internal
      */
      constructor(
      /**
      The type of this mark.
      */
      type, 
      /**
      The attributes associated with this mark.
      */
      attrs) {
          this.type = type;
          this.attrs = attrs;
      }
      /**
      Given a set of marks, create a new set which contains this one as
      well, in the right position. If this mark is already in the set,
      the set itself is returned. If any marks that are set to be
      [exclusive](https://prosemirror.net/docs/ref/#model.MarkSpec.excludes) with this mark are present,
      those are replaced by this one.
      */
      addToSet(set) {
          let copy, placed = false;
          for (let i = 0; i < set.length; i++) {
              let other = set[i];
              if (this.eq(other))
                  return set;
              if (this.type.excludes(other.type)) {
                  if (!copy)
                      copy = set.slice(0, i);
              }
              else if (other.type.excludes(this.type)) {
                  return set;
              }
              else {
                  if (!placed && other.type.rank > this.type.rank) {
                      if (!copy)
                          copy = set.slice(0, i);
                      copy.push(this);
                      placed = true;
                  }
                  if (copy)
                      copy.push(other);
              }
          }
          if (!copy)
              copy = set.slice();
          if (!placed)
              copy.push(this);
          return copy;
      }
      /**
      Remove this mark from the given set, returning a new set. If this
      mark is not in the set, the set itself is returned.
      */
      removeFromSet(set) {
          for (let i = 0; i < set.length; i++)
              if (this.eq(set[i]))
                  return set.slice(0, i).concat(set.slice(i + 1));
          return set;
      }
      /**
      Test whether this mark is in the given set of marks.
      */
      isInSet(set) {
          for (let i = 0; i < set.length; i++)
              if (this.eq(set[i]))
                  return true;
          return false;
      }
      /**
      Test whether this mark has the same type and attributes as
      another mark.
      */
      eq(other) {
          return this == other ||
              (this.type == other.type && compareDeep(this.attrs, other.attrs));
      }
      /**
      Convert this mark to a JSON-serializeable representation.
      */
      toJSON() {
          let obj = { type: this.type.name };
          for (let _ in this.attrs) {
              obj.attrs = this.attrs;
              break;
          }
          return obj;
      }
      /**
      Deserialize a mark from JSON.
      */
      static fromJSON(schema, json) {
          if (!json)
              throw new RangeError("Invalid input for Mark.fromJSON");
          let type = schema.marks[json.type];
          if (!type)
              throw new RangeError(`There is no mark type ${json.type} in this schema`);
          return type.create(json.attrs);
      }
      /**
      Test whether two sets of marks are identical.
      */
      static sameSet(a, b) {
          if (a == b)
              return true;
          if (a.length != b.length)
              return false;
          for (let i = 0; i < a.length; i++)
              if (!a[i].eq(b[i]))
                  return false;
          return true;
      }
      /**
      Create a properly sorted mark set from null, a single mark, or an
      unsorted array of marks.
      */
      static setFrom(marks) {
          if (!marks || Array.isArray(marks) && marks.length == 0)
              return Mark.none;
          if (marks instanceof Mark)
              return [marks];
          let copy = marks.slice();
          copy.sort((a, b) => a.type.rank - b.type.rank);
          return copy;
      }
  }
  /**
  The empty set of marks.
  */
  Mark.none = [];

  /**
  Error type raised by [`Node.replace`](https://prosemirror.net/docs/ref/#model.Node.replace) when
  given an invalid replacement.
  */
  class ReplaceError extends Error {
  }
  /*
  ReplaceError = function(this: any, message: string) {
    let err = Error.call(this, message)
    ;(err as any).__proto__ = ReplaceError.prototype
    return err
  } as any

  ReplaceError.prototype = Object.create(Error.prototype)
  ReplaceError.prototype.constructor = ReplaceError
  ReplaceError.prototype.name = "ReplaceError"
  */
  /**
  A slice represents a piece cut out of a larger document. It
  stores not only a fragment, but also the depth up to which nodes on
  both side are ‘open’ (cut through).
  */
  class Slice {
      /**
      Create a slice. When specifying a non-zero open depth, you must
      make sure that there are nodes of at least that depth at the
      appropriate side of the fragment—i.e. if the fragment is an
      empty paragraph node, `openStart` and `openEnd` can't be greater
      than 1.
      
      It is not necessary for the content of open nodes to conform to
      the schema's content constraints, though it should be a valid
      start/end/middle for such a node, depending on which sides are
      open.
      */
      constructor(
      /**
      The slice's content.
      */
      content, 
      /**
      The open depth at the start of the fragment.
      */
      openStart, 
      /**
      The open depth at the end.
      */
      openEnd) {
          this.content = content;
          this.openStart = openStart;
          this.openEnd = openEnd;
      }
      /**
      The size this slice would add when inserted into a document.
      */
      get size() {
          return this.content.size - this.openStart - this.openEnd;
      }
      /**
      @internal
      */
      insertAt(pos, fragment) {
          let content = insertInto(this.content, pos + this.openStart, fragment);
          return content && new Slice(content, this.openStart, this.openEnd);
      }
      /**
      @internal
      */
      removeBetween(from, to) {
          return new Slice(removeRange(this.content, from + this.openStart, to + this.openStart), this.openStart, this.openEnd);
      }
      /**
      Tests whether this slice is equal to another slice.
      */
      eq(other) {
          return this.content.eq(other.content) && this.openStart == other.openStart && this.openEnd == other.openEnd;
      }
      /**
      @internal
      */
      toString() {
          return this.content + "(" + this.openStart + "," + this.openEnd + ")";
      }
      /**
      Convert a slice to a JSON-serializable representation.
      */
      toJSON() {
          if (!this.content.size)
              return null;
          let json = { content: this.content.toJSON() };
          if (this.openStart > 0)
              json.openStart = this.openStart;
          if (this.openEnd > 0)
              json.openEnd = this.openEnd;
          return json;
      }
      /**
      Deserialize a slice from its JSON representation.
      */
      static fromJSON(schema, json) {
          if (!json)
              return Slice.empty;
          let openStart = json.openStart || 0, openEnd = json.openEnd || 0;
          if (typeof openStart != "number" || typeof openEnd != "number")
              throw new RangeError("Invalid input for Slice.fromJSON");
          return new Slice(Fragment.fromJSON(schema, json.content), openStart, openEnd);
      }
      /**
      Create a slice from a fragment by taking the maximum possible
      open value on both side of the fragment.
      */
      static maxOpen(fragment, openIsolating = true) {
          let openStart = 0, openEnd = 0;
          for (let n = fragment.firstChild; n && !n.isLeaf && (openIsolating || !n.type.spec.isolating); n = n.firstChild)
              openStart++;
          for (let n = fragment.lastChild; n && !n.isLeaf && (openIsolating || !n.type.spec.isolating); n = n.lastChild)
              openEnd++;
          return new Slice(fragment, openStart, openEnd);
      }
  }
  /**
  The empty slice.
  */
  Slice.empty = new Slice(Fragment.empty, 0, 0);
  function removeRange(content, from, to) {
      let { index, offset } = content.findIndex(from), child = content.maybeChild(index);
      let { index: indexTo, offset: offsetTo } = content.findIndex(to);
      if (offset == from || child.isText) {
          if (offsetTo != to && !content.child(indexTo).isText)
              throw new RangeError("Removing non-flat range");
          return content.cut(0, from).append(content.cut(to));
      }
      if (index != indexTo)
          throw new RangeError("Removing non-flat range");
      return content.replaceChild(index, child.copy(removeRange(child.content, from - offset - 1, to - offset - 1)));
  }
  function insertInto(content, dist, insert, parent) {
      let { index, offset } = content.findIndex(dist), child = content.maybeChild(index);
      if (offset == dist || child.isText) {
          return content.cut(0, dist).append(insert).append(content.cut(dist));
      }
      let inner = insertInto(child.content, dist - offset - 1, insert);
      return inner && content.replaceChild(index, child.copy(inner));
  }
  function replace($from, $to, slice) {
      if (slice.openStart > $from.depth)
          throw new ReplaceError("Inserted content deeper than insertion position");
      if ($from.depth - slice.openStart != $to.depth - slice.openEnd)
          throw new ReplaceError("Inconsistent open depths");
      return replaceOuter($from, $to, slice, 0);
  }
  function replaceOuter($from, $to, slice, depth) {
      let index = $from.index(depth), node = $from.node(depth);
      if (index == $to.index(depth) && depth < $from.depth - slice.openStart) {
          let inner = replaceOuter($from, $to, slice, depth + 1);
          return node.copy(node.content.replaceChild(index, inner));
      }
      else if (!slice.content.size) {
          return close(node, replaceTwoWay($from, $to, depth));
      }
      else if (!slice.openStart && !slice.openEnd && $from.depth == depth && $to.depth == depth) { // Simple, flat case
          let parent = $from.parent, content = parent.content;
          return close(parent, content.cut(0, $from.parentOffset).append(slice.content).append(content.cut($to.parentOffset)));
      }
      else {
          let { start, end } = prepareSliceForReplace(slice, $from);
          return close(node, replaceThreeWay($from, start, end, $to, depth));
      }
  }
  function checkJoin(main, sub) {
      if (!sub.type.compatibleContent(main.type))
          throw new ReplaceError("Cannot join " + sub.type.name + " onto " + main.type.name);
  }
  function joinable$1($before, $after, depth) {
      let node = $before.node(depth);
      checkJoin(node, $after.node(depth));
      return node;
  }
  function addNode(child, target) {
      let last = target.length - 1;
      if (last >= 0 && child.isText && child.sameMarkup(target[last]))
          target[last] = child.withText(target[last].text + child.text);
      else
          target.push(child);
  }
  function addRange($start, $end, depth, target) {
      let node = ($end || $start).node(depth);
      let startIndex = 0, endIndex = $end ? $end.index(depth) : node.childCount;
      if ($start) {
          startIndex = $start.index(depth);
          if ($start.depth > depth) {
              startIndex++;
          }
          else if ($start.textOffset) {
              addNode($start.nodeAfter, target);
              startIndex++;
          }
      }
      for (let i = startIndex; i < endIndex; i++)
          addNode(node.child(i), target);
      if ($end && $end.depth == depth && $end.textOffset)
          addNode($end.nodeBefore, target);
  }
  function close(node, content) {
      node.type.checkContent(content);
      return node.copy(content);
  }
  function replaceThreeWay($from, $start, $end, $to, depth) {
      let openStart = $from.depth > depth && joinable$1($from, $start, depth + 1);
      let openEnd = $to.depth > depth && joinable$1($end, $to, depth + 1);
      let content = [];
      addRange(null, $from, depth, content);
      if (openStart && openEnd && $start.index(depth) == $end.index(depth)) {
          checkJoin(openStart, openEnd);
          addNode(close(openStart, replaceThreeWay($from, $start, $end, $to, depth + 1)), content);
      }
      else {
          if (openStart)
              addNode(close(openStart, replaceTwoWay($from, $start, depth + 1)), content);
          addRange($start, $end, depth, content);
          if (openEnd)
              addNode(close(openEnd, replaceTwoWay($end, $to, depth + 1)), content);
      }
      addRange($to, null, depth, content);
      return new Fragment(content);
  }
  function replaceTwoWay($from, $to, depth) {
      let content = [];
      addRange(null, $from, depth, content);
      if ($from.depth > depth) {
          let type = joinable$1($from, $to, depth + 1);
          addNode(close(type, replaceTwoWay($from, $to, depth + 1)), content);
      }
      addRange($to, null, depth, content);
      return new Fragment(content);
  }
  function prepareSliceForReplace(slice, $along) {
      let extra = $along.depth - slice.openStart, parent = $along.node(extra);
      let node = parent.copy(slice.content);
      for (let i = extra - 1; i >= 0; i--)
          node = $along.node(i).copy(Fragment.from(node));
      return { start: node.resolveNoCache(slice.openStart + extra),
          end: node.resolveNoCache(node.content.size - slice.openEnd - extra) };
  }

  /**
  You can [_resolve_](https://prosemirror.net/docs/ref/#model.Node.resolve) a position to get more
  information about it. Objects of this class represent such a
  resolved position, providing various pieces of context
  information, and some helper methods.

  Throughout this interface, methods that take an optional `depth`
  parameter will interpret undefined as `this.depth` and negative
  numbers as `this.depth + value`.
  */
  class ResolvedPos {
      /**
      @internal
      */
      constructor(
      /**
      The position that was resolved.
      */
      pos, 
      /**
      @internal
      */
      path, 
      /**
      The offset this position has into its parent node.
      */
      parentOffset) {
          this.pos = pos;
          this.path = path;
          this.parentOffset = parentOffset;
          this.depth = path.length / 3 - 1;
      }
      /**
      @internal
      */
      resolveDepth(val) {
          if (val == null)
              return this.depth;
          if (val < 0)
              return this.depth + val;
          return val;
      }
      /**
      The parent node that the position points into. Note that even if
      a position points into a text node, that node is not considered
      the parent—text nodes are ‘flat’ in this model, and have no content.
      */
      get parent() { return this.node(this.depth); }
      /**
      The root node in which the position was resolved.
      */
      get doc() { return this.node(0); }
      /**
      The ancestor node at the given level. `p.node(p.depth)` is the
      same as `p.parent`.
      */
      node(depth) { return this.path[this.resolveDepth(depth) * 3]; }
      /**
      The index into the ancestor at the given level. If this points
      at the 3rd node in the 2nd paragraph on the top level, for
      example, `p.index(0)` is 1 and `p.index(1)` is 2.
      */
      index(depth) { return this.path[this.resolveDepth(depth) * 3 + 1]; }
      /**
      The index pointing after this position into the ancestor at the
      given level.
      */
      indexAfter(depth) {
          depth = this.resolveDepth(depth);
          return this.index(depth) + (depth == this.depth && !this.textOffset ? 0 : 1);
      }
      /**
      The (absolute) position at the start of the node at the given
      level.
      */
      start(depth) {
          depth = this.resolveDepth(depth);
          return depth == 0 ? 0 : this.path[depth * 3 - 1] + 1;
      }
      /**
      The (absolute) position at the end of the node at the given
      level.
      */
      end(depth) {
          depth = this.resolveDepth(depth);
          return this.start(depth) + this.node(depth).content.size;
      }
      /**
      The (absolute) position directly before the wrapping node at the
      given level, or, when `depth` is `this.depth + 1`, the original
      position.
      */
      before(depth) {
          depth = this.resolveDepth(depth);
          if (!depth)
              throw new RangeError("There is no position before the top-level node");
          return depth == this.depth + 1 ? this.pos : this.path[depth * 3 - 1];
      }
      /**
      The (absolute) position directly after the wrapping node at the
      given level, or the original position when `depth` is `this.depth + 1`.
      */
      after(depth) {
          depth = this.resolveDepth(depth);
          if (!depth)
              throw new RangeError("There is no position after the top-level node");
          return depth == this.depth + 1 ? this.pos : this.path[depth * 3 - 1] + this.path[depth * 3].nodeSize;
      }
      /**
      When this position points into a text node, this returns the
      distance between the position and the start of the text node.
      Will be zero for positions that point between nodes.
      */
      get textOffset() { return this.pos - this.path[this.path.length - 1]; }
      /**
      Get the node directly after the position, if any. If the position
      points into a text node, only the part of that node after the
      position is returned.
      */
      get nodeAfter() {
          let parent = this.parent, index = this.index(this.depth);
          if (index == parent.childCount)
              return null;
          let dOff = this.pos - this.path[this.path.length - 1], child = parent.child(index);
          return dOff ? parent.child(index).cut(dOff) : child;
      }
      /**
      Get the node directly before the position, if any. If the
      position points into a text node, only the part of that node
      before the position is returned.
      */
      get nodeBefore() {
          let index = this.index(this.depth);
          let dOff = this.pos - this.path[this.path.length - 1];
          if (dOff)
              return this.parent.child(index).cut(0, dOff);
          return index == 0 ? null : this.parent.child(index - 1);
      }
      /**
      Get the position at the given index in the parent node at the
      given depth (which defaults to `this.depth`).
      */
      posAtIndex(index, depth) {
          depth = this.resolveDepth(depth);
          let node = this.path[depth * 3], pos = depth == 0 ? 0 : this.path[depth * 3 - 1] + 1;
          for (let i = 0; i < index; i++)
              pos += node.child(i).nodeSize;
          return pos;
      }
      /**
      Get the marks at this position, factoring in the surrounding
      marks' [`inclusive`](https://prosemirror.net/docs/ref/#model.MarkSpec.inclusive) property. If the
      position is at the start of a non-empty node, the marks of the
      node after it (if any) are returned.
      */
      marks() {
          let parent = this.parent, index = this.index();
          // In an empty parent, return the empty array
          if (parent.content.size == 0)
              return Mark.none;
          // When inside a text node, just return the text node's marks
          if (this.textOffset)
              return parent.child(index).marks;
          let main = parent.maybeChild(index - 1), other = parent.maybeChild(index);
          // If the `after` flag is true of there is no node before, make
          // the node after this position the main reference.
          if (!main) {
              let tmp = main;
              main = other;
              other = tmp;
          }
          // Use all marks in the main node, except those that have
          // `inclusive` set to false and are not present in the other node.
          let marks = main.marks;
          for (var i = 0; i < marks.length; i++)
              if (marks[i].type.spec.inclusive === false && (!other || !marks[i].isInSet(other.marks)))
                  marks = marks[i--].removeFromSet(marks);
          return marks;
      }
      /**
      Get the marks after the current position, if any, except those
      that are non-inclusive and not present at position `$end`. This
      is mostly useful for getting the set of marks to preserve after a
      deletion. Will return `null` if this position is at the end of
      its parent node or its parent node isn't a textblock (in which
      case no marks should be preserved).
      */
      marksAcross($end) {
          let after = this.parent.maybeChild(this.index());
          if (!after || !after.isInline)
              return null;
          let marks = after.marks, next = $end.parent.maybeChild($end.index());
          for (var i = 0; i < marks.length; i++)
              if (marks[i].type.spec.inclusive === false && (!next || !marks[i].isInSet(next.marks)))
                  marks = marks[i--].removeFromSet(marks);
          return marks;
      }
      /**
      The depth up to which this position and the given (non-resolved)
      position share the same parent nodes.
      */
      sharedDepth(pos) {
          for (let depth = this.depth; depth > 0; depth--)
              if (this.start(depth) <= pos && this.end(depth) >= pos)
                  return depth;
          return 0;
      }
      /**
      Returns a range based on the place where this position and the
      given position diverge around block content. If both point into
      the same textblock, for example, a range around that textblock
      will be returned. If they point into different blocks, the range
      around those blocks in their shared ancestor is returned. You can
      pass in an optional predicate that will be called with a parent
      node to see if a range into that parent is acceptable.
      */
      blockRange(other = this, pred) {
          if (other.pos < this.pos)
              return other.blockRange(this);
          for (let d = this.depth - (this.parent.inlineContent || this.pos == other.pos ? 1 : 0); d >= 0; d--)
              if (other.pos <= this.end(d) && (!pred || pred(this.node(d))))
                  return new NodeRange(this, other, d);
          return null;
      }
      /**
      Query whether the given position shares the same parent node.
      */
      sameParent(other) {
          return this.pos - this.parentOffset == other.pos - other.parentOffset;
      }
      /**
      Return the greater of this and the given position.
      */
      max(other) {
          return other.pos > this.pos ? other : this;
      }
      /**
      Return the smaller of this and the given position.
      */
      min(other) {
          return other.pos < this.pos ? other : this;
      }
      /**
      @internal
      */
      toString() {
          let str = "";
          for (let i = 1; i <= this.depth; i++)
              str += (str ? "/" : "") + this.node(i).type.name + "_" + this.index(i - 1);
          return str + ":" + this.parentOffset;
      }
      /**
      @internal
      */
      static resolve(doc, pos) {
          if (!(pos >= 0 && pos <= doc.content.size))
              throw new RangeError("Position " + pos + " out of range");
          let path = [];
          let start = 0, parentOffset = pos;
          for (let node = doc;;) {
              let { index, offset } = node.content.findIndex(parentOffset);
              let rem = parentOffset - offset;
              path.push(node, index, start + offset);
              if (!rem)
                  break;
              node = node.child(index);
              if (node.isText)
                  break;
              parentOffset = rem - 1;
              start += offset + 1;
          }
          return new ResolvedPos(pos, path, parentOffset);
      }
      /**
      @internal
      */
      static resolveCached(doc, pos) {
          for (let i = 0; i < resolveCache.length; i++) {
              let cached = resolveCache[i];
              if (cached.pos == pos && cached.doc == doc)
                  return cached;
          }
          let result = resolveCache[resolveCachePos] = ResolvedPos.resolve(doc, pos);
          resolveCachePos = (resolveCachePos + 1) % resolveCacheSize;
          return result;
      }
  }
  let resolveCache = [], resolveCachePos = 0, resolveCacheSize = 12;
  /**
  Represents a flat range of content, i.e. one that starts and
  ends in the same node.
  */
  class NodeRange {
      /**
      Construct a node range. `$from` and `$to` should point into the
      same node until at least the given `depth`, since a node range
      denotes an adjacent set of nodes in a single parent node.
      */
      constructor(
      /**
      A resolved position along the start of the content. May have a
      `depth` greater than this object's `depth` property, since
      these are the positions that were used to compute the range,
      not re-resolved positions directly at its boundaries.
      */
      $from, 
      /**
      A position along the end of the content. See
      caveat for [`$from`](https://prosemirror.net/docs/ref/#model.NodeRange.$from).
      */
      $to, 
      /**
      The depth of the node that this range points into.
      */
      depth) {
          this.$from = $from;
          this.$to = $to;
          this.depth = depth;
      }
      /**
      The position at the start of the range.
      */
      get start() { return this.$from.before(this.depth + 1); }
      /**
      The position at the end of the range.
      */
      get end() { return this.$to.after(this.depth + 1); }
      /**
      The parent node that the range points into.
      */
      get parent() { return this.$from.node(this.depth); }
      /**
      The start index of the range in the parent node.
      */
      get startIndex() { return this.$from.index(this.depth); }
      /**
      The end index of the range in the parent node.
      */
      get endIndex() { return this.$to.indexAfter(this.depth); }
  }

  const emptyAttrs = Object.create(null);
  /**
  This class represents a node in the tree that makes up a
  ProseMirror document. So a document is an instance of `Node`, with
  children that are also instances of `Node`.

  Nodes are persistent data structures. Instead of changing them, you
  create new ones with the content you want. Old ones keep pointing
  at the old document shape. This is made cheaper by sharing
  structure between the old and new data as much as possible, which a
  tree shape like this (without back pointers) makes easy.

  **Do not** directly mutate the properties of a `Node` object. See
  [the guide](/docs/guide/#doc) for more information.
  */
  class Node {
      /**
      @internal
      */
      constructor(
      /**
      The type of node that this is.
      */
      type, 
      /**
      An object mapping attribute names to values. The kind of
      attributes allowed and required are
      [determined](https://prosemirror.net/docs/ref/#model.NodeSpec.attrs) by the node type.
      */
      attrs, 
      // A fragment holding the node's children.
      content, 
      /**
      The marks (things like whether it is emphasized or part of a
      link) applied to this node.
      */
      marks = Mark.none) {
          this.type = type;
          this.attrs = attrs;
          this.marks = marks;
          this.content = content || Fragment.empty;
      }
      /**
      The size of this node, as defined by the integer-based [indexing
      scheme](/docs/guide/#doc.indexing). For text nodes, this is the
      amount of characters. For other leaf nodes, it is one. For
      non-leaf nodes, it is the size of the content plus two (the
      start and end token).
      */
      get nodeSize() { return this.isLeaf ? 1 : 2 + this.content.size; }
      /**
      The number of children that the node has.
      */
      get childCount() { return this.content.childCount; }
      /**
      Get the child node at the given index. Raises an error when the
      index is out of range.
      */
      child(index) { return this.content.child(index); }
      /**
      Get the child node at the given index, if it exists.
      */
      maybeChild(index) { return this.content.maybeChild(index); }
      /**
      Call `f` for every child node, passing the node, its offset
      into this parent node, and its index.
      */
      forEach(f) { this.content.forEach(f); }
      /**
      Invoke a callback for all descendant nodes recursively between
      the given two positions that are relative to start of this
      node's content. The callback is invoked with the node, its
      position relative to the original node (method receiver),
      its parent node, and its child index. When the callback returns
      false for a given node, that node's children will not be
      recursed over. The last parameter can be used to specify a
      starting position to count from.
      */
      nodesBetween(from, to, f, startPos = 0) {
          this.content.nodesBetween(from, to, f, startPos, this);
      }
      /**
      Call the given callback for every descendant node. Doesn't
      descend into a node when the callback returns `false`.
      */
      descendants(f) {
          this.nodesBetween(0, this.content.size, f);
      }
      /**
      Concatenates all the text nodes found in this fragment and its
      children.
      */
      get textContent() {
          return (this.isLeaf && this.type.spec.leafText)
              ? this.type.spec.leafText(this)
              : this.textBetween(0, this.content.size, "");
      }
      /**
      Get all text between positions `from` and `to`. When
      `blockSeparator` is given, it will be inserted to separate text
      from different block nodes. If `leafText` is given, it'll be
      inserted for every non-text leaf node encountered, otherwise
      [`leafText`](https://prosemirror.net/docs/ref/#model.NodeSpec^leafText) will be used.
      */
      textBetween(from, to, blockSeparator, leafText) {
          return this.content.textBetween(from, to, blockSeparator, leafText);
      }
      /**
      Returns this node's first child, or `null` if there are no
      children.
      */
      get firstChild() { return this.content.firstChild; }
      /**
      Returns this node's last child, or `null` if there are no
      children.
      */
      get lastChild() { return this.content.lastChild; }
      /**
      Test whether two nodes represent the same piece of document.
      */
      eq(other) {
          return this == other || (this.sameMarkup(other) && this.content.eq(other.content));
      }
      /**
      Compare the markup (type, attributes, and marks) of this node to
      those of another. Returns `true` if both have the same markup.
      */
      sameMarkup(other) {
          return this.hasMarkup(other.type, other.attrs, other.marks);
      }
      /**
      Check whether this node's markup correspond to the given type,
      attributes, and marks.
      */
      hasMarkup(type, attrs, marks) {
          return this.type == type &&
              compareDeep(this.attrs, attrs || type.defaultAttrs || emptyAttrs) &&
              Mark.sameSet(this.marks, marks || Mark.none);
      }
      /**
      Create a new node with the same markup as this node, containing
      the given content (or empty, if no content is given).
      */
      copy(content = null) {
          if (content == this.content)
              return this;
          return new Node(this.type, this.attrs, content, this.marks);
      }
      /**
      Create a copy of this node, with the given set of marks instead
      of the node's own marks.
      */
      mark(marks) {
          return marks == this.marks ? this : new Node(this.type, this.attrs, this.content, marks);
      }
      /**
      Create a copy of this node with only the content between the
      given positions. If `to` is not given, it defaults to the end of
      the node.
      */
      cut(from, to = this.content.size) {
          if (from == 0 && to == this.content.size)
              return this;
          return this.copy(this.content.cut(from, to));
      }
      /**
      Cut out the part of the document between the given positions, and
      return it as a `Slice` object.
      */
      slice(from, to = this.content.size, includeParents = false) {
          if (from == to)
              return Slice.empty;
          let $from = this.resolve(from), $to = this.resolve(to);
          let depth = includeParents ? 0 : $from.sharedDepth(to);
          let start = $from.start(depth), node = $from.node(depth);
          let content = node.content.cut($from.pos - start, $to.pos - start);
          return new Slice(content, $from.depth - depth, $to.depth - depth);
      }
      /**
      Replace the part of the document between the given positions with
      the given slice. The slice must 'fit', meaning its open sides
      must be able to connect to the surrounding content, and its
      content nodes must be valid children for the node they are placed
      into. If any of this is violated, an error of type
      [`ReplaceError`](https://prosemirror.net/docs/ref/#model.ReplaceError) is thrown.
      */
      replace(from, to, slice) {
          return replace(this.resolve(from), this.resolve(to), slice);
      }
      /**
      Find the node directly after the given position.
      */
      nodeAt(pos) {
          for (let node = this;;) {
              let { index, offset } = node.content.findIndex(pos);
              node = node.maybeChild(index);
              if (!node)
                  return null;
              if (offset == pos || node.isText)
                  return node;
              pos -= offset + 1;
          }
      }
      /**
      Find the (direct) child node after the given offset, if any,
      and return it along with its index and offset relative to this
      node.
      */
      childAfter(pos) {
          let { index, offset } = this.content.findIndex(pos);
          return { node: this.content.maybeChild(index), index, offset };
      }
      /**
      Find the (direct) child node before the given offset, if any,
      and return it along with its index and offset relative to this
      node.
      */
      childBefore(pos) {
          if (pos == 0)
              return { node: null, index: 0, offset: 0 };
          let { index, offset } = this.content.findIndex(pos);
          if (offset < pos)
              return { node: this.content.child(index), index, offset };
          let node = this.content.child(index - 1);
          return { node, index: index - 1, offset: offset - node.nodeSize };
      }
      /**
      Resolve the given position in the document, returning an
      [object](https://prosemirror.net/docs/ref/#model.ResolvedPos) with information about its context.
      */
      resolve(pos) { return ResolvedPos.resolveCached(this, pos); }
      /**
      @internal
      */
      resolveNoCache(pos) { return ResolvedPos.resolve(this, pos); }
      /**
      Test whether a given mark or mark type occurs in this document
      between the two given positions.
      */
      rangeHasMark(from, to, type) {
          let found = false;
          if (to > from)
              this.nodesBetween(from, to, node => {
                  if (type.isInSet(node.marks))
                      found = true;
                  return !found;
              });
          return found;
      }
      /**
      True when this is a block (non-inline node)
      */
      get isBlock() { return this.type.isBlock; }
      /**
      True when this is a textblock node, a block node with inline
      content.
      */
      get isTextblock() { return this.type.isTextblock; }
      /**
      True when this node allows inline content.
      */
      get inlineContent() { return this.type.inlineContent; }
      /**
      True when this is an inline node (a text node or a node that can
      appear among text).
      */
      get isInline() { return this.type.isInline; }
      /**
      True when this is a text node.
      */
      get isText() { return this.type.isText; }
      /**
      True when this is a leaf node.
      */
      get isLeaf() { return this.type.isLeaf; }
      /**
      True when this is an atom, i.e. when it does not have directly
      editable content. This is usually the same as `isLeaf`, but can
      be configured with the [`atom` property](https://prosemirror.net/docs/ref/#model.NodeSpec.atom)
      on a node's spec (typically used when the node is displayed as
      an uneditable [node view](https://prosemirror.net/docs/ref/#view.NodeView)).
      */
      get isAtom() { return this.type.isAtom; }
      /**
      Return a string representation of this node for debugging
      purposes.
      */
      toString() {
          if (this.type.spec.toDebugString)
              return this.type.spec.toDebugString(this);
          let name = this.type.name;
          if (this.content.size)
              name += "(" + this.content.toStringInner() + ")";
          return wrapMarks(this.marks, name);
      }
      /**
      Get the content match in this node at the given index.
      */
      contentMatchAt(index) {
          let match = this.type.contentMatch.matchFragment(this.content, 0, index);
          if (!match)
              throw new Error("Called contentMatchAt on a node with invalid content");
          return match;
      }
      /**
      Test whether replacing the range between `from` and `to` (by
      child index) with the given replacement fragment (which defaults
      to the empty fragment) would leave the node's content valid. You
      can optionally pass `start` and `end` indices into the
      replacement fragment.
      */
      canReplace(from, to, replacement = Fragment.empty, start = 0, end = replacement.childCount) {
          let one = this.contentMatchAt(from).matchFragment(replacement, start, end);
          let two = one && one.matchFragment(this.content, to);
          if (!two || !two.validEnd)
              return false;
          for (let i = start; i < end; i++)
              if (!this.type.allowsMarks(replacement.child(i).marks))
                  return false;
          return true;
      }
      /**
      Test whether replacing the range `from` to `to` (by index) with
      a node of the given type would leave the node's content valid.
      */
      canReplaceWith(from, to, type, marks) {
          if (marks && !this.type.allowsMarks(marks))
              return false;
          let start = this.contentMatchAt(from).matchType(type);
          let end = start && start.matchFragment(this.content, to);
          return end ? end.validEnd : false;
      }
      /**
      Test whether the given node's content could be appended to this
      node. If that node is empty, this will only return true if there
      is at least one node type that can appear in both nodes (to avoid
      merging completely incompatible nodes).
      */
      canAppend(other) {
          if (other.content.size)
              return this.canReplace(this.childCount, this.childCount, other.content);
          else
              return this.type.compatibleContent(other.type);
      }
      /**
      Check whether this node and its descendants conform to the
      schema, and raise error when they do not.
      */
      check() {
          this.type.checkContent(this.content);
          let copy = Mark.none;
          for (let i = 0; i < this.marks.length; i++)
              copy = this.marks[i].addToSet(copy);
          if (!Mark.sameSet(copy, this.marks))
              throw new RangeError(`Invalid collection of marks for node ${this.type.name}: ${this.marks.map(m => m.type.name)}`);
          this.content.forEach(node => node.check());
      }
      /**
      Return a JSON-serializeable representation of this node.
      */
      toJSON() {
          let obj = { type: this.type.name };
          for (let _ in this.attrs) {
              obj.attrs = this.attrs;
              break;
          }
          if (this.content.size)
              obj.content = this.content.toJSON();
          if (this.marks.length)
              obj.marks = this.marks.map(n => n.toJSON());
          return obj;
      }
      /**
      Deserialize a node from its JSON representation.
      */
      static fromJSON(schema, json) {
          if (!json)
              throw new RangeError("Invalid input for Node.fromJSON");
          let marks = null;
          if (json.marks) {
              if (!Array.isArray(json.marks))
                  throw new RangeError("Invalid mark data for Node.fromJSON");
              marks = json.marks.map(schema.markFromJSON);
          }
          if (json.type == "text") {
              if (typeof json.text != "string")
                  throw new RangeError("Invalid text node in JSON");
              return schema.text(json.text, marks);
          }
          let content = Fragment.fromJSON(schema, json.content);
          return schema.nodeType(json.type).create(json.attrs, content, marks);
      }
  }
  Node.prototype.text = undefined;
  class TextNode extends Node {
      /**
      @internal
      */
      constructor(type, attrs, content, marks) {
          super(type, attrs, null, marks);
          if (!content)
              throw new RangeError("Empty text nodes are not allowed");
          this.text = content;
      }
      toString() {
          if (this.type.spec.toDebugString)
              return this.type.spec.toDebugString(this);
          return wrapMarks(this.marks, JSON.stringify(this.text));
      }
      get textContent() { return this.text; }
      textBetween(from, to) { return this.text.slice(from, to); }
      get nodeSize() { return this.text.length; }
      mark(marks) {
          return marks == this.marks ? this : new TextNode(this.type, this.attrs, this.text, marks);
      }
      withText(text) {
          if (text == this.text)
              return this;
          return new TextNode(this.type, this.attrs, text, this.marks);
      }
      cut(from = 0, to = this.text.length) {
          if (from == 0 && to == this.text.length)
              return this;
          return this.withText(this.text.slice(from, to));
      }
      eq(other) {
          return this.sameMarkup(other) && this.text == other.text;
      }
      toJSON() {
          let base = super.toJSON();
          base.text = this.text;
          return base;
      }
  }
  function wrapMarks(marks, str) {
      for (let i = marks.length - 1; i >= 0; i--)
          str = marks[i].type.name + "(" + str + ")";
      return str;
  }

  /**
  Instances of this class represent a match state of a node type's
  [content expression](https://prosemirror.net/docs/ref/#model.NodeSpec.content), and can be used to
  find out whether further content matches here, and whether a given
  position is a valid end of the node.
  */
  class ContentMatch {
      /**
      @internal
      */
      constructor(
      /**
      True when this match state represents a valid end of the node.
      */
      validEnd) {
          this.validEnd = validEnd;
          /**
          @internal
          */
          this.next = [];
          /**
          @internal
          */
          this.wrapCache = [];
      }
      /**
      @internal
      */
      static parse(string, nodeTypes) {
          let stream = new TokenStream(string, nodeTypes);
          if (stream.next == null)
              return ContentMatch.empty;
          let expr = parseExpr(stream);
          if (stream.next)
              stream.err("Unexpected trailing text");
          let match = dfa(nfa(expr));
          checkForDeadEnds(match, stream);
          return match;
      }
      /**
      Match a node type, returning a match after that node if
      successful.
      */
      matchType(type) {
          for (let i = 0; i < this.next.length; i++)
              if (this.next[i].type == type)
                  return this.next[i].next;
          return null;
      }
      /**
      Try to match a fragment. Returns the resulting match when
      successful.
      */
      matchFragment(frag, start = 0, end = frag.childCount) {
          let cur = this;
          for (let i = start; cur && i < end; i++)
              cur = cur.matchType(frag.child(i).type);
          return cur;
      }
      /**
      @internal
      */
      get inlineContent() {
          return this.next.length != 0 && this.next[0].type.isInline;
      }
      /**
      Get the first matching node type at this match position that can
      be generated.
      */
      get defaultType() {
          for (let i = 0; i < this.next.length; i++) {
              let { type } = this.next[i];
              if (!(type.isText || type.hasRequiredAttrs()))
                  return type;
          }
          return null;
      }
      /**
      @internal
      */
      compatible(other) {
          for (let i = 0; i < this.next.length; i++)
              for (let j = 0; j < other.next.length; j++)
                  if (this.next[i].type == other.next[j].type)
                      return true;
          return false;
      }
      /**
      Try to match the given fragment, and if that fails, see if it can
      be made to match by inserting nodes in front of it. When
      successful, return a fragment of inserted nodes (which may be
      empty if nothing had to be inserted). When `toEnd` is true, only
      return a fragment if the resulting match goes to the end of the
      content expression.
      */
      fillBefore(after, toEnd = false, startIndex = 0) {
          let seen = [this];
          function search(match, types) {
              let finished = match.matchFragment(after, startIndex);
              if (finished && (!toEnd || finished.validEnd))
                  return Fragment.from(types.map(tp => tp.createAndFill()));
              for (let i = 0; i < match.next.length; i++) {
                  let { type, next } = match.next[i];
                  if (!(type.isText || type.hasRequiredAttrs()) && seen.indexOf(next) == -1) {
                      seen.push(next);
                      let found = search(next, types.concat(type));
                      if (found)
                          return found;
                  }
              }
              return null;
          }
          return search(this, []);
      }
      /**
      Find a set of wrapping node types that would allow a node of the
      given type to appear at this position. The result may be empty
      (when it fits directly) and will be null when no such wrapping
      exists.
      */
      findWrapping(target) {
          for (let i = 0; i < this.wrapCache.length; i += 2)
              if (this.wrapCache[i] == target)
                  return this.wrapCache[i + 1];
          let computed = this.computeWrapping(target);
          this.wrapCache.push(target, computed);
          return computed;
      }
      /**
      @internal
      */
      computeWrapping(target) {
          let seen = Object.create(null), active = [{ match: this, type: null, via: null }];
          while (active.length) {
              let current = active.shift(), match = current.match;
              if (match.matchType(target)) {
                  let result = [];
                  for (let obj = current; obj.type; obj = obj.via)
                      result.push(obj.type);
                  return result.reverse();
              }
              for (let i = 0; i < match.next.length; i++) {
                  let { type, next } = match.next[i];
                  if (!type.isLeaf && !type.hasRequiredAttrs() && !(type.name in seen) && (!current.type || next.validEnd)) {
                      active.push({ match: type.contentMatch, type, via: current });
                      seen[type.name] = true;
                  }
              }
          }
          return null;
      }
      /**
      The number of outgoing edges this node has in the finite
      automaton that describes the content expression.
      */
      get edgeCount() {
          return this.next.length;
      }
      /**
      Get the _n_​th outgoing edge from this node in the finite
      automaton that describes the content expression.
      */
      edge(n) {
          if (n >= this.next.length)
              throw new RangeError(`There's no ${n}th edge in this content match`);
          return this.next[n];
      }
      /**
      @internal
      */
      toString() {
          let seen = [];
          function scan(m) {
              seen.push(m);
              for (let i = 0; i < m.next.length; i++)
                  if (seen.indexOf(m.next[i].next) == -1)
                      scan(m.next[i].next);
          }
          scan(this);
          return seen.map((m, i) => {
              let out = i + (m.validEnd ? "*" : " ") + " ";
              for (let i = 0; i < m.next.length; i++)
                  out += (i ? ", " : "") + m.next[i].type.name + "->" + seen.indexOf(m.next[i].next);
              return out;
          }).join("\n");
      }
  }
  /**
  @internal
  */
  ContentMatch.empty = new ContentMatch(true);
  class TokenStream {
      constructor(string, nodeTypes) {
          this.string = string;
          this.nodeTypes = nodeTypes;
          this.inline = null;
          this.pos = 0;
          this.tokens = string.split(/\s*(?=\b|\W|$)/);
          if (this.tokens[this.tokens.length - 1] == "")
              this.tokens.pop();
          if (this.tokens[0] == "")
              this.tokens.shift();
      }
      get next() { return this.tokens[this.pos]; }
      eat(tok) { return this.next == tok && (this.pos++ || true); }
      err(str) { throw new SyntaxError(str + " (in content expression '" + this.string + "')"); }
  }
  function parseExpr(stream) {
      let exprs = [];
      do {
          exprs.push(parseExprSeq(stream));
      } while (stream.eat("|"));
      return exprs.length == 1 ? exprs[0] : { type: "choice", exprs };
  }
  function parseExprSeq(stream) {
      let exprs = [];
      do {
          exprs.push(parseExprSubscript(stream));
      } while (stream.next && stream.next != ")" && stream.next != "|");
      return exprs.length == 1 ? exprs[0] : { type: "seq", exprs };
  }
  function parseExprSubscript(stream) {
      let expr = parseExprAtom(stream);
      for (;;) {
          if (stream.eat("+"))
              expr = { type: "plus", expr };
          else if (stream.eat("*"))
              expr = { type: "star", expr };
          else if (stream.eat("?"))
              expr = { type: "opt", expr };
          else if (stream.eat("{"))
              expr = parseExprRange(stream, expr);
          else
              break;
      }
      return expr;
  }
  function parseNum(stream) {
      if (/\D/.test(stream.next))
          stream.err("Expected number, got '" + stream.next + "'");
      let result = Number(stream.next);
      stream.pos++;
      return result;
  }
  function parseExprRange(stream, expr) {
      let min = parseNum(stream), max = min;
      if (stream.eat(",")) {
          if (stream.next != "}")
              max = parseNum(stream);
          else
              max = -1;
      }
      if (!stream.eat("}"))
          stream.err("Unclosed braced range");
      return { type: "range", min, max, expr };
  }
  function resolveName(stream, name) {
      let types = stream.nodeTypes, type = types[name];
      if (type)
          return [type];
      let result = [];
      for (let typeName in types) {
          let type = types[typeName];
          if (type.groups.indexOf(name) > -1)
              result.push(type);
      }
      if (result.length == 0)
          stream.err("No node type or group '" + name + "' found");
      return result;
  }
  function parseExprAtom(stream) {
      if (stream.eat("(")) {
          let expr = parseExpr(stream);
          if (!stream.eat(")"))
              stream.err("Missing closing paren");
          return expr;
      }
      else if (!/\W/.test(stream.next)) {
          let exprs = resolveName(stream, stream.next).map(type => {
              if (stream.inline == null)
                  stream.inline = type.isInline;
              else if (stream.inline != type.isInline)
                  stream.err("Mixing inline and block content");
              return { type: "name", value: type };
          });
          stream.pos++;
          return exprs.length == 1 ? exprs[0] : { type: "choice", exprs };
      }
      else {
          stream.err("Unexpected token '" + stream.next + "'");
      }
  }
  /**
  Construct an NFA from an expression as returned by the parser. The
  NFA is represented as an array of states, which are themselves
  arrays of edges, which are `{term, to}` objects. The first state is
  the entry state and the last node is the success state.

  Note that unlike typical NFAs, the edge ordering in this one is
  significant, in that it is used to contruct filler content when
  necessary.
  */
  function nfa(expr) {
      let nfa = [[]];
      connect(compile(expr, 0), node());
      return nfa;
      function node() { return nfa.push([]) - 1; }
      function edge(from, to, term) {
          let edge = { term, to };
          nfa[from].push(edge);
          return edge;
      }
      function connect(edges, to) {
          edges.forEach(edge => edge.to = to);
      }
      function compile(expr, from) {
          if (expr.type == "choice") {
              return expr.exprs.reduce((out, expr) => out.concat(compile(expr, from)), []);
          }
          else if (expr.type == "seq") {
              for (let i = 0;; i++) {
                  let next = compile(expr.exprs[i], from);
                  if (i == expr.exprs.length - 1)
                      return next;
                  connect(next, from = node());
              }
          }
          else if (expr.type == "star") {
              let loop = node();
              edge(from, loop);
              connect(compile(expr.expr, loop), loop);
              return [edge(loop)];
          }
          else if (expr.type == "plus") {
              let loop = node();
              connect(compile(expr.expr, from), loop);
              connect(compile(expr.expr, loop), loop);
              return [edge(loop)];
          }
          else if (expr.type == "opt") {
              return [edge(from)].concat(compile(expr.expr, from));
          }
          else if (expr.type == "range") {
              let cur = from;
              for (let i = 0; i < expr.min; i++) {
                  let next = node();
                  connect(compile(expr.expr, cur), next);
                  cur = next;
              }
              if (expr.max == -1) {
                  connect(compile(expr.expr, cur), cur);
              }
              else {
                  for (let i = expr.min; i < expr.max; i++) {
                      let next = node();
                      edge(cur, next);
                      connect(compile(expr.expr, cur), next);
                      cur = next;
                  }
              }
              return [edge(cur)];
          }
          else if (expr.type == "name") {
              return [edge(from, undefined, expr.value)];
          }
          else {
              throw new Error("Unknown expr type");
          }
      }
  }
  function cmp(a, b) { return b - a; }
  // Get the set of nodes reachable by null edges from `node`. Omit
  // nodes with only a single null-out-edge, since they may lead to
  // needless duplicated nodes.
  function nullFrom(nfa, node) {
      let result = [];
      scan(node);
      return result.sort(cmp);
      function scan(node) {
          let edges = nfa[node];
          if (edges.length == 1 && !edges[0].term)
              return scan(edges[0].to);
          result.push(node);
          for (let i = 0; i < edges.length; i++) {
              let { term, to } = edges[i];
              if (!term && result.indexOf(to) == -1)
                  scan(to);
          }
      }
  }
  // Compiles an NFA as produced by `nfa` into a DFA, modeled as a set
  // of state objects (`ContentMatch` instances) with transitions
  // between them.
  function dfa(nfa) {
      let labeled = Object.create(null);
      return explore(nullFrom(nfa, 0));
      function explore(states) {
          let out = [];
          states.forEach(node => {
              nfa[node].forEach(({ term, to }) => {
                  if (!term)
                      return;
                  let set;
                  for (let i = 0; i < out.length; i++)
                      if (out[i][0] == term)
                          set = out[i][1];
                  nullFrom(nfa, to).forEach(node => {
                      if (!set)
                          out.push([term, set = []]);
                      if (set.indexOf(node) == -1)
                          set.push(node);
                  });
              });
          });
          let state = labeled[states.join(",")] = new ContentMatch(states.indexOf(nfa.length - 1) > -1);
          for (let i = 0; i < out.length; i++) {
              let states = out[i][1].sort(cmp);
              state.next.push({ type: out[i][0], next: labeled[states.join(",")] || explore(states) });
          }
          return state;
      }
  }
  function checkForDeadEnds(match, stream) {
      for (let i = 0, work = [match]; i < work.length; i++) {
          let state = work[i], dead = !state.validEnd, nodes = [];
          for (let j = 0; j < state.next.length; j++) {
              let { type, next } = state.next[j];
              nodes.push(type.name);
              if (dead && !(type.isText || type.hasRequiredAttrs()))
                  dead = false;
              if (work.indexOf(next) == -1)
                  work.push(next);
          }
          if (dead)
              stream.err("Only non-generatable nodes (" + nodes.join(", ") + ") in a required position (see https://prosemirror.net/docs/guide/#generatable)");
      }
  }

  // For node types where all attrs have a default value (or which don't
  // have any attributes), build up a single reusable default attribute
  // object, and use it for all nodes that don't specify specific
  // attributes.
  function defaultAttrs(attrs) {
      let defaults = Object.create(null);
      for (let attrName in attrs) {
          let attr = attrs[attrName];
          if (!attr.hasDefault)
              return null;
          defaults[attrName] = attr.default;
      }
      return defaults;
  }
  function computeAttrs(attrs, value) {
      let built = Object.create(null);
      for (let name in attrs) {
          let given = value && value[name];
          if (given === undefined) {
              let attr = attrs[name];
              if (attr.hasDefault)
                  given = attr.default;
              else
                  throw new RangeError("No value supplied for attribute " + name);
          }
          built[name] = given;
      }
      return built;
  }
  function initAttrs(attrs) {
      let result = Object.create(null);
      if (attrs)
          for (let name in attrs)
              result[name] = new Attribute(attrs[name]);
      return result;
  }
  /**
  Node types are objects allocated once per `Schema` and used to
  [tag](https://prosemirror.net/docs/ref/#model.Node.type) `Node` instances. They contain information
  about the node type, such as its name and what kind of node it
  represents.
  */
  let NodeType$1 = class NodeType {
      /**
      @internal
      */
      constructor(
      /**
      The name the node type has in this schema.
      */
      name, 
      /**
      A link back to the `Schema` the node type belongs to.
      */
      schema, 
      /**
      The spec that this type is based on
      */
      spec) {
          this.name = name;
          this.schema = schema;
          this.spec = spec;
          /**
          The set of marks allowed in this node. `null` means all marks
          are allowed.
          */
          this.markSet = null;
          this.groups = spec.group ? spec.group.split(" ") : [];
          this.attrs = initAttrs(spec.attrs);
          this.defaultAttrs = defaultAttrs(this.attrs);
          this.contentMatch = null;
          this.inlineContent = null;
          this.isBlock = !(spec.inline || name == "text");
          this.isText = name == "text";
      }
      /**
      True if this is an inline type.
      */
      get isInline() { return !this.isBlock; }
      /**
      True if this is a textblock type, a block that contains inline
      content.
      */
      get isTextblock() { return this.isBlock && this.inlineContent; }
      /**
      True for node types that allow no content.
      */
      get isLeaf() { return this.contentMatch == ContentMatch.empty; }
      /**
      True when this node is an atom, i.e. when it does not have
      directly editable content.
      */
      get isAtom() { return this.isLeaf || !!this.spec.atom; }
      /**
      The node type's [whitespace](https://prosemirror.net/docs/ref/#model.NodeSpec.whitespace) option.
      */
      get whitespace() {
          return this.spec.whitespace || (this.spec.code ? "pre" : "normal");
      }
      /**
      Tells you whether this node type has any required attributes.
      */
      hasRequiredAttrs() {
          for (let n in this.attrs)
              if (this.attrs[n].isRequired)
                  return true;
          return false;
      }
      /**
      Indicates whether this node allows some of the same content as
      the given node type.
      */
      compatibleContent(other) {
          return this == other || this.contentMatch.compatible(other.contentMatch);
      }
      /**
      @internal
      */
      computeAttrs(attrs) {
          if (!attrs && this.defaultAttrs)
              return this.defaultAttrs;
          else
              return computeAttrs(this.attrs, attrs);
      }
      /**
      Create a `Node` of this type. The given attributes are
      checked and defaulted (you can pass `null` to use the type's
      defaults entirely, if no required attributes exist). `content`
      may be a `Fragment`, a node, an array of nodes, or
      `null`. Similarly `marks` may be `null` to default to the empty
      set of marks.
      */
      create(attrs = null, content, marks) {
          if (this.isText)
              throw new Error("NodeType.create can't construct text nodes");
          return new Node(this, this.computeAttrs(attrs), Fragment.from(content), Mark.setFrom(marks));
      }
      /**
      Like [`create`](https://prosemirror.net/docs/ref/#model.NodeType.create), but check the given content
      against the node type's content restrictions, and throw an error
      if it doesn't match.
      */
      createChecked(attrs = null, content, marks) {
          content = Fragment.from(content);
          this.checkContent(content);
          return new Node(this, this.computeAttrs(attrs), content, Mark.setFrom(marks));
      }
      /**
      Like [`create`](https://prosemirror.net/docs/ref/#model.NodeType.create), but see if it is
      necessary to add nodes to the start or end of the given fragment
      to make it fit the node. If no fitting wrapping can be found,
      return null. Note that, due to the fact that required nodes can
      always be created, this will always succeed if you pass null or
      `Fragment.empty` as content.
      */
      createAndFill(attrs = null, content, marks) {
          attrs = this.computeAttrs(attrs);
          content = Fragment.from(content);
          if (content.size) {
              let before = this.contentMatch.fillBefore(content);
              if (!before)
                  return null;
              content = before.append(content);
          }
          let matched = this.contentMatch.matchFragment(content);
          let after = matched && matched.fillBefore(Fragment.empty, true);
          if (!after)
              return null;
          return new Node(this, attrs, content.append(after), Mark.setFrom(marks));
      }
      /**
      Returns true if the given fragment is valid content for this node
      type with the given attributes.
      */
      validContent(content) {
          let result = this.contentMatch.matchFragment(content);
          if (!result || !result.validEnd)
              return false;
          for (let i = 0; i < content.childCount; i++)
              if (!this.allowsMarks(content.child(i).marks))
                  return false;
          return true;
      }
      /**
      Throws a RangeError if the given fragment is not valid content for this
      node type.
      @internal
      */
      checkContent(content) {
          if (!this.validContent(content))
              throw new RangeError(`Invalid content for node ${this.name}: ${content.toString().slice(0, 50)}`);
      }
      /**
      Check whether the given mark type is allowed in this node.
      */
      allowsMarkType(markType) {
          return this.markSet == null || this.markSet.indexOf(markType) > -1;
      }
      /**
      Test whether the given set of marks are allowed in this node.
      */
      allowsMarks(marks) {
          if (this.markSet == null)
              return true;
          for (let i = 0; i < marks.length; i++)
              if (!this.allowsMarkType(marks[i].type))
                  return false;
          return true;
      }
      /**
      Removes the marks that are not allowed in this node from the given set.
      */
      allowedMarks(marks) {
          if (this.markSet == null)
              return marks;
          let copy;
          for (let i = 0; i < marks.length; i++) {
              if (!this.allowsMarkType(marks[i].type)) {
                  if (!copy)
                      copy = marks.slice(0, i);
              }
              else if (copy) {
                  copy.push(marks[i]);
              }
          }
          return !copy ? marks : copy.length ? copy : Mark.none;
      }
      /**
      @internal
      */
      static compile(nodes, schema) {
          let result = Object.create(null);
          nodes.forEach((name, spec) => result[name] = new NodeType(name, schema, spec));
          let topType = schema.spec.topNode || "doc";
          if (!result[topType])
              throw new RangeError("Schema is missing its top node type ('" + topType + "')");
          if (!result.text)
              throw new RangeError("Every schema needs a 'text' type");
          for (let _ in result.text.attrs)
              throw new RangeError("The text node type should not have attributes");
          return result;
      }
  };
  // Attribute descriptors
  class Attribute {
      constructor(options) {
          this.hasDefault = Object.prototype.hasOwnProperty.call(options, "default");
          this.default = options.default;
      }
      get isRequired() {
          return !this.hasDefault;
      }
  }
  // Marks
  /**
  Like nodes, marks (which are associated with nodes to signify
  things like emphasis or being part of a link) are
  [tagged](https://prosemirror.net/docs/ref/#model.Mark.type) with type objects, which are
  instantiated once per `Schema`.
  */
  class MarkType {
      /**
      @internal
      */
      constructor(
      /**
      The name of the mark type.
      */
      name, 
      /**
      @internal
      */
      rank, 
      /**
      The schema that this mark type instance is part of.
      */
      schema, 
      /**
      The spec on which the type is based.
      */
      spec) {
          this.name = name;
          this.rank = rank;
          this.schema = schema;
          this.spec = spec;
          this.attrs = initAttrs(spec.attrs);
          this.excluded = null;
          let defaults = defaultAttrs(this.attrs);
          this.instance = defaults ? new Mark(this, defaults) : null;
      }
      /**
      Create a mark of this type. `attrs` may be `null` or an object
      containing only some of the mark's attributes. The others, if
      they have defaults, will be added.
      */
      create(attrs = null) {
          if (!attrs && this.instance)
              return this.instance;
          return new Mark(this, computeAttrs(this.attrs, attrs));
      }
      /**
      @internal
      */
      static compile(marks, schema) {
          let result = Object.create(null), rank = 0;
          marks.forEach((name, spec) => result[name] = new MarkType(name, rank++, schema, spec));
          return result;
      }
      /**
      When there is a mark of this type in the given set, a new set
      without it is returned. Otherwise, the input set is returned.
      */
      removeFromSet(set) {
          for (var i = 0; i < set.length; i++)
              if (set[i].type == this) {
                  set = set.slice(0, i).concat(set.slice(i + 1));
                  i--;
              }
          return set;
      }
      /**
      Tests whether there is a mark of this type in the given set.
      */
      isInSet(set) {
          for (let i = 0; i < set.length; i++)
              if (set[i].type == this)
                  return set[i];
      }
      /**
      Queries whether a given mark type is
      [excluded](https://prosemirror.net/docs/ref/#model.MarkSpec.excludes) by this one.
      */
      excludes(other) {
          return this.excluded.indexOf(other) > -1;
      }
  }
  /**
  A document schema. Holds [node](https://prosemirror.net/docs/ref/#model.NodeType) and [mark
  type](https://prosemirror.net/docs/ref/#model.MarkType) objects for the nodes and marks that may
  occur in conforming documents, and provides functionality for
  creating and deserializing such documents.

  When given, the type parameters provide the names of the nodes and
  marks in this schema.
  */
  class Schema {
      /**
      Construct a schema from a schema [specification](https://prosemirror.net/docs/ref/#model.SchemaSpec).
      */
      constructor(spec) {
          /**
          The [linebreak
          replacement](https://prosemirror.net/docs/ref/#model.NodeSpec.linebreakReplacement) node defined
          in this schema, if any.
          */
          this.linebreakReplacement = null;
          /**
          An object for storing whatever values modules may want to
          compute and cache per schema. (If you want to store something
          in it, try to use property names unlikely to clash.)
          */
          this.cached = Object.create(null);
          let instanceSpec = this.spec = {};
          for (let prop in spec)
              instanceSpec[prop] = spec[prop];
          instanceSpec.nodes = OrderedMap.from(spec.nodes),
              instanceSpec.marks = OrderedMap.from(spec.marks || {}),
              this.nodes = NodeType$1.compile(this.spec.nodes, this);
          this.marks = MarkType.compile(this.spec.marks, this);
          let contentExprCache = Object.create(null);
          for (let prop in this.nodes) {
              if (prop in this.marks)
                  throw new RangeError(prop + " can not be both a node and a mark");
              let type = this.nodes[prop], contentExpr = type.spec.content || "", markExpr = type.spec.marks;
              type.contentMatch = contentExprCache[contentExpr] ||
                  (contentExprCache[contentExpr] = ContentMatch.parse(contentExpr, this.nodes));
              type.inlineContent = type.contentMatch.inlineContent;
              if (type.spec.linebreakReplacement) {
                  if (this.linebreakReplacement)
                      throw new RangeError("Multiple linebreak nodes defined");
                  if (!type.isInline || !type.isLeaf)
                      throw new RangeError("Linebreak replacement nodes must be inline leaf nodes");
                  this.linebreakReplacement = type;
              }
              type.markSet = markExpr == "_" ? null :
                  markExpr ? gatherMarks(this, markExpr.split(" ")) :
                      markExpr == "" || !type.inlineContent ? [] : null;
          }
          for (let prop in this.marks) {
              let type = this.marks[prop], excl = type.spec.excludes;
              type.excluded = excl == null ? [type] : excl == "" ? [] : gatherMarks(this, excl.split(" "));
          }
          this.nodeFromJSON = this.nodeFromJSON.bind(this);
          this.markFromJSON = this.markFromJSON.bind(this);
          this.topNodeType = this.nodes[this.spec.topNode || "doc"];
          this.cached.wrappings = Object.create(null);
      }
      /**
      Create a node in this schema. The `type` may be a string or a
      `NodeType` instance. Attributes will be extended with defaults,
      `content` may be a `Fragment`, `null`, a `Node`, or an array of
      nodes.
      */
      node(type, attrs = null, content, marks) {
          if (typeof type == "string")
              type = this.nodeType(type);
          else if (!(type instanceof NodeType$1))
              throw new RangeError("Invalid node type: " + type);
          else if (type.schema != this)
              throw new RangeError("Node type from different schema used (" + type.name + ")");
          return type.createChecked(attrs, content, marks);
      }
      /**
      Create a text node in the schema. Empty text nodes are not
      allowed.
      */
      text(text, marks) {
          let type = this.nodes.text;
          return new TextNode(type, type.defaultAttrs, text, Mark.setFrom(marks));
      }
      /**
      Create a mark with the given type and attributes.
      */
      mark(type, attrs) {
          if (typeof type == "string")
              type = this.marks[type];
          return type.create(attrs);
      }
      /**
      Deserialize a node from its JSON representation. This method is
      bound.
      */
      nodeFromJSON(json) {
          return Node.fromJSON(this, json);
      }
      /**
      Deserialize a mark from its JSON representation. This method is
      bound.
      */
      markFromJSON(json) {
          return Mark.fromJSON(this, json);
      }
      /**
      @internal
      */
      nodeType(name) {
          let found = this.nodes[name];
          if (!found)
              throw new RangeError("Unknown node type: " + name);
          return found;
      }
  }
  function gatherMarks(schema, marks) {
      let found = [];
      for (let i = 0; i < marks.length; i++) {
          let name = marks[i], mark = schema.marks[name], ok = mark;
          if (mark) {
              found.push(mark);
          }
          else {
              for (let prop in schema.marks) {
                  let mark = schema.marks[prop];
                  if (name == "_" || (mark.spec.group && mark.spec.group.split(" ").indexOf(name) > -1))
                      found.push(ok = mark);
              }
          }
          if (!ok)
              throw new SyntaxError("Unknown mark type: '" + marks[i] + "'");
      }
      return found;
  }

  function isTagRule(rule) { return rule.tag != null; }
  function isStyleRule(rule) { return rule.style != null; }
  /**
  A DOM parser represents a strategy for parsing DOM content into a
  ProseMirror document conforming to a given schema. Its behavior is
  defined by an array of [rules](https://prosemirror.net/docs/ref/#model.ParseRule).
  */
  let DOMParser$1 = class DOMParser {
      /**
      Create a parser that targets the given schema, using the given
      parsing rules.
      */
      constructor(
      /**
      The schema into which the parser parses.
      */
      schema, 
      /**
      The set of [parse rules](https://prosemirror.net/docs/ref/#model.ParseRule) that the parser
      uses, in order of precedence.
      */
      rules) {
          this.schema = schema;
          this.rules = rules;
          /**
          @internal
          */
          this.tags = [];
          /**
          @internal
          */
          this.styles = [];
          rules.forEach(rule => {
              if (isTagRule(rule))
                  this.tags.push(rule);
              else if (isStyleRule(rule))
                  this.styles.push(rule);
          });
          // Only normalize list elements when lists in the schema can't directly contain themselves
          this.normalizeLists = !this.tags.some(r => {
              if (!/^(ul|ol)\b/.test(r.tag) || !r.node)
                  return false;
              let node = schema.nodes[r.node];
              return node.contentMatch.matchType(node);
          });
      }
      /**
      Parse a document from the content of a DOM node.
      */
      parse(dom, options = {}) {
          let context = new ParseContext(this, options, false);
          context.addAll(dom, options.from, options.to);
          return context.finish();
      }
      /**
      Parses the content of the given DOM node, like
      [`parse`](https://prosemirror.net/docs/ref/#model.DOMParser.parse), and takes the same set of
      options. But unlike that method, which produces a whole node,
      this one returns a slice that is open at the sides, meaning that
      the schema constraints aren't applied to the start of nodes to
      the left of the input and the end of nodes at the end.
      */
      parseSlice(dom, options = {}) {
          let context = new ParseContext(this, options, true);
          context.addAll(dom, options.from, options.to);
          return Slice.maxOpen(context.finish());
      }
      /**
      @internal
      */
      matchTag(dom, context, after) {
          for (let i = after ? this.tags.indexOf(after) + 1 : 0; i < this.tags.length; i++) {
              let rule = this.tags[i];
              if (matches(dom, rule.tag) &&
                  (rule.namespace === undefined || dom.namespaceURI == rule.namespace) &&
                  (!rule.context || context.matchesContext(rule.context))) {
                  if (rule.getAttrs) {
                      let result = rule.getAttrs(dom);
                      if (result === false)
                          continue;
                      rule.attrs = result || undefined;
                  }
                  return rule;
              }
          }
      }
      /**
      @internal
      */
      matchStyle(prop, value, context, after) {
          for (let i = after ? this.styles.indexOf(after) + 1 : 0; i < this.styles.length; i++) {
              let rule = this.styles[i], style = rule.style;
              if (style.indexOf(prop) != 0 ||
                  rule.context && !context.matchesContext(rule.context) ||
                  // Test that the style string either precisely matches the prop,
                  // or has an '=' sign after the prop, followed by the given
                  // value.
                  style.length > prop.length &&
                      (style.charCodeAt(prop.length) != 61 || style.slice(prop.length + 1) != value))
                  continue;
              if (rule.getAttrs) {
                  let result = rule.getAttrs(value);
                  if (result === false)
                      continue;
                  rule.attrs = result || undefined;
              }
              return rule;
          }
      }
      /**
      @internal
      */
      static schemaRules(schema) {
          let result = [];
          function insert(rule) {
              let priority = rule.priority == null ? 50 : rule.priority, i = 0;
              for (; i < result.length; i++) {
                  let next = result[i], nextPriority = next.priority == null ? 50 : next.priority;
                  if (nextPriority < priority)
                      break;
              }
              result.splice(i, 0, rule);
          }
          for (let name in schema.marks) {
              let rules = schema.marks[name].spec.parseDOM;
              if (rules)
                  rules.forEach(rule => {
                      insert(rule = copy(rule));
                      if (!(rule.mark || rule.ignore || rule.clearMark))
                          rule.mark = name;
                  });
          }
          for (let name in schema.nodes) {
              let rules = schema.nodes[name].spec.parseDOM;
              if (rules)
                  rules.forEach(rule => {
                      insert(rule = copy(rule));
                      if (!(rule.node || rule.ignore || rule.mark))
                          rule.node = name;
                  });
          }
          return result;
      }
      /**
      Construct a DOM parser using the parsing rules listed in a
      schema's [node specs](https://prosemirror.net/docs/ref/#model.NodeSpec.parseDOM), reordered by
      [priority](https://prosemirror.net/docs/ref/#model.ParseRule.priority).
      */
      static fromSchema(schema) {
          return schema.cached.domParser ||
              (schema.cached.domParser = new DOMParser(schema, DOMParser.schemaRules(schema)));
      }
  };
  const blockTags = {
      address: true, article: true, aside: true, blockquote: true, canvas: true,
      dd: true, div: true, dl: true, fieldset: true, figcaption: true, figure: true,
      footer: true, form: true, h1: true, h2: true, h3: true, h4: true, h5: true,
      h6: true, header: true, hgroup: true, hr: true, li: true, noscript: true, ol: true,
      output: true, p: true, pre: true, section: true, table: true, tfoot: true, ul: true
  };
  const ignoreTags = {
      head: true, noscript: true, object: true, script: true, style: true, title: true
  };
  const listTags = { ol: true, ul: true };
  // Using a bitfield for node context options
  const OPT_PRESERVE_WS = 1, OPT_PRESERVE_WS_FULL = 2, OPT_OPEN_LEFT = 4;
  function wsOptionsFor(type, preserveWhitespace, base) {
      if (preserveWhitespace != null)
          return (preserveWhitespace ? OPT_PRESERVE_WS : 0) |
              (preserveWhitespace === "full" ? OPT_PRESERVE_WS_FULL : 0);
      return type && type.whitespace == "pre" ? OPT_PRESERVE_WS | OPT_PRESERVE_WS_FULL : base & ~OPT_OPEN_LEFT;
  }
  class NodeContext {
      constructor(type, attrs, 
      // Marks applied to this node itself
      marks, 
      // Marks that can't apply here, but will be used in children if possible
      pendingMarks, solid, match, options) {
          this.type = type;
          this.attrs = attrs;
          this.marks = marks;
          this.pendingMarks = pendingMarks;
          this.solid = solid;
          this.options = options;
          this.content = [];
          // Marks applied to the node's children
          this.activeMarks = Mark.none;
          // Nested Marks with same type
          this.stashMarks = [];
          this.match = match || (options & OPT_OPEN_LEFT ? null : type.contentMatch);
      }
      findWrapping(node) {
          if (!this.match) {
              if (!this.type)
                  return [];
              let fill = this.type.contentMatch.fillBefore(Fragment.from(node));
              if (fill) {
                  this.match = this.type.contentMatch.matchFragment(fill);
              }
              else {
                  let start = this.type.contentMatch, wrap;
                  if (wrap = start.findWrapping(node.type)) {
                      this.match = start;
                      return wrap;
                  }
                  else {
                      return null;
                  }
              }
          }
          return this.match.findWrapping(node.type);
      }
      finish(openEnd) {
          if (!(this.options & OPT_PRESERVE_WS)) { // Strip trailing whitespace
              let last = this.content[this.content.length - 1], m;
              if (last && last.isText && (m = /[ \t\r\n\u000c]+$/.exec(last.text))) {
                  let text = last;
                  if (last.text.length == m[0].length)
                      this.content.pop();
                  else
                      this.content[this.content.length - 1] = text.withText(text.text.slice(0, text.text.length - m[0].length));
              }
          }
          let content = Fragment.from(this.content);
          if (!openEnd && this.match)
              content = content.append(this.match.fillBefore(Fragment.empty, true));
          return this.type ? this.type.create(this.attrs, content, this.marks) : content;
      }
      popFromStashMark(mark) {
          for (let i = this.stashMarks.length - 1; i >= 0; i--)
              if (mark.eq(this.stashMarks[i]))
                  return this.stashMarks.splice(i, 1)[0];
      }
      applyPending(nextType) {
          for (let i = 0, pending = this.pendingMarks; i < pending.length; i++) {
              let mark = pending[i];
              if ((this.type ? this.type.allowsMarkType(mark.type) : markMayApply(mark.type, nextType)) &&
                  !mark.isInSet(this.activeMarks)) {
                  this.activeMarks = mark.addToSet(this.activeMarks);
                  this.pendingMarks = mark.removeFromSet(this.pendingMarks);
              }
          }
      }
      inlineContext(node) {
          if (this.type)
              return this.type.inlineContent;
          if (this.content.length)
              return this.content[0].isInline;
          return node.parentNode && !blockTags.hasOwnProperty(node.parentNode.nodeName.toLowerCase());
      }
  }
  class ParseContext {
      constructor(
      // The parser we are using.
      parser, 
      // The options passed to this parse.
      options, isOpen) {
          this.parser = parser;
          this.options = options;
          this.isOpen = isOpen;
          this.open = 0;
          let topNode = options.topNode, topContext;
          let topOptions = wsOptionsFor(null, options.preserveWhitespace, 0) | (isOpen ? OPT_OPEN_LEFT : 0);
          if (topNode)
              topContext = new NodeContext(topNode.type, topNode.attrs, Mark.none, Mark.none, true, options.topMatch || topNode.type.contentMatch, topOptions);
          else if (isOpen)
              topContext = new NodeContext(null, null, Mark.none, Mark.none, true, null, topOptions);
          else
              topContext = new NodeContext(parser.schema.topNodeType, null, Mark.none, Mark.none, true, null, topOptions);
          this.nodes = [topContext];
          this.find = options.findPositions;
          this.needsBlock = false;
      }
      get top() {
          return this.nodes[this.open];
      }
      // Add a DOM node to the content. Text is inserted as text node,
      // otherwise, the node is passed to `addElement` or, if it has a
      // `style` attribute, `addElementWithStyles`.
      addDOM(dom) {
          if (dom.nodeType == 3)
              this.addTextNode(dom);
          else if (dom.nodeType == 1)
              this.addElement(dom);
      }
      withStyleRules(dom, f) {
          let style = dom.getAttribute("style");
          if (!style)
              return f();
          let marks = this.readStyles(parseStyles(style));
          if (!marks)
              return; // A style with ignore: true
          let [addMarks, removeMarks] = marks, top = this.top;
          for (let i = 0; i < removeMarks.length; i++)
              this.removePendingMark(removeMarks[i], top);
          for (let i = 0; i < addMarks.length; i++)
              this.addPendingMark(addMarks[i]);
          f();
          for (let i = 0; i < addMarks.length; i++)
              this.removePendingMark(addMarks[i], top);
          for (let i = 0; i < removeMarks.length; i++)
              this.addPendingMark(removeMarks[i]);
      }
      addTextNode(dom) {
          let value = dom.nodeValue;
          let top = this.top;
          if (top.options & OPT_PRESERVE_WS_FULL ||
              top.inlineContext(dom) ||
              /[^ \t\r\n\u000c]/.test(value)) {
              if (!(top.options & OPT_PRESERVE_WS)) {
                  value = value.replace(/[ \t\r\n\u000c]+/g, " ");
                  // If this starts with whitespace, and there is no node before it, or
                  // a hard break, or a text node that ends with whitespace, strip the
                  // leading space.
                  if (/^[ \t\r\n\u000c]/.test(value) && this.open == this.nodes.length - 1) {
                      let nodeBefore = top.content[top.content.length - 1];
                      let domNodeBefore = dom.previousSibling;
                      if (!nodeBefore ||
                          (domNodeBefore && domNodeBefore.nodeName == 'BR') ||
                          (nodeBefore.isText && /[ \t\r\n\u000c]$/.test(nodeBefore.text)))
                          value = value.slice(1);
                  }
              }
              else if (!(top.options & OPT_PRESERVE_WS_FULL)) {
                  value = value.replace(/\r?\n|\r/g, " ");
              }
              else {
                  value = value.replace(/\r\n?/g, "\n");
              }
              if (value)
                  this.insertNode(this.parser.schema.text(value));
              this.findInText(dom);
          }
          else {
              this.findInside(dom);
          }
      }
      // Try to find a handler for the given tag and use that to parse. If
      // none is found, the element's content nodes are added directly.
      addElement(dom, matchAfter) {
          let name = dom.nodeName.toLowerCase(), ruleID;
          if (listTags.hasOwnProperty(name) && this.parser.normalizeLists)
              normalizeList(dom);
          let rule = (this.options.ruleFromNode && this.options.ruleFromNode(dom)) ||
              (ruleID = this.parser.matchTag(dom, this, matchAfter));
          if (rule ? rule.ignore : ignoreTags.hasOwnProperty(name)) {
              this.findInside(dom);
              this.ignoreFallback(dom);
          }
          else if (!rule || rule.skip || rule.closeParent) {
              if (rule && rule.closeParent)
                  this.open = Math.max(0, this.open - 1);
              else if (rule && rule.skip.nodeType)
                  dom = rule.skip;
              let sync, top = this.top, oldNeedsBlock = this.needsBlock;
              if (blockTags.hasOwnProperty(name)) {
                  if (top.content.length && top.content[0].isInline && this.open) {
                      this.open--;
                      top = this.top;
                  }
                  sync = true;
                  if (!top.type)
                      this.needsBlock = true;
              }
              else if (!dom.firstChild) {
                  this.leafFallback(dom);
                  return;
              }
              if (rule && rule.skip)
                  this.addAll(dom);
              else
                  this.withStyleRules(dom, () => this.addAll(dom));
              if (sync)
                  this.sync(top);
              this.needsBlock = oldNeedsBlock;
          }
          else {
              this.withStyleRules(dom, () => {
                  this.addElementByRule(dom, rule, rule.consuming === false ? ruleID : undefined);
              });
          }
      }
      // Called for leaf DOM nodes that would otherwise be ignored
      leafFallback(dom) {
          if (dom.nodeName == "BR" && this.top.type && this.top.type.inlineContent)
              this.addTextNode(dom.ownerDocument.createTextNode("\n"));
      }
      // Called for ignored nodes
      ignoreFallback(dom) {
          // Ignored BR nodes should at least create an inline context
          if (dom.nodeName == "BR" && (!this.top.type || !this.top.type.inlineContent))
              this.findPlace(this.parser.schema.text("-"));
      }
      // Run any style parser associated with the node's styles. Either
      // return an array of marks, or null to indicate some of the styles
      // had a rule with `ignore` set.
      readStyles(styles) {
          let add = Mark.none, remove = Mark.none;
          for (let i = 0; i < styles.length; i += 2) {
              for (let after = undefined;;) {
                  let rule = this.parser.matchStyle(styles[i], styles[i + 1], this, after);
                  if (!rule)
                      break;
                  if (rule.ignore)
                      return null;
                  if (rule.clearMark) {
                      this.top.pendingMarks.concat(this.top.activeMarks).forEach(m => {
                          if (rule.clearMark(m))
                              remove = m.addToSet(remove);
                      });
                  }
                  else {
                      add = this.parser.schema.marks[rule.mark].create(rule.attrs).addToSet(add);
                  }
                  if (rule.consuming === false)
                      after = rule;
                  else
                      break;
              }
          }
          return [add, remove];
      }
      // Look up a handler for the given node. If none are found, return
      // false. Otherwise, apply it, use its return value to drive the way
      // the node's content is wrapped, and return true.
      addElementByRule(dom, rule, continueAfter) {
          let sync, nodeType, mark;
          if (rule.node) {
              nodeType = this.parser.schema.nodes[rule.node];
              if (!nodeType.isLeaf) {
                  sync = this.enter(nodeType, rule.attrs || null, rule.preserveWhitespace);
              }
              else if (!this.insertNode(nodeType.create(rule.attrs))) {
                  this.leafFallback(dom);
              }
          }
          else {
              let markType = this.parser.schema.marks[rule.mark];
              mark = markType.create(rule.attrs);
              this.addPendingMark(mark);
          }
          let startIn = this.top;
          if (nodeType && nodeType.isLeaf) {
              this.findInside(dom);
          }
          else if (continueAfter) {
              this.addElement(dom, continueAfter);
          }
          else if (rule.getContent) {
              this.findInside(dom);
              rule.getContent(dom, this.parser.schema).forEach(node => this.insertNode(node));
          }
          else {
              let contentDOM = dom;
              if (typeof rule.contentElement == "string")
                  contentDOM = dom.querySelector(rule.contentElement);
              else if (typeof rule.contentElement == "function")
                  contentDOM = rule.contentElement(dom);
              else if (rule.contentElement)
                  contentDOM = rule.contentElement;
              this.findAround(dom, contentDOM, true);
              this.addAll(contentDOM);
          }
          if (sync && this.sync(startIn))
              this.open--;
          if (mark)
              this.removePendingMark(mark, startIn);
      }
      // Add all child nodes between `startIndex` and `endIndex` (or the
      // whole node, if not given). If `sync` is passed, use it to
      // synchronize after every block element.
      addAll(parent, startIndex, endIndex) {
          let index = startIndex || 0;
          for (let dom = startIndex ? parent.childNodes[startIndex] : parent.firstChild, end = endIndex == null ? null : parent.childNodes[endIndex]; dom != end; dom = dom.nextSibling, ++index) {
              this.findAtPoint(parent, index);
              this.addDOM(dom);
          }
          this.findAtPoint(parent, index);
      }
      // Try to find a way to fit the given node type into the current
      // context. May add intermediate wrappers and/or leave non-solid
      // nodes that we're in.
      findPlace(node) {
          let route, sync;
          for (let depth = this.open; depth >= 0; depth--) {
              let cx = this.nodes[depth];
              let found = cx.findWrapping(node);
              if (found && (!route || route.length > found.length)) {
                  route = found;
                  sync = cx;
                  if (!found.length)
                      break;
              }
              if (cx.solid)
                  break;
          }
          if (!route)
              return false;
          this.sync(sync);
          for (let i = 0; i < route.length; i++)
              this.enterInner(route[i], null, false);
          return true;
      }
      // Try to insert the given node, adjusting the context when needed.
      insertNode(node) {
          if (node.isInline && this.needsBlock && !this.top.type) {
              let block = this.textblockFromContext();
              if (block)
                  this.enterInner(block);
          }
          if (this.findPlace(node)) {
              this.closeExtra();
              let top = this.top;
              top.applyPending(node.type);
              if (top.match)
                  top.match = top.match.matchType(node.type);
              let marks = top.activeMarks;
              for (let i = 0; i < node.marks.length; i++)
                  if (!top.type || top.type.allowsMarkType(node.marks[i].type))
                      marks = node.marks[i].addToSet(marks);
              top.content.push(node.mark(marks));
              return true;
          }
          return false;
      }
      // Try to start a node of the given type, adjusting the context when
      // necessary.
      enter(type, attrs, preserveWS) {
          let ok = this.findPlace(type.create(attrs));
          if (ok)
              this.enterInner(type, attrs, true, preserveWS);
          return ok;
      }
      // Open a node of the given type
      enterInner(type, attrs = null, solid = false, preserveWS) {
          this.closeExtra();
          let top = this.top;
          top.applyPending(type);
          top.match = top.match && top.match.matchType(type);
          let options = wsOptionsFor(type, preserveWS, top.options);
          if ((top.options & OPT_OPEN_LEFT) && top.content.length == 0)
              options |= OPT_OPEN_LEFT;
          this.nodes.push(new NodeContext(type, attrs, top.activeMarks, top.pendingMarks, solid, null, options));
          this.open++;
      }
      // Make sure all nodes above this.open are finished and added to
      // their parents
      closeExtra(openEnd = false) {
          let i = this.nodes.length - 1;
          if (i > this.open) {
              for (; i > this.open; i--)
                  this.nodes[i - 1].content.push(this.nodes[i].finish(openEnd));
              this.nodes.length = this.open + 1;
          }
      }
      finish() {
          this.open = 0;
          this.closeExtra(this.isOpen);
          return this.nodes[0].finish(this.isOpen || this.options.topOpen);
      }
      sync(to) {
          for (let i = this.open; i >= 0; i--)
              if (this.nodes[i] == to) {
                  this.open = i;
                  return true;
              }
          return false;
      }
      get currentPos() {
          this.closeExtra();
          let pos = 0;
          for (let i = this.open; i >= 0; i--) {
              let content = this.nodes[i].content;
              for (let j = content.length - 1; j >= 0; j--)
                  pos += content[j].nodeSize;
              if (i)
                  pos++;
          }
          return pos;
      }
      findAtPoint(parent, offset) {
          if (this.find)
              for (let i = 0; i < this.find.length; i++) {
                  if (this.find[i].node == parent && this.find[i].offset == offset)
                      this.find[i].pos = this.currentPos;
              }
      }
      findInside(parent) {
          if (this.find)
              for (let i = 0; i < this.find.length; i++) {
                  if (this.find[i].pos == null && parent.nodeType == 1 && parent.contains(this.find[i].node))
                      this.find[i].pos = this.currentPos;
              }
      }
      findAround(parent, content, before) {
          if (parent != content && this.find)
              for (let i = 0; i < this.find.length; i++) {
                  if (this.find[i].pos == null && parent.nodeType == 1 && parent.contains(this.find[i].node)) {
                      let pos = content.compareDocumentPosition(this.find[i].node);
                      if (pos & (before ? 2 : 4))
                          this.find[i].pos = this.currentPos;
                  }
              }
      }
      findInText(textNode) {
          if (this.find)
              for (let i = 0; i < this.find.length; i++) {
                  if (this.find[i].node == textNode)
                      this.find[i].pos = this.currentPos - (textNode.nodeValue.length - this.find[i].offset);
              }
      }
      // Determines whether the given context string matches this context.
      matchesContext(context) {
          if (context.indexOf("|") > -1)
              return context.split(/\s*\|\s*/).some(this.matchesContext, this);
          let parts = context.split("/");
          let option = this.options.context;
          let useRoot = !this.isOpen && (!option || option.parent.type == this.nodes[0].type);
          let minDepth = -(option ? option.depth + 1 : 0) + (useRoot ? 0 : 1);
          let match = (i, depth) => {
              for (; i >= 0; i--) {
                  let part = parts[i];
                  if (part == "") {
                      if (i == parts.length - 1 || i == 0)
                          continue;
                      for (; depth >= minDepth; depth--)
                          if (match(i - 1, depth))
                              return true;
                      return false;
                  }
                  else {
                      let next = depth > 0 || (depth == 0 && useRoot) ? this.nodes[depth].type
                          : option && depth >= minDepth ? option.node(depth - minDepth).type
                              : null;
                      if (!next || (next.name != part && next.groups.indexOf(part) == -1))
                          return false;
                      depth--;
                  }
              }
              return true;
          };
          return match(parts.length - 1, this.open);
      }
      textblockFromContext() {
          let $context = this.options.context;
          if ($context)
              for (let d = $context.depth; d >= 0; d--) {
                  let deflt = $context.node(d).contentMatchAt($context.indexAfter(d)).defaultType;
                  if (deflt && deflt.isTextblock && deflt.defaultAttrs)
                      return deflt;
              }
          for (let name in this.parser.schema.nodes) {
              let type = this.parser.schema.nodes[name];
              if (type.isTextblock && type.defaultAttrs)
                  return type;
          }
      }
      addPendingMark(mark) {
          let found = findSameMarkInSet(mark, this.top.pendingMarks);
          if (found)
              this.top.stashMarks.push(found);
          this.top.pendingMarks = mark.addToSet(this.top.pendingMarks);
      }
      removePendingMark(mark, upto) {
          for (let depth = this.open; depth >= 0; depth--) {
              let level = this.nodes[depth];
              let found = level.pendingMarks.lastIndexOf(mark);
              if (found > -1) {
                  level.pendingMarks = mark.removeFromSet(level.pendingMarks);
              }
              else {
                  level.activeMarks = mark.removeFromSet(level.activeMarks);
                  let stashMark = level.popFromStashMark(mark);
                  if (stashMark && level.type && level.type.allowsMarkType(stashMark.type))
                      level.activeMarks = stashMark.addToSet(level.activeMarks);
              }
              if (level == upto)
                  break;
          }
      }
  }
  // Kludge to work around directly nested list nodes produced by some
  // tools and allowed by browsers to mean that the nested list is
  // actually part of the list item above it.
  function normalizeList(dom) {
      for (let child = dom.firstChild, prevItem = null; child; child = child.nextSibling) {
          let name = child.nodeType == 1 ? child.nodeName.toLowerCase() : null;
          if (name && listTags.hasOwnProperty(name) && prevItem) {
              prevItem.appendChild(child);
              child = prevItem;
          }
          else if (name == "li") {
              prevItem = child;
          }
          else if (name) {
              prevItem = null;
          }
      }
  }
  // Apply a CSS selector.
  function matches(dom, selector) {
      return (dom.matches || dom.msMatchesSelector || dom.webkitMatchesSelector || dom.mozMatchesSelector).call(dom, selector);
  }
  // Tokenize a style attribute into property/value pairs.
  function parseStyles(style) {
      let re = /\s*([\w-]+)\s*:\s*([^;]+)/g, m, result = [];
      while (m = re.exec(style))
          result.push(m[1], m[2].trim());
      return result;
  }
  function copy(obj) {
      let copy = {};
      for (let prop in obj)
          copy[prop] = obj[prop];
      return copy;
  }
  // Used when finding a mark at the top level of a fragment parse.
  // Checks whether it would be reasonable to apply a given mark type to
  // a given node, by looking at the way the mark occurs in the schema.
  function markMayApply(markType, nodeType) {
      let nodes = nodeType.schema.nodes;
      for (let name in nodes) {
          let parent = nodes[name];
          if (!parent.allowsMarkType(markType))
              continue;
          let seen = [], scan = (match) => {
              seen.push(match);
              for (let i = 0; i < match.edgeCount; i++) {
                  let { type, next } = match.edge(i);
                  if (type == nodeType)
                      return true;
                  if (seen.indexOf(next) < 0 && scan(next))
                      return true;
              }
          };
          if (scan(parent.contentMatch))
              return true;
      }
  }
  function findSameMarkInSet(mark, set) {
      for (let i = 0; i < set.length; i++) {
          if (mark.eq(set[i]))
              return set[i];
      }
  }

  /**
  A DOM serializer knows how to convert ProseMirror nodes and
  marks of various types to DOM nodes.
  */
  class DOMSerializer {
      /**
      Create a serializer. `nodes` should map node names to functions
      that take a node and return a description of the corresponding
      DOM. `marks` does the same for mark names, but also gets an
      argument that tells it whether the mark's content is block or
      inline content (for typical use, it'll always be inline). A mark
      serializer may be `null` to indicate that marks of that type
      should not be serialized.
      */
      constructor(
      /**
      The node serialization functions.
      */
      nodes, 
      /**
      The mark serialization functions.
      */
      marks) {
          this.nodes = nodes;
          this.marks = marks;
      }
      /**
      Serialize the content of this fragment to a DOM fragment. When
      not in the browser, the `document` option, containing a DOM
      document, should be passed so that the serializer can create
      nodes.
      */
      serializeFragment(fragment, options = {}, target) {
          if (!target)
              target = doc$2(options).createDocumentFragment();
          let top = target, active = [];
          fragment.forEach(node => {
              if (active.length || node.marks.length) {
                  let keep = 0, rendered = 0;
                  while (keep < active.length && rendered < node.marks.length) {
                      let next = node.marks[rendered];
                      if (!this.marks[next.type.name]) {
                          rendered++;
                          continue;
                      }
                      if (!next.eq(active[keep][0]) || next.type.spec.spanning === false)
                          break;
                      keep++;
                      rendered++;
                  }
                  while (keep < active.length)
                      top = active.pop()[1];
                  while (rendered < node.marks.length) {
                      let add = node.marks[rendered++];
                      let markDOM = this.serializeMark(add, node.isInline, options);
                      if (markDOM) {
                          active.push([add, top]);
                          top.appendChild(markDOM.dom);
                          top = markDOM.contentDOM || markDOM.dom;
                      }
                  }
              }
              top.appendChild(this.serializeNodeInner(node, options));
          });
          return target;
      }
      /**
      @internal
      */
      serializeNodeInner(node, options) {
          let { dom, contentDOM } = DOMSerializer.renderSpec(doc$2(options), this.nodes[node.type.name](node));
          if (contentDOM) {
              if (node.isLeaf)
                  throw new RangeError("Content hole not allowed in a leaf node spec");
              this.serializeFragment(node.content, options, contentDOM);
          }
          return dom;
      }
      /**
      Serialize this node to a DOM node. This can be useful when you
      need to serialize a part of a document, as opposed to the whole
      document. To serialize a whole document, use
      [`serializeFragment`](https://prosemirror.net/docs/ref/#model.DOMSerializer.serializeFragment) on
      its [content](https://prosemirror.net/docs/ref/#model.Node.content).
      */
      serializeNode(node, options = {}) {
          let dom = this.serializeNodeInner(node, options);
          for (let i = node.marks.length - 1; i >= 0; i--) {
              let wrap = this.serializeMark(node.marks[i], node.isInline, options);
              if (wrap) {
                  (wrap.contentDOM || wrap.dom).appendChild(dom);
                  dom = wrap.dom;
              }
          }
          return dom;
      }
      /**
      @internal
      */
      serializeMark(mark, inline, options = {}) {
          let toDOM = this.marks[mark.type.name];
          return toDOM && DOMSerializer.renderSpec(doc$2(options), toDOM(mark, inline));
      }
      /**
      Render an [output spec](https://prosemirror.net/docs/ref/#model.DOMOutputSpec) to a DOM node. If
      the spec has a hole (zero) in it, `contentDOM` will point at the
      node with the hole.
      */
      static renderSpec(doc, structure, xmlNS = null) {
          if (typeof structure == "string")
              return { dom: doc.createTextNode(structure) };
          if (structure.nodeType != null)
              return { dom: structure };
          if (structure.dom && structure.dom.nodeType != null)
              return structure;
          let tagName = structure[0], space = tagName.indexOf(" ");
          if (space > 0) {
              xmlNS = tagName.slice(0, space);
              tagName = tagName.slice(space + 1);
          }
          let contentDOM;
          let dom = (xmlNS ? doc.createElementNS(xmlNS, tagName) : doc.createElement(tagName));
          let attrs = structure[1], start = 1;
          if (attrs && typeof attrs == "object" && attrs.nodeType == null && !Array.isArray(attrs)) {
              start = 2;
              for (let name in attrs)
                  if (attrs[name] != null) {
                      let space = name.indexOf(" ");
                      if (space > 0)
                          dom.setAttributeNS(name.slice(0, space), name.slice(space + 1), attrs[name]);
                      else
                          dom.setAttribute(name, attrs[name]);
                  }
          }
          for (let i = start; i < structure.length; i++) {
              let child = structure[i];
              if (child === 0) {
                  if (i < structure.length - 1 || i > start)
                      throw new RangeError("Content hole must be the only child of its parent node");
                  return { dom, contentDOM: dom };
              }
              else {
                  let { dom: inner, contentDOM: innerContent } = DOMSerializer.renderSpec(doc, child, xmlNS);
                  dom.appendChild(inner);
                  if (innerContent) {
                      if (contentDOM)
                          throw new RangeError("Multiple content holes");
                      contentDOM = innerContent;
                  }
              }
          }
          return { dom, contentDOM };
      }
      /**
      Build a serializer using the [`toDOM`](https://prosemirror.net/docs/ref/#model.NodeSpec.toDOM)
      properties in a schema's node and mark specs.
      */
      static fromSchema(schema) {
          return schema.cached.domSerializer ||
              (schema.cached.domSerializer = new DOMSerializer(this.nodesFromSchema(schema), this.marksFromSchema(schema)));
      }
      /**
      Gather the serializers in a schema's node specs into an object.
      This can be useful as a base to build a custom serializer from.
      */
      static nodesFromSchema(schema) {
          let result = gatherToDOM(schema.nodes);
          if (!result.text)
              result.text = node => node.text;
          return result;
      }
      /**
      Gather the serializers in a schema's mark specs into an object.
      */
      static marksFromSchema(schema) {
          return gatherToDOM(schema.marks);
      }
  }
  function gatherToDOM(obj) {
      let result = {};
      for (let name in obj) {
          let toDOM = obj[name].spec.toDOM;
          if (toDOM)
              result[name] = toDOM;
      }
      return result;
  }
  function doc$2(options) {
      return options.document || window.document;
  }

  // Recovery values encode a range index and an offset. They are
  // represented as numbers, because tons of them will be created when
  // mapping, for example, a large number of decorations. The number's
  // lower 16 bits provide the index, the remaining bits the offset.
  //
  // Note: We intentionally don't use bit shift operators to en- and
  // decode these, since those clip to 32 bits, which we might in rare
  // cases want to overflow. A 64-bit float can represent 48-bit
  // integers precisely.
  const lower16 = 0xffff;
  const factor16 = Math.pow(2, 16);
  function makeRecover(index, offset) { return index + offset * factor16; }
  function recoverIndex(value) { return value & lower16; }
  function recoverOffset(value) { return (value - (value & lower16)) / factor16; }
  const DEL_BEFORE = 1, DEL_AFTER = 2, DEL_ACROSS = 4, DEL_SIDE = 8;
  /**
  An object representing a mapped position with extra
  information.
  */
  class MapResult {
      /**
      @internal
      */
      constructor(
      /**
      The mapped version of the position.
      */
      pos, 
      /**
      @internal
      */
      delInfo, 
      /**
      @internal
      */
      recover) {
          this.pos = pos;
          this.delInfo = delInfo;
          this.recover = recover;
      }
      /**
      Tells you whether the position was deleted, that is, whether the
      step removed the token on the side queried (via the `assoc`)
      argument from the document.
      */
      get deleted() { return (this.delInfo & DEL_SIDE) > 0; }
      /**
      Tells you whether the token before the mapped position was deleted.
      */
      get deletedBefore() { return (this.delInfo & (DEL_BEFORE | DEL_ACROSS)) > 0; }
      /**
      True when the token after the mapped position was deleted.
      */
      get deletedAfter() { return (this.delInfo & (DEL_AFTER | DEL_ACROSS)) > 0; }
      /**
      Tells whether any of the steps mapped through deletes across the
      position (including both the token before and after the
      position).
      */
      get deletedAcross() { return (this.delInfo & DEL_ACROSS) > 0; }
  }
  /**
  A map describing the deletions and insertions made by a step, which
  can be used to find the correspondence between positions in the
  pre-step version of a document and the same position in the
  post-step version.
  */
  class StepMap {
      /**
      Create a position map. The modifications to the document are
      represented as an array of numbers, in which each group of three
      represents a modified chunk as `[start, oldSize, newSize]`.
      */
      constructor(
      /**
      @internal
      */
      ranges, 
      /**
      @internal
      */
      inverted = false) {
          this.ranges = ranges;
          this.inverted = inverted;
          if (!ranges.length && StepMap.empty)
              return StepMap.empty;
      }
      /**
      @internal
      */
      recover(value) {
          let diff = 0, index = recoverIndex(value);
          if (!this.inverted)
              for (let i = 0; i < index; i++)
                  diff += this.ranges[i * 3 + 2] - this.ranges[i * 3 + 1];
          return this.ranges[index * 3] + diff + recoverOffset(value);
      }
      mapResult(pos, assoc = 1) { return this._map(pos, assoc, false); }
      map(pos, assoc = 1) { return this._map(pos, assoc, true); }
      /**
      @internal
      */
      _map(pos, assoc, simple) {
          let diff = 0, oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
          for (let i = 0; i < this.ranges.length; i += 3) {
              let start = this.ranges[i] - (this.inverted ? diff : 0);
              if (start > pos)
                  break;
              let oldSize = this.ranges[i + oldIndex], newSize = this.ranges[i + newIndex], end = start + oldSize;
              if (pos <= end) {
                  let side = !oldSize ? assoc : pos == start ? -1 : pos == end ? 1 : assoc;
                  let result = start + diff + (side < 0 ? 0 : newSize);
                  if (simple)
                      return result;
                  let recover = pos == (assoc < 0 ? start : end) ? null : makeRecover(i / 3, pos - start);
                  let del = pos == start ? DEL_AFTER : pos == end ? DEL_BEFORE : DEL_ACROSS;
                  if (assoc < 0 ? pos != start : pos != end)
                      del |= DEL_SIDE;
                  return new MapResult(result, del, recover);
              }
              diff += newSize - oldSize;
          }
          return simple ? pos + diff : new MapResult(pos + diff, 0, null);
      }
      /**
      @internal
      */
      touches(pos, recover) {
          let diff = 0, index = recoverIndex(recover);
          let oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
          for (let i = 0; i < this.ranges.length; i += 3) {
              let start = this.ranges[i] - (this.inverted ? diff : 0);
              if (start > pos)
                  break;
              let oldSize = this.ranges[i + oldIndex], end = start + oldSize;
              if (pos <= end && i == index * 3)
                  return true;
              diff += this.ranges[i + newIndex] - oldSize;
          }
          return false;
      }
      /**
      Calls the given function on each of the changed ranges included in
      this map.
      */
      forEach(f) {
          let oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
          for (let i = 0, diff = 0; i < this.ranges.length; i += 3) {
              let start = this.ranges[i], oldStart = start - (this.inverted ? diff : 0), newStart = start + (this.inverted ? 0 : diff);
              let oldSize = this.ranges[i + oldIndex], newSize = this.ranges[i + newIndex];
              f(oldStart, oldStart + oldSize, newStart, newStart + newSize);
              diff += newSize - oldSize;
          }
      }
      /**
      Create an inverted version of this map. The result can be used to
      map positions in the post-step document to the pre-step document.
      */
      invert() {
          return new StepMap(this.ranges, !this.inverted);
      }
      /**
      @internal
      */
      toString() {
          return (this.inverted ? "-" : "") + JSON.stringify(this.ranges);
      }
      /**
      Create a map that moves all positions by offset `n` (which may be
      negative). This can be useful when applying steps meant for a
      sub-document to a larger document, or vice-versa.
      */
      static offset(n) {
          return n == 0 ? StepMap.empty : new StepMap(n < 0 ? [0, -n, 0] : [0, 0, n]);
      }
  }
  /**
  A StepMap that contains no changed ranges.
  */
  StepMap.empty = new StepMap([]);
  /**
  A mapping represents a pipeline of zero or more [step
  maps](https://prosemirror.net/docs/ref/#transform.StepMap). It has special provisions for losslessly
  handling mapping positions through a series of steps in which some
  steps are inverted versions of earlier steps. (This comes up when
  ‘[rebasing](/docs/guide/#transform.rebasing)’ steps for
  collaboration or history management.)
  */
  class Mapping {
      /**
      Create a new mapping with the given position maps.
      */
      constructor(
      /**
      The step maps in this mapping.
      */
      maps = [], 
      /**
      @internal
      */
      mirror, 
      /**
      The starting position in the `maps` array, used when `map` or
      `mapResult` is called.
      */
      from = 0, 
      /**
      The end position in the `maps` array.
      */
      to = maps.length) {
          this.maps = maps;
          this.mirror = mirror;
          this.from = from;
          this.to = to;
      }
      /**
      Create a mapping that maps only through a part of this one.
      */
      slice(from = 0, to = this.maps.length) {
          return new Mapping(this.maps, this.mirror, from, to);
      }
      /**
      @internal
      */
      copy() {
          return new Mapping(this.maps.slice(), this.mirror && this.mirror.slice(), this.from, this.to);
      }
      /**
      Add a step map to the end of this mapping. If `mirrors` is
      given, it should be the index of the step map that is the mirror
      image of this one.
      */
      appendMap(map, mirrors) {
          this.to = this.maps.push(map);
          if (mirrors != null)
              this.setMirror(this.maps.length - 1, mirrors);
      }
      /**
      Add all the step maps in a given mapping to this one (preserving
      mirroring information).
      */
      appendMapping(mapping) {
          for (let i = 0, startSize = this.maps.length; i < mapping.maps.length; i++) {
              let mirr = mapping.getMirror(i);
              this.appendMap(mapping.maps[i], mirr != null && mirr < i ? startSize + mirr : undefined);
          }
      }
      /**
      Finds the offset of the step map that mirrors the map at the
      given offset, in this mapping (as per the second argument to
      `appendMap`).
      */
      getMirror(n) {
          if (this.mirror)
              for (let i = 0; i < this.mirror.length; i++)
                  if (this.mirror[i] == n)
                      return this.mirror[i + (i % 2 ? -1 : 1)];
      }
      /**
      @internal
      */
      setMirror(n, m) {
          if (!this.mirror)
              this.mirror = [];
          this.mirror.push(n, m);
      }
      /**
      Append the inverse of the given mapping to this one.
      */
      appendMappingInverted(mapping) {
          for (let i = mapping.maps.length - 1, totalSize = this.maps.length + mapping.maps.length; i >= 0; i--) {
              let mirr = mapping.getMirror(i);
              this.appendMap(mapping.maps[i].invert(), mirr != null && mirr > i ? totalSize - mirr - 1 : undefined);
          }
      }
      /**
      Create an inverted version of this mapping.
      */
      invert() {
          let inverse = new Mapping;
          inverse.appendMappingInverted(this);
          return inverse;
      }
      /**
      Map a position through this mapping.
      */
      map(pos, assoc = 1) {
          if (this.mirror)
              return this._map(pos, assoc, true);
          for (let i = this.from; i < this.to; i++)
              pos = this.maps[i].map(pos, assoc);
          return pos;
      }
      /**
      Map a position through this mapping, returning a mapping
      result.
      */
      mapResult(pos, assoc = 1) { return this._map(pos, assoc, false); }
      /**
      @internal
      */
      _map(pos, assoc, simple) {
          let delInfo = 0;
          for (let i = this.from; i < this.to; i++) {
              let map = this.maps[i], result = map.mapResult(pos, assoc);
              if (result.recover != null) {
                  let corr = this.getMirror(i);
                  if (corr != null && corr > i && corr < this.to) {
                      i = corr;
                      pos = this.maps[corr].recover(result.recover);
                      continue;
                  }
              }
              delInfo |= result.delInfo;
              pos = result.pos;
          }
          return simple ? pos : new MapResult(pos, delInfo, null);
      }
  }

  const stepsByID = Object.create(null);
  /**
  A step object represents an atomic change. It generally applies
  only to the document it was created for, since the positions
  stored in it will only make sense for that document.

  New steps are defined by creating classes that extend `Step`,
  overriding the `apply`, `invert`, `map`, `getMap` and `fromJSON`
  methods, and registering your class with a unique
  JSON-serialization identifier using
  [`Step.jsonID`](https://prosemirror.net/docs/ref/#transform.Step^jsonID).
  */
  class Step {
      /**
      Get the step map that represents the changes made by this step,
      and which can be used to transform between positions in the old
      and the new document.
      */
      getMap() { return StepMap.empty; }
      /**
      Try to merge this step with another one, to be applied directly
      after it. Returns the merged step when possible, null if the
      steps can't be merged.
      */
      merge(other) { return null; }
      /**
      Deserialize a step from its JSON representation. Will call
      through to the step class' own implementation of this method.
      */
      static fromJSON(schema, json) {
          if (!json || !json.stepType)
              throw new RangeError("Invalid input for Step.fromJSON");
          let type = stepsByID[json.stepType];
          if (!type)
              throw new RangeError(`No step type ${json.stepType} defined`);
          return type.fromJSON(schema, json);
      }
      /**
      To be able to serialize steps to JSON, each step needs a string
      ID to attach to its JSON representation. Use this method to
      register an ID for your step classes. Try to pick something
      that's unlikely to clash with steps from other modules.
      */
      static jsonID(id, stepClass) {
          if (id in stepsByID)
              throw new RangeError("Duplicate use of step JSON ID " + id);
          stepsByID[id] = stepClass;
          stepClass.prototype.jsonID = id;
          return stepClass;
      }
  }
  /**
  The result of [applying](https://prosemirror.net/docs/ref/#transform.Step.apply) a step. Contains either a
  new document or a failure value.
  */
  class StepResult {
      /**
      @internal
      */
      constructor(
      /**
      The transformed document, if successful.
      */
      doc, 
      /**
      The failure message, if unsuccessful.
      */
      failed) {
          this.doc = doc;
          this.failed = failed;
      }
      /**
      Create a successful step result.
      */
      static ok(doc) { return new StepResult(doc, null); }
      /**
      Create a failed step result.
      */
      static fail(message) { return new StepResult(null, message); }
      /**
      Call [`Node.replace`](https://prosemirror.net/docs/ref/#model.Node.replace) with the given
      arguments. Create a successful result if it succeeds, and a
      failed one if it throws a `ReplaceError`.
      */
      static fromReplace(doc, from, to, slice) {
          try {
              return StepResult.ok(doc.replace(from, to, slice));
          }
          catch (e) {
              if (e instanceof ReplaceError)
                  return StepResult.fail(e.message);
              throw e;
          }
      }
  }

  function mapFragment(fragment, f, parent) {
      let mapped = [];
      for (let i = 0; i < fragment.childCount; i++) {
          let child = fragment.child(i);
          if (child.content.size)
              child = child.copy(mapFragment(child.content, f, child));
          if (child.isInline)
              child = f(child, parent, i);
          mapped.push(child);
      }
      return Fragment.fromArray(mapped);
  }
  /**
  Add a mark to all inline content between two positions.
  */
  class AddMarkStep extends Step {
      /**
      Create a mark step.
      */
      constructor(
      /**
      The start of the marked range.
      */
      from, 
      /**
      The end of the marked range.
      */
      to, 
      /**
      The mark to add.
      */
      mark) {
          super();
          this.from = from;
          this.to = to;
          this.mark = mark;
      }
      apply(doc) {
          let oldSlice = doc.slice(this.from, this.to), $from = doc.resolve(this.from);
          let parent = $from.node($from.sharedDepth(this.to));
          let slice = new Slice(mapFragment(oldSlice.content, (node, parent) => {
              if (!node.isAtom || !parent.type.allowsMarkType(this.mark.type))
                  return node;
              return node.mark(this.mark.addToSet(node.marks));
          }, parent), oldSlice.openStart, oldSlice.openEnd);
          return StepResult.fromReplace(doc, this.from, this.to, slice);
      }
      invert() {
          return new RemoveMarkStep(this.from, this.to, this.mark);
      }
      map(mapping) {
          let from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
          if (from.deleted && to.deleted || from.pos >= to.pos)
              return null;
          return new AddMarkStep(from.pos, to.pos, this.mark);
      }
      merge(other) {
          if (other instanceof AddMarkStep &&
              other.mark.eq(this.mark) &&
              this.from <= other.to && this.to >= other.from)
              return new AddMarkStep(Math.min(this.from, other.from), Math.max(this.to, other.to), this.mark);
          return null;
      }
      toJSON() {
          return { stepType: "addMark", mark: this.mark.toJSON(),
              from: this.from, to: this.to };
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.from != "number" || typeof json.to != "number")
              throw new RangeError("Invalid input for AddMarkStep.fromJSON");
          return new AddMarkStep(json.from, json.to, schema.markFromJSON(json.mark));
      }
  }
  Step.jsonID("addMark", AddMarkStep);
  /**
  Remove a mark from all inline content between two positions.
  */
  class RemoveMarkStep extends Step {
      /**
      Create a mark-removing step.
      */
      constructor(
      /**
      The start of the unmarked range.
      */
      from, 
      /**
      The end of the unmarked range.
      */
      to, 
      /**
      The mark to remove.
      */
      mark) {
          super();
          this.from = from;
          this.to = to;
          this.mark = mark;
      }
      apply(doc) {
          let oldSlice = doc.slice(this.from, this.to);
          let slice = new Slice(mapFragment(oldSlice.content, node => {
              return node.mark(this.mark.removeFromSet(node.marks));
          }, doc), oldSlice.openStart, oldSlice.openEnd);
          return StepResult.fromReplace(doc, this.from, this.to, slice);
      }
      invert() {
          return new AddMarkStep(this.from, this.to, this.mark);
      }
      map(mapping) {
          let from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
          if (from.deleted && to.deleted || from.pos >= to.pos)
              return null;
          return new RemoveMarkStep(from.pos, to.pos, this.mark);
      }
      merge(other) {
          if (other instanceof RemoveMarkStep &&
              other.mark.eq(this.mark) &&
              this.from <= other.to && this.to >= other.from)
              return new RemoveMarkStep(Math.min(this.from, other.from), Math.max(this.to, other.to), this.mark);
          return null;
      }
      toJSON() {
          return { stepType: "removeMark", mark: this.mark.toJSON(),
              from: this.from, to: this.to };
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.from != "number" || typeof json.to != "number")
              throw new RangeError("Invalid input for RemoveMarkStep.fromJSON");
          return new RemoveMarkStep(json.from, json.to, schema.markFromJSON(json.mark));
      }
  }
  Step.jsonID("removeMark", RemoveMarkStep);
  /**
  Add a mark to a specific node.
  */
  class AddNodeMarkStep extends Step {
      /**
      Create a node mark step.
      */
      constructor(
      /**
      The position of the target node.
      */
      pos, 
      /**
      The mark to add.
      */
      mark) {
          super();
          this.pos = pos;
          this.mark = mark;
      }
      apply(doc) {
          let node = doc.nodeAt(this.pos);
          if (!node)
              return StepResult.fail("No node at mark step's position");
          let updated = node.type.create(node.attrs, null, this.mark.addToSet(node.marks));
          return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
      }
      invert(doc) {
          let node = doc.nodeAt(this.pos);
          if (node) {
              let newSet = this.mark.addToSet(node.marks);
              if (newSet.length == node.marks.length) {
                  for (let i = 0; i < node.marks.length; i++)
                      if (!node.marks[i].isInSet(newSet))
                          return new AddNodeMarkStep(this.pos, node.marks[i]);
                  return new AddNodeMarkStep(this.pos, this.mark);
              }
          }
          return new RemoveNodeMarkStep(this.pos, this.mark);
      }
      map(mapping) {
          let pos = mapping.mapResult(this.pos, 1);
          return pos.deletedAfter ? null : new AddNodeMarkStep(pos.pos, this.mark);
      }
      toJSON() {
          return { stepType: "addNodeMark", pos: this.pos, mark: this.mark.toJSON() };
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.pos != "number")
              throw new RangeError("Invalid input for AddNodeMarkStep.fromJSON");
          return new AddNodeMarkStep(json.pos, schema.markFromJSON(json.mark));
      }
  }
  Step.jsonID("addNodeMark", AddNodeMarkStep);
  /**
  Remove a mark from a specific node.
  */
  class RemoveNodeMarkStep extends Step {
      /**
      Create a mark-removing step.
      */
      constructor(
      /**
      The position of the target node.
      */
      pos, 
      /**
      The mark to remove.
      */
      mark) {
          super();
          this.pos = pos;
          this.mark = mark;
      }
      apply(doc) {
          let node = doc.nodeAt(this.pos);
          if (!node)
              return StepResult.fail("No node at mark step's position");
          let updated = node.type.create(node.attrs, null, this.mark.removeFromSet(node.marks));
          return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
      }
      invert(doc) {
          let node = doc.nodeAt(this.pos);
          if (!node || !this.mark.isInSet(node.marks))
              return this;
          return new AddNodeMarkStep(this.pos, this.mark);
      }
      map(mapping) {
          let pos = mapping.mapResult(this.pos, 1);
          return pos.deletedAfter ? null : new RemoveNodeMarkStep(pos.pos, this.mark);
      }
      toJSON() {
          return { stepType: "removeNodeMark", pos: this.pos, mark: this.mark.toJSON() };
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.pos != "number")
              throw new RangeError("Invalid input for RemoveNodeMarkStep.fromJSON");
          return new RemoveNodeMarkStep(json.pos, schema.markFromJSON(json.mark));
      }
  }
  Step.jsonID("removeNodeMark", RemoveNodeMarkStep);

  /**
  Replace a part of the document with a slice of new content.
  */
  class ReplaceStep extends Step {
      /**
      The given `slice` should fit the 'gap' between `from` and
      `to`—the depths must line up, and the surrounding nodes must be
      able to be joined with the open sides of the slice. When
      `structure` is true, the step will fail if the content between
      from and to is not just a sequence of closing and then opening
      tokens (this is to guard against rebased replace steps
      overwriting something they weren't supposed to).
      */
      constructor(
      /**
      The start position of the replaced range.
      */
      from, 
      /**
      The end position of the replaced range.
      */
      to, 
      /**
      The slice to insert.
      */
      slice, 
      /**
      @internal
      */
      structure = false) {
          super();
          this.from = from;
          this.to = to;
          this.slice = slice;
          this.structure = structure;
      }
      apply(doc) {
          if (this.structure && contentBetween(doc, this.from, this.to))
              return StepResult.fail("Structure replace would overwrite content");
          return StepResult.fromReplace(doc, this.from, this.to, this.slice);
      }
      getMap() {
          return new StepMap([this.from, this.to - this.from, this.slice.size]);
      }
      invert(doc) {
          return new ReplaceStep(this.from, this.from + this.slice.size, doc.slice(this.from, this.to));
      }
      map(mapping) {
          let from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
          if (from.deletedAcross && to.deletedAcross)
              return null;
          return new ReplaceStep(from.pos, Math.max(from.pos, to.pos), this.slice);
      }
      merge(other) {
          if (!(other instanceof ReplaceStep) || other.structure || this.structure)
              return null;
          if (this.from + this.slice.size == other.from && !this.slice.openEnd && !other.slice.openStart) {
              let slice = this.slice.size + other.slice.size == 0 ? Slice.empty
                  : new Slice(this.slice.content.append(other.slice.content), this.slice.openStart, other.slice.openEnd);
              return new ReplaceStep(this.from, this.to + (other.to - other.from), slice, this.structure);
          }
          else if (other.to == this.from && !this.slice.openStart && !other.slice.openEnd) {
              let slice = this.slice.size + other.slice.size == 0 ? Slice.empty
                  : new Slice(other.slice.content.append(this.slice.content), other.slice.openStart, this.slice.openEnd);
              return new ReplaceStep(other.from, this.to, slice, this.structure);
          }
          else {
              return null;
          }
      }
      toJSON() {
          let json = { stepType: "replace", from: this.from, to: this.to };
          if (this.slice.size)
              json.slice = this.slice.toJSON();
          if (this.structure)
              json.structure = true;
          return json;
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.from != "number" || typeof json.to != "number")
              throw new RangeError("Invalid input for ReplaceStep.fromJSON");
          return new ReplaceStep(json.from, json.to, Slice.fromJSON(schema, json.slice), !!json.structure);
      }
  }
  Step.jsonID("replace", ReplaceStep);
  /**
  Replace a part of the document with a slice of content, but
  preserve a range of the replaced content by moving it into the
  slice.
  */
  class ReplaceAroundStep extends Step {
      /**
      Create a replace-around step with the given range and gap.
      `insert` should be the point in the slice into which the content
      of the gap should be moved. `structure` has the same meaning as
      it has in the [`ReplaceStep`](https://prosemirror.net/docs/ref/#transform.ReplaceStep) class.
      */
      constructor(
      /**
      The start position of the replaced range.
      */
      from, 
      /**
      The end position of the replaced range.
      */
      to, 
      /**
      The start of preserved range.
      */
      gapFrom, 
      /**
      The end of preserved range.
      */
      gapTo, 
      /**
      The slice to insert.
      */
      slice, 
      /**
      The position in the slice where the preserved range should be
      inserted.
      */
      insert, 
      /**
      @internal
      */
      structure = false) {
          super();
          this.from = from;
          this.to = to;
          this.gapFrom = gapFrom;
          this.gapTo = gapTo;
          this.slice = slice;
          this.insert = insert;
          this.structure = structure;
      }
      apply(doc) {
          if (this.structure && (contentBetween(doc, this.from, this.gapFrom) ||
              contentBetween(doc, this.gapTo, this.to)))
              return StepResult.fail("Structure gap-replace would overwrite content");
          let gap = doc.slice(this.gapFrom, this.gapTo);
          if (gap.openStart || gap.openEnd)
              return StepResult.fail("Gap is not a flat range");
          let inserted = this.slice.insertAt(this.insert, gap.content);
          if (!inserted)
              return StepResult.fail("Content does not fit in gap");
          return StepResult.fromReplace(doc, this.from, this.to, inserted);
      }
      getMap() {
          return new StepMap([this.from, this.gapFrom - this.from, this.insert,
              this.gapTo, this.to - this.gapTo, this.slice.size - this.insert]);
      }
      invert(doc) {
          let gap = this.gapTo - this.gapFrom;
          return new ReplaceAroundStep(this.from, this.from + this.slice.size + gap, this.from + this.insert, this.from + this.insert + gap, doc.slice(this.from, this.to).removeBetween(this.gapFrom - this.from, this.gapTo - this.from), this.gapFrom - this.from, this.structure);
      }
      map(mapping) {
          let from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
          let gapFrom = this.from == this.gapFrom ? from.pos : mapping.map(this.gapFrom, -1);
          let gapTo = this.to == this.gapTo ? to.pos : mapping.map(this.gapTo, 1);
          if ((from.deletedAcross && to.deletedAcross) || gapFrom < from.pos || gapTo > to.pos)
              return null;
          return new ReplaceAroundStep(from.pos, to.pos, gapFrom, gapTo, this.slice, this.insert, this.structure);
      }
      toJSON() {
          let json = { stepType: "replaceAround", from: this.from, to: this.to,
              gapFrom: this.gapFrom, gapTo: this.gapTo, insert: this.insert };
          if (this.slice.size)
              json.slice = this.slice.toJSON();
          if (this.structure)
              json.structure = true;
          return json;
      }
      /**
      @internal
      */
      static fromJSON(schema, json) {
          if (typeof json.from != "number" || typeof json.to != "number" ||
              typeof json.gapFrom != "number" || typeof json.gapTo != "number" || typeof json.insert != "number")
              throw new RangeError("Invalid input for ReplaceAroundStep.fromJSON");
          return new ReplaceAroundStep(json.from, json.to, json.gapFrom, json.gapTo, Slice.fromJSON(schema, json.slice), json.insert, !!json.structure);
      }
  }
  Step.jsonID("replaceAround", ReplaceAroundStep);
  function contentBetween(doc, from, to) {
      let $from = doc.resolve(from), dist = to - from, depth = $from.depth;
      while (dist > 0 && depth > 0 && $from.indexAfter(depth) == $from.node(depth).childCount) {
          depth--;
          dist--;
      }
      if (dist > 0) {
          let next = $from.node(depth).maybeChild($from.indexAfter(depth));
          while (dist > 0) {
              if (!next || next.isLeaf)
                  return true;
              next = next.firstChild;
              dist--;
          }
      }
      return false;
  }

  function addMark(tr, from, to, mark) {
      let removed = [], added = [];
      let removing, adding;
      tr.doc.nodesBetween(from, to, (node, pos, parent) => {
          if (!node.isInline)
              return;
          let marks = node.marks;
          if (!mark.isInSet(marks) && parent.type.allowsMarkType(mark.type)) {
              let start = Math.max(pos, from), end = Math.min(pos + node.nodeSize, to);
              let newSet = mark.addToSet(marks);
              for (let i = 0; i < marks.length; i++) {
                  if (!marks[i].isInSet(newSet)) {
                      if (removing && removing.to == start && removing.mark.eq(marks[i]))
                          removing.to = end;
                      else
                          removed.push(removing = new RemoveMarkStep(start, end, marks[i]));
                  }
              }
              if (adding && adding.to == start)
                  adding.to = end;
              else
                  added.push(adding = new AddMarkStep(start, end, mark));
          }
      });
      removed.forEach(s => tr.step(s));
      added.forEach(s => tr.step(s));
  }
  function removeMark(tr, from, to, mark) {
      let matched = [], step = 0;
      tr.doc.nodesBetween(from, to, (node, pos) => {
          if (!node.isInline)
              return;
          step++;
          let toRemove = null;
          if (mark instanceof MarkType) {
              let set = node.marks, found;
              while (found = mark.isInSet(set)) {
                  (toRemove || (toRemove = [])).push(found);
                  set = found.removeFromSet(set);
              }
          }
          else if (mark) {
              if (mark.isInSet(node.marks))
                  toRemove = [mark];
          }
          else {
              toRemove = node.marks;
          }
          if (toRemove && toRemove.length) {
              let end = Math.min(pos + node.nodeSize, to);
              for (let i = 0; i < toRemove.length; i++) {
                  let style = toRemove[i], found;
                  for (let j = 0; j < matched.length; j++) {
                      let m = matched[j];
                      if (m.step == step - 1 && style.eq(matched[j].style))
                          found = m;
                  }
                  if (found) {
                      found.to = end;
                      found.step = step;
                  }
                  else {
                      matched.push({ style, from: Math.max(pos, from), to: end, step });
                  }
              }
          }
      });
      matched.forEach(m => tr.step(new RemoveMarkStep(m.from, m.to, m.style)));
  }
  function clearIncompatible(tr, pos, parentType, match = parentType.contentMatch, clearNewlines = true) {
      let node = tr.doc.nodeAt(pos);
      let replSteps = [], cur = pos + 1;
      for (let i = 0; i < node.childCount; i++) {
          let child = node.child(i), end = cur + child.nodeSize;
          let allowed = match.matchType(child.type);
          if (!allowed) {
              replSteps.push(new ReplaceStep(cur, end, Slice.empty));
          }
          else {
              match = allowed;
              for (let j = 0; j < child.marks.length; j++)
                  if (!parentType.allowsMarkType(child.marks[j].type))
                      tr.step(new RemoveMarkStep(cur, end, child.marks[j]));
              if (clearNewlines && child.isText && parentType.whitespace != "pre") {
                  let m, newline = /\r?\n|\r/g, slice;
                  while (m = newline.exec(child.text)) {
                      if (!slice)
                          slice = new Slice(Fragment.from(parentType.schema.text(" ", parentType.allowedMarks(child.marks))), 0, 0);
                      replSteps.push(new ReplaceStep(cur + m.index, cur + m.index + m[0].length, slice));
                  }
              }
          }
          cur = end;
      }
      if (!match.validEnd) {
          let fill = match.fillBefore(Fragment.empty, true);
          tr.replace(cur, cur, new Slice(fill, 0, 0));
      }
      for (let i = replSteps.length - 1; i >= 0; i--)
          tr.step(replSteps[i]);
  }

  function canCut(node, start, end) {
      return (start == 0 || node.canReplace(start, node.childCount)) &&
          (end == node.childCount || node.canReplace(0, end));
  }
  /**
  Try to find a target depth to which the content in the given range
  can be lifted. Will not go across
  [isolating](https://prosemirror.net/docs/ref/#model.NodeSpec.isolating) parent nodes.
  */
  function liftTarget(range) {
      let parent = range.parent;
      let content = parent.content.cutByIndex(range.startIndex, range.endIndex);
      for (let depth = range.depth;; --depth) {
          let node = range.$from.node(depth);
          let index = range.$from.index(depth), endIndex = range.$to.indexAfter(depth);
          if (depth < range.depth && node.canReplace(index, endIndex, content))
              return depth;
          if (depth == 0 || node.type.spec.isolating || !canCut(node, index, endIndex))
              break;
      }
      return null;
  }
  function lift$1(tr, range, target) {
      let { $from, $to, depth } = range;
      let gapStart = $from.before(depth + 1), gapEnd = $to.after(depth + 1);
      let start = gapStart, end = gapEnd;
      let before = Fragment.empty, openStart = 0;
      for (let d = depth, splitting = false; d > target; d--)
          if (splitting || $from.index(d) > 0) {
              splitting = true;
              before = Fragment.from($from.node(d).copy(before));
              openStart++;
          }
          else {
              start--;
          }
      let after = Fragment.empty, openEnd = 0;
      for (let d = depth, splitting = false; d > target; d--)
          if (splitting || $to.after(d + 1) < $to.end(d)) {
              splitting = true;
              after = Fragment.from($to.node(d).copy(after));
              openEnd++;
          }
          else {
              end++;
          }
      tr.step(new ReplaceAroundStep(start, end, gapStart, gapEnd, new Slice(before.append(after), openStart, openEnd), before.size - openStart, true));
  }
  /**
  Try to find a valid way to wrap the content in the given range in a
  node of the given type. May introduce extra nodes around and inside
  the wrapper node, if necessary. Returns null if no valid wrapping
  could be found. When `innerRange` is given, that range's content is
  used as the content to fit into the wrapping, instead of the
  content of `range`.
  */
  function findWrapping(range, nodeType, attrs = null, innerRange = range) {
      let around = findWrappingOutside(range, nodeType);
      let inner = around && findWrappingInside(innerRange, nodeType);
      if (!inner)
          return null;
      return around.map(withAttrs)
          .concat({ type: nodeType, attrs }).concat(inner.map(withAttrs));
  }
  function withAttrs(type) { return { type, attrs: null }; }
  function findWrappingOutside(range, type) {
      let { parent, startIndex, endIndex } = range;
      let around = parent.contentMatchAt(startIndex).findWrapping(type);
      if (!around)
          return null;
      let outer = around.length ? around[0] : type;
      return parent.canReplaceWith(startIndex, endIndex, outer) ? around : null;
  }
  function findWrappingInside(range, type) {
      let { parent, startIndex, endIndex } = range;
      let inner = parent.child(startIndex);
      let inside = type.contentMatch.findWrapping(inner.type);
      if (!inside)
          return null;
      let lastType = inside.length ? inside[inside.length - 1] : type;
      let innerMatch = lastType.contentMatch;
      for (let i = startIndex; innerMatch && i < endIndex; i++)
          innerMatch = innerMatch.matchType(parent.child(i).type);
      if (!innerMatch || !innerMatch.validEnd)
          return null;
      return inside;
  }
  function wrap(tr, range, wrappers) {
      let content = Fragment.empty;
      for (let i = wrappers.length - 1; i >= 0; i--) {
          if (content.size) {
              let match = wrappers[i].type.contentMatch.matchFragment(content);
              if (!match || !match.validEnd)
                  throw new RangeError("Wrapper type given to Transform.wrap does not form valid content of its parent wrapper");
          }
          content = Fragment.from(wrappers[i].type.create(wrappers[i].attrs, content));
      }
      let start = range.start, end = range.end;
      tr.step(new ReplaceAroundStep(start, end, start, end, new Slice(content, 0, 0), wrappers.length, true));
  }
  function setBlockType$1(tr, from, to, type, attrs) {
      if (!type.isTextblock)
          throw new RangeError("Type given to setBlockType should be a textblock");
      let mapFrom = tr.steps.length;
      tr.doc.nodesBetween(from, to, (node, pos) => {
          if (node.isTextblock && !node.hasMarkup(type, attrs) && canChangeType(tr.doc, tr.mapping.slice(mapFrom).map(pos), type)) {
              let convertNewlines = null;
              if (type.schema.linebreakReplacement) {
                  let pre = type.whitespace == "pre", supportLinebreak = !!type.contentMatch.matchType(type.schema.linebreakReplacement);
                  if (pre && !supportLinebreak)
                      convertNewlines = false;
                  else if (!pre && supportLinebreak)
                      convertNewlines = true;
              }
              // Ensure all markup that isn't allowed in the new node type is cleared
              if (convertNewlines === false)
                  replaceLinebreaks(tr, node, pos, mapFrom);
              clearIncompatible(tr, tr.mapping.slice(mapFrom).map(pos, 1), type, undefined, convertNewlines === null);
              let mapping = tr.mapping.slice(mapFrom);
              let startM = mapping.map(pos, 1), endM = mapping.map(pos + node.nodeSize, 1);
              tr.step(new ReplaceAroundStep(startM, endM, startM + 1, endM - 1, new Slice(Fragment.from(type.create(attrs, null, node.marks)), 0, 0), 1, true));
              if (convertNewlines === true)
                  replaceNewlines(tr, node, pos, mapFrom);
              return false;
          }
      });
  }
  function replaceNewlines(tr, node, pos, mapFrom) {
      node.forEach((child, offset) => {
          if (child.isText) {
              let m, newline = /\r?\n|\r/g;
              while (m = newline.exec(child.text)) {
                  let start = tr.mapping.slice(mapFrom).map(pos + 1 + offset + m.index);
                  tr.replaceWith(start, start + 1, node.type.schema.linebreakReplacement.create());
              }
          }
      });
  }
  function replaceLinebreaks(tr, node, pos, mapFrom) {
      node.forEach((child, offset) => {
          if (child.type == child.type.schema.linebreakReplacement) {
              let start = tr.mapping.slice(mapFrom).map(pos + 1 + offset);
              tr.replaceWith(start, start + 1, node.type.schema.text("\n"));
          }
      });
  }
  function canChangeType(doc, pos, type) {
      let $pos = doc.resolve(pos), index = $pos.index();
      return $pos.parent.canReplaceWith(index, index + 1, type);
  }
  /**
  Change the type, attributes, and/or marks of the node at `pos`.
  When `type` isn't given, the existing node type is preserved,
  */
  function setNodeMarkup(tr, pos, type, attrs, marks) {
      let node = tr.doc.nodeAt(pos);
      if (!node)
          throw new RangeError("No node at given position");
      if (!type)
          type = node.type;
      let newNode = type.create(attrs, null, marks || node.marks);
      if (node.isLeaf)
          return tr.replaceWith(pos, pos + node.nodeSize, newNode);
      if (!type.validContent(node.content))
          throw new RangeError("Invalid content for node type " + type.name);
      tr.step(new ReplaceAroundStep(pos, pos + node.nodeSize, pos + 1, pos + node.nodeSize - 1, new Slice(Fragment.from(newNode), 0, 0), 1, true));
  }
  /**
  Check whether splitting at the given position is allowed.
  */
  function canSplit(doc, pos, depth = 1, typesAfter) {
      let $pos = doc.resolve(pos), base = $pos.depth - depth;
      let innerType = (typesAfter && typesAfter[typesAfter.length - 1]) || $pos.parent;
      if (base < 0 || $pos.parent.type.spec.isolating ||
          !$pos.parent.canReplace($pos.index(), $pos.parent.childCount) ||
          !innerType.type.validContent($pos.parent.content.cutByIndex($pos.index(), $pos.parent.childCount)))
          return false;
      for (let d = $pos.depth - 1, i = depth - 2; d > base; d--, i--) {
          let node = $pos.node(d), index = $pos.index(d);
          if (node.type.spec.isolating)
              return false;
          let rest = node.content.cutByIndex(index, node.childCount);
          let overrideChild = typesAfter && typesAfter[i + 1];
          if (overrideChild)
              rest = rest.replaceChild(0, overrideChild.type.create(overrideChild.attrs));
          let after = (typesAfter && typesAfter[i]) || node;
          if (!node.canReplace(index + 1, node.childCount) || !after.type.validContent(rest))
              return false;
      }
      let index = $pos.indexAfter(base);
      let baseType = typesAfter && typesAfter[0];
      return $pos.node(base).canReplaceWith(index, index, baseType ? baseType.type : $pos.node(base + 1).type);
  }
  function split(tr, pos, depth = 1, typesAfter) {
      let $pos = tr.doc.resolve(pos), before = Fragment.empty, after = Fragment.empty;
      for (let d = $pos.depth, e = $pos.depth - depth, i = depth - 1; d > e; d--, i--) {
          before = Fragment.from($pos.node(d).copy(before));
          let typeAfter = typesAfter && typesAfter[i];
          after = Fragment.from(typeAfter ? typeAfter.type.create(typeAfter.attrs, after) : $pos.node(d).copy(after));
      }
      tr.step(new ReplaceStep(pos, pos, new Slice(before.append(after), depth, depth), true));
  }
  /**
  Test whether the blocks before and after a given position can be
  joined.
  */
  function canJoin(doc, pos) {
      let $pos = doc.resolve(pos), index = $pos.index();
      return joinable($pos.nodeBefore, $pos.nodeAfter) &&
          $pos.parent.canReplace(index, index + 1);
  }
  function joinable(a, b) {
      return !!(a && b && !a.isLeaf && a.canAppend(b));
  }
  /**
  Find an ancestor of the given position that can be joined to the
  block before (or after if `dir` is positive). Returns the joinable
  point, if any.
  */
  function joinPoint(doc, pos, dir = -1) {
      let $pos = doc.resolve(pos);
      for (let d = $pos.depth;; d--) {
          let before, after, index = $pos.index(d);
          if (d == $pos.depth) {
              before = $pos.nodeBefore;
              after = $pos.nodeAfter;
          }
          else if (dir > 0) {
              before = $pos.node(d + 1);
              index++;
              after = $pos.node(d).maybeChild(index);
          }
          else {
              before = $pos.node(d).maybeChild(index - 1);
              after = $pos.node(d + 1);
          }
          if (before && !before.isTextblock && joinable(before, after) &&
              $pos.node(d).canReplace(index, index + 1))
              return pos;
          if (d == 0)
              break;
          pos = dir < 0 ? $pos.before(d) : $pos.after(d);
      }
  }
  function join(tr, pos, depth) {
      let step = new ReplaceStep(pos - depth, pos + depth, Slice.empty, true);
      tr.step(step);
  }
  /**
  Try to find a point where a node of the given type can be inserted
  near `pos`, by searching up the node hierarchy when `pos` itself
  isn't a valid place but is at the start or end of a node. Return
  null if no position was found.
  */
  function insertPoint(doc, pos, nodeType) {
      let $pos = doc.resolve(pos);
      if ($pos.parent.canReplaceWith($pos.index(), $pos.index(), nodeType))
          return pos;
      if ($pos.parentOffset == 0)
          for (let d = $pos.depth - 1; d >= 0; d--) {
              let index = $pos.index(d);
              if ($pos.node(d).canReplaceWith(index, index, nodeType))
                  return $pos.before(d + 1);
              if (index > 0)
                  return null;
          }
      if ($pos.parentOffset == $pos.parent.content.size)
          for (let d = $pos.depth - 1; d >= 0; d--) {
              let index = $pos.indexAfter(d);
              if ($pos.node(d).canReplaceWith(index, index, nodeType))
                  return $pos.after(d + 1);
              if (index < $pos.node(d).childCount)
                  return null;
          }
      return null;
  }
  /**
  Finds a position at or around the given position where the given
  slice can be inserted. Will look at parent nodes' nearest boundary
  and try there, even if the original position wasn't directly at the
  start or end of that node. Returns null when no position was found.
  */
  function dropPoint(doc, pos, slice) {
      let $pos = doc.resolve(pos);
      if (!slice.content.size)
          return pos;
      let content = slice.content;
      for (let i = 0; i < slice.openStart; i++)
          content = content.firstChild.content;
      for (let pass = 1; pass <= (slice.openStart == 0 && slice.size ? 2 : 1); pass++) {
          for (let d = $pos.depth; d >= 0; d--) {
              let bias = d == $pos.depth ? 0 : $pos.pos <= ($pos.start(d + 1) + $pos.end(d + 1)) / 2 ? -1 : 1;
              let insertPos = $pos.index(d) + (bias > 0 ? 1 : 0);
              let parent = $pos.node(d), fits = false;
              if (pass == 1) {
                  fits = parent.canReplace(insertPos, insertPos, content);
              }
              else {
                  let wrapping = parent.contentMatchAt(insertPos).findWrapping(content.firstChild.type);
                  fits = wrapping && parent.canReplaceWith(insertPos, insertPos, wrapping[0]);
              }
              if (fits)
                  return bias == 0 ? $pos.pos : bias < 0 ? $pos.before(d + 1) : $pos.after(d + 1);
          }
      }
      return null;
  }

  /**
  ‘Fit’ a slice into a given position in the document, producing a
  [step](https://prosemirror.net/docs/ref/#transform.Step) that inserts it. Will return null if
  there's no meaningful way to insert the slice here, or inserting it
  would be a no-op (an empty slice over an empty range).
  */
  function replaceStep(doc, from, to = from, slice = Slice.empty) {
      if (from == to && !slice.size)
          return null;
      let $from = doc.resolve(from), $to = doc.resolve(to);
      // Optimization -- avoid work if it's obvious that it's not needed.
      if (fitsTrivially($from, $to, slice))
          return new ReplaceStep(from, to, slice);
      return new Fitter($from, $to, slice).fit();
  }
  function fitsTrivially($from, $to, slice) {
      return !slice.openStart && !slice.openEnd && $from.start() == $to.start() &&
          $from.parent.canReplace($from.index(), $to.index(), slice.content);
  }
  // Algorithm for 'placing' the elements of a slice into a gap:
  //
  // We consider the content of each node that is open to the left to be
  // independently placeable. I.e. in <p("foo"), p("bar")>, when the
  // paragraph on the left is open, "foo" can be placed (somewhere on
  // the left side of the replacement gap) independently from p("bar").
  //
  // This class tracks the state of the placement progress in the
  // following properties:
  //
  //  - `frontier` holds a stack of `{type, match}` objects that
  //    represent the open side of the replacement. It starts at
  //    `$from`, then moves forward as content is placed, and is finally
  //    reconciled with `$to`.
  //
  //  - `unplaced` is a slice that represents the content that hasn't
  //    been placed yet.
  //
  //  - `placed` is a fragment of placed content. Its open-start value
  //    is implicit in `$from`, and its open-end value in `frontier`.
  class Fitter {
      constructor($from, $to, unplaced) {
          this.$from = $from;
          this.$to = $to;
          this.unplaced = unplaced;
          this.frontier = [];
          this.placed = Fragment.empty;
          for (let i = 0; i <= $from.depth; i++) {
              let node = $from.node(i);
              this.frontier.push({
                  type: node.type,
                  match: node.contentMatchAt($from.indexAfter(i))
              });
          }
          for (let i = $from.depth; i > 0; i--)
              this.placed = Fragment.from($from.node(i).copy(this.placed));
      }
      get depth() { return this.frontier.length - 1; }
      fit() {
          // As long as there's unplaced content, try to place some of it.
          // If that fails, either increase the open score of the unplaced
          // slice, or drop nodes from it, and then try again.
          while (this.unplaced.size) {
              let fit = this.findFittable();
              if (fit)
                  this.placeNodes(fit);
              else
                  this.openMore() || this.dropNode();
          }
          // When there's inline content directly after the frontier _and_
          // directly after `this.$to`, we must generate a `ReplaceAround`
          // step that pulls that content into the node after the frontier.
          // That means the fitting must be done to the end of the textblock
          // node after `this.$to`, not `this.$to` itself.
          let moveInline = this.mustMoveInline(), placedSize = this.placed.size - this.depth - this.$from.depth;
          let $from = this.$from, $to = this.close(moveInline < 0 ? this.$to : $from.doc.resolve(moveInline));
          if (!$to)
              return null;
          // If closing to `$to` succeeded, create a step
          let content = this.placed, openStart = $from.depth, openEnd = $to.depth;
          while (openStart && openEnd && content.childCount == 1) { // Normalize by dropping open parent nodes
              content = content.firstChild.content;
              openStart--;
              openEnd--;
          }
          let slice = new Slice(content, openStart, openEnd);
          if (moveInline > -1)
              return new ReplaceAroundStep($from.pos, moveInline, this.$to.pos, this.$to.end(), slice, placedSize);
          if (slice.size || $from.pos != this.$to.pos) // Don't generate no-op steps
              return new ReplaceStep($from.pos, $to.pos, slice);
          return null;
      }
      // Find a position on the start spine of `this.unplaced` that has
      // content that can be moved somewhere on the frontier. Returns two
      // depths, one for the slice and one for the frontier.
      findFittable() {
          let startDepth = this.unplaced.openStart;
          for (let cur = this.unplaced.content, d = 0, openEnd = this.unplaced.openEnd; d < startDepth; d++) {
              let node = cur.firstChild;
              if (cur.childCount > 1)
                  openEnd = 0;
              if (node.type.spec.isolating && openEnd <= d) {
                  startDepth = d;
                  break;
              }
              cur = node.content;
          }
          // Only try wrapping nodes (pass 2) after finding a place without
          // wrapping failed.
          for (let pass = 1; pass <= 2; pass++) {
              for (let sliceDepth = pass == 1 ? startDepth : this.unplaced.openStart; sliceDepth >= 0; sliceDepth--) {
                  let fragment, parent = null;
                  if (sliceDepth) {
                      parent = contentAt(this.unplaced.content, sliceDepth - 1).firstChild;
                      fragment = parent.content;
                  }
                  else {
                      fragment = this.unplaced.content;
                  }
                  let first = fragment.firstChild;
                  for (let frontierDepth = this.depth; frontierDepth >= 0; frontierDepth--) {
                      let { type, match } = this.frontier[frontierDepth], wrap, inject = null;
                      // In pass 1, if the next node matches, or there is no next
                      // node but the parents look compatible, we've found a
                      // place.
                      if (pass == 1 && (first ? match.matchType(first.type) || (inject = match.fillBefore(Fragment.from(first), false))
                          : parent && type.compatibleContent(parent.type)))
                          return { sliceDepth, frontierDepth, parent, inject };
                      // In pass 2, look for a set of wrapping nodes that make
                      // `first` fit here.
                      else if (pass == 2 && first && (wrap = match.findWrapping(first.type)))
                          return { sliceDepth, frontierDepth, parent, wrap };
                      // Don't continue looking further up if the parent node
                      // would fit here.
                      if (parent && match.matchType(parent.type))
                          break;
                  }
              }
          }
      }
      openMore() {
          let { content, openStart, openEnd } = this.unplaced;
          let inner = contentAt(content, openStart);
          if (!inner.childCount || inner.firstChild.isLeaf)
              return false;
          this.unplaced = new Slice(content, openStart + 1, Math.max(openEnd, inner.size + openStart >= content.size - openEnd ? openStart + 1 : 0));
          return true;
      }
      dropNode() {
          let { content, openStart, openEnd } = this.unplaced;
          let inner = contentAt(content, openStart);
          if (inner.childCount <= 1 && openStart > 0) {
              let openAtEnd = content.size - openStart <= openStart + inner.size;
              this.unplaced = new Slice(dropFromFragment(content, openStart - 1, 1), openStart - 1, openAtEnd ? openStart - 1 : openEnd);
          }
          else {
              this.unplaced = new Slice(dropFromFragment(content, openStart, 1), openStart, openEnd);
          }
      }
      // Move content from the unplaced slice at `sliceDepth` to the
      // frontier node at `frontierDepth`. Close that frontier node when
      // applicable.
      placeNodes({ sliceDepth, frontierDepth, parent, inject, wrap }) {
          while (this.depth > frontierDepth)
              this.closeFrontierNode();
          if (wrap)
              for (let i = 0; i < wrap.length; i++)
                  this.openFrontierNode(wrap[i]);
          let slice = this.unplaced, fragment = parent ? parent.content : slice.content;
          let openStart = slice.openStart - sliceDepth;
          let taken = 0, add = [];
          let { match, type } = this.frontier[frontierDepth];
          if (inject) {
              for (let i = 0; i < inject.childCount; i++)
                  add.push(inject.child(i));
              match = match.matchFragment(inject);
          }
          // Computes the amount of (end) open nodes at the end of the
          // fragment. When 0, the parent is open, but no more. When
          // negative, nothing is open.
          let openEndCount = (fragment.size + sliceDepth) - (slice.content.size - slice.openEnd);
          // Scan over the fragment, fitting as many child nodes as
          // possible.
          while (taken < fragment.childCount) {
              let next = fragment.child(taken), matches = match.matchType(next.type);
              if (!matches)
                  break;
              taken++;
              if (taken > 1 || openStart == 0 || next.content.size) { // Drop empty open nodes
                  match = matches;
                  add.push(closeNodeStart(next.mark(type.allowedMarks(next.marks)), taken == 1 ? openStart : 0, taken == fragment.childCount ? openEndCount : -1));
              }
          }
          let toEnd = taken == fragment.childCount;
          if (!toEnd)
              openEndCount = -1;
          this.placed = addToFragment(this.placed, frontierDepth, Fragment.from(add));
          this.frontier[frontierDepth].match = match;
          // If the parent types match, and the entire node was moved, and
          // it's not open, close this frontier node right away.
          if (toEnd && openEndCount < 0 && parent && parent.type == this.frontier[this.depth].type && this.frontier.length > 1)
              this.closeFrontierNode();
          // Add new frontier nodes for any open nodes at the end.
          for (let i = 0, cur = fragment; i < openEndCount; i++) {
              let node = cur.lastChild;
              this.frontier.push({ type: node.type, match: node.contentMatchAt(node.childCount) });
              cur = node.content;
          }
          // Update `this.unplaced`. Drop the entire node from which we
          // placed it we got to its end, otherwise just drop the placed
          // nodes.
          this.unplaced = !toEnd ? new Slice(dropFromFragment(slice.content, sliceDepth, taken), slice.openStart, slice.openEnd)
              : sliceDepth == 0 ? Slice.empty
                  : new Slice(dropFromFragment(slice.content, sliceDepth - 1, 1), sliceDepth - 1, openEndCount < 0 ? slice.openEnd : sliceDepth - 1);
      }
      mustMoveInline() {
          if (!this.$to.parent.isTextblock)
              return -1;
          let top = this.frontier[this.depth], level;
          if (!top.type.isTextblock || !contentAfterFits(this.$to, this.$to.depth, top.type, top.match, false) ||
              (this.$to.depth == this.depth && (level = this.findCloseLevel(this.$to)) && level.depth == this.depth))
              return -1;
          let { depth } = this.$to, after = this.$to.after(depth);
          while (depth > 1 && after == this.$to.end(--depth))
              ++after;
          return after;
      }
      findCloseLevel($to) {
          scan: for (let i = Math.min(this.depth, $to.depth); i >= 0; i--) {
              let { match, type } = this.frontier[i];
              let dropInner = i < $to.depth && $to.end(i + 1) == $to.pos + ($to.depth - (i + 1));
              let fit = contentAfterFits($to, i, type, match, dropInner);
              if (!fit)
                  continue;
              for (let d = i - 1; d >= 0; d--) {
                  let { match, type } = this.frontier[d];
                  let matches = contentAfterFits($to, d, type, match, true);
                  if (!matches || matches.childCount)
                      continue scan;
              }
              return { depth: i, fit, move: dropInner ? $to.doc.resolve($to.after(i + 1)) : $to };
          }
      }
      close($to) {
          let close = this.findCloseLevel($to);
          if (!close)
              return null;
          while (this.depth > close.depth)
              this.closeFrontierNode();
          if (close.fit.childCount)
              this.placed = addToFragment(this.placed, close.depth, close.fit);
          $to = close.move;
          for (let d = close.depth + 1; d <= $to.depth; d++) {
              let node = $to.node(d), add = node.type.contentMatch.fillBefore(node.content, true, $to.index(d));
              this.openFrontierNode(node.type, node.attrs, add);
          }
          return $to;
      }
      openFrontierNode(type, attrs = null, content) {
          let top = this.frontier[this.depth];
          top.match = top.match.matchType(type);
          this.placed = addToFragment(this.placed, this.depth, Fragment.from(type.create(attrs, content)));
          this.frontier.push({ type, match: type.contentMatch });
      }
      closeFrontierNode() {
          let open = this.frontier.pop();
          let add = open.match.fillBefore(Fragment.empty, true);
          if (add.childCount)
              this.placed = addToFragment(this.placed, this.frontier.length, add);
      }
  }
  function dropFromFragment(fragment, depth, count) {
      if (depth == 0)
          return fragment.cutByIndex(count, fragment.childCount);
      return fragment.replaceChild(0, fragment.firstChild.copy(dropFromFragment(fragment.firstChild.content, depth - 1, count)));
  }
  function addToFragment(fragment, depth, content) {
      if (depth == 0)
          return fragment.append(content);
      return fragment.replaceChild(fragment.childCount - 1, fragment.lastChild.copy(addToFragment(fragment.lastChild.content, depth - 1, content)));
  }
  function contentAt(fragment, depth) {
      for (let i = 0; i < depth; i++)
          fragment = fragment.firstChild.content;
      return fragment;
  }
  function closeNodeStart(node, openStart, openEnd) {
      if (openStart <= 0)
          return node;
      let frag = node.content;
      if (openStart > 1)
          frag = frag.replaceChild(0, closeNodeStart(frag.firstChild, openStart - 1, frag.childCount == 1 ? openEnd - 1 : 0));
      if (openStart > 0) {
          frag = node.type.contentMatch.fillBefore(frag).append(frag);
          if (openEnd <= 0)
              frag = frag.append(node.type.contentMatch.matchFragment(frag).fillBefore(Fragment.empty, true));
      }
      return node.copy(frag);
  }
  function contentAfterFits($to, depth, type, match, open) {
      let node = $to.node(depth), index = open ? $to.indexAfter(depth) : $to.index(depth);
      if (index == node.childCount && !type.compatibleContent(node.type))
          return null;
      let fit = match.fillBefore(node.content, true, index);
      return fit && !invalidMarks(type, node.content, index) ? fit : null;
  }
  function invalidMarks(type, fragment, start) {
      for (let i = start; i < fragment.childCount; i++)
          if (!type.allowsMarks(fragment.child(i).marks))
              return true;
      return false;
  }
  function definesContent(type) {
      return type.spec.defining || type.spec.definingForContent;
  }
  function replaceRange(tr, from, to, slice) {
      if (!slice.size)
          return tr.deleteRange(from, to);
      let $from = tr.doc.resolve(from), $to = tr.doc.resolve(to);
      if (fitsTrivially($from, $to, slice))
          return tr.step(new ReplaceStep(from, to, slice));
      let targetDepths = coveredDepths($from, tr.doc.resolve(to));
      // Can't replace the whole document, so remove 0 if it's present
      if (targetDepths[targetDepths.length - 1] == 0)
          targetDepths.pop();
      // Negative numbers represent not expansion over the whole node at
      // that depth, but replacing from $from.before(-D) to $to.pos.
      let preferredTarget = -($from.depth + 1);
      targetDepths.unshift(preferredTarget);
      // This loop picks a preferred target depth, if one of the covering
      // depths is not outside of a defining node, and adds negative
      // depths for any depth that has $from at its start and does not
      // cross a defining node.
      for (let d = $from.depth, pos = $from.pos - 1; d > 0; d--, pos--) {
          let spec = $from.node(d).type.spec;
          if (spec.defining || spec.definingAsContext || spec.isolating)
              break;
          if (targetDepths.indexOf(d) > -1)
              preferredTarget = d;
          else if ($from.before(d) == pos)
              targetDepths.splice(1, 0, -d);
      }
      // Try to fit each possible depth of the slice into each possible
      // target depth, starting with the preferred depths.
      let preferredTargetIndex = targetDepths.indexOf(preferredTarget);
      let leftNodes = [], preferredDepth = slice.openStart;
      for (let content = slice.content, i = 0;; i++) {
          let node = content.firstChild;
          leftNodes.push(node);
          if (i == slice.openStart)
              break;
          content = node.content;
      }
      // Back up preferredDepth to cover defining textblocks directly
      // above it, possibly skipping a non-defining textblock.
      for (let d = preferredDepth - 1; d >= 0; d--) {
          let leftNode = leftNodes[d], def = definesContent(leftNode.type);
          if (def && !leftNode.sameMarkup($from.node(Math.abs(preferredTarget) - 1)))
              preferredDepth = d;
          else if (def || !leftNode.type.isTextblock)
              break;
      }
      for (let j = slice.openStart; j >= 0; j--) {
          let openDepth = (j + preferredDepth + 1) % (slice.openStart + 1);
          let insert = leftNodes[openDepth];
          if (!insert)
              continue;
          for (let i = 0; i < targetDepths.length; i++) {
              // Loop over possible expansion levels, starting with the
              // preferred one
              let targetDepth = targetDepths[(i + preferredTargetIndex) % targetDepths.length], expand = true;
              if (targetDepth < 0) {
                  expand = false;
                  targetDepth = -targetDepth;
              }
              let parent = $from.node(targetDepth - 1), index = $from.index(targetDepth - 1);
              if (parent.canReplaceWith(index, index, insert.type, insert.marks))
                  return tr.replace($from.before(targetDepth), expand ? $to.after(targetDepth) : to, new Slice(closeFragment(slice.content, 0, slice.openStart, openDepth), openDepth, slice.openEnd));
          }
      }
      let startSteps = tr.steps.length;
      for (let i = targetDepths.length - 1; i >= 0; i--) {
          tr.replace(from, to, slice);
          if (tr.steps.length > startSteps)
              break;
          let depth = targetDepths[i];
          if (depth < 0)
              continue;
          from = $from.before(depth);
          to = $to.after(depth);
      }
  }
  function closeFragment(fragment, depth, oldOpen, newOpen, parent) {
      if (depth < oldOpen) {
          let first = fragment.firstChild;
          fragment = fragment.replaceChild(0, first.copy(closeFragment(first.content, depth + 1, oldOpen, newOpen, first)));
      }
      if (depth > newOpen) {
          let match = parent.contentMatchAt(0);
          let start = match.fillBefore(fragment).append(fragment);
          fragment = start.append(match.matchFragment(start).fillBefore(Fragment.empty, true));
      }
      return fragment;
  }
  function replaceRangeWith(tr, from, to, node) {
      if (!node.isInline && from == to && tr.doc.resolve(from).parent.content.size) {
          let point = insertPoint(tr.doc, from, node.type);
          if (point != null)
              from = to = point;
      }
      tr.replaceRange(from, to, new Slice(Fragment.from(node), 0, 0));
  }
  function deleteRange(tr, from, to) {
      let $from = tr.doc.resolve(from), $to = tr.doc.resolve(to);
      let covered = coveredDepths($from, $to);
      for (let i = 0; i < covered.length; i++) {
          let depth = covered[i], last = i == covered.length - 1;
          if ((last && depth == 0) || $from.node(depth).type.contentMatch.validEnd)
              return tr.delete($from.start(depth), $to.end(depth));
          if (depth > 0 && (last || $from.node(depth - 1).canReplace($from.index(depth - 1), $to.indexAfter(depth - 1))))
              return tr.delete($from.before(depth), $to.after(depth));
      }
      for (let d = 1; d <= $from.depth && d <= $to.depth; d++) {
          if (from - $from.start(d) == $from.depth - d && to > $from.end(d) && $to.end(d) - to != $to.depth - d)
              return tr.delete($from.before(d), to);
      }
      tr.delete(from, to);
  }
  // Returns an array of all depths for which $from - $to spans the
  // whole content of the nodes at that depth.
  function coveredDepths($from, $to) {
      let result = [], minDepth = Math.min($from.depth, $to.depth);
      for (let d = minDepth; d >= 0; d--) {
          let start = $from.start(d);
          if (start < $from.pos - ($from.depth - d) ||
              $to.end(d) > $to.pos + ($to.depth - d) ||
              $from.node(d).type.spec.isolating ||
              $to.node(d).type.spec.isolating)
              break;
          if (start == $to.start(d) ||
              (d == $from.depth && d == $to.depth && $from.parent.inlineContent && $to.parent.inlineContent &&
                  d && $to.start(d - 1) == start - 1))
              result.push(d);
      }
      return result;
  }

  /**
  Update an attribute in a specific node.
  */
  class AttrStep extends Step {
      /**
      Construct an attribute step.
      */
      constructor(
      /**
      The position of the target node.
      */
      pos, 
      /**
      The attribute to set.
      */
      attr, 
      // The attribute's new value.
      value) {
          super();
          this.pos = pos;
          this.attr = attr;
          this.value = value;
      }
      apply(doc) {
          let node = doc.nodeAt(this.pos);
          if (!node)
              return StepResult.fail("No node at attribute step's position");
          let attrs = Object.create(null);
          for (let name in node.attrs)
              attrs[name] = node.attrs[name];
          attrs[this.attr] = this.value;
          let updated = node.type.create(attrs, null, node.marks);
          return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
      }
      getMap() {
          return StepMap.empty;
      }
      invert(doc) {
          return new AttrStep(this.pos, this.attr, doc.nodeAt(this.pos).attrs[this.attr]);
      }
      map(mapping) {
          let pos = mapping.mapResult(this.pos, 1);
          return pos.deletedAfter ? null : new AttrStep(pos.pos, this.attr, this.value);
      }
      toJSON() {
          return { stepType: "attr", pos: this.pos, attr: this.attr, value: this.value };
      }
      static fromJSON(schema, json) {
          if (typeof json.pos != "number" || typeof json.attr != "string")
              throw new RangeError("Invalid input for AttrStep.fromJSON");
          return new AttrStep(json.pos, json.attr, json.value);
      }
  }
  Step.jsonID("attr", AttrStep);
  /**
  Update an attribute in the doc node.
  */
  class DocAttrStep extends Step {
      /**
      Construct an attribute step.
      */
      constructor(
      /**
      The attribute to set.
      */
      attr, 
      // The attribute's new value.
      value) {
          super();
          this.attr = attr;
          this.value = value;
      }
      apply(doc) {
          let attrs = Object.create(null);
          for (let name in doc.attrs)
              attrs[name] = doc.attrs[name];
          attrs[this.attr] = this.value;
          let updated = doc.type.create(attrs, doc.content, doc.marks);
          return StepResult.ok(updated);
      }
      getMap() {
          return StepMap.empty;
      }
      invert(doc) {
          return new DocAttrStep(this.attr, doc.attrs[this.attr]);
      }
      map(mapping) {
          return this;
      }
      toJSON() {
          return { stepType: "docAttr", attr: this.attr, value: this.value };
      }
      static fromJSON(schema, json) {
          if (typeof json.attr != "string")
              throw new RangeError("Invalid input for DocAttrStep.fromJSON");
          return new DocAttrStep(json.attr, json.value);
      }
  }
  Step.jsonID("docAttr", DocAttrStep);

  /**
  @internal
  */
  let TransformError = class extends Error {
  };
  TransformError = function TransformError(message) {
      let err = Error.call(this, message);
      err.__proto__ = TransformError.prototype;
      return err;
  };
  TransformError.prototype = Object.create(Error.prototype);
  TransformError.prototype.constructor = TransformError;
  TransformError.prototype.name = "TransformError";
  /**
  Abstraction to build up and track an array of
  [steps](https://prosemirror.net/docs/ref/#transform.Step) representing a document transformation.

  Most transforming methods return the `Transform` object itself, so
  that they can be chained.
  */
  class Transform {
      /**
      Create a transform that starts with the given document.
      */
      constructor(
      /**
      The current document (the result of applying the steps in the
      transform).
      */
      doc) {
          this.doc = doc;
          /**
          The steps in this transform.
          */
          this.steps = [];
          /**
          The documents before each of the steps.
          */
          this.docs = [];
          /**
          A mapping with the maps for each of the steps in this transform.
          */
          this.mapping = new Mapping;
      }
      /**
      The starting document.
      */
      get before() { return this.docs.length ? this.docs[0] : this.doc; }
      /**
      Apply a new step in this transform, saving the result. Throws an
      error when the step fails.
      */
      step(step) {
          let result = this.maybeStep(step);
          if (result.failed)
              throw new TransformError(result.failed);
          return this;
      }
      /**
      Try to apply a step in this transformation, ignoring it if it
      fails. Returns the step result.
      */
      maybeStep(step) {
          let result = step.apply(this.doc);
          if (!result.failed)
              this.addStep(step, result.doc);
          return result;
      }
      /**
      True when the document has been changed (when there are any
      steps).
      */
      get docChanged() {
          return this.steps.length > 0;
      }
      /**
      @internal
      */
      addStep(step, doc) {
          this.docs.push(this.doc);
          this.steps.push(step);
          this.mapping.appendMap(step.getMap());
          this.doc = doc;
      }
      /**
      Replace the part of the document between `from` and `to` with the
      given `slice`.
      */
      replace(from, to = from, slice = Slice.empty) {
          let step = replaceStep(this.doc, from, to, slice);
          if (step)
              this.step(step);
          return this;
      }
      /**
      Replace the given range with the given content, which may be a
      fragment, node, or array of nodes.
      */
      replaceWith(from, to, content) {
          return this.replace(from, to, new Slice(Fragment.from(content), 0, 0));
      }
      /**
      Delete the content between the given positions.
      */
      delete(from, to) {
          return this.replace(from, to, Slice.empty);
      }
      /**
      Insert the given content at the given position.
      */
      insert(pos, content) {
          return this.replaceWith(pos, pos, content);
      }
      /**
      Replace a range of the document with a given slice, using
      `from`, `to`, and the slice's
      [`openStart`](https://prosemirror.net/docs/ref/#model.Slice.openStart) property as hints, rather
      than fixed start and end points. This method may grow the
      replaced area or close open nodes in the slice in order to get a
      fit that is more in line with WYSIWYG expectations, by dropping
      fully covered parent nodes of the replaced region when they are
      marked [non-defining as
      context](https://prosemirror.net/docs/ref/#model.NodeSpec.definingAsContext), or including an
      open parent node from the slice that _is_ marked as [defining
      its content](https://prosemirror.net/docs/ref/#model.NodeSpec.definingForContent).
      
      This is the method, for example, to handle paste. The similar
      [`replace`](https://prosemirror.net/docs/ref/#transform.Transform.replace) method is a more
      primitive tool which will _not_ move the start and end of its given
      range, and is useful in situations where you need more precise
      control over what happens.
      */
      replaceRange(from, to, slice) {
          replaceRange(this, from, to, slice);
          return this;
      }
      /**
      Replace the given range with a node, but use `from` and `to` as
      hints, rather than precise positions. When from and to are the same
      and are at the start or end of a parent node in which the given
      node doesn't fit, this method may _move_ them out towards a parent
      that does allow the given node to be placed. When the given range
      completely covers a parent node, this method may completely replace
      that parent node.
      */
      replaceRangeWith(from, to, node) {
          replaceRangeWith(this, from, to, node);
          return this;
      }
      /**
      Delete the given range, expanding it to cover fully covered
      parent nodes until a valid replace is found.
      */
      deleteRange(from, to) {
          deleteRange(this, from, to);
          return this;
      }
      /**
      Split the content in the given range off from its parent, if there
      is sibling content before or after it, and move it up the tree to
      the depth specified by `target`. You'll probably want to use
      [`liftTarget`](https://prosemirror.net/docs/ref/#transform.liftTarget) to compute `target`, to make
      sure the lift is valid.
      */
      lift(range, target) {
          lift$1(this, range, target);
          return this;
      }
      /**
      Join the blocks around the given position. If depth is 2, their
      last and first siblings are also joined, and so on.
      */
      join(pos, depth = 1) {
          join(this, pos, depth);
          return this;
      }
      /**
      Wrap the given [range](https://prosemirror.net/docs/ref/#model.NodeRange) in the given set of wrappers.
      The wrappers are assumed to be valid in this position, and should
      probably be computed with [`findWrapping`](https://prosemirror.net/docs/ref/#transform.findWrapping).
      */
      wrap(range, wrappers) {
          wrap(this, range, wrappers);
          return this;
      }
      /**
      Set the type of all textblocks (partly) between `from` and `to` to
      the given node type with the given attributes.
      */
      setBlockType(from, to = from, type, attrs = null) {
          setBlockType$1(this, from, to, type, attrs);
          return this;
      }
      /**
      Change the type, attributes, and/or marks of the node at `pos`.
      When `type` isn't given, the existing node type is preserved,
      */
      setNodeMarkup(pos, type, attrs = null, marks) {
          setNodeMarkup(this, pos, type, attrs, marks);
          return this;
      }
      /**
      Set a single attribute on a given node to a new value.
      The `pos` addresses the document content. Use `setDocAttribute`
      to set attributes on the document itself.
      */
      setNodeAttribute(pos, attr, value) {
          this.step(new AttrStep(pos, attr, value));
          return this;
      }
      /**
      Set a single attribute on the document to a new value.
      */
      setDocAttribute(attr, value) {
          this.step(new DocAttrStep(attr, value));
          return this;
      }
      /**
      Add a mark to the node at position `pos`.
      */
      addNodeMark(pos, mark) {
          this.step(new AddNodeMarkStep(pos, mark));
          return this;
      }
      /**
      Remove a mark (or a mark of the given type) from the node at
      position `pos`.
      */
      removeNodeMark(pos, mark) {
          if (!(mark instanceof Mark)) {
              let node = this.doc.nodeAt(pos);
              if (!node)
                  throw new RangeError("No node at position " + pos);
              mark = mark.isInSet(node.marks);
              if (!mark)
                  return this;
          }
          this.step(new RemoveNodeMarkStep(pos, mark));
          return this;
      }
      /**
      Split the node at the given position, and optionally, if `depth` is
      greater than one, any number of nodes above that. By default, the
      parts split off will inherit the node type of the original node.
      This can be changed by passing an array of types and attributes to
      use after the split.
      */
      split(pos, depth = 1, typesAfter) {
          split(this, pos, depth, typesAfter);
          return this;
      }
      /**
      Add the given mark to the inline content between `from` and `to`.
      */
      addMark(from, to, mark) {
          addMark(this, from, to, mark);
          return this;
      }
      /**
      Remove marks from inline nodes between `from` and `to`. When
      `mark` is a single mark, remove precisely that mark. When it is
      a mark type, remove all marks of that type. When it is null,
      remove all marks of any type.
      */
      removeMark(from, to, mark) {
          removeMark(this, from, to, mark);
          return this;
      }
      /**
      Removes all marks and nodes from the content of the node at
      `pos` that don't match the given new parent node type. Accepts
      an optional starting [content match](https://prosemirror.net/docs/ref/#model.ContentMatch) as
      third argument.
      */
      clearIncompatible(pos, parentType, match) {
          clearIncompatible(this, pos, parentType, match);
          return this;
      }
  }

  var index$6 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    AddMarkStep: AddMarkStep,
    AddNodeMarkStep: AddNodeMarkStep,
    AttrStep: AttrStep,
    DocAttrStep: DocAttrStep,
    MapResult: MapResult,
    Mapping: Mapping,
    RemoveMarkStep: RemoveMarkStep,
    RemoveNodeMarkStep: RemoveNodeMarkStep,
    ReplaceAroundStep: ReplaceAroundStep,
    ReplaceStep: ReplaceStep,
    Step: Step,
    StepMap: StepMap,
    StepResult: StepResult,
    Transform: Transform,
    get TransformError () { return TransformError; },
    canJoin: canJoin,
    canSplit: canSplit,
    dropPoint: dropPoint,
    findWrapping: findWrapping,
    insertPoint: insertPoint,
    joinPoint: joinPoint,
    liftTarget: liftTarget,
    replaceStep: replaceStep
  });

  const classesById = Object.create(null);
  /**
  Superclass for editor selections. Every selection type should
  extend this. Should not be instantiated directly.
  */
  class Selection {
      /**
      Initialize a selection with the head and anchor and ranges. If no
      ranges are given, constructs a single range across `$anchor` and
      `$head`.
      */
      constructor(
      /**
      The resolved anchor of the selection (the side that stays in
      place when the selection is modified).
      */
      $anchor, 
      /**
      The resolved head of the selection (the side that moves when
      the selection is modified).
      */
      $head, ranges) {
          this.$anchor = $anchor;
          this.$head = $head;
          this.ranges = ranges || [new SelectionRange($anchor.min($head), $anchor.max($head))];
      }
      /**
      The selection's anchor, as an unresolved position.
      */
      get anchor() { return this.$anchor.pos; }
      /**
      The selection's head.
      */
      get head() { return this.$head.pos; }
      /**
      The lower bound of the selection's main range.
      */
      get from() { return this.$from.pos; }
      /**
      The upper bound of the selection's main range.
      */
      get to() { return this.$to.pos; }
      /**
      The resolved lower  bound of the selection's main range.
      */
      get $from() {
          return this.ranges[0].$from;
      }
      /**
      The resolved upper bound of the selection's main range.
      */
      get $to() {
          return this.ranges[0].$to;
      }
      /**
      Indicates whether the selection contains any content.
      */
      get empty() {
          let ranges = this.ranges;
          for (let i = 0; i < ranges.length; i++)
              if (ranges[i].$from.pos != ranges[i].$to.pos)
                  return false;
          return true;
      }
      /**
      Get the content of this selection as a slice.
      */
      content() {
          return this.$from.doc.slice(this.from, this.to, true);
      }
      /**
      Replace the selection with a slice or, if no slice is given,
      delete the selection. Will append to the given transaction.
      */
      replace(tr, content = Slice.empty) {
          // Put the new selection at the position after the inserted
          // content. When that ended in an inline node, search backwards,
          // to get the position after that node. If not, search forward.
          let lastNode = content.content.lastChild, lastParent = null;
          for (let i = 0; i < content.openEnd; i++) {
              lastParent = lastNode;
              lastNode = lastNode.lastChild;
          }
          let mapFrom = tr.steps.length, ranges = this.ranges;
          for (let i = 0; i < ranges.length; i++) {
              let { $from, $to } = ranges[i], mapping = tr.mapping.slice(mapFrom);
              tr.replaceRange(mapping.map($from.pos), mapping.map($to.pos), i ? Slice.empty : content);
              if (i == 0)
                  selectionToInsertionEnd(tr, mapFrom, (lastNode ? lastNode.isInline : lastParent && lastParent.isTextblock) ? -1 : 1);
          }
      }
      /**
      Replace the selection with the given node, appending the changes
      to the given transaction.
      */
      replaceWith(tr, node) {
          let mapFrom = tr.steps.length, ranges = this.ranges;
          for (let i = 0; i < ranges.length; i++) {
              let { $from, $to } = ranges[i], mapping = tr.mapping.slice(mapFrom);
              let from = mapping.map($from.pos), to = mapping.map($to.pos);
              if (i) {
                  tr.deleteRange(from, to);
              }
              else {
                  tr.replaceRangeWith(from, to, node);
                  selectionToInsertionEnd(tr, mapFrom, node.isInline ? -1 : 1);
              }
          }
      }
      /**
      Find a valid cursor or leaf node selection starting at the given
      position and searching back if `dir` is negative, and forward if
      positive. When `textOnly` is true, only consider cursor
      selections. Will return null when no valid selection position is
      found.
      */
      static findFrom($pos, dir, textOnly = false) {
          let inner = $pos.parent.inlineContent ? new TextSelection($pos)
              : findSelectionIn($pos.node(0), $pos.parent, $pos.pos, $pos.index(), dir, textOnly);
          if (inner)
              return inner;
          for (let depth = $pos.depth - 1; depth >= 0; depth--) {
              let found = dir < 0
                  ? findSelectionIn($pos.node(0), $pos.node(depth), $pos.before(depth + 1), $pos.index(depth), dir, textOnly)
                  : findSelectionIn($pos.node(0), $pos.node(depth), $pos.after(depth + 1), $pos.index(depth) + 1, dir, textOnly);
              if (found)
                  return found;
          }
          return null;
      }
      /**
      Find a valid cursor or leaf node selection near the given
      position. Searches forward first by default, but if `bias` is
      negative, it will search backwards first.
      */
      static near($pos, bias = 1) {
          return this.findFrom($pos, bias) || this.findFrom($pos, -bias) || new AllSelection($pos.node(0));
      }
      /**
      Find the cursor or leaf node selection closest to the start of
      the given document. Will return an
      [`AllSelection`](https://prosemirror.net/docs/ref/#state.AllSelection) if no valid position
      exists.
      */
      static atStart(doc) {
          return findSelectionIn(doc, doc, 0, 0, 1) || new AllSelection(doc);
      }
      /**
      Find the cursor or leaf node selection closest to the end of the
      given document.
      */
      static atEnd(doc) {
          return findSelectionIn(doc, doc, doc.content.size, doc.childCount, -1) || new AllSelection(doc);
      }
      /**
      Deserialize the JSON representation of a selection. Must be
      implemented for custom classes (as a static class method).
      */
      static fromJSON(doc, json) {
          if (!json || !json.type)
              throw new RangeError("Invalid input for Selection.fromJSON");
          let cls = classesById[json.type];
          if (!cls)
              throw new RangeError(`No selection type ${json.type} defined`);
          return cls.fromJSON(doc, json);
      }
      /**
      To be able to deserialize selections from JSON, custom selection
      classes must register themselves with an ID string, so that they
      can be disambiguated. Try to pick something that's unlikely to
      clash with classes from other modules.
      */
      static jsonID(id, selectionClass) {
          if (id in classesById)
              throw new RangeError("Duplicate use of selection JSON ID " + id);
          classesById[id] = selectionClass;
          selectionClass.prototype.jsonID = id;
          return selectionClass;
      }
      /**
      Get a [bookmark](https://prosemirror.net/docs/ref/#state.SelectionBookmark) for this selection,
      which is a value that can be mapped without having access to a
      current document, and later resolved to a real selection for a
      given document again. (This is used mostly by the history to
      track and restore old selections.) The default implementation of
      this method just converts the selection to a text selection and
      returns the bookmark for that.
      */
      getBookmark() {
          return TextSelection.between(this.$anchor, this.$head).getBookmark();
      }
  }
  Selection.prototype.visible = true;
  /**
  Represents a selected range in a document.
  */
  class SelectionRange {
      /**
      Create a range.
      */
      constructor(
      /**
      The lower bound of the range.
      */
      $from, 
      /**
      The upper bound of the range.
      */
      $to) {
          this.$from = $from;
          this.$to = $to;
      }
  }
  let warnedAboutTextSelection = false;
  function checkTextSelection($pos) {
      if (!warnedAboutTextSelection && !$pos.parent.inlineContent) {
          warnedAboutTextSelection = true;
          console["warn"]("TextSelection endpoint not pointing into a node with inline content (" + $pos.parent.type.name + ")");
      }
  }
  /**
  A text selection represents a classical editor selection, with a
  head (the moving side) and anchor (immobile side), both of which
  point into textblock nodes. It can be empty (a regular cursor
  position).
  */
  class TextSelection extends Selection {
      /**
      Construct a text selection between the given points.
      */
      constructor($anchor, $head = $anchor) {
          checkTextSelection($anchor);
          checkTextSelection($head);
          super($anchor, $head);
      }
      /**
      Returns a resolved position if this is a cursor selection (an
      empty text selection), and null otherwise.
      */
      get $cursor() { return this.$anchor.pos == this.$head.pos ? this.$head : null; }
      map(doc, mapping) {
          let $head = doc.resolve(mapping.map(this.head));
          if (!$head.parent.inlineContent)
              return Selection.near($head);
          let $anchor = doc.resolve(mapping.map(this.anchor));
          return new TextSelection($anchor.parent.inlineContent ? $anchor : $head, $head);
      }
      replace(tr, content = Slice.empty) {
          super.replace(tr, content);
          if (content == Slice.empty) {
              let marks = this.$from.marksAcross(this.$to);
              if (marks)
                  tr.ensureMarks(marks);
          }
      }
      eq(other) {
          return other instanceof TextSelection && other.anchor == this.anchor && other.head == this.head;
      }
      getBookmark() {
          return new TextBookmark(this.anchor, this.head);
      }
      toJSON() {
          return { type: "text", anchor: this.anchor, head: this.head };
      }
      /**
      @internal
      */
      static fromJSON(doc, json) {
          if (typeof json.anchor != "number" || typeof json.head != "number")
              throw new RangeError("Invalid input for TextSelection.fromJSON");
          return new TextSelection(doc.resolve(json.anchor), doc.resolve(json.head));
      }
      /**
      Create a text selection from non-resolved positions.
      */
      static create(doc, anchor, head = anchor) {
          let $anchor = doc.resolve(anchor);
          return new this($anchor, head == anchor ? $anchor : doc.resolve(head));
      }
      /**
      Return a text selection that spans the given positions or, if
      they aren't text positions, find a text selection near them.
      `bias` determines whether the method searches forward (default)
      or backwards (negative number) first. Will fall back to calling
      [`Selection.near`](https://prosemirror.net/docs/ref/#state.Selection^near) when the document
      doesn't contain a valid text position.
      */
      static between($anchor, $head, bias) {
          let dPos = $anchor.pos - $head.pos;
          if (!bias || dPos)
              bias = dPos >= 0 ? 1 : -1;
          if (!$head.parent.inlineContent) {
              let found = Selection.findFrom($head, bias, true) || Selection.findFrom($head, -bias, true);
              if (found)
                  $head = found.$head;
              else
                  return Selection.near($head, bias);
          }
          if (!$anchor.parent.inlineContent) {
              if (dPos == 0) {
                  $anchor = $head;
              }
              else {
                  $anchor = (Selection.findFrom($anchor, -bias, true) || Selection.findFrom($anchor, bias, true)).$anchor;
                  if (($anchor.pos < $head.pos) != (dPos < 0))
                      $anchor = $head;
              }
          }
          return new TextSelection($anchor, $head);
      }
  }
  Selection.jsonID("text", TextSelection);
  class TextBookmark {
      constructor(anchor, head) {
          this.anchor = anchor;
          this.head = head;
      }
      map(mapping) {
          return new TextBookmark(mapping.map(this.anchor), mapping.map(this.head));
      }
      resolve(doc) {
          return TextSelection.between(doc.resolve(this.anchor), doc.resolve(this.head));
      }
  }
  /**
  A node selection is a selection that points at a single node. All
  nodes marked [selectable](https://prosemirror.net/docs/ref/#model.NodeSpec.selectable) can be the
  target of a node selection. In such a selection, `from` and `to`
  point directly before and after the selected node, `anchor` equals
  `from`, and `head` equals `to`..
  */
  class NodeSelection extends Selection {
      /**
      Create a node selection. Does not verify the validity of its
      argument.
      */
      constructor($pos) {
          let node = $pos.nodeAfter;
          let $end = $pos.node(0).resolve($pos.pos + node.nodeSize);
          super($pos, $end);
          this.node = node;
      }
      map(doc, mapping) {
          let { deleted, pos } = mapping.mapResult(this.anchor);
          let $pos = doc.resolve(pos);
          if (deleted)
              return Selection.near($pos);
          return new NodeSelection($pos);
      }
      content() {
          return new Slice(Fragment.from(this.node), 0, 0);
      }
      eq(other) {
          return other instanceof NodeSelection && other.anchor == this.anchor;
      }
      toJSON() {
          return { type: "node", anchor: this.anchor };
      }
      getBookmark() { return new NodeBookmark(this.anchor); }
      /**
      @internal
      */
      static fromJSON(doc, json) {
          if (typeof json.anchor != "number")
              throw new RangeError("Invalid input for NodeSelection.fromJSON");
          return new NodeSelection(doc.resolve(json.anchor));
      }
      /**
      Create a node selection from non-resolved positions.
      */
      static create(doc, from) {
          return new NodeSelection(doc.resolve(from));
      }
      /**
      Determines whether the given node may be selected as a node
      selection.
      */
      static isSelectable(node) {
          return !node.isText && node.type.spec.selectable !== false;
      }
  }
  NodeSelection.prototype.visible = false;
  Selection.jsonID("node", NodeSelection);
  class NodeBookmark {
      constructor(anchor) {
          this.anchor = anchor;
      }
      map(mapping) {
          let { deleted, pos } = mapping.mapResult(this.anchor);
          return deleted ? new TextBookmark(pos, pos) : new NodeBookmark(pos);
      }
      resolve(doc) {
          let $pos = doc.resolve(this.anchor), node = $pos.nodeAfter;
          if (node && NodeSelection.isSelectable(node))
              return new NodeSelection($pos);
          return Selection.near($pos);
      }
  }
  /**
  A selection type that represents selecting the whole document
  (which can not necessarily be expressed with a text selection, when
  there are for example leaf block nodes at the start or end of the
  document).
  */
  class AllSelection extends Selection {
      /**
      Create an all-selection over the given document.
      */
      constructor(doc) {
          super(doc.resolve(0), doc.resolve(doc.content.size));
      }
      replace(tr, content = Slice.empty) {
          if (content == Slice.empty) {
              tr.delete(0, tr.doc.content.size);
              let sel = Selection.atStart(tr.doc);
              if (!sel.eq(tr.selection))
                  tr.setSelection(sel);
          }
          else {
              super.replace(tr, content);
          }
      }
      toJSON() { return { type: "all" }; }
      /**
      @internal
      */
      static fromJSON(doc) { return new AllSelection(doc); }
      map(doc) { return new AllSelection(doc); }
      eq(other) { return other instanceof AllSelection; }
      getBookmark() { return AllBookmark; }
  }
  Selection.jsonID("all", AllSelection);
  const AllBookmark = {
      map() { return this; },
      resolve(doc) { return new AllSelection(doc); }
  };
  // FIXME we'll need some awareness of text direction when scanning for selections
  // Try to find a selection inside the given node. `pos` points at the
  // position where the search starts. When `text` is true, only return
  // text selections.
  function findSelectionIn(doc, node, pos, index, dir, text = false) {
      if (node.inlineContent)
          return TextSelection.create(doc, pos);
      for (let i = index - (dir > 0 ? 0 : 1); dir > 0 ? i < node.childCount : i >= 0; i += dir) {
          let child = node.child(i);
          if (!child.isAtom) {
              let inner = findSelectionIn(doc, child, pos + dir, dir < 0 ? child.childCount : 0, dir, text);
              if (inner)
                  return inner;
          }
          else if (!text && NodeSelection.isSelectable(child)) {
              return NodeSelection.create(doc, pos - (dir < 0 ? child.nodeSize : 0));
          }
          pos += child.nodeSize * dir;
      }
      return null;
  }
  function selectionToInsertionEnd(tr, startLen, bias) {
      let last = tr.steps.length - 1;
      if (last < startLen)
          return;
      let step = tr.steps[last];
      if (!(step instanceof ReplaceStep || step instanceof ReplaceAroundStep))
          return;
      let map = tr.mapping.maps[last], end;
      map.forEach((_from, _to, _newFrom, newTo) => { if (end == null)
          end = newTo; });
      tr.setSelection(Selection.near(tr.doc.resolve(end), bias));
  }

  const UPDATED_SEL = 1, UPDATED_MARKS = 2, UPDATED_SCROLL = 4;
  /**
  An editor state transaction, which can be applied to a state to
  create an updated state. Use
  [`EditorState.tr`](https://prosemirror.net/docs/ref/#state.EditorState.tr) to create an instance.

  Transactions track changes to the document (they are a subclass of
  [`Transform`](https://prosemirror.net/docs/ref/#transform.Transform)), but also other state changes,
  like selection updates and adjustments of the set of [stored
  marks](https://prosemirror.net/docs/ref/#state.EditorState.storedMarks). In addition, you can store
  metadata properties in a transaction, which are extra pieces of
  information that client code or plugins can use to describe what a
  transaction represents, so that they can update their [own
  state](https://prosemirror.net/docs/ref/#state.StateField) accordingly.

  The [editor view](https://prosemirror.net/docs/ref/#view.EditorView) uses a few metadata
  properties: it will attach a property `"pointer"` with the value
  `true` to selection transactions directly caused by mouse or touch
  input, a `"composition"` property holding an ID identifying the
  composition that caused it to transactions caused by composed DOM
  input, and a `"uiEvent"` property of that may be `"paste"`,
  `"cut"`, or `"drop"`.
  */
  class Transaction extends Transform {
      /**
      @internal
      */
      constructor(state) {
          super(state.doc);
          // The step count for which the current selection is valid.
          this.curSelectionFor = 0;
          // Bitfield to track which aspects of the state were updated by
          // this transaction.
          this.updated = 0;
          // Object used to store metadata properties for the transaction.
          this.meta = Object.create(null);
          this.time = Date.now();
          this.curSelection = state.selection;
          this.storedMarks = state.storedMarks;
      }
      /**
      The transaction's current selection. This defaults to the editor
      selection [mapped](https://prosemirror.net/docs/ref/#state.Selection.map) through the steps in the
      transaction, but can be overwritten with
      [`setSelection`](https://prosemirror.net/docs/ref/#state.Transaction.setSelection).
      */
      get selection() {
          if (this.curSelectionFor < this.steps.length) {
              this.curSelection = this.curSelection.map(this.doc, this.mapping.slice(this.curSelectionFor));
              this.curSelectionFor = this.steps.length;
          }
          return this.curSelection;
      }
      /**
      Update the transaction's current selection. Will determine the
      selection that the editor gets when the transaction is applied.
      */
      setSelection(selection) {
          if (selection.$from.doc != this.doc)
              throw new RangeError("Selection passed to setSelection must point at the current document");
          this.curSelection = selection;
          this.curSelectionFor = this.steps.length;
          this.updated = (this.updated | UPDATED_SEL) & ~UPDATED_MARKS;
          this.storedMarks = null;
          return this;
      }
      /**
      Whether the selection was explicitly updated by this transaction.
      */
      get selectionSet() {
          return (this.updated & UPDATED_SEL) > 0;
      }
      /**
      Set the current stored marks.
      */
      setStoredMarks(marks) {
          this.storedMarks = marks;
          this.updated |= UPDATED_MARKS;
          return this;
      }
      /**
      Make sure the current stored marks or, if that is null, the marks
      at the selection, match the given set of marks. Does nothing if
      this is already the case.
      */
      ensureMarks(marks) {
          if (!Mark.sameSet(this.storedMarks || this.selection.$from.marks(), marks))
              this.setStoredMarks(marks);
          return this;
      }
      /**
      Add a mark to the set of stored marks.
      */
      addStoredMark(mark) {
          return this.ensureMarks(mark.addToSet(this.storedMarks || this.selection.$head.marks()));
      }
      /**
      Remove a mark or mark type from the set of stored marks.
      */
      removeStoredMark(mark) {
          return this.ensureMarks(mark.removeFromSet(this.storedMarks || this.selection.$head.marks()));
      }
      /**
      Whether the stored marks were explicitly set for this transaction.
      */
      get storedMarksSet() {
          return (this.updated & UPDATED_MARKS) > 0;
      }
      /**
      @internal
      */
      addStep(step, doc) {
          super.addStep(step, doc);
          this.updated = this.updated & ~UPDATED_MARKS;
          this.storedMarks = null;
      }
      /**
      Update the timestamp for the transaction.
      */
      setTime(time) {
          this.time = time;
          return this;
      }
      /**
      Replace the current selection with the given slice.
      */
      replaceSelection(slice) {
          this.selection.replace(this, slice);
          return this;
      }
      /**
      Replace the selection with the given node. When `inheritMarks` is
      true and the content is inline, it inherits the marks from the
      place where it is inserted.
      */
      replaceSelectionWith(node, inheritMarks = true) {
          let selection = this.selection;
          if (inheritMarks)
              node = node.mark(this.storedMarks || (selection.empty ? selection.$from.marks() : (selection.$from.marksAcross(selection.$to) || Mark.none)));
          selection.replaceWith(this, node);
          return this;
      }
      /**
      Delete the selection.
      */
      deleteSelection() {
          this.selection.replace(this);
          return this;
      }
      /**
      Replace the given range, or the selection if no range is given,
      with a text node containing the given string.
      */
      insertText(text, from, to) {
          let schema = this.doc.type.schema;
          if (from == null) {
              if (!text)
                  return this.deleteSelection();
              return this.replaceSelectionWith(schema.text(text), true);
          }
          else {
              if (to == null)
                  to = from;
              to = to == null ? from : to;
              if (!text)
                  return this.deleteRange(from, to);
              let marks = this.storedMarks;
              if (!marks) {
                  let $from = this.doc.resolve(from);
                  marks = to == from ? $from.marks() : $from.marksAcross(this.doc.resolve(to));
              }
              this.replaceRangeWith(from, to, schema.text(text, marks));
              if (!this.selection.empty)
                  this.setSelection(Selection.near(this.selection.$to));
              return this;
          }
      }
      /**
      Store a metadata property in this transaction, keyed either by
      name or by plugin.
      */
      setMeta(key, value) {
          this.meta[typeof key == "string" ? key : key.key] = value;
          return this;
      }
      /**
      Retrieve a metadata property for a given name or plugin.
      */
      getMeta(key) {
          return this.meta[typeof key == "string" ? key : key.key];
      }
      /**
      Returns true if this transaction doesn't contain any metadata,
      and can thus safely be extended.
      */
      get isGeneric() {
          for (let _ in this.meta)
              return false;
          return true;
      }
      /**
      Indicate that the editor should scroll the selection into view
      when updated to the state produced by this transaction.
      */
      scrollIntoView() {
          this.updated |= UPDATED_SCROLL;
          return this;
      }
      /**
      True when this transaction has had `scrollIntoView` called on it.
      */
      get scrolledIntoView() {
          return (this.updated & UPDATED_SCROLL) > 0;
      }
  }

  function bind(f, self) {
      return !self || !f ? f : f.bind(self);
  }
  class FieldDesc {
      constructor(name, desc, self) {
          this.name = name;
          this.init = bind(desc.init, self);
          this.apply = bind(desc.apply, self);
      }
  }
  const baseFields = [
      new FieldDesc("doc", {
          init(config) { return config.doc || config.schema.topNodeType.createAndFill(); },
          apply(tr) { return tr.doc; }
      }),
      new FieldDesc("selection", {
          init(config, instance) { return config.selection || Selection.atStart(instance.doc); },
          apply(tr) { return tr.selection; }
      }),
      new FieldDesc("storedMarks", {
          init(config) { return config.storedMarks || null; },
          apply(tr, _marks, _old, state) { return state.selection.$cursor ? tr.storedMarks : null; }
      }),
      new FieldDesc("scrollToSelection", {
          init() { return 0; },
          apply(tr, prev) { return tr.scrolledIntoView ? prev + 1 : prev; }
      })
  ];
  // Object wrapping the part of a state object that stays the same
  // across transactions. Stored in the state's `config` property.
  class Configuration {
      constructor(schema, plugins) {
          this.schema = schema;
          this.plugins = [];
          this.pluginsByKey = Object.create(null);
          this.fields = baseFields.slice();
          if (plugins)
              plugins.forEach(plugin => {
                  if (this.pluginsByKey[plugin.key])
                      throw new RangeError("Adding different instances of a keyed plugin (" + plugin.key + ")");
                  this.plugins.push(plugin);
                  this.pluginsByKey[plugin.key] = plugin;
                  if (plugin.spec.state)
                      this.fields.push(new FieldDesc(plugin.key, plugin.spec.state, plugin));
              });
      }
  }
  /**
  The state of a ProseMirror editor is represented by an object of
  this type. A state is a persistent data structure—it isn't
  updated, but rather a new state value is computed from an old one
  using the [`apply`](https://prosemirror.net/docs/ref/#state.EditorState.apply) method.

  A state holds a number of built-in fields, and plugins can
  [define](https://prosemirror.net/docs/ref/#state.PluginSpec.state) additional fields.
  */
  class EditorState {
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      config) {
          this.config = config;
      }
      /**
      The schema of the state's document.
      */
      get schema() {
          return this.config.schema;
      }
      /**
      The plugins that are active in this state.
      */
      get plugins() {
          return this.config.plugins;
      }
      /**
      Apply the given transaction to produce a new state.
      */
      apply(tr) {
          return this.applyTransaction(tr).state;
      }
      /**
      @internal
      */
      filterTransaction(tr, ignore = -1) {
          for (let i = 0; i < this.config.plugins.length; i++)
              if (i != ignore) {
                  let plugin = this.config.plugins[i];
                  if (plugin.spec.filterTransaction && !plugin.spec.filterTransaction.call(plugin, tr, this))
                      return false;
              }
          return true;
      }
      /**
      Verbose variant of [`apply`](https://prosemirror.net/docs/ref/#state.EditorState.apply) that
      returns the precise transactions that were applied (which might
      be influenced by the [transaction
      hooks](https://prosemirror.net/docs/ref/#state.PluginSpec.filterTransaction) of
      plugins) along with the new state.
      */
      applyTransaction(rootTr) {
          if (!this.filterTransaction(rootTr))
              return { state: this, transactions: [] };
          let trs = [rootTr], newState = this.applyInner(rootTr), seen = null;
          // This loop repeatedly gives plugins a chance to respond to
          // transactions as new transactions are added, making sure to only
          // pass the transactions the plugin did not see before.
          for (;;) {
              let haveNew = false;
              for (let i = 0; i < this.config.plugins.length; i++) {
                  let plugin = this.config.plugins[i];
                  if (plugin.spec.appendTransaction) {
                      let n = seen ? seen[i].n : 0, oldState = seen ? seen[i].state : this;
                      let tr = n < trs.length &&
                          plugin.spec.appendTransaction.call(plugin, n ? trs.slice(n) : trs, oldState, newState);
                      if (tr && newState.filterTransaction(tr, i)) {
                          tr.setMeta("appendedTransaction", rootTr);
                          if (!seen) {
                              seen = [];
                              for (let j = 0; j < this.config.plugins.length; j++)
                                  seen.push(j < i ? { state: newState, n: trs.length } : { state: this, n: 0 });
                          }
                          trs.push(tr);
                          newState = newState.applyInner(tr);
                          haveNew = true;
                      }
                      if (seen)
                          seen[i] = { state: newState, n: trs.length };
                  }
              }
              if (!haveNew)
                  return { state: newState, transactions: trs };
          }
      }
      /**
      @internal
      */
      applyInner(tr) {
          if (!tr.before.eq(this.doc))
              throw new RangeError("Applying a mismatched transaction");
          let newInstance = new EditorState(this.config), fields = this.config.fields;
          for (let i = 0; i < fields.length; i++) {
              let field = fields[i];
              newInstance[field.name] = field.apply(tr, this[field.name], this, newInstance);
          }
          return newInstance;
      }
      /**
      Start a [transaction](https://prosemirror.net/docs/ref/#state.Transaction) from this state.
      */
      get tr() { return new Transaction(this); }
      /**
      Create a new state.
      */
      static create(config) {
          let $config = new Configuration(config.doc ? config.doc.type.schema : config.schema, config.plugins);
          let instance = new EditorState($config);
          for (let i = 0; i < $config.fields.length; i++)
              instance[$config.fields[i].name] = $config.fields[i].init(config, instance);
          return instance;
      }
      /**
      Create a new state based on this one, but with an adjusted set
      of active plugins. State fields that exist in both sets of
      plugins are kept unchanged. Those that no longer exist are
      dropped, and those that are new are initialized using their
      [`init`](https://prosemirror.net/docs/ref/#state.StateField.init) method, passing in the new
      configuration object..
      */
      reconfigure(config) {
          let $config = new Configuration(this.schema, config.plugins);
          let fields = $config.fields, instance = new EditorState($config);
          for (let i = 0; i < fields.length; i++) {
              let name = fields[i].name;
              instance[name] = this.hasOwnProperty(name) ? this[name] : fields[i].init(config, instance);
          }
          return instance;
      }
      /**
      Serialize this state to JSON. If you want to serialize the state
      of plugins, pass an object mapping property names to use in the
      resulting JSON object to plugin objects. The argument may also be
      a string or number, in which case it is ignored, to support the
      way `JSON.stringify` calls `toString` methods.
      */
      toJSON(pluginFields) {
          let result = { doc: this.doc.toJSON(), selection: this.selection.toJSON() };
          if (this.storedMarks)
              result.storedMarks = this.storedMarks.map(m => m.toJSON());
          if (pluginFields && typeof pluginFields == 'object')
              for (let prop in pluginFields) {
                  if (prop == "doc" || prop == "selection")
                      throw new RangeError("The JSON fields `doc` and `selection` are reserved");
                  let plugin = pluginFields[prop], state = plugin.spec.state;
                  if (state && state.toJSON)
                      result[prop] = state.toJSON.call(plugin, this[plugin.key]);
              }
          return result;
      }
      /**
      Deserialize a JSON representation of a state. `config` should
      have at least a `schema` field, and should contain array of
      plugins to initialize the state with. `pluginFields` can be used
      to deserialize the state of plugins, by associating plugin
      instances with the property names they use in the JSON object.
      */
      static fromJSON(config, json, pluginFields) {
          if (!json)
              throw new RangeError("Invalid input for EditorState.fromJSON");
          if (!config.schema)
              throw new RangeError("Required config field 'schema' missing");
          let $config = new Configuration(config.schema, config.plugins);
          let instance = new EditorState($config);
          $config.fields.forEach(field => {
              if (field.name == "doc") {
                  instance.doc = Node.fromJSON(config.schema, json.doc);
              }
              else if (field.name == "selection") {
                  instance.selection = Selection.fromJSON(instance.doc, json.selection);
              }
              else if (field.name == "storedMarks") {
                  if (json.storedMarks)
                      instance.storedMarks = json.storedMarks.map(config.schema.markFromJSON);
              }
              else {
                  if (pluginFields)
                      for (let prop in pluginFields) {
                          let plugin = pluginFields[prop], state = plugin.spec.state;
                          if (plugin.key == field.name && state && state.fromJSON &&
                              Object.prototype.hasOwnProperty.call(json, prop)) {
                              instance[field.name] = state.fromJSON.call(plugin, config, json[prop], instance);
                              return;
                          }
                      }
                  instance[field.name] = field.init(config, instance);
              }
          });
          return instance;
      }
  }

  function bindProps(obj, self, target) {
      for (let prop in obj) {
          let val = obj[prop];
          if (val instanceof Function)
              val = val.bind(self);
          else if (prop == "handleDOMEvents")
              val = bindProps(val, self, {});
          target[prop] = val;
      }
      return target;
  }
  /**
  Plugins bundle functionality that can be added to an editor.
  They are part of the [editor state](https://prosemirror.net/docs/ref/#state.EditorState) and
  may influence that state and the view that contains it.
  */
  class Plugin {
      /**
      Create a plugin.
      */
      constructor(
      /**
      The plugin's [spec object](https://prosemirror.net/docs/ref/#state.PluginSpec).
      */
      spec) {
          this.spec = spec;
          /**
          The [props](https://prosemirror.net/docs/ref/#view.EditorProps) exported by this plugin.
          */
          this.props = {};
          if (spec.props)
              bindProps(spec.props, this, this.props);
          this.key = spec.key ? spec.key.key : createKey("plugin");
      }
      /**
      Extract the plugin's state field from an editor state.
      */
      getState(state) { return state[this.key]; }
  }
  const keys = Object.create(null);
  function createKey(name) {
      if (name in keys)
          return name + "$" + ++keys[name];
      keys[name] = 0;
      return name + "$";
  }
  /**
  A key is used to [tag](https://prosemirror.net/docs/ref/#state.PluginSpec.key) plugins in a way
  that makes it possible to find them, given an editor state.
  Assigning a key does mean only one plugin of that type can be
  active in a state.
  */
  class PluginKey {
      /**
      Create a plugin key.
      */
      constructor(name = "key") { this.key = createKey(name); }
      /**
      Get the active plugin with this key, if any, from an editor
      state.
      */
      get(state) { return state.config.pluginsByKey[this.key]; }
      /**
      Get the plugin's state from an editor state.
      */
      getState(state) { return state[this.key]; }
  }

  var index$5 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    AllSelection: AllSelection,
    EditorState: EditorState,
    NodeSelection: NodeSelection,
    Plugin: Plugin,
    PluginKey: PluginKey,
    Selection: Selection,
    SelectionRange: SelectionRange,
    TextSelection: TextSelection,
    Transaction: Transaction
  });

  const domIndex = function (node) {
      for (var index = 0;; index++) {
          node = node.previousSibling;
          if (!node)
              return index;
      }
  };
  const parentNode = function (node) {
      let parent = node.assignedSlot || node.parentNode;
      return parent && parent.nodeType == 11 ? parent.host : parent;
  };
  let reusedRange = null;
  // Note that this will always return the same range, because DOM range
  // objects are every expensive, and keep slowing down subsequent DOM
  // updates, for some reason.
  const textRange = function (node, from, to) {
      let range = reusedRange || (reusedRange = document.createRange());
      range.setEnd(node, to == null ? node.nodeValue.length : to);
      range.setStart(node, from || 0);
      return range;
  };
  const clearReusedRange = function () {
      reusedRange = null;
  };
  // Scans forward and backward through DOM positions equivalent to the
  // given one to see if the two are in the same place (i.e. after a
  // text node vs at the end of that text node)
  const isEquivalentPosition = function (node, off, targetNode, targetOff) {
      return targetNode && (scanFor(node, off, targetNode, targetOff, -1) ||
          scanFor(node, off, targetNode, targetOff, 1));
  };
  const atomElements = /^(img|br|input|textarea|hr)$/i;
  function scanFor(node, off, targetNode, targetOff, dir) {
      for (;;) {
          if (node == targetNode && off == targetOff)
              return true;
          if (off == (dir < 0 ? 0 : nodeSize(node))) {
              let parent = node.parentNode;
              if (!parent || parent.nodeType != 1 || hasBlockDesc(node) || atomElements.test(node.nodeName) ||
                  node.contentEditable == "false")
                  return false;
              off = domIndex(node) + (dir < 0 ? 0 : 1);
              node = parent;
          }
          else if (node.nodeType == 1) {
              node = node.childNodes[off + (dir < 0 ? -1 : 0)];
              if (node.contentEditable == "false")
                  return false;
              off = dir < 0 ? nodeSize(node) : 0;
          }
          else {
              return false;
          }
      }
  }
  function nodeSize(node) {
      return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
  }
  function textNodeBefore$1(node, offset) {
      for (;;) {
          if (node.nodeType == 3 && offset)
              return node;
          if (node.nodeType == 1 && offset > 0) {
              if (node.contentEditable == "false")
                  return null;
              node = node.childNodes[offset - 1];
              offset = nodeSize(node);
          }
          else if (node.parentNode && !hasBlockDesc(node)) {
              offset = domIndex(node);
              node = node.parentNode;
          }
          else {
              return null;
          }
      }
  }
  function textNodeAfter$1(node, offset) {
      for (;;) {
          if (node.nodeType == 3 && offset < node.nodeValue.length)
              return node;
          if (node.nodeType == 1 && offset < node.childNodes.length) {
              if (node.contentEditable == "false")
                  return null;
              node = node.childNodes[offset];
              offset = 0;
          }
          else if (node.parentNode && !hasBlockDesc(node)) {
              offset = domIndex(node) + 1;
              node = node.parentNode;
          }
          else {
              return null;
          }
      }
  }
  function isOnEdge(node, offset, parent) {
      for (let atStart = offset == 0, atEnd = offset == nodeSize(node); atStart || atEnd;) {
          if (node == parent)
              return true;
          let index = domIndex(node);
          node = node.parentNode;
          if (!node)
              return false;
          atStart = atStart && index == 0;
          atEnd = atEnd && index == nodeSize(node);
      }
  }
  function hasBlockDesc(dom) {
      let desc;
      for (let cur = dom; cur; cur = cur.parentNode)
          if (desc = cur.pmViewDesc)
              break;
      return desc && desc.node && desc.node.isBlock && (desc.dom == dom || desc.contentDOM == dom);
  }
  // Work around Chrome issue https://bugs.chromium.org/p/chromium/issues/detail?id=447523
  // (isCollapsed inappropriately returns true in shadow dom)
  const selectionCollapsed = function (domSel) {
      return domSel.focusNode && isEquivalentPosition(domSel.focusNode, domSel.focusOffset, domSel.anchorNode, domSel.anchorOffset);
  };
  function keyEvent(keyCode, key) {
      let event = document.createEvent("Event");
      event.initEvent("keydown", true, true);
      event.keyCode = keyCode;
      event.key = event.code = key;
      return event;
  }
  function deepActiveElement(doc) {
      let elt = doc.activeElement;
      while (elt && elt.shadowRoot)
          elt = elt.shadowRoot.activeElement;
      return elt;
  }
  function caretFromPoint(doc, x, y) {
      if (doc.caretPositionFromPoint) {
          try { // Firefox throws for this call in hard-to-predict circumstances (#994)
              let pos = doc.caretPositionFromPoint(x, y);
              if (pos)
                  return { node: pos.offsetNode, offset: pos.offset };
          }
          catch (_) { }
      }
      if (doc.caretRangeFromPoint) {
          let range = doc.caretRangeFromPoint(x, y);
          if (range)
              return { node: range.startContainer, offset: range.startOffset };
      }
  }

  const nav = typeof navigator != "undefined" ? navigator : null;
  const doc$1 = typeof document != "undefined" ? document : null;
  const agent = (nav && nav.userAgent) || "";
  const ie_edge = /Edge\/(\d+)/.exec(agent);
  const ie_upto10 = /MSIE \d/.exec(agent);
  const ie_11up = /Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(agent);
  const ie$1 = !!(ie_upto10 || ie_11up || ie_edge);
  const ie_version = ie_upto10 ? document.documentMode : ie_11up ? +ie_11up[1] : ie_edge ? +ie_edge[1] : 0;
  const gecko = !ie$1 && /gecko\/(\d+)/i.test(agent);
  gecko && +(/Firefox\/(\d+)/.exec(agent) || [0, 0])[1];
  const _chrome = !ie$1 && /Chrome\/(\d+)/.exec(agent);
  const chrome = !!_chrome;
  const chrome_version = _chrome ? +_chrome[1] : 0;
  const safari = !ie$1 && !!nav && /Apple Computer/.test(nav.vendor);
  // Is true for both iOS and iPadOS for convenience
  const ios = safari && (/Mobile\/\w+/.test(agent) || !!nav && nav.maxTouchPoints > 2);
  const mac$3 = ios || (nav ? /Mac/.test(nav.platform) : false);
  const windows = nav ? /Win/.test(nav.platform) : false;
  const android = /Android \d/.test(agent);
  const webkit = !!doc$1 && "webkitFontSmoothing" in doc$1.documentElement.style;
  const webkit_version = webkit ? +(/\bAppleWebKit\/(\d+)/.exec(navigator.userAgent) || [0, 0])[1] : 0;

  function windowRect(doc) {
      let vp = doc.defaultView && doc.defaultView.visualViewport;
      if (vp)
          return {
              left: 0, right: vp.width,
              top: 0, bottom: vp.height
          };
      return { left: 0, right: doc.documentElement.clientWidth,
          top: 0, bottom: doc.documentElement.clientHeight };
  }
  function getSide(value, side) {
      return typeof value == "number" ? value : value[side];
  }
  function clientRect(node) {
      let rect = node.getBoundingClientRect();
      // Adjust for elements with style "transform: scale()"
      let scaleX = (rect.width / node.offsetWidth) || 1;
      let scaleY = (rect.height / node.offsetHeight) || 1;
      // Make sure scrollbar width isn't included in the rectangle
      return { left: rect.left, right: rect.left + node.clientWidth * scaleX,
          top: rect.top, bottom: rect.top + node.clientHeight * scaleY };
  }
  function scrollRectIntoView(view, rect, startDOM) {
      let scrollThreshold = view.someProp("scrollThreshold") || 0, scrollMargin = view.someProp("scrollMargin") || 5;
      let doc = view.dom.ownerDocument;
      for (let parent = startDOM || view.dom;; parent = parentNode(parent)) {
          if (!parent)
              break;
          if (parent.nodeType != 1)
              continue;
          let elt = parent;
          let atTop = elt == doc.body;
          let bounding = atTop ? windowRect(doc) : clientRect(elt);
          let moveX = 0, moveY = 0;
          if (rect.top < bounding.top + getSide(scrollThreshold, "top"))
              moveY = -(bounding.top - rect.top + getSide(scrollMargin, "top"));
          else if (rect.bottom > bounding.bottom - getSide(scrollThreshold, "bottom"))
              moveY = rect.bottom - rect.top > bounding.bottom - bounding.top
                  ? rect.top + getSide(scrollMargin, "top") - bounding.top
                  : rect.bottom - bounding.bottom + getSide(scrollMargin, "bottom");
          if (rect.left < bounding.left + getSide(scrollThreshold, "left"))
              moveX = -(bounding.left - rect.left + getSide(scrollMargin, "left"));
          else if (rect.right > bounding.right - getSide(scrollThreshold, "right"))
              moveX = rect.right - bounding.right + getSide(scrollMargin, "right");
          if (moveX || moveY) {
              if (atTop) {
                  doc.defaultView.scrollBy(moveX, moveY);
              }
              else {
                  let startX = elt.scrollLeft, startY = elt.scrollTop;
                  if (moveY)
                      elt.scrollTop += moveY;
                  if (moveX)
                      elt.scrollLeft += moveX;
                  let dX = elt.scrollLeft - startX, dY = elt.scrollTop - startY;
                  rect = { left: rect.left - dX, top: rect.top - dY, right: rect.right - dX, bottom: rect.bottom - dY };
              }
          }
          if (atTop || /^(fixed|sticky)$/.test(getComputedStyle(parent).position))
              break;
      }
  }
  // Store the scroll position of the editor's parent nodes, along with
  // the top position of an element near the top of the editor, which
  // will be used to make sure the visible viewport remains stable even
  // when the size of the content above changes.
  function storeScrollPos(view) {
      let rect = view.dom.getBoundingClientRect(), startY = Math.max(0, rect.top);
      let refDOM, refTop;
      for (let x = (rect.left + rect.right) / 2, y = startY + 1; y < Math.min(innerHeight, rect.bottom); y += 5) {
          let dom = view.root.elementFromPoint(x, y);
          if (!dom || dom == view.dom || !view.dom.contains(dom))
              continue;
          let localRect = dom.getBoundingClientRect();
          if (localRect.top >= startY - 20) {
              refDOM = dom;
              refTop = localRect.top;
              break;
          }
      }
      return { refDOM: refDOM, refTop: refTop, stack: scrollStack(view.dom) };
  }
  function scrollStack(dom) {
      let stack = [], doc = dom.ownerDocument;
      for (let cur = dom; cur; cur = parentNode(cur)) {
          stack.push({ dom: cur, top: cur.scrollTop, left: cur.scrollLeft });
          if (dom == doc)
              break;
      }
      return stack;
  }
  // Reset the scroll position of the editor's parent nodes to that what
  // it was before, when storeScrollPos was called.
  function resetScrollPos({ refDOM, refTop, stack }) {
      let newRefTop = refDOM ? refDOM.getBoundingClientRect().top : 0;
      restoreScrollStack(stack, newRefTop == 0 ? 0 : newRefTop - refTop);
  }
  function restoreScrollStack(stack, dTop) {
      for (let i = 0; i < stack.length; i++) {
          let { dom, top, left } = stack[i];
          if (dom.scrollTop != top + dTop)
              dom.scrollTop = top + dTop;
          if (dom.scrollLeft != left)
              dom.scrollLeft = left;
      }
  }
  let preventScrollSupported = null;
  // Feature-detects support for .focus({preventScroll: true}), and uses
  // a fallback kludge when not supported.
  function focusPreventScroll(dom) {
      if (dom.setActive)
          return dom.setActive(); // in IE
      if (preventScrollSupported)
          return dom.focus(preventScrollSupported);
      let stored = scrollStack(dom);
      dom.focus(preventScrollSupported == null ? {
          get preventScroll() {
              preventScrollSupported = { preventScroll: true };
              return true;
          }
      } : undefined);
      if (!preventScrollSupported) {
          preventScrollSupported = false;
          restoreScrollStack(stored, 0);
      }
  }
  function findOffsetInNode(node, coords) {
      let closest, dxClosest = 2e8, coordsClosest, offset = 0;
      let rowBot = coords.top, rowTop = coords.top;
      let firstBelow, coordsBelow;
      for (let child = node.firstChild, childIndex = 0; child; child = child.nextSibling, childIndex++) {
          let rects;
          if (child.nodeType == 1)
              rects = child.getClientRects();
          else if (child.nodeType == 3)
              rects = textRange(child).getClientRects();
          else
              continue;
          for (let i = 0; i < rects.length; i++) {
              let rect = rects[i];
              if (rect.top <= rowBot && rect.bottom >= rowTop) {
                  rowBot = Math.max(rect.bottom, rowBot);
                  rowTop = Math.min(rect.top, rowTop);
                  let dx = rect.left > coords.left ? rect.left - coords.left
                      : rect.right < coords.left ? coords.left - rect.right : 0;
                  if (dx < dxClosest) {
                      closest = child;
                      dxClosest = dx;
                      coordsClosest = dx && closest.nodeType == 3 ? {
                          left: rect.right < coords.left ? rect.right : rect.left,
                          top: coords.top
                      } : coords;
                      if (child.nodeType == 1 && dx)
                          offset = childIndex + (coords.left >= (rect.left + rect.right) / 2 ? 1 : 0);
                      continue;
                  }
              }
              else if (rect.top > coords.top && !firstBelow && rect.left <= coords.left && rect.right >= coords.left) {
                  firstBelow = child;
                  coordsBelow = { left: Math.max(rect.left, Math.min(rect.right, coords.left)), top: rect.top };
              }
              if (!closest && (coords.left >= rect.right && coords.top >= rect.top ||
                  coords.left >= rect.left && coords.top >= rect.bottom))
                  offset = childIndex + 1;
          }
      }
      if (!closest && firstBelow) {
          closest = firstBelow;
          coordsClosest = coordsBelow;
          dxClosest = 0;
      }
      if (closest && closest.nodeType == 3)
          return findOffsetInText(closest, coordsClosest);
      if (!closest || (dxClosest && closest.nodeType == 1))
          return { node, offset };
      return findOffsetInNode(closest, coordsClosest);
  }
  function findOffsetInText(node, coords) {
      let len = node.nodeValue.length;
      let range = document.createRange();
      for (let i = 0; i < len; i++) {
          range.setEnd(node, i + 1);
          range.setStart(node, i);
          let rect = singleRect(range, 1);
          if (rect.top == rect.bottom)
              continue;
          if (inRect(coords, rect))
              return { node, offset: i + (coords.left >= (rect.left + rect.right) / 2 ? 1 : 0) };
      }
      return { node, offset: 0 };
  }
  function inRect(coords, rect) {
      return coords.left >= rect.left - 1 && coords.left <= rect.right + 1 &&
          coords.top >= rect.top - 1 && coords.top <= rect.bottom + 1;
  }
  function targetKludge(dom, coords) {
      let parent = dom.parentNode;
      if (parent && /^li$/i.test(parent.nodeName) && coords.left < dom.getBoundingClientRect().left)
          return parent;
      return dom;
  }
  function posFromElement(view, elt, coords) {
      let { node, offset } = findOffsetInNode(elt, coords), bias = -1;
      if (node.nodeType == 1 && !node.firstChild) {
          let rect = node.getBoundingClientRect();
          bias = rect.left != rect.right && coords.left > (rect.left + rect.right) / 2 ? 1 : -1;
      }
      return view.docView.posFromDOM(node, offset, bias);
  }
  function posFromCaret(view, node, offset, coords) {
      // Browser (in caretPosition/RangeFromPoint) will agressively
      // normalize towards nearby inline nodes. Since we are interested in
      // positions between block nodes too, we first walk up the hierarchy
      // of nodes to see if there are block nodes that the coordinates
      // fall outside of. If so, we take the position before/after that
      // block. If not, we call `posFromDOM` on the raw node/offset.
      let outsideBlock = -1;
      for (let cur = node, sawBlock = false;;) {
          if (cur == view.dom)
              break;
          let desc = view.docView.nearestDesc(cur, true);
          if (!desc)
              return null;
          if (desc.dom.nodeType == 1 && (desc.node.isBlock && desc.parent && !sawBlock || !desc.contentDOM)) {
              let rect = desc.dom.getBoundingClientRect();
              if (desc.node.isBlock && desc.parent && !sawBlock) {
                  sawBlock = true;
                  if (rect.left > coords.left || rect.top > coords.top)
                      outsideBlock = desc.posBefore;
                  else if (rect.right < coords.left || rect.bottom < coords.top)
                      outsideBlock = desc.posAfter;
              }
              if (!desc.contentDOM && outsideBlock < 0 && !desc.node.isText) {
                  // If we are inside a leaf, return the side of the leaf closer to the coords
                  let before = desc.node.isBlock ? coords.top < (rect.top + rect.bottom) / 2
                      : coords.left < (rect.left + rect.right) / 2;
                  return before ? desc.posBefore : desc.posAfter;
              }
          }
          cur = desc.dom.parentNode;
      }
      return outsideBlock > -1 ? outsideBlock : view.docView.posFromDOM(node, offset, -1);
  }
  function elementFromPoint(element, coords, box) {
      let len = element.childNodes.length;
      if (len && box.top < box.bottom) {
          for (let startI = Math.max(0, Math.min(len - 1, Math.floor(len * (coords.top - box.top) / (box.bottom - box.top)) - 2)), i = startI;;) {
              let child = element.childNodes[i];
              if (child.nodeType == 1) {
                  let rects = child.getClientRects();
                  for (let j = 0; j < rects.length; j++) {
                      let rect = rects[j];
                      if (inRect(coords, rect))
                          return elementFromPoint(child, coords, rect);
                  }
              }
              if ((i = (i + 1) % len) == startI)
                  break;
          }
      }
      return element;
  }
  // Given an x,y position on the editor, get the position in the document.
  function posAtCoords(view, coords) {
      let doc = view.dom.ownerDocument, node, offset = 0;
      let caret = caretFromPoint(doc, coords.left, coords.top);
      if (caret)
          ({ node, offset } = caret);
      let elt = (view.root.elementFromPoint ? view.root : doc)
          .elementFromPoint(coords.left, coords.top);
      let pos;
      if (!elt || !view.dom.contains(elt.nodeType != 1 ? elt.parentNode : elt)) {
          let box = view.dom.getBoundingClientRect();
          if (!inRect(coords, box))
              return null;
          elt = elementFromPoint(view.dom, coords, box);
          if (!elt)
              return null;
      }
      // Safari's caretRangeFromPoint returns nonsense when on a draggable element
      if (safari) {
          for (let p = elt; node && p; p = parentNode(p))
              if (p.draggable)
                  node = undefined;
      }
      elt = targetKludge(elt, coords);
      if (node) {
          if (gecko && node.nodeType == 1) {
              // Firefox will sometimes return offsets into <input> nodes, which
              // have no actual children, from caretPositionFromPoint (#953)
              offset = Math.min(offset, node.childNodes.length);
              // It'll also move the returned position before image nodes,
              // even if those are behind it.
              if (offset < node.childNodes.length) {
                  let next = node.childNodes[offset], box;
                  if (next.nodeName == "IMG" && (box = next.getBoundingClientRect()).right <= coords.left &&
                      box.bottom > coords.top)
                      offset++;
              }
          }
          let prev;
          // When clicking above the right side of an uneditable node, Chrome will report a cursor position after that node.
          if (webkit && offset && node.nodeType == 1 && (prev = node.childNodes[offset - 1]).nodeType == 1 &&
              prev.contentEditable == "false" && prev.getBoundingClientRect().top >= coords.top)
              offset--;
          // Suspiciously specific kludge to work around caret*FromPoint
          // never returning a position at the end of the document
          if (node == view.dom && offset == node.childNodes.length - 1 && node.lastChild.nodeType == 1 &&
              coords.top > node.lastChild.getBoundingClientRect().bottom)
              pos = view.state.doc.content.size;
          // Ignore positions directly after a BR, since caret*FromPoint
          // 'round up' positions that would be more accurately placed
          // before the BR node.
          else if (offset == 0 || node.nodeType != 1 || node.childNodes[offset - 1].nodeName != "BR")
              pos = posFromCaret(view, node, offset, coords);
      }
      if (pos == null)
          pos = posFromElement(view, elt, coords);
      let desc = view.docView.nearestDesc(elt, true);
      return { pos, inside: desc ? desc.posAtStart - desc.border : -1 };
  }
  function nonZero(rect) {
      return rect.top < rect.bottom || rect.left < rect.right;
  }
  function singleRect(target, bias) {
      let rects = target.getClientRects();
      if (rects.length) {
          let first = rects[bias < 0 ? 0 : rects.length - 1];
          if (nonZero(first))
              return first;
      }
      return Array.prototype.find.call(rects, nonZero) || target.getBoundingClientRect();
  }
  const BIDI = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac]/;
  // Given a position in the document model, get a bounding box of the
  // character at that position, relative to the window.
  function coordsAtPos(view, pos, side) {
      let { node, offset, atom } = view.docView.domFromPos(pos, side < 0 ? -1 : 1);
      let supportEmptyRange = webkit || gecko;
      if (node.nodeType == 3) {
          // These browsers support querying empty text ranges. Prefer that in
          // bidi context or when at the end of a node.
          if (supportEmptyRange && (BIDI.test(node.nodeValue) || (side < 0 ? !offset : offset == node.nodeValue.length))) {
              let rect = singleRect(textRange(node, offset, offset), side);
              // Firefox returns bad results (the position before the space)
              // when querying a position directly after line-broken
              // whitespace. Detect this situation and and kludge around it
              if (gecko && offset && /\s/.test(node.nodeValue[offset - 1]) && offset < node.nodeValue.length) {
                  let rectBefore = singleRect(textRange(node, offset - 1, offset - 1), -1);
                  if (rectBefore.top == rect.top) {
                      let rectAfter = singleRect(textRange(node, offset, offset + 1), -1);
                      if (rectAfter.top != rect.top)
                          return flattenV(rectAfter, rectAfter.left < rectBefore.left);
                  }
              }
              return rect;
          }
          else {
              let from = offset, to = offset, takeSide = side < 0 ? 1 : -1;
              if (side < 0 && !offset) {
                  to++;
                  takeSide = -1;
              }
              else if (side >= 0 && offset == node.nodeValue.length) {
                  from--;
                  takeSide = 1;
              }
              else if (side < 0) {
                  from--;
              }
              else {
                  to++;
              }
              return flattenV(singleRect(textRange(node, from, to), takeSide), takeSide < 0);
          }
      }
      let $dom = view.state.doc.resolve(pos - (atom || 0));
      // Return a horizontal line in block context
      if (!$dom.parent.inlineContent) {
          if (atom == null && offset && (side < 0 || offset == nodeSize(node))) {
              let before = node.childNodes[offset - 1];
              if (before.nodeType == 1)
                  return flattenH(before.getBoundingClientRect(), false);
          }
          if (atom == null && offset < nodeSize(node)) {
              let after = node.childNodes[offset];
              if (after.nodeType == 1)
                  return flattenH(after.getBoundingClientRect(), true);
          }
          return flattenH(node.getBoundingClientRect(), side >= 0);
      }
      // Inline, not in text node (this is not Bidi-safe)
      if (atom == null && offset && (side < 0 || offset == nodeSize(node))) {
          let before = node.childNodes[offset - 1];
          let target = before.nodeType == 3 ? textRange(before, nodeSize(before) - (supportEmptyRange ? 0 : 1))
              // BR nodes tend to only return the rectangle before them.
              // Only use them if they are the last element in their parent
              : before.nodeType == 1 && (before.nodeName != "BR" || !before.nextSibling) ? before : null;
          if (target)
              return flattenV(singleRect(target, 1), false);
      }
      if (atom == null && offset < nodeSize(node)) {
          let after = node.childNodes[offset];
          while (after.pmViewDesc && after.pmViewDesc.ignoreForCoords)
              after = after.nextSibling;
          let target = !after ? null : after.nodeType == 3 ? textRange(after, 0, (supportEmptyRange ? 0 : 1))
              : after.nodeType == 1 ? after : null;
          if (target)
              return flattenV(singleRect(target, -1), true);
      }
      // All else failed, just try to get a rectangle for the target node
      return flattenV(singleRect(node.nodeType == 3 ? textRange(node) : node, -side), side >= 0);
  }
  function flattenV(rect, left) {
      if (rect.width == 0)
          return rect;
      let x = left ? rect.left : rect.right;
      return { top: rect.top, bottom: rect.bottom, left: x, right: x };
  }
  function flattenH(rect, top) {
      if (rect.height == 0)
          return rect;
      let y = top ? rect.top : rect.bottom;
      return { top: y, bottom: y, left: rect.left, right: rect.right };
  }
  function withFlushedState(view, state, f) {
      let viewState = view.state, active = view.root.activeElement;
      if (viewState != state)
          view.updateState(state);
      if (active != view.dom)
          view.focus();
      try {
          return f();
      }
      finally {
          if (viewState != state)
              view.updateState(viewState);
          if (active != view.dom && active)
              active.focus();
      }
  }
  // Whether vertical position motion in a given direction
  // from a position would leave a text block.
  function endOfTextblockVertical(view, state, dir) {
      let sel = state.selection;
      let $pos = dir == "up" ? sel.$from : sel.$to;
      return withFlushedState(view, state, () => {
          let { node: dom } = view.docView.domFromPos($pos.pos, dir == "up" ? -1 : 1);
          for (;;) {
              let nearest = view.docView.nearestDesc(dom, true);
              if (!nearest)
                  break;
              if (nearest.node.isBlock) {
                  dom = nearest.contentDOM || nearest.dom;
                  break;
              }
              dom = nearest.dom.parentNode;
          }
          let coords = coordsAtPos(view, $pos.pos, 1);
          for (let child = dom.firstChild; child; child = child.nextSibling) {
              let boxes;
              if (child.nodeType == 1)
                  boxes = child.getClientRects();
              else if (child.nodeType == 3)
                  boxes = textRange(child, 0, child.nodeValue.length).getClientRects();
              else
                  continue;
              for (let i = 0; i < boxes.length; i++) {
                  let box = boxes[i];
                  if (box.bottom > box.top + 1 &&
                      (dir == "up" ? coords.top - box.top > (box.bottom - coords.top) * 2
                          : box.bottom - coords.bottom > (coords.bottom - box.top) * 2))
                      return false;
              }
          }
          return true;
      });
  }
  const maybeRTL = /[\u0590-\u08ac]/;
  function endOfTextblockHorizontal(view, state, dir) {
      let { $head } = state.selection;
      if (!$head.parent.isTextblock)
          return false;
      let offset = $head.parentOffset, atStart = !offset, atEnd = offset == $head.parent.content.size;
      let sel = view.domSelection();
      // If the textblock is all LTR, or the browser doesn't support
      // Selection.modify (Edge), fall back to a primitive approach
      if (!maybeRTL.test($head.parent.textContent) || !sel.modify)
          return dir == "left" || dir == "backward" ? atStart : atEnd;
      return withFlushedState(view, state, () => {
          // This is a huge hack, but appears to be the best we can
          // currently do: use `Selection.modify` to move the selection by
          // one character, and see if that moves the cursor out of the
          // textblock (or doesn't move it at all, when at the start/end of
          // the document).
          let { focusNode: oldNode, focusOffset: oldOff, anchorNode, anchorOffset } = view.domSelectionRange();
          let oldBidiLevel = sel.caretBidiLevel // Only for Firefox
          ;
          sel.modify("move", dir, "character");
          let parentDOM = $head.depth ? view.docView.domAfterPos($head.before()) : view.dom;
          let { focusNode: newNode, focusOffset: newOff } = view.domSelectionRange();
          let result = newNode && !parentDOM.contains(newNode.nodeType == 1 ? newNode : newNode.parentNode) ||
              (oldNode == newNode && oldOff == newOff);
          // Restore the previous selection
          try {
              sel.collapse(anchorNode, anchorOffset);
              if (oldNode && (oldNode != anchorNode || oldOff != anchorOffset) && sel.extend)
                  sel.extend(oldNode, oldOff);
          }
          catch (_) { }
          if (oldBidiLevel != null)
              sel.caretBidiLevel = oldBidiLevel;
          return result;
      });
  }
  let cachedState = null;
  let cachedDir = null;
  let cachedResult = false;
  function endOfTextblock(view, state, dir) {
      if (cachedState == state && cachedDir == dir)
          return cachedResult;
      cachedState = state;
      cachedDir = dir;
      return cachedResult = dir == "up" || dir == "down"
          ? endOfTextblockVertical(view, state, dir)
          : endOfTextblockHorizontal(view, state, dir);
  }

  // View descriptions are data structures that describe the DOM that is
  // used to represent the editor's content. They are used for:
  //
  // - Incremental redrawing when the document changes
  //
  // - Figuring out what part of the document a given DOM position
  //   corresponds to
  //
  // - Wiring in custom implementations of the editing interface for a
  //   given node
  //
  // They form a doubly-linked mutable tree, starting at `view.docView`.
  const NOT_DIRTY = 0, CHILD_DIRTY = 1, CONTENT_DIRTY = 2, NODE_DIRTY = 3;
  // Superclass for the various kinds of descriptions. Defines their
  // basic structure and shared methods.
  class ViewDesc {
      constructor(parent, children, dom, 
      // This is the node that holds the child views. It may be null for
      // descs that don't have children.
      contentDOM) {
          this.parent = parent;
          this.children = children;
          this.dom = dom;
          this.contentDOM = contentDOM;
          this.dirty = NOT_DIRTY;
          // An expando property on the DOM node provides a link back to its
          // description.
          dom.pmViewDesc = this;
      }
      // Used to check whether a given description corresponds to a
      // widget/mark/node.
      matchesWidget(widget) { return false; }
      matchesMark(mark) { return false; }
      matchesNode(node, outerDeco, innerDeco) { return false; }
      matchesHack(nodeName) { return false; }
      // When parsing in-editor content (in domchange.js), we allow
      // descriptions to determine the parse rules that should be used to
      // parse them.
      parseRule() { return null; }
      // Used by the editor's event handler to ignore events that come
      // from certain descs.
      stopEvent(event) { return false; }
      // The size of the content represented by this desc.
      get size() {
          let size = 0;
          for (let i = 0; i < this.children.length; i++)
              size += this.children[i].size;
          return size;
      }
      // For block nodes, this represents the space taken up by their
      // start/end tokens.
      get border() { return 0; }
      destroy() {
          this.parent = undefined;
          if (this.dom.pmViewDesc == this)
              this.dom.pmViewDesc = undefined;
          for (let i = 0; i < this.children.length; i++)
              this.children[i].destroy();
      }
      posBeforeChild(child) {
          for (let i = 0, pos = this.posAtStart;; i++) {
              let cur = this.children[i];
              if (cur == child)
                  return pos;
              pos += cur.size;
          }
      }
      get posBefore() {
          return this.parent.posBeforeChild(this);
      }
      get posAtStart() {
          return this.parent ? this.parent.posBeforeChild(this) + this.border : 0;
      }
      get posAfter() {
          return this.posBefore + this.size;
      }
      get posAtEnd() {
          return this.posAtStart + this.size - 2 * this.border;
      }
      localPosFromDOM(dom, offset, bias) {
          // If the DOM position is in the content, use the child desc after
          // it to figure out a position.
          if (this.contentDOM && this.contentDOM.contains(dom.nodeType == 1 ? dom : dom.parentNode)) {
              if (bias < 0) {
                  let domBefore, desc;
                  if (dom == this.contentDOM) {
                      domBefore = dom.childNodes[offset - 1];
                  }
                  else {
                      while (dom.parentNode != this.contentDOM)
                          dom = dom.parentNode;
                      domBefore = dom.previousSibling;
                  }
                  while (domBefore && !((desc = domBefore.pmViewDesc) && desc.parent == this))
                      domBefore = domBefore.previousSibling;
                  return domBefore ? this.posBeforeChild(desc) + desc.size : this.posAtStart;
              }
              else {
                  let domAfter, desc;
                  if (dom == this.contentDOM) {
                      domAfter = dom.childNodes[offset];
                  }
                  else {
                      while (dom.parentNode != this.contentDOM)
                          dom = dom.parentNode;
                      domAfter = dom.nextSibling;
                  }
                  while (domAfter && !((desc = domAfter.pmViewDesc) && desc.parent == this))
                      domAfter = domAfter.nextSibling;
                  return domAfter ? this.posBeforeChild(desc) : this.posAtEnd;
              }
          }
          // Otherwise, use various heuristics, falling back on the bias
          // parameter, to determine whether to return the position at the
          // start or at the end of this view desc.
          let atEnd;
          if (dom == this.dom && this.contentDOM) {
              atEnd = offset > domIndex(this.contentDOM);
          }
          else if (this.contentDOM && this.contentDOM != this.dom && this.dom.contains(this.contentDOM)) {
              atEnd = dom.compareDocumentPosition(this.contentDOM) & 2;
          }
          else if (this.dom.firstChild) {
              if (offset == 0)
                  for (let search = dom;; search = search.parentNode) {
                      if (search == this.dom) {
                          atEnd = false;
                          break;
                      }
                      if (search.previousSibling)
                          break;
                  }
              if (atEnd == null && offset == dom.childNodes.length)
                  for (let search = dom;; search = search.parentNode) {
                      if (search == this.dom) {
                          atEnd = true;
                          break;
                      }
                      if (search.nextSibling)
                          break;
                  }
          }
          return (atEnd == null ? bias > 0 : atEnd) ? this.posAtEnd : this.posAtStart;
      }
      nearestDesc(dom, onlyNodes = false) {
          for (let first = true, cur = dom; cur; cur = cur.parentNode) {
              let desc = this.getDesc(cur), nodeDOM;
              if (desc && (!onlyNodes || desc.node)) {
                  // If dom is outside of this desc's nodeDOM, don't count it.
                  if (first && (nodeDOM = desc.nodeDOM) &&
                      !(nodeDOM.nodeType == 1 ? nodeDOM.contains(dom.nodeType == 1 ? dom : dom.parentNode) : nodeDOM == dom))
                      first = false;
                  else
                      return desc;
              }
          }
      }
      getDesc(dom) {
          let desc = dom.pmViewDesc;
          for (let cur = desc; cur; cur = cur.parent)
              if (cur == this)
                  return desc;
      }
      posFromDOM(dom, offset, bias) {
          for (let scan = dom; scan; scan = scan.parentNode) {
              let desc = this.getDesc(scan);
              if (desc)
                  return desc.localPosFromDOM(dom, offset, bias);
          }
          return -1;
      }
      // Find the desc for the node after the given pos, if any. (When a
      // parent node overrode rendering, there might not be one.)
      descAt(pos) {
          for (let i = 0, offset = 0; i < this.children.length; i++) {
              let child = this.children[i], end = offset + child.size;
              if (offset == pos && end != offset) {
                  while (!child.border && child.children.length)
                      child = child.children[0];
                  return child;
              }
              if (pos < end)
                  return child.descAt(pos - offset - child.border);
              offset = end;
          }
      }
      domFromPos(pos, side) {
          if (!this.contentDOM)
              return { node: this.dom, offset: 0, atom: pos + 1 };
          // First find the position in the child array
          let i = 0, offset = 0;
          for (let curPos = 0; i < this.children.length; i++) {
              let child = this.children[i], end = curPos + child.size;
              if (end > pos || child instanceof TrailingHackViewDesc) {
                  offset = pos - curPos;
                  break;
              }
              curPos = end;
          }
          // If this points into the middle of a child, call through
          if (offset)
              return this.children[i].domFromPos(offset - this.children[i].border, side);
          // Go back if there were any zero-length widgets with side >= 0 before this point
          for (let prev; i && !(prev = this.children[i - 1]).size && prev instanceof WidgetViewDesc && prev.side >= 0; i--) { }
          // Scan towards the first useable node
          if (side <= 0) {
              let prev, enter = true;
              for (;; i--, enter = false) {
                  prev = i ? this.children[i - 1] : null;
                  if (!prev || prev.dom.parentNode == this.contentDOM)
                      break;
              }
              if (prev && side && enter && !prev.border && !prev.domAtom)
                  return prev.domFromPos(prev.size, side);
              return { node: this.contentDOM, offset: prev ? domIndex(prev.dom) + 1 : 0 };
          }
          else {
              let next, enter = true;
              for (;; i++, enter = false) {
                  next = i < this.children.length ? this.children[i] : null;
                  if (!next || next.dom.parentNode == this.contentDOM)
                      break;
              }
              if (next && enter && !next.border && !next.domAtom)
                  return next.domFromPos(0, side);
              return { node: this.contentDOM, offset: next ? domIndex(next.dom) : this.contentDOM.childNodes.length };
          }
      }
      // Used to find a DOM range in a single parent for a given changed
      // range.
      parseRange(from, to, base = 0) {
          if (this.children.length == 0)
              return { node: this.contentDOM, from, to, fromOffset: 0, toOffset: this.contentDOM.childNodes.length };
          let fromOffset = -1, toOffset = -1;
          for (let offset = base, i = 0;; i++) {
              let child = this.children[i], end = offset + child.size;
              if (fromOffset == -1 && from <= end) {
                  let childBase = offset + child.border;
                  // FIXME maybe descend mark views to parse a narrower range?
                  if (from >= childBase && to <= end - child.border && child.node &&
                      child.contentDOM && this.contentDOM.contains(child.contentDOM))
                      return child.parseRange(from, to, childBase);
                  from = offset;
                  for (let j = i; j > 0; j--) {
                      let prev = this.children[j - 1];
                      if (prev.size && prev.dom.parentNode == this.contentDOM && !prev.emptyChildAt(1)) {
                          fromOffset = domIndex(prev.dom) + 1;
                          break;
                      }
                      from -= prev.size;
                  }
                  if (fromOffset == -1)
                      fromOffset = 0;
              }
              if (fromOffset > -1 && (end > to || i == this.children.length - 1)) {
                  to = end;
                  for (let j = i + 1; j < this.children.length; j++) {
                      let next = this.children[j];
                      if (next.size && next.dom.parentNode == this.contentDOM && !next.emptyChildAt(-1)) {
                          toOffset = domIndex(next.dom);
                          break;
                      }
                      to += next.size;
                  }
                  if (toOffset == -1)
                      toOffset = this.contentDOM.childNodes.length;
                  break;
              }
              offset = end;
          }
          return { node: this.contentDOM, from, to, fromOffset, toOffset };
      }
      emptyChildAt(side) {
          if (this.border || !this.contentDOM || !this.children.length)
              return false;
          let child = this.children[side < 0 ? 0 : this.children.length - 1];
          return child.size == 0 || child.emptyChildAt(side);
      }
      domAfterPos(pos) {
          let { node, offset } = this.domFromPos(pos, 0);
          if (node.nodeType != 1 || offset == node.childNodes.length)
              throw new RangeError("No node after pos " + pos);
          return node.childNodes[offset];
      }
      // View descs are responsible for setting any selection that falls
      // entirely inside of them, so that custom implementations can do
      // custom things with the selection. Note that this falls apart when
      // a selection starts in such a node and ends in another, in which
      // case we just use whatever domFromPos produces as a best effort.
      setSelection(anchor, head, root, force = false) {
          // If the selection falls entirely in a child, give it to that child
          let from = Math.min(anchor, head), to = Math.max(anchor, head);
          for (let i = 0, offset = 0; i < this.children.length; i++) {
              let child = this.children[i], end = offset + child.size;
              if (from > offset && to < end)
                  return child.setSelection(anchor - offset - child.border, head - offset - child.border, root, force);
              offset = end;
          }
          let anchorDOM = this.domFromPos(anchor, anchor ? -1 : 1);
          let headDOM = head == anchor ? anchorDOM : this.domFromPos(head, head ? -1 : 1);
          let domSel = root.getSelection();
          let brKludge = false;
          // On Firefox, using Selection.collapse to put the cursor after a
          // BR node for some reason doesn't always work (#1073). On Safari,
          // the cursor sometimes inexplicable visually lags behind its
          // reported position in such situations (#1092).
          if ((gecko || safari) && anchor == head) {
              let { node, offset } = anchorDOM;
              if (node.nodeType == 3) {
                  brKludge = !!(offset && node.nodeValue[offset - 1] == "\n");
                  // Issue #1128
                  if (brKludge && offset == node.nodeValue.length) {
                      for (let scan = node, after; scan; scan = scan.parentNode) {
                          if (after = scan.nextSibling) {
                              if (after.nodeName == "BR")
                                  anchorDOM = headDOM = { node: after.parentNode, offset: domIndex(after) + 1 };
                              break;
                          }
                          let desc = scan.pmViewDesc;
                          if (desc && desc.node && desc.node.isBlock)
                              break;
                      }
                  }
              }
              else {
                  let prev = node.childNodes[offset - 1];
                  brKludge = prev && (prev.nodeName == "BR" || prev.contentEditable == "false");
              }
          }
          // Firefox can act strangely when the selection is in front of an
          // uneditable node. See #1163 and https://bugzilla.mozilla.org/show_bug.cgi?id=1709536
          if (gecko && domSel.focusNode && domSel.focusNode != headDOM.node && domSel.focusNode.nodeType == 1) {
              let after = domSel.focusNode.childNodes[domSel.focusOffset];
              if (after && after.contentEditable == "false")
                  force = true;
          }
          if (!(force || brKludge && safari) &&
              isEquivalentPosition(anchorDOM.node, anchorDOM.offset, domSel.anchorNode, domSel.anchorOffset) &&
              isEquivalentPosition(headDOM.node, headDOM.offset, domSel.focusNode, domSel.focusOffset))
              return;
          // Selection.extend can be used to create an 'inverted' selection
          // (one where the focus is before the anchor), but not all
          // browsers support it yet.
          let domSelExtended = false;
          if ((domSel.extend || anchor == head) && !brKludge) {
              domSel.collapse(anchorDOM.node, anchorDOM.offset);
              try {
                  if (anchor != head)
                      domSel.extend(headDOM.node, headDOM.offset);
                  domSelExtended = true;
              }
              catch (_) {
                  // In some cases with Chrome the selection is empty after calling
                  // collapse, even when it should be valid. This appears to be a bug, but
                  // it is difficult to isolate. If this happens fallback to the old path
                  // without using extend.
                  // Similarly, this could crash on Safari if the editor is hidden, and
                  // there was no selection.
              }
          }
          if (!domSelExtended) {
              if (anchor > head) {
                  let tmp = anchorDOM;
                  anchorDOM = headDOM;
                  headDOM = tmp;
              }
              let range = document.createRange();
              range.setEnd(headDOM.node, headDOM.offset);
              range.setStart(anchorDOM.node, anchorDOM.offset);
              domSel.removeAllRanges();
              domSel.addRange(range);
          }
      }
      ignoreMutation(mutation) {
          return !this.contentDOM && mutation.type != "selection";
      }
      get contentLost() {
          return this.contentDOM && this.contentDOM != this.dom && !this.dom.contains(this.contentDOM);
      }
      // Remove a subtree of the element tree that has been touched
      // by a DOM change, so that the next update will redraw it.
      markDirty(from, to) {
          for (let offset = 0, i = 0; i < this.children.length; i++) {
              let child = this.children[i], end = offset + child.size;
              if (offset == end ? from <= end && to >= offset : from < end && to > offset) {
                  let startInside = offset + child.border, endInside = end - child.border;
                  if (from >= startInside && to <= endInside) {
                      this.dirty = from == offset || to == end ? CONTENT_DIRTY : CHILD_DIRTY;
                      if (from == startInside && to == endInside &&
                          (child.contentLost || child.dom.parentNode != this.contentDOM))
                          child.dirty = NODE_DIRTY;
                      else
                          child.markDirty(from - startInside, to - startInside);
                      return;
                  }
                  else {
                      child.dirty = child.dom == child.contentDOM && child.dom.parentNode == this.contentDOM && !child.children.length
                          ? CONTENT_DIRTY : NODE_DIRTY;
                  }
              }
              offset = end;
          }
          this.dirty = CONTENT_DIRTY;
      }
      markParentsDirty() {
          let level = 1;
          for (let node = this.parent; node; node = node.parent, level++) {
              let dirty = level == 1 ? CONTENT_DIRTY : CHILD_DIRTY;
              if (node.dirty < dirty)
                  node.dirty = dirty;
          }
      }
      get domAtom() { return false; }
      get ignoreForCoords() { return false; }
      isText(text) { return false; }
  }
  // A widget desc represents a widget decoration, which is a DOM node
  // drawn between the document nodes.
  class WidgetViewDesc extends ViewDesc {
      constructor(parent, widget, view, pos) {
          let self, dom = widget.type.toDOM;
          if (typeof dom == "function")
              dom = dom(view, () => {
                  if (!self)
                      return pos;
                  if (self.parent)
                      return self.parent.posBeforeChild(self);
              });
          if (!widget.type.spec.raw) {
              if (dom.nodeType != 1) {
                  let wrap = document.createElement("span");
                  wrap.appendChild(dom);
                  dom = wrap;
              }
              dom.contentEditable = "false";
              dom.classList.add("ProseMirror-widget");
          }
          super(parent, [], dom, null);
          this.widget = widget;
          this.widget = widget;
          self = this;
      }
      matchesWidget(widget) {
          return this.dirty == NOT_DIRTY && widget.type.eq(this.widget.type);
      }
      parseRule() { return { ignore: true }; }
      stopEvent(event) {
          let stop = this.widget.spec.stopEvent;
          return stop ? stop(event) : false;
      }
      ignoreMutation(mutation) {
          return mutation.type != "selection" || this.widget.spec.ignoreSelection;
      }
      destroy() {
          this.widget.type.destroy(this.dom);
          super.destroy();
      }
      get domAtom() { return true; }
      get side() { return this.widget.type.side; }
  }
  class CompositionViewDesc extends ViewDesc {
      constructor(parent, dom, textDOM, text) {
          super(parent, [], dom, null);
          this.textDOM = textDOM;
          this.text = text;
      }
      get size() { return this.text.length; }
      localPosFromDOM(dom, offset) {
          if (dom != this.textDOM)
              return this.posAtStart + (offset ? this.size : 0);
          return this.posAtStart + offset;
      }
      domFromPos(pos) {
          return { node: this.textDOM, offset: pos };
      }
      ignoreMutation(mut) {
          return mut.type === 'characterData' && mut.target.nodeValue == mut.oldValue;
      }
  }
  // A mark desc represents a mark. May have multiple children,
  // depending on how the mark is split. Note that marks are drawn using
  // a fixed nesting order, for simplicity and predictability, so in
  // some cases they will be split more often than would appear
  // necessary.
  class MarkViewDesc extends ViewDesc {
      constructor(parent, mark, dom, contentDOM) {
          super(parent, [], dom, contentDOM);
          this.mark = mark;
      }
      static create(parent, mark, inline, view) {
          let custom = view.nodeViews[mark.type.name];
          let spec = custom && custom(mark, view, inline);
          if (!spec || !spec.dom)
              spec = DOMSerializer.renderSpec(document, mark.type.spec.toDOM(mark, inline));
          return new MarkViewDesc(parent, mark, spec.dom, spec.contentDOM || spec.dom);
      }
      parseRule() {
          if ((this.dirty & NODE_DIRTY) || this.mark.type.spec.reparseInView)
              return null;
          return { mark: this.mark.type.name, attrs: this.mark.attrs, contentElement: this.contentDOM };
      }
      matchesMark(mark) { return this.dirty != NODE_DIRTY && this.mark.eq(mark); }
      markDirty(from, to) {
          super.markDirty(from, to);
          // Move dirty info to nearest node view
          if (this.dirty != NOT_DIRTY) {
              let parent = this.parent;
              while (!parent.node)
                  parent = parent.parent;
              if (parent.dirty < this.dirty)
                  parent.dirty = this.dirty;
              this.dirty = NOT_DIRTY;
          }
      }
      slice(from, to, view) {
          let copy = MarkViewDesc.create(this.parent, this.mark, true, view);
          let nodes = this.children, size = this.size;
          if (to < size)
              nodes = replaceNodes(nodes, to, size, view);
          if (from > 0)
              nodes = replaceNodes(nodes, 0, from, view);
          for (let i = 0; i < nodes.length; i++)
              nodes[i].parent = copy;
          copy.children = nodes;
          return copy;
      }
  }
  // Node view descs are the main, most common type of view desc, and
  // correspond to an actual node in the document. Unlike mark descs,
  // they populate their child array themselves.
  class NodeViewDesc extends ViewDesc {
      constructor(parent, node, outerDeco, innerDeco, dom, contentDOM, nodeDOM, view, pos) {
          super(parent, [], dom, contentDOM);
          this.node = node;
          this.outerDeco = outerDeco;
          this.innerDeco = innerDeco;
          this.nodeDOM = nodeDOM;
      }
      // By default, a node is rendered using the `toDOM` method from the
      // node type spec. But client code can use the `nodeViews` spec to
      // supply a custom node view, which can influence various aspects of
      // the way the node works.
      //
      // (Using subclassing for this was intentionally decided against,
      // since it'd require exposing a whole slew of finicky
      // implementation details to the user code that they probably will
      // never need.)
      static create(parent, node, outerDeco, innerDeco, view, pos) {
          let custom = view.nodeViews[node.type.name], descObj;
          let spec = custom && custom(node, view, () => {
              // (This is a function that allows the custom view to find its
              // own position)
              if (!descObj)
                  return pos;
              if (descObj.parent)
                  return descObj.parent.posBeforeChild(descObj);
          }, outerDeco, innerDeco);
          let dom = spec && spec.dom, contentDOM = spec && spec.contentDOM;
          if (node.isText) {
              if (!dom)
                  dom = document.createTextNode(node.text);
              else if (dom.nodeType != 3)
                  throw new RangeError("Text must be rendered as a DOM text node");
          }
          else if (!dom) {
              ({ dom, contentDOM } = DOMSerializer.renderSpec(document, node.type.spec.toDOM(node)));
          }
          if (!contentDOM && !node.isText && dom.nodeName != "BR") { // Chrome gets confused by <br contenteditable=false>
              if (!dom.hasAttribute("contenteditable"))
                  dom.contentEditable = "false";
              if (node.type.spec.draggable)
                  dom.draggable = true;
          }
          let nodeDOM = dom;
          dom = applyOuterDeco(dom, outerDeco, node);
          if (spec)
              return descObj = new CustomNodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM || null, nodeDOM, spec, view, pos + 1);
          else if (node.isText)
              return new TextViewDesc(parent, node, outerDeco, innerDeco, dom, nodeDOM, view);
          else
              return new NodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM || null, nodeDOM, view, pos + 1);
      }
      parseRule() {
          // Experimental kludge to allow opt-in re-parsing of nodes
          if (this.node.type.spec.reparseInView)
              return null;
          // FIXME the assumption that this can always return the current
          // attrs means that if the user somehow manages to change the
          // attrs in the dom, that won't be picked up. Not entirely sure
          // whether this is a problem
          let rule = { node: this.node.type.name, attrs: this.node.attrs };
          if (this.node.type.whitespace == "pre")
              rule.preserveWhitespace = "full";
          if (!this.contentDOM) {
              rule.getContent = () => this.node.content;
          }
          else if (!this.contentLost) {
              rule.contentElement = this.contentDOM;
          }
          else {
              // Chrome likes to randomly recreate parent nodes when
              // backspacing things. When that happens, this tries to find the
              // new parent.
              for (let i = this.children.length - 1; i >= 0; i--) {
                  let child = this.children[i];
                  if (this.dom.contains(child.dom.parentNode)) {
                      rule.contentElement = child.dom.parentNode;
                      break;
                  }
              }
              if (!rule.contentElement)
                  rule.getContent = () => Fragment.empty;
          }
          return rule;
      }
      matchesNode(node, outerDeco, innerDeco) {
          return this.dirty == NOT_DIRTY && node.eq(this.node) &&
              sameOuterDeco(outerDeco, this.outerDeco) && innerDeco.eq(this.innerDeco);
      }
      get size() { return this.node.nodeSize; }
      get border() { return this.node.isLeaf ? 0 : 1; }
      // Syncs `this.children` to match `this.node.content` and the local
      // decorations, possibly introducing nesting for marks. Then, in a
      // separate step, syncs the DOM inside `this.contentDOM` to
      // `this.children`.
      updateChildren(view, pos) {
          let inline = this.node.inlineContent, off = pos;
          let composition = view.composing ? this.localCompositionInfo(view, pos) : null;
          let localComposition = composition && composition.pos > -1 ? composition : null;
          let compositionInChild = composition && composition.pos < 0;
          let updater = new ViewTreeUpdater(this, localComposition && localComposition.node, view);
          iterDeco(this.node, this.innerDeco, (widget, i, insideNode) => {
              if (widget.spec.marks)
                  updater.syncToMarks(widget.spec.marks, inline, view);
              else if (widget.type.side >= 0 && !insideNode)
                  updater.syncToMarks(i == this.node.childCount ? Mark.none : this.node.child(i).marks, inline, view);
              // If the next node is a desc matching this widget, reuse it,
              // otherwise insert the widget as a new view desc.
              updater.placeWidget(widget, view, off);
          }, (child, outerDeco, innerDeco, i) => {
              // Make sure the wrapping mark descs match the node's marks.
              updater.syncToMarks(child.marks, inline, view);
              // Try several strategies for drawing this node
              let compIndex;
              if (updater.findNodeMatch(child, outerDeco, innerDeco, i)) ;
              else if (compositionInChild && view.state.selection.from > off &&
                  view.state.selection.to < off + child.nodeSize &&
                  (compIndex = updater.findIndexWithChild(composition.node)) > -1 &&
                  updater.updateNodeAt(child, outerDeco, innerDeco, compIndex, view)) ;
              else if (updater.updateNextNode(child, outerDeco, innerDeco, view, i, off)) ;
              else {
                  // Add it as a new view
                  updater.addNode(child, outerDeco, innerDeco, view, off);
              }
              off += child.nodeSize;
          });
          // Drop all remaining descs after the current position.
          updater.syncToMarks([], inline, view);
          if (this.node.isTextblock)
              updater.addTextblockHacks();
          updater.destroyRest();
          // Sync the DOM if anything changed
          if (updater.changed || this.dirty == CONTENT_DIRTY) {
              // May have to protect focused DOM from being changed if a composition is active
              if (localComposition)
                  this.protectLocalComposition(view, localComposition);
              renderDescs(this.contentDOM, this.children, view);
              if (ios)
                  iosHacks(this.dom);
          }
      }
      localCompositionInfo(view, pos) {
          // Only do something if both the selection and a focused text node
          // are inside of this node
          let { from, to } = view.state.selection;
          if (!(view.state.selection instanceof TextSelection) || from < pos || to > pos + this.node.content.size)
              return null;
          let textNode = view.input.compositionNode;
          if (!textNode || !this.dom.contains(textNode.parentNode))
              return null;
          if (this.node.inlineContent) {
              // Find the text in the focused node in the node, stop if it's not
              // there (may have been modified through other means, in which
              // case it should overwritten)
              let text = textNode.nodeValue;
              let textPos = findTextInFragment(this.node.content, text, from - pos, to - pos);
              return textPos < 0 ? null : { node: textNode, pos: textPos, text };
          }
          else {
              return { node: textNode, pos: -1, text: "" };
          }
      }
      protectLocalComposition(view, { node, pos, text }) {
          // The node is already part of a local view desc, leave it there
          if (this.getDesc(node))
              return;
          // Create a composition view for the orphaned nodes
          let topNode = node;
          for (;; topNode = topNode.parentNode) {
              if (topNode.parentNode == this.contentDOM)
                  break;
              while (topNode.previousSibling)
                  topNode.parentNode.removeChild(topNode.previousSibling);
              while (topNode.nextSibling)
                  topNode.parentNode.removeChild(topNode.nextSibling);
              if (topNode.pmViewDesc)
                  topNode.pmViewDesc = undefined;
          }
          let desc = new CompositionViewDesc(this, topNode, node, text);
          view.input.compositionNodes.push(desc);
          // Patch up this.children to contain the composition view
          this.children = replaceNodes(this.children, pos, pos + text.length, view, desc);
      }
      // If this desc must be updated to match the given node decoration,
      // do so and return true.
      update(node, outerDeco, innerDeco, view) {
          if (this.dirty == NODE_DIRTY ||
              !node.sameMarkup(this.node))
              return false;
          this.updateInner(node, outerDeco, innerDeco, view);
          return true;
      }
      updateInner(node, outerDeco, innerDeco, view) {
          this.updateOuterDeco(outerDeco);
          this.node = node;
          this.innerDeco = innerDeco;
          if (this.contentDOM)
              this.updateChildren(view, this.posAtStart);
          this.dirty = NOT_DIRTY;
      }
      updateOuterDeco(outerDeco) {
          if (sameOuterDeco(outerDeco, this.outerDeco))
              return;
          let needsWrap = this.nodeDOM.nodeType != 1;
          let oldDOM = this.dom;
          this.dom = patchOuterDeco(this.dom, this.nodeDOM, computeOuterDeco(this.outerDeco, this.node, needsWrap), computeOuterDeco(outerDeco, this.node, needsWrap));
          if (this.dom != oldDOM) {
              oldDOM.pmViewDesc = undefined;
              this.dom.pmViewDesc = this;
          }
          this.outerDeco = outerDeco;
      }
      // Mark this node as being the selected node.
      selectNode() {
          if (this.nodeDOM.nodeType == 1)
              this.nodeDOM.classList.add("ProseMirror-selectednode");
          if (this.contentDOM || !this.node.type.spec.draggable)
              this.dom.draggable = true;
      }
      // Remove selected node marking from this node.
      deselectNode() {
          if (this.nodeDOM.nodeType == 1)
              this.nodeDOM.classList.remove("ProseMirror-selectednode");
          if (this.contentDOM || !this.node.type.spec.draggable)
              this.dom.removeAttribute("draggable");
      }
      get domAtom() { return this.node.isAtom; }
  }
  // Create a view desc for the top-level document node, to be exported
  // and used by the view class.
  function docViewDesc(doc, outerDeco, innerDeco, dom, view) {
      applyOuterDeco(dom, outerDeco, doc);
      let docView = new NodeViewDesc(undefined, doc, outerDeco, innerDeco, dom, dom, dom, view, 0);
      if (docView.contentDOM)
          docView.updateChildren(view, 0);
      return docView;
  }
  class TextViewDesc extends NodeViewDesc {
      constructor(parent, node, outerDeco, innerDeco, dom, nodeDOM, view) {
          super(parent, node, outerDeco, innerDeco, dom, null, nodeDOM, view, 0);
      }
      parseRule() {
          let skip = this.nodeDOM.parentNode;
          while (skip && skip != this.dom && !skip.pmIsDeco)
              skip = skip.parentNode;
          return { skip: (skip || true) };
      }
      update(node, outerDeco, innerDeco, view) {
          if (this.dirty == NODE_DIRTY || (this.dirty != NOT_DIRTY && !this.inParent()) ||
              !node.sameMarkup(this.node))
              return false;
          this.updateOuterDeco(outerDeco);
          if ((this.dirty != NOT_DIRTY || node.text != this.node.text) && node.text != this.nodeDOM.nodeValue) {
              this.nodeDOM.nodeValue = node.text;
              if (view.trackWrites == this.nodeDOM)
                  view.trackWrites = null;
          }
          this.node = node;
          this.dirty = NOT_DIRTY;
          return true;
      }
      inParent() {
          let parentDOM = this.parent.contentDOM;
          for (let n = this.nodeDOM; n; n = n.parentNode)
              if (n == parentDOM)
                  return true;
          return false;
      }
      domFromPos(pos) {
          return { node: this.nodeDOM, offset: pos };
      }
      localPosFromDOM(dom, offset, bias) {
          if (dom == this.nodeDOM)
              return this.posAtStart + Math.min(offset, this.node.text.length);
          return super.localPosFromDOM(dom, offset, bias);
      }
      ignoreMutation(mutation) {
          return mutation.type != "characterData" && mutation.type != "selection";
      }
      slice(from, to, view) {
          let node = this.node.cut(from, to), dom = document.createTextNode(node.text);
          return new TextViewDesc(this.parent, node, this.outerDeco, this.innerDeco, dom, dom, view);
      }
      markDirty(from, to) {
          super.markDirty(from, to);
          if (this.dom != this.nodeDOM && (from == 0 || to == this.nodeDOM.nodeValue.length))
              this.dirty = NODE_DIRTY;
      }
      get domAtom() { return false; }
      isText(text) { return this.node.text == text; }
  }
  // A dummy desc used to tag trailing BR or IMG nodes created to work
  // around contentEditable terribleness.
  class TrailingHackViewDesc extends ViewDesc {
      parseRule() { return { ignore: true }; }
      matchesHack(nodeName) { return this.dirty == NOT_DIRTY && this.dom.nodeName == nodeName; }
      get domAtom() { return true; }
      get ignoreForCoords() { return this.dom.nodeName == "IMG"; }
  }
  // A separate subclass is used for customized node views, so that the
  // extra checks only have to be made for nodes that are actually
  // customized.
  class CustomNodeViewDesc extends NodeViewDesc {
      constructor(parent, node, outerDeco, innerDeco, dom, contentDOM